/******************************************************************************
**
**     File/Function name:	make_identity.c
**
**				Copyright 1998 Tadayoshi Kohno
**				All rights reserved.  See the LICENSE file.
**
**     Purpose:			make an RSA identity for SSH
**
**     Author/Date:		Tadayoshi Kohno, 23 February 1998
**
**     References:
**
**     Notes:
**	passphrase echo'd when its read, ...  but this program does
**	provide a means to make RSA keys to test with
**
******************************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/unix_ssh/RCS/make_identity.c,v 1.4 1998/05/21 16:22:01 kohno Exp $";
#endif


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "ssh.h"


/*
**	number of arguments
*/
#define ARGC_NUM	3

/*
**	identity file to generate
*/
#define ARGV_PRIV	1
#define ARGV_PUB	2

/*
**	length of fine names
*/
#define MAX_NAME	1000

/*
**	max length of passphrase
*/
#define PASS_LEN	1000


/*
**	define a random name to stick on the end of the public file
*/
#define COMMENT_NAME	"libSSH Generated identity file for <name>"

int print_debug(const char * header, const char * body);

/******************************************************************************
**
**     File/Function name:	main
**
**     Purpose:			handle the whole ssh identity generation thing
**
**     Preconditions:		none
**
**     Parameters:		private file name
**				public file name
**
**     Exit (post) conditions:	0
**
**     Error conditions:	< 0 , message to stderr
**
**     Side effects:		identity generated with private information
**				in the file specified by the first
**				arguement and the public information
**				in the file specified by the second arguement
**
**     Author/Date:		Tadayoshi Kohno, 23 February 1998
**
******************************************************************************/

int main
(
	int argc,
	char * argv[]
)
{
	char priv_name[MAX_NAME];	/* name of private identity file */
	char pub_name[MAX_NAME];	/* name of public identity file */

	char passphrase[PASS_LEN];	/* passphrase */

	struct ssh_struct ssh_info;	/* info about debugging, ... */

/*
**	make sure right number of arguments
*/
	if (argc != ARGC_NUM)
	{
		fprintf(stderr, "invalid arguements\n");
		fprintf(stderr, "%s priv_file pub_file\n",
			argv[0]);
		return(-1);
	}

/*
**	and that the arguments are not too long
*/
	if (strlen(argv[ARGV_PRIV]) >= MAX_NAME)
	{
		fprintf(stderr, "private file name too long\n");
		return(-1);
	}
	if (strlen(argv[ARGV_PUB]) >= MAX_NAME)
	{
		fprintf(stderr, "public file name too long\n");
		return(-1);
	}

/*
**	copy the argvs over
*/
	strcpy(priv_name, argv[ARGV_PRIV]);
	strcpy(pub_name, argv[ARGV_PUB]);

/*
**	get the passphrase (with echo and everything -- yucky)
*/
	printf("enter passphrase: ");
	fgets(passphrase, PASS_LEN, stdin);

/*
**	make sure passphrase wasn't too long, and chop off tailing '\n'
*/
	if (passphrase[strlen(passphrase) - 1] != '\n')
	{
		fprintf(stderr, "passphrase too long\n");
		return(-1);
	}
	passphrase[strlen(passphrase) - 1] = 0;

/*
**	even though not connecting, still setup ssh_struct
*/
	ssh_presetup_client(&ssh_info, print_debug, SSH_YES_DEBUG);

/*
**	and now generate the RSA keys
*/
	if (ssh_identity_gen(&ssh_info, priv_name, pub_name, COMMENT_NAME,
		passphrase, SSH_IDENT_MOD_BITS, SSH_CIPHER_3DES) != S_GOOD)
	{
		fprintf(stderr, "ssh errno: %d on ssh_identity_gen\n",
			ssh_errno_get());
		return(-1);
	}

	printf("identity generated in files %s and %s\n",
		priv_name, pub_name);

	return(0);
}

/******************************************************************************
**
**     File/Function name:	print_debug
**
**     Purpose:			function to pass to libSSH to show
**				debug messages
**
**     Parameters:		header		message section
**				body		message body
**
**     Exit (post) conditions:	0
**
**     Side effects:		libSSH debug message displayed
**
**     Author/Date:		Tadayoshi Kohno, 23 February 1998
**
******************************************************************************/

int print_debug
(
	const char * header,		/* debug message header */
	const char * body		/* debug message body */
)
{
	fprintf(stderr, "DEBUG identity MAKE %s: %s\n",
		header, body);
	fflush(stderr);

	return(0);
}

