/*********************************************************************
**
**     File name:               ssh_proto.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 protocol hand shaking over ssh
**
**     Author/Date:             Tadayoshi Kohno, 20 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	Do I want to change recvs, sends to read, writes. xxx ?
**
**	The routines in this file ask for and sends protocol information.
**	This is the first step in the SSH-Protocol handshaking
**
**     Functions:
**	ssh_proto_get		get protocol info from remote host
**	ssh_proto_put		put protocol info to remote host
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_proto.c,v 3.1 1998/05/09 14:46:56 kohno Exp $";
#endif

#include <string.h>
#include <stdio.h>
#include <assert.h>

#include "ssh.h"
#include "ssh_proto.h"

/*********************************************************************
**
**     Function:                ssh_proto_get
**
**     Purpose:                 receive protocol information from
**				remote host
**
**     Entry (pre) conditions:  sockfd valid socket
**
**     Parameters:              sockfd		socket to remote host
**				ssh_info	info about connection
**
**				*major		major number of remote host
**				*minor		minor number of remote host
**				*version	version of remote host
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for
**					SSH_ERRNO_RECV
**					SSH_ERRNO_PROTO_PARSE
**
**     Side effects:            *major contains major number of remote host
**				*minor contains minor number of remote host
**				*version contains software version of rhost
**
**     Author/Date:             Tadayoshi Kohno, 20 November 1997
**
**     References:                  
**
**     Notes:
**
*********************************************************************/

int ssh_proto_get
(
	socket_type sockfd,		/* socket to read from */
	struct ssh_struct * ssh_info,	/* info about current connection */
	int * major,		/* major number of remote host */
	int * minor,		/* minor number of remote host */
	char * version		/* version of remote host */
)
{
	char protocol[SSH_PROTOCOL_LEN+1];	/* protocol received */
	char ssh_str[STR_MISC_LEN];	/* string to store "SSH" */
	char major_str[STR_MISC_LEN];	/* string for the major section */
	char minor_str[STR_MISC_LEN];	/* string for the minor section */

	char * str_ptr;			/* string pointer to the data read */
	char * str_next;		/* next pointer */

#ifdef SSH_CORE_DEBUG
	/* set to all 0 incase we want to print the string */
	for (int i = 0; i < SSH_PROTOCOL_LEN + 1; i++)
	{
		protocol[i] = 0;
	}
#endif

	/*
	** Get the protocol information.  This NEEDS to be timed out xxx.
	*/
	if (recv(sockfd, protocol, STR_MISC_LEN, SR_FLAGS) <= 0)
	{
		ssh_errno_set(SSH_ERRNO_RECV);
		return(S_BAD);
	}

	/*
	** First lets chop out the leading SSH_PROTO_HEAD and make sure its ok
	*/
	if ((str_ptr = strstr(protocol, SSH_PROTO_SEP)) == (char *) NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"could not find protocol head", "ssh_proto_get");

		ssh_errno_set(SSH_ERRNO_PROTO_PARSE);
		return(S_BAD);
	}
	*str_ptr = 0;

	strcpy(ssh_str, protocol);
	if (strcmp(ssh_str, SSH_PROTO_HEAD) != 0)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"protocol head bad", "ssh_proto_get");

		ssh_errno_set(SSH_ERRNO_PROTO_PARSE);
		return(S_BAD);
	}

	/*
	** Now lets grab the major number
	*/
	str_ptr++;
	if ((str_next = strstr(str_ptr, SSH_PROTO_MMSEP)) == (char *) NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"bad major number", "ssh_proto_get");

		ssh_errno_set(SSH_ERRNO_PROTO_PARSE);
		return(S_BAD);
	}
	*str_next = 0;
	strcpy(major_str, str_ptr);

	/*
	** Now grab the minor numbers
	*/
	str_ptr = str_next + 1;
	if ((str_next = strstr(str_ptr, SSH_PROTO_SEP)) == (char *) NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"bad minor number", "ssh_proto_get");

		ssh_errno_set(SSH_ERRNO_PROTO_PARSE);
		return(S_BAD);
	}
	*str_next = 0;
	strcpy(minor_str, str_ptr);

	/*
	** Now grab the version
	*/
	str_ptr = str_next + 1;
	if ((str_next = strstr(str_ptr, SSH_PROTO_END)) == (char *) NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"bad program version", "ssh_proto_get");

		ssh_errno_set(SSH_ERRNO_PROTO_PARSE);
		return(S_BAD);
	}
	*str_next = 0;
	strcpy(version, str_ptr);

	/*
	** Finally convert from strings to pointers
	*/
	*major = atoi(major_str);
	*minor = atoi(minor_str);

	ssh_debugger_new(&(ssh_info->debug_info),
		"remote protocol version information received",
		"ssh_proto_get");

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_proto_put
**
**     Purpose:                 send protocol information to remote host
**
**     Entry (pre) conditions:  socket opened
**
**     Parameters:              sockfd		socket to remote host
**				ssh_info	info about connection
**
**				major		our major number
**				minor		our minor number
**				version		our (software) version
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for SSH_ERRNO_SEND
**
**     Side effects:            SSH protocol version exchange sent.
**
**     Author/Date:             Tadayoshi Kohno, 20 November 1997
**
**     References:                  
**
**     Notes:
**	*Need* to timeout send()s, recv()s xxx
**
*********************************************************************/

int ssh_proto_put
(
	socket_type sockfd,		/* socket to write to */
	struct ssh_struct * ssh_info,	/* info about current connection */
	int major,			/* our major number */
	int minor,			/* our minor number */
	const char * version		/* our version */
)
{
	char str[STR_MISC_LEN];		/* temporary storage */

	(void) sprintf(str, "%s%s%d%s%d%s%s%s", SSH_PROTO_HEAD, 
		SSH_PROTO_SEP, major,
		SSH_PROTO_MMSEP, minor,
		SSH_PROTO_SEP, version,
		SSH_PROTO_END);

	/*
	**	Send our protocol information.  This NEEDS to be timed out xxx
	*/
	if (send(sockfd, str, strlen(str), SR_FLAGS) != (int) strlen(str))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending to client", "ssh_proto_put");

		ssh_errno_set(SSH_ERRNO_SEND);
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"protocol version information sent", "ssh_proto_put");

	return(S_GOOD);
}

