/******************************************************************************
**
**     File/Function name:	ssh_compress.c
**
**				Copyright 1998 Tadayoshi Kohno
**				All rights reserved.  See the LICENSE file.
**
**     Purpose:			handle SSH compression
**
**     Author/Date:		Tadayoshi Kohno, 4 April 1998
**
**     References:
**	http://www.cdrom.com/pub/infozip/zlib zlib compression library
**		home page and version 1.1.2 zlib.h header file
**
**     Notes:
**	The routines in this file make heavy use of the zlib compression
**	library by Jean-loup Gailly and Mark Adler.  The version that
**	libSSH has been test with is version 1.1.2 (19 March 1998)
**	and is available from http://www.cdrom.com/pub/infozip/zlib/ or
**	ftp://ftp.cdrom.com/pub/infozip/zlib/
**
**     Functions:
**	ssh_compression_active	is compression currently set (macro in header)
**
**	ssh_compression_none	sets compression to "none"
**	ssh_compression_set	sets compression to a specific level
**	ssh_compression_unset	turns off compression (cleans up)
**
**	ssh_compression_unset	deactive compression (not implemented)
**
**	ssh_compression_compress	compress data
**	ssh_compression_uncompress	uncompress data
**
******************************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_compress.c,v 1.19 1998/04/18 17:03:02 kohno Exp $";
#endif

#include "ssh_compress.h"

/******************************************************************************
**
**     File/Function name:	ssh_compression_none
**
**     Purpose:			set ssh compression to none
**
**     Preconditions:		ssh_info valid
**
**     Parameters:		ssh_info	pointer to ssh_struct structure
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:        S_BAD
**
**     Side effects:		ssh_info's compress_info's do_compress
**				set to SSH_COMPRESS_NO
**
**     Author/Date:		Tadayoshi Kohno, 4 April 1998
**
**     References:
**
**     Notes:
**	For some reason this was causing problems until I added a blank line
**	to the sample unix client.  Weird.  Associated with the call from
**	ssh_presetup.c xxx
**
******************************************************************************/

int ssh_compression_none
(
	struct ssh_struct * ssh_info	/* ssh connection information */
)
{
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	ssh_info->compress_info.do_compress = SSH_COMPRESS_NO;

	ssh_debugger_new(&(ssh_info->debug_info),
		"compression set to none",
		"ssh_compression_none");
	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_compression_set
**
**     Purpose:			turn on compression for the ssh_info connection
**
**     Preconditions:		pointers valid
**				level between SSH_COMPRESS_{MIN,MAX}_LEVEL
**
**     Parameters:		ssh_info	ssh connection information
**				level		desired compression level
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error
**						NULL pointer
**						illegal compression level
**
**     Side effects:		ssh_info's compression flag set
**				compression streams initialized
**
**     Author/Date:		Tadayoshi Kohno, 4 April 1998
**     Modified:                Tadayoshi Kohno, 18 April 1998
**					Added data_type, extra initialization
**
**     Notes:
**	This function does not send any request for compression to the
**	remote host.  It merely says to the ssh_info that compression
**	should now be used.  This needs to be called after the server
**	has replied (without compression) to the client's request for
**	compression.
**
******************************************************************************/

int ssh_compression_set
(
	struct ssh_struct * ssh_info,	/* ssh connection information */
	int level			/* desired compression level */
)
{
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	if (level < SSH_COMPRESS_MIN_LEVEL || level > SSH_COMPRESS_MAX_LEVEL)
	{
		ssh_errno_set(SSH_ERRNO_COMPRESS_LEVEL);
		return(S_BAD);
	}

	if (ssh_info->compress_info.do_compress == SSH_COMPRESS_YES)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"compression ALREADY set",
			"ssh_compression_set");
		return(S_GOOD);
	}

/*
**	initialize some zlib state
*/
	ssh_info->compress_info.inflate_stream.zalloc = (alloc_func) Z_NULL;
	ssh_info->compress_info.inflate_stream.zfree = (free_func) Z_NULL;
	ssh_info->compress_info.inflate_stream.opaque = (voidpf) Z_NULL;
	ssh_info->compress_info.inflate_stream.next_in = Z_NULL;
	ssh_info->compress_info.inflate_stream.avail_in = 0;
	ssh_info->compress_info.inflate_stream.data_type = Z_BINARY;

	ssh_info->compress_info.deflate_stream.zalloc = (alloc_func) Z_NULL;
	ssh_info->compress_info.deflate_stream.zfree = (free_func) Z_NULL;
	ssh_info->compress_info.deflate_stream.opaque = (voidpf) Z_NULL;
	ssh_info->compress_info.deflate_stream.next_in = Z_NULL;
	ssh_info->compress_info.deflate_stream.avail_in = 0;
	ssh_info->compress_info.deflate_stream.data_type = Z_BINARY;

/*
**	initialize deflation (compression)
*/
	switch (deflateInit(&(ssh_info->compress_info.deflate_stream), level))
	{
	case Z_OK:
		/* continue with execution -- all the rest return() */
		break;

	case Z_MEM_ERROR:
		ssh_errno_set(SSH_ERRNO_ZLIB_MEM_ERROR);

		return(S_BAD);
		break;

	case Z_STREAM_ERROR:
		ssh_errno_set(SSH_ERRNO_ZLIB_STREAM_ERROR);
		
		return(S_BAD);
		break;

	case Z_VERSION_ERROR:
		ssh_errno_set(SSH_ERRNO_ZLIB_VERSION_ERROR);

		return(S_BAD);
		break;

	default:
		ssh_errno_set(SSH_ERRNO_ZLIB_MISC);
		
		return(S_BAD);
		break;
	}
	/* deflate initialization was OK */


/*
**	initialize inflation (uncompression)
*/
	switch(inflateInit(&(ssh_info->compress_info.inflate_stream)))
	{
	case Z_OK:
		/* continue with execution -- all the rest return() */
		break;

	case Z_VERSION_ERROR:
		ssh_errno_set(SSH_ERRNO_ZLIB_VERSION_ERROR);

		return(S_BAD);
		break;

	case Z_MEM_ERROR:
		ssh_errno_set(SSH_ERRNO_ZLIB_MEM_ERROR);

		return(S_BAD);
		break;

	default:
		ssh_errno_set(SSH_ERRNO_ZLIB_MISC);
		
		return(S_BAD);
		break;
	}
	/* inflate initialization was OK */

/*
**	compression initialized
*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"internal compression initialized", "ssh_compression_set");

	ssh_info->compress_info.do_compress = SSH_COMPRESS_YES;
	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_compression_unset
**
**     Purpose:			turns off compression
**
**     Preconditions:		compression turned on (ssh_compression_set)
**				pointers valid
**
**     Parameters:		ssh_info	ssh connection information
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error
**
**     Side effects:		compression turned off and zlib data structures
**				deallocated
**
**     Author/Date:		Tadayoshi Kohno, 18 April 1998
**
**     References:
**
**     Notes:
**
******************************************************************************/

int ssh_compression_unset
(
	struct ssh_struct * ssh_info	/* ssh connection information */
)
{
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}
/*
**	turn of compression
*/
	ssh_info->compress_info.do_compress = SSH_COMPRESS_NO;

/*
**	tell zlib to deallocate memory
*/
	if (inflateEnd(&(ssh_info->compress_info.inflate_stream)) == Z_OK)
	{
		return(S_GOOD);
	} else
	{
		ssh_errno_set(SSH_ERRNO_ZLIB_DEALLOC);
		return(S_BAD);
	}

	if (deflateEnd(&(ssh_info->compress_info.deflate_stream)) == Z_OK)
	{
		return(S_GOOD);
	} else
	{
		ssh_errno_set(SSH_ERRNO_ZLIB_DEALLOC);
		return(S_BAD);
	}

	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_compression_compress
**
**     Purpose:			compress data
**
**     Preconditions:		ssh_compression_set called
**				pointers valid
**				*output_str and *input_str have enough space
**
**     Parameters:		ssh_info	ssh connection information
**
**				input_str	stream to compress
**				*output_str	resulting compressed stream
**				
**				input_len	length of stream to compress	
**				output_len	*available* output size
**
**				*final_len	resulting length of compressed
**						stream
**
**     Exit (post) conditions:  S_GOOD
**
**     Error conditions:	S_BAD		error
**						NULL pointer
**						compression not set
**						...
**
**     Side effects:		input_str compressed into output_str
**				final_len contains the length of the
**				compressed stream
**
**     Author/Date:		Tadayoshi Kohno, 11 April 1998
**     Modified:                Tadayoshi Kohno, 18 April 1998
**					Changed Z_PARTIAL_FLUSH o Z_SYNC_FLUSH
**					[following suggestion from zlib.h]
**
**     Notes:
**
******************************************************************************/

int ssh_compression_compress
(
	struct ssh_struct * ssh_info,	/* ssh connection information */

	uint8_t * input_str,		/* input string */
	uint8_t * output_str,		/* pointer to output string */

	int input_len,			/* length of input */
	int output_len,			/* space allocated for output */

	int * final_len		/* length of output used */
)
{
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	if (ssh_info->compress_info.do_compress != SSH_COMPRESS_YES)
	{
		ssh_errno_set(SSH_ERRNO_COMPRESSION_WRONG);
		return(S_BAD);
	}

	ssh_info->compress_info.deflate_stream.next_in = input_str;
	ssh_info->compress_info.deflate_stream.avail_in = input_len;

	ssh_info->compress_info.deflate_stream.next_out = output_str;
	ssh_info->compress_info.deflate_stream.avail_out = output_len;

	switch (deflate(&(ssh_info->compress_info.deflate_stream),
		Z_SYNC_FLUSH))
	{
	case Z_OK:
		/* compressed, fall through */
		break;

	case Z_BUF_ERROR:
	/*
	**	no progress was possible.  
	**
	**	we should hopefully never get here because we will
	**	always want to compress at least the packet type.
	**
	**	we can get here if output_len == 0
	*/
		ssh_errno_set(SSH_ERRNO_ZLIB_STALL_ERROR);

		return(S_BAD);
		break;

	case Z_STREAM_ERROR:
		/* inconsisten state */
		ssh_errno_set(SSH_ERRNO_ZLIB_STATE_ERROR);

		return(S_BAD);
		break;

	default:
		ssh_errno_set(SSH_ERRNO_ZLIB_MISC);
		
		return(S_BAD);
		break;
	}

/*
**	if we didn't have enough room to output the entire deflated
**	input (and not a Z_BUF_ERROR because we must have made some
**	progress)
*/
	if (ssh_info->compress_info.deflate_stream.avail_out == 0)
	{
		ssh_errno_set(SSH_ERRNO_ZLIB_MORE_OUT_ERROR);

		return(S_BAD);
	}

/*
**	set *final_len to the length of the deflated stream
*/
	*final_len = output_len
		- ssh_info->compress_info.deflate_stream.avail_out;

	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_compression_uncompress
**
**     Purpose:			uncompress data
**
**     Preconditions:		ssh_compression_set() called
**				pointers valid, memory allocated
**
**     Parameters:		ssh_info	ssh connection information
**				
**				input_str	string to uncompress
**				*output_str	resulting uncompressed str
**
**				input_len	length of input string
**				output_len	available space for *output_str
**
**				*final_len	resulting length of uncompressed
**						string
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error
**						NULL pointer
**						compression not set
**						...
**
**     Side effects:		input_str uncompressed (inflated) into
**				the output_str
**
**				final_len set to the length of the uncompressed
**				string stored in output_str
**
**     Author/Date:		Tadayoshi Kohno, 11 April 1998
**     Modified:                Tadayoshi Kohno, 18 April 1998
**					Changed Z_PARTIAL_FLUSH o Z_SYNC_FLUSH
**					[following suggestion from zlib.h]
**
**					Not doing while() loop (see notes)
**
**     Notes:
**	The zlib.h header file says that inflate() should typically
**	be called until and Z_STREAM_END or and error is returned.
**	The rought-draft of this routine did that waiting for a Z_BUF_ERROR
**	which showed that no progress was possible.
**
**	But because we are using Z_SYNC_FLUSH inflation and deflation,
**	I don't *think* we really need the while loop.  So, for this reason,
**	the uncompress function now uses a switch statement similar to
**	the compress function.
**
******************************************************************************/

int ssh_compression_uncompress
(
	struct ssh_struct * ssh_info,	/* ssh connection information */

	uint8_t * input_str,		/* input string */
	uint8_t * output_str,		/* pointer to output string */

	int input_len,			/* length of input */
	int output_len,			/* space allocated for output */

	int * final_len			/* length of output used */
)
{
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	if (ssh_info->compress_info.do_compress != SSH_COMPRESS_YES)
	{

		ssh_errno_set(SSH_ERRNO_COMPRESSION_WRONG);
		return(S_BAD);
	}

	ssh_info->compress_info.inflate_stream.next_in = input_str;
	ssh_info->compress_info.inflate_stream.avail_in = input_len;

	ssh_info->compress_info.inflate_stream.next_out = output_str;
	ssh_info->compress_info.inflate_stream.avail_out = output_len;

/*
**	inflate the input with Z_SYNC_FLUSH (Z_PARTIAL_FLUSH)
*/
	switch (inflate(&(ssh_info->compress_info.inflate_stream),
		Z_SYNC_FLUSH))
	{
	case Z_OK:
		/* compressed, fall through */
		break;

	case Z_BUF_ERROR:
	/*
	**	no progress was possible.  
	**
	**	we should hopefully never get here because we will
	**	always want to decompress at least the packet type.
	*/
		ssh_errno_set(SSH_ERRNO_ZLIB_STALL_ERROR);

		return(S_BAD);
		break;

	case Z_STREAM_ERROR:
		/* inconsistent state */
		ssh_errno_set(SSH_ERRNO_ZLIB_STATE_ERROR);

		return(S_BAD);
		break;

	case Z_MEM_ERROR:
		/* not enough memory */
		ssh_errno_set(SSH_ERRNO_ZLIB_MEM_ERROR);

		return(S_BAD);
		break;

	default:
		/* some other error (need dict, data error, stream end) */
		ssh_errno_set(SSH_ERRNO_ZLIB_MISC);
		
		return(S_BAD);
		break;
	}

/*
**	if we didn't have enough room to output the entire deflated
**	input (and not a Z_BUF_ERROR because we must have made some
**	progress)
*/
	if (ssh_info->compress_info.inflate_stream.avail_out == 0)
	{
		ssh_errno_set(SSH_ERRNO_ZLIB_MORE_OUT_ERROR);

		return(S_BAD);
	}

	*final_len = output_len
		- ssh_info->compress_info.inflate_stream.avail_out;

	return(S_GOOD);
}

