/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)es_interactive.c	1.22 96/10/14 Sun Microsystems"


/*
 * System includes
 */
#include <skip_os.h>

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/scrollbar.h>
#include <xview/svrimage.h>
#include <xview/notice.h>
#include <xview/xv_xrect.h>

/*
 * SKIP includes
 */
#include <skip_conf.h>
#include <skip_types.h>
#include <skip_proto.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_lib.h>

#include "skiptool_ui.h"
#include "es_glue.h"
#include "es_ras.h"
#include "es_systems.h"
#include "es_timers.h"
#include "es_interactive.h"

typedef struct listentry {
	struct listentry	*next;
	struct in_addr		addr;
} listentry_t;

#define	MAXMSGS			10
const int			inaddrsz = 4;
int				skip_fd = -1;
static int			add_automatically = 0;
static listentry_t		*informed_hosts = NULL;

static char			*versions[]  = {
					"cleartext",
					"SKIP (version 1)",
					"SKIP",
					"RAW ESP/AH"
				};
/*
 * cancel any further popup messages
 */
static void
cancel_interactive_mode()
{
	extern void		access_update();

	(void) xv_set(Skiptool_base_window->access_mode,
			PANEL_VALUE, NoAccess,
			NULL);
	access_update();
}

/*
 * ask the user for a yes/no response
 */
static int
es_yes_no(Panel_item owner, char *msgs[], char *yes, char *no)
{
	Panel				panel;
	Xv_notice			notice; 
	int				notice_stat;

	if (owner == XV_NULL) {
		panel = Skiptool_base_window->base_window;
	} else {
		panel = xv_get(owner, PANEL_PARENT_PANEL);
	}
	notice = xv_create(panel,
			NOTICE,
			XV_SHOW, TRUE,
			NOTICE_LOCK_SCREEN, FALSE,
			NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msgs,
			NOTICE_BUTTON, yes, NOTICE_YES,
			NOTICE_BUTTON, no, NOTICE_NO,
			NOTICE_STATUS, &notice_stat,
			NULL);
	xv_destroy_safe(notice);
	return notice_stat;
}

/*
 * popup a security warning message
 */
static void
acl_warning(Panel_item owner, char *msgs[])
{
	int			notice_stat;

	notice_stat = es_yes_no(owner, msgs, SKIP_CONFIRM, SKIP_CANCEL);
	if (notice_stat == NOTICE_NO) {
		cancel_interactive_mode();
	}
}

/*
 * prompt the user to accept a host or not
 */
static int
ask_user(Panel_item owner, char *msgs[])
{
	extern skiptool_base_window_objects	*Skiptool_base_window;
	const int			notice_cancel = 3;
	Panel				panel;
	Xv_notice			notice; 
	int				notice_stat;

	if (owner == XV_NULL) {
		panel = Skiptool_base_window->base_window;
	} else {
		panel = xv_get(owner, PANEL_PARENT_PANEL);
	}
	(void) xv_set(Skiptool_base_window->base_window,
			FRAME_CLOSED, FALSE,
			NULL);
	notice = xv_create(Skiptool_base_window->base_window, 
			NOTICE,
			XV_SHOW, TRUE,
			NOTICE_LOCK_SCREEN, FALSE,
			NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msgs,
			NOTICE_BUTTON, SKIP_ACCESS, NOTICE_YES,
			NOTICE_BUTTON, SKIP_NO_ACCESS, NOTICE_NO,
			NOTICE_BUTTON, SKIP_CANCEL, notice_cancel,
			NOTICE_STATUS, &notice_stat,
			NULL);
	xv_destroy_safe(notice);
	if (notice_stat == notice_cancel) {
		cancel_interactive_mode();
		notice_stat = NOTICE_NO;
	}
	return (notice_stat);
}

/*
 * keep track of systems we have already informed the user about
 */
static boolean_t
already_notified(struct in_addr addr)
{
	listentry_t	*lp;

	for (lp = informed_hosts; lp; lp = lp->next) {
		if (memcmp((const char *)&addr, 
				(const char *)&lp->addr, inaddrsz) == 0) {
			return (B_TRUE);
		}
	}
	lp = (listentry_t *)calloc(1, sizeof(*lp));
	if (lp == NULL) {
		return (B_TRUE);
	}
	memcpy((char *)&lp->addr, (char *)&addr, inaddrsz);
	lp->next = informed_hosts;
	informed_hosts = lp;

	return (B_FALSE);
}

/*
 * delete entry from informed list
 */
void
inform_list_delete(struct in_addr addr)
{
	listentry_t	*lp, **prevlp;

	for (prevlp = &informed_hosts; (lp = *prevlp); prevlp = &lp->next) {
		if (memcmp((const char *)&addr,
				(const char *)&lp->addr, inaddrsz) == 0) {
			break;
		}
	}
	if (lp == NULL) {
		return;
	}
	*prevlp = lp->next;

	free(lp);
}

/*
 * free informed list
 */
static void
inform_list_free()
{
	listentry_t	*lp, *nlp;

	for (lp = informed_hosts; lp; lp = nlp) {
		nlp = lp->next;
		free(lp);
	}
	informed_hosts = NULL;
}

/*
 * Build a message array
 */
static void
add_msg(char *msgs[], char *msg)
{
	register int	i;

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i] == NULL) {
			msgs[i] = strdup(msg);
			break;
		}
	}
}

/*
 * Destroy a message array
 */
static void
destroy_msgs(char *msgs[])
{
	register int	i;

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i] != NULL) {
			free(msgs[i]);
			msgs[i] = NULL;
		}
	}
}

/*
 * handle tunnel address checking
 */
void
es_check_tunnel(es_acl_t *access)
{
	skip_param_t		p, *ref = &p, *params = &access->params;
	int			notice_stat;
	char			*kij_algname, *kp_algname, *mac_algname,
				*versionstr;
	char			hostname[STRSZ], s[STRSZ],
				msg[STRSZ], *msgs[MAXMSGS] = { NULL };

	memset(ref, 0, sizeof (skip_param_t));

	if (params->ip_addr.s_addr == params->tunnel_addr.s_addr) {
		/*
		 * no tunnel address specified
		 */
		return;
	}

	if (params->flags & SKIP_NOMADIC) {
		/*
		 * don't do this for nomadic entries
		 */
		return;
	}

	IPADDRSET(&params->tunnel_addr, ref);

	if (skip_acl_get(ifname, ref) == 0) {
		if (ref->ip_addr.s_addr == params->tunnel_addr.s_addr) {
			/*
			 * acl entry for tunnel exists
			 */
			return;
		}
	}

	/*
	 * need to advise user about a missing tunnel address
	 */

	switch (params->version) {
	case SKIP_NONE:
	case SKIP_V1:
	case SKIP_V2:
	case SKIP_RAW:
		versionstr = versions[params->version];
		break;

	default:
		sprintf(s, "bad version #%d\n", params->version);
		versionstr = s;
	}

	if (skip_addr_to_host(&params->tunnel_addr, hostname) == 0) {
		sprintf(msg, SKIP_TUNNEL_1,
			hostname, inet_ntoa(params->tunnel_addr));
	} else {
		sprintf(msg, SKIP_TUNNEL_1,
			inet_ntoa(params->tunnel_addr), SKIP_UNKNOWN);
	}
	add_msg(msgs, msg);

	sprintf(msg, SKIP_TUNNEL_2, versionstr);

	add_msg(msgs, msg);

	kij_algname = skip_kij_alg_to_name(params->kij_alg, params->version);
	kp_algname = skip_kp_alg_to_name(params->kp_alg, params->version);
	mac_algname = skip_mac_alg_to_name(params->mac_alg, params->version);

	/*
	 * build algorithm/key information messages
	 */
	if (params->kij_alg) {
		sprintf(msg, "- %s (key encryption) ", kij_algname);
		add_msg(msgs, msg);
	}

	if (params->mac_alg) {
		sprintf(msg, "- %s (authentication)", mac_algname); 
		add_msg(msgs, msg);
	}

	if (params->kp_alg) {
		sprintf(msg, "- %s (traffic encryption)", kp_algname); 
		add_msg(msgs, msg);
	}

	(void) skip_keyid_to_s(&params->r_mkeyid, params->r_nsid, s);
	sprintf(msg, "- remote key id %s %s",
			skip_nsids[params->r_nsid].name,
			s);
	add_msg(msgs, msg);

	(void) skip_keyid_to_s(&params->s_mkeyid, params->s_nsid, s);
	sprintf(msg, "- local key id %s",
			skip_nsids[params->s_nsid].name);
	add_msg(msgs, msg);

	notice_stat = es_yes_no(access->win, msgs,
						SKIP_TUNNEL_3, SKIP_TUNNEL_4);

	if (notice_stat == NOTICE_YES) {
		*ref = *params;
		IPADDRSET(&ref->tunnel_addr, ref);
		if (skip_acl_add(ifname, ref)) {
			sprintf(msg, "Failed to add tunnel host %s (%s)",
				inet_ntoa(ref->ip_addr), skip_errmsg);
			alert(XV_NULL, msg, B_TRUE);
		} else {
			acl_build();
		}
	}

	destroy_msgs(msgs);
}

/*
 * handle incoming packet from an unknown destination
 */
static void
unknown_src(skip_es_notification_t	*skipmsgp)
{
	boolean_t		secure, is_bad = B_FALSE;
	skip_param_t		*params = &skipmsgp->params;
	struct ip		*ip;
	char			*kij_algname, *kp_algname, *mac_algname,
				*versionstr;
	char			*msgs[MAXMSGS] = { NULL };
	char			msg[STRSZ], s[STRSZ], hostname[STRSZ];

	ip = (struct ip *) &skipmsgp->iphdr;

	switch (params->version) {
	case SKIP_NONE:
		versionstr = "in the clear";
		break;

	case SKIP_V1:
	case SKIP_V2:
	case SKIP_RAW:
		sprintf(s, "with %s using\n", versions[params->version]);
		versionstr = s;
		break;
	default:
		sprintf(s, "with bad version #%d\n", params->version);
		versionstr = s;
	}

	secure = (params->version != SKIP_NONE);

	if (skip_addr_to_host(&ip->ip_src, hostname) == 0) {
		sprintf(msg, "\"%s\" (%s) is trying to connect %s",
			hostname, inet_ntoa(ip->ip_src), versionstr);
	} else {
		sprintf(msg, "%s (unknown) is trying to connect %s",
			inet_ntoa(ip->ip_src), versionstr);
	}
	add_msg(msgs, msg);

	if (secure) {
		/*
		 * check we could support this host
		 */
		kij_algname = skip_kij_alg_to_name(params->kij_alg,
							params->version);
		kp_algname = skip_kp_alg_to_name(params->kp_alg,
							params->version);
		mac_algname = skip_mac_alg_to_name(params->mac_alg,
							params->version);

		if (!skip_supported_kij_alg(params->kij_alg, params->version)) {
			sprintf(msg, "and unsupported key encryption "
				"algorithm #%d (%s)",
				params->kij_alg, kij_algname);

			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if (!skip_supported_kp_alg(params->kp_alg, params->version)) {
			sprintf(msg, "and unsupported traffic encryption "
				"algorithm #%d (%s)",
				params->kp_alg, kp_algname);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if (!skip_supported_mac_alg(params->mac_alg,
			params->version)) {
			sprintf(msg, "and unsupported authenication "
				"algorithm #%d (%s)",
				params->mac_alg, mac_algname);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if ((int) params->r_nsid >= skip_max_nsids) {
			sprintf(msg, "and unsupported sender NSID #%d",
				params->r_nsid);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}

		if ((int) params->s_nsid >= skip_max_nsids) {
			sprintf(msg, "and unsupported receiver NSID #%d ",
				params->s_nsid);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}

		if (is_bad) {
			acl_warning(XV_NULL, msgs);
			destroy_msgs(msgs);
			return;
		}

		/*
		 * build algorithm/key information messages
		 */
		sprintf(msg, "- %s (key encryption) ", kij_algname);
		add_msg(msgs, msg);

		if (params->mac_alg) {
			sprintf(msg, "- %s (authentication)", mac_algname); 
			add_msg(msgs, msg);
		}

		if (params->kp_alg) {
			sprintf(msg, "- %s (traffic encryption)", kp_algname); 
			add_msg(msgs, msg);
		}

		(void) skip_keyid_to_s(&params->r_mkeyid, params->r_nsid, s);
		sprintf(msg, "- its key id is %s %s",
				skip_nsids[params->r_nsid].name,
				s);
		add_msg(msgs, msg);

		sprintf(msg, "- local key id specified as %s",
				skip_nsids[params->s_nsid].name);
		add_msg(msgs, msg);
	}

	if (add_automatically || ask_user(XV_NULL, msgs) == NOTICE_YES) {
		if (skip_acl_add(ifname, params)) {
			sprintf(msg, "Failed to add host %s (%s)",
				inet_ntoa(ip->ip_src), skip_errmsg);
			alert(XV_NULL, msg, B_TRUE);
		} else {
			acl_build();
		}
	}
	destroy_msgs(msgs);
}

/*
 * handle incoming packet which doesn't match ACL
 */
static void
bad_params(skip_es_notification_t *skipmsgp)
{
	skip_param_t		*in = &skipmsgp->params;
	skip_param_t		acl_params, *ref = &acl_params;
	struct ip		*ip;
	char			in_str[STRSZ], ref_str[STRSZ];
	char			*msgs[MAXMSGS] = { NULL };
	char			msg[STRSZ], s[STRSZ];

	ip = (struct ip *) &skipmsgp->iphdr;

	(void) skip_addr_to_host(&ip->ip_src, s);

	ref->ip_addr = ip->ip_src;

	if (skip_acl_get(ifname, ref) != 0) {
		sprintf(msg, "Failed to get ACL entry for host %s (%s) - %s",
			s, inet_ntoa(ip->ip_src), skip_errmsg);
		alert(XV_NULL, msg, B_TRUE);
		return;
	}

	if ((ref->mask.s_addr != SKIP_HOSTMASK) ||
						(ref->ip_addr.s_addr == 0x0)) {
		/*
		 * BadParams against a wildcard or default entry means that
		 * the host was unknown, treat as an unknown source
		 */
		unknown_src(skipmsgp);
		return;
	}

	destroy_msgs(msgs);
	sprintf(msg, "\"%s\" (%s) is trying to connect using",
					s, inet_ntoa(ip->ip_src));

	add_msg(msgs, msg);

	/*
	 * check SKIP version
	 */
	if (in->version != ref->version) {
		if ((int) in->version <= SKIP_RAW) {
			strcpy(in_str, versions[in->version]);
		} else {
			sprintf(s, "<bad SKIP version %d>", in->version);
		}

		if ((int) ref->version <= SKIP_RAW) {
			strcpy(ref_str, versions[ref->version]);
		} else {
			sprintf(ref_str, "<bad SKIP version %d>", in->version);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
		acl_warning(XV_NULL, msgs);
		destroy_msgs(msgs);
		return;
	}

	/*
	 * check remote NSID
	 */
	if (in->r_nsid != ref->r_nsid) {
		if ((int) in->r_nsid >= skip_max_nsids) {
			sprintf(in_str, "remote NSID #%d "
				"<unknown>", in->r_nsid);
		} else {
			sprintf(in_str, "remote NSID #%d " "(%s)",
				in->r_nsid, skip_nsids[in->r_nsid].name);
		}
		if ((int) ref->r_nsid >= skip_max_nsids) {
			sprintf(ref_str, "#%d <unknown>",
				ref->r_nsid);
		} else {
			sprintf(ref_str, "#%d (%s)",
				ref->r_nsid, skip_nsids[ref->r_nsid].name);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
	} else {
		/*
		 * remote NSID is correct, check key value
		 */
		if (memcmp(in->r_mkeyid.buf, ref->r_mkeyid.buf,
							in->r_mkeyid.len)) {
			/*
			 * Key values do not match.  Check the key size is
			 * valid.
			 */
			if (skip_keyid_to_s(&in->r_mkeyid, in->r_nsid, in_str)
									< 0) {
				sprintf(in_str, "<bad key of length %d>",
						in->r_mkeyid.len);
			}
			if (skip_keyid_to_s(&ref->r_mkeyid, ref->r_nsid,
								ref_str) < 0) {
				sprintf(ref_str, "<bad key of length %d>",
						ref->r_mkeyid.len);
			}
			sprintf(msg, "with remote key %s", in_str);
			add_msg(msgs, msg);
			sprintf(msg, "instead of %s", ref_str);
			add_msg(msgs, msg);
		}
	}

	/*
	 * check local NSID
	 */
	if (in->s_nsid != ref->s_nsid) {
		if ((int) in->s_nsid >= skip_max_nsids) {
			sprintf(in_str, "local NSID #%d "
				"<unknown>", in->s_nsid);
		} else {
			sprintf(in_str, "local NSID #%d (%s)",
				in->s_nsid, skip_nsids[in->s_nsid].name);
		}
		if ((int) ref->s_nsid >= skip_max_nsids) {
			sprintf(ref_str, "#%d <unknown>",
				ref->s_nsid);
		} else {
			sprintf(ref_str, "#%d (%s)",
				ref->s_nsid, skip_nsids[ref->s_nsid].name);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
	}

	/*
	 * check Kij algs
	 */
	if (in->kij_alg != ref->kij_alg) {
		sprintf(msg, "key encryption algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->kij_alg,
			skip_kij_alg_to_name(in->kij_alg, in->version),
			ref->kij_alg,
			skip_kij_alg_to_name(ref->kij_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check Kp algs
	 */
	if (in->kp_alg != ref->kp_alg) {
		sprintf(msg, "traffic encryption algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->kp_alg,
			skip_kp_alg_to_name(in->kp_alg, in->version),
			ref->kp_alg,
			skip_kp_alg_to_name(ref->kp_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check MAC algs
	 */
	if (in->mac_alg != ref->mac_alg) {
		sprintf(msg, "authentication algorithm #%d "
				"instead of #%d",
			in->mac_alg,
			ref->mac_alg);
		add_msg(msgs, msg);
	}
	/*
	 * check compression algs
	 */
	if (in->comp_alg != ref->comp_alg) {
		sprintf(msg, "compression algorithm #%d "
				"instead of #%d",
			in->comp_alg,
			ref->comp_alg);
		add_msg(msgs, msg);
	}

	acl_warning(XV_NULL, msgs);
	destroy_msgs(msgs);
}

/*
 * handle outgoing packet to an unknown destination
 */
static void
unknown_dst(skip_es_notification_t	*skipmsgp)
{
	struct ip		*ip;
	char			*msgs[MAXMSGS] = { NULL }, msg[STRSZ],
				hostname[STRSZ];

	ip = (struct ip *) &skipmsgp->iphdr;

	if (skip_addr_to_host(&ip->ip_dst, hostname) == 0) {
		sprintf(msg,
			"This system is trying to connect to \"%s\" "
			"(%s)", hostname, inet_ntoa(ip->ip_dst));
	} else {
		sprintf(msg,
			"This system is trying to connect to \"%s\" "
			"(unknown)", inet_ntoa(ip->ip_dst));
		strcpy(hostname, inet_ntoa(ip->ip_dst));
	}
	add_msg(msgs, msg);

	if (add_automatically || ask_user(XV_NULL, msgs) == NOTICE_YES) {
		if (skip_acl_add(ifname, &skipmsgp->params)) {
			sprintf(msg, "Failed to add host %s (%s)",
				hostname, skip_errmsg);
			alert(XV_NULL, msg, B_TRUE);
		} else {
			acl_build();
		}
	}
	destroy_msgs(msgs);
}

static Notify_value
skip_msg(Notify_client client, int skip_fd)
{
	int			rc;
	skip_es_notification_t	skipmsg;
	struct ip		*ip;
	char			msg[STRSZ], *msgs[MAXMSGS] = { NULL };
	struct hostent		*hp;

	rc = skip_getmsg(skip_fd, (void *) &skipmsg, sizeof(skipmsg));
	if (rc < 0) {
		perror("skip_get_message failed");
		return (NOTIFY_DONE);
	}

	ip = (struct ip *) &skipmsg.iphdr;

	switch(skipmsg.what) {

	case SkipUnknownSrc:
		if (already_notified(ip->ip_src)) {
			break;
		}
		unknown_src(&skipmsg);
		break;

	case SkipUnknownDst:
		if (already_notified(ip->ip_dst)) {
			break;
		}
		unknown_dst(&skipmsg);
		break;

	case SkipBadParams:
		if (already_notified(ip->ip_src)) {
			break;
		}
		bad_params(&skipmsg);
		break;

	case SkipAuthFailed:
		if (already_notified(ip->ip_src)) {
			break;
		}
		hp = gethostbyaddr((char *) &ip->ip_src, inaddrsz, AF_INET);
		if (hp) {
			sprintf(msg,
				"Warning: \"%s\" (%s) failed authentication",
				hp->h_name, inet_ntoa(ip->ip_src));
		} else {
			sprintf(msg,
				"Warning: %s (unknown) failed authentication",
				inet_ntoa(ip->ip_src));
		}
		add_msg(msgs, msg);

		sprintf(msg, "Its certificate/key id may be invalid");
		add_msg(msgs, msg);

		acl_warning(XV_NULL, msgs);
		destroy_msgs(msgs);
		break;

	}
	return (NOTIFY_DONE);
}

static void
make_skip_interactive()
{
	extern skiptool_base_window_objects	*Skiptool_base_window;
	char					*msgs[MAXMSGS] = { NULL };

	if (skip_fd >= 0) {
		fprintf(stderr, "make_skip_interactive: internal error\n");
		return;
	}

	skip_fd = skip_set_mode(ifname, SkipInteractive);
	if (skip_fd < 0) {
		if (errno == EEXIST) {
			add_msg(msgs, "This mode is not available "
							"(already in use)");
		} else {
			add_msg(msgs, skip_errmsg);
		}
		acl_warning(XV_NULL, msgs);
		cancel_interactive_mode();
		destroy_msgs(msgs);
		return;
	}
	notify_set_input_func(Skiptool_base_window->base_window,
							skip_msg, skip_fd);
}

static void
make_skip_noninteractive()
{
	extern skiptool_base_window_objects	*Skiptool_base_window;

	if (skip_fd < 0) {
		fprintf(stderr, "make_skip_noninteractive: internal error\n");
		return;
	}
	notify_set_input_func(Skiptool_base_window->base_window,
						NOTIFY_FUNC_NULL, skip_fd);
	close(skip_fd);
	skip_fd = -1;
	return;
}

/*
 * modify SKIP access control mode
 */
void
access_update()
{
	extern skiptool_base_window_objects	*Skiptool_base_window;
	int					enable, mode, current_mode;
	char					*msgs[MAXMSGS] = { NULL };

	enable = xv_get(Skiptool_base_window->access_ctrl, PANEL_VALUE);
	current_mode = skip_get_mode(ifname);
	mode = xv_get(Skiptool_base_window->access_mode, PANEL_VALUE);

	if (enable) {
		inform_list_free();
		if (current_mode == SkipAccessControlOff) {
			/*
			 * need to check for required servers
			 */
			if (enable_checks() == EBUSY) {
				/*
				 * wait for user to answer
				 */
				return;
			}
		}
		acl_build();
		switch (mode) {
		case NoAccess:
			if (skip_fd >= 0) {
				make_skip_noninteractive();
			}
			if (skip_access_ctrl_on(ifname) < 0) {
				add_msg(msgs, skip_errmsg);
				acl_warning(Skiptool_base_window->access_ctrl,
					msgs);
			}
			break;

		case AskFirst:
			add_automatically = 0;
			if (skip_access_ctrl_on(ifname) < 0) {
				add_msg(msgs, skip_errmsg);
				acl_warning(Skiptool_base_window->access_ctrl,
					msgs);
			}
			if (skip_fd < 0) {
				make_skip_interactive();
			}
			break;

		case AddAutomatically:
			add_automatically = 1;
			if (skip_access_ctrl_on(ifname) < 0) {
				add_msg(msgs, skip_errmsg);
				acl_warning(Skiptool_base_window->access_ctrl,
					msgs);
			}
			if (skip_fd < 0) {
				make_skip_interactive();
			}
		}
	} else {
		if (skip_fd >= 0) {
			make_skip_noninteractive();
		}
		if (skip_access_ctrl_off(ifname) < 0) {
			add_msg(msgs, skip_errmsg);
			acl_warning(Skiptool_base_window->access_ctrl, msgs);
		}
	}
	destroy_msgs(msgs);
	show_mode();
}
