/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)SlotParse.C	1.22 96/10/15"

#include <skip_os.h>
#include <skip_proto.h>
#include "skip_log.h"

#include "skip_keymgr.h"
#include "SlotParse.h"
#include "X509skip.h"
#include "HashCert.h"
#include "Passwd.h"

#define		sp_iswhite(c)		(c == ' ' || c == '\t')
#define		sp_iswhite_equals(c)	(c == ' ' || c == '\t' || c == '=')
#define		sp_isdigit(c)		(c >= '0' && c <= '9')


Bstream passwd_key;


SlotParse::SlotParse()
{
	cert = 0;
	clear();
}

SlotParse::~SlotParse()
{
	clear();
}

void SlotParse::clear()
{
	if (cert)
		delete cert;

	cert = 0;
	type = 0;
	nsid = 0;
	mkid = Bstream();
	secret = Bigint();

	enable_cdp = 1;
	enable_certd = 1;
	require_cdp_cookie = 0;
	cdp_max_retry = 3;
	cdp_retry_time = 5;
	max_cdp_queue = 10;
	serve_only_local = 0;
	disallow_cdp_put = 0;
	max_certdb_size = 0;		// unlimited
	rng_dev_audio = 1;

	for (char **p = cdp_resolvers.first(); p; p = cdp_resolvers.next()) {
		free(*p);
		cdp_resolvers.delete_current();
	}
}

static char *slot_file;
static int lineno;

static char white_save;
static char *white_p;

void next_white(char*& s)
{
	char *p;

	while (*s && sp_iswhite(*s))
		s++;

	for (p = s; *p && !sp_iswhite(*p); p++)
		;

	white_save = *p;
	*p = '\0';
	white_p = p;
}

char *unwhite()
{
	*white_p = white_save;

	while (*white_p && sp_iswhite(*white_p))
		white_p++;

	return white_p;
}

void next_white_equals(char*& s)
{
	char *p;

	while (*s && sp_iswhite(*s))
		s++;

	for (p = s; *p && !sp_iswhite_equals(*p); p++)
		;

	white_save = *p;
	*p = '\0';
	white_p = p;
}

char *unwhite_equals()
{
	*white_p = white_save;

	while (*white_p && sp_iswhite_equals(*white_p))
		white_p++;

	return white_p;
}

#define CMD_NONE		0
#define CMD_TYPE		1
#define CMD_NAME		2
#define CMD_NSID		3
#define CMD_V1COMPAT		4

#define	CMD_ENABLE_CDP		5
#define	CMD_ENABLE_CERTD	6
#define	CMD_REQUIRE_CDP_COOKIE	7
#define	CMD_CDP_SERVER		8
#define CMD_CDP_MAX_RETRY	9
#define CMD_CDP_RETRY_TIME	10
#define CMD_MAX_CDP_QUEUE	11
#define CMD_SERVE_ONLY_LOCAL	12
#define CMD_DISALLOW_CDP_PUT	13
#define CMD_MAX_CERTDB_SIZE	14
#define CMD_RNG_DEV_AUDIO	15


int parse_top(char*& s)
{
	int ret = 0;

	next_white(s);

	if (*s == '\0')		// blank line
		return CMD_NONE;

	if (strcmp(s, "type") == 0)
		ret = CMD_TYPE;
	else if (strcmp(s, "name") == 0 || strcmp(s, "mkid") == 0)
		ret = CMD_NAME;
	else if (strcmp(s, "nsid") == 0)
		ret = CMD_NSID;
	else if (strcmp(s, "enable_cdp") == 0)
		ret = CMD_ENABLE_CDP;
	else if (strcmp(s, "enable_certd") == 0)
		ret = CMD_ENABLE_CERTD;
	else if (strcmp(s, "require_cdp_cookie") == 0)
		ret = CMD_REQUIRE_CDP_COOKIE;
	else if (strcmp(s, "rng_dev_audio") == 0)
		ret = CMD_RNG_DEV_AUDIO;
	else if (strcmp(s, "cdp_retry_time") == 0)
		ret = CMD_CDP_RETRY_TIME;
	else if (strcmp(s, "cdp_max_retry") == 0)
		ret = CMD_CDP_MAX_RETRY;
	else if (strcmp(s, "cdp_server") == 0)
		ret = CMD_CDP_SERVER;
	else if (strcmp(s, "max_cdp_queue") == 0)
		ret = CMD_MAX_CDP_QUEUE;
	else if (strcmp(s, "serve_only_local") == 0)
		ret = CMD_SERVE_ONLY_LOCAL;
	else if (strcmp(s, "disallow_cdp_put") == 0)
		ret = CMD_DISALLOW_CDP_PUT;
	else if (strcmp(s, "max_certdb_size") == 0)
		ret = CMD_MAX_CERTDB_SIZE;
	else
		skip_log(SKIP_ERROR, "%s(%d): unrecognized command '%s'",
				slot_file, lineno, s);

	s = unwhite();

	return ret;
}

int parse_type(char*& s)
{
	int ret = 0;

	next_white(s);

	if (strcmp(s, "soft") == 0)
		ret = SLOT_SOFT;
	else if (strcmp(s, "card") == 0)
		ret = SLOT_CARD;
	else
		skip_log(SKIP_ERROR, "%s(%d): unknown type '%s'",
				slot_file, lineno, s);
	
	s = unwhite();

	return ret;
}

int parse_nsid(char*& s)
{
	int ret = 0;

	next_white(s);

	if (!sp_isdigit(*s))
		skip_log(SKIP_ERROR, "%s(%d): bad nsid '%s'",
				slot_file, lineno, s);
	else
		ret = atoi(s);

	s = unwhite();

	return ret;
}

int parse_int(char*& s)
{
	int ret = 0;

	next_white(s);

	if (!sp_isdigit(*s))
		skip_log(SKIP_ERROR, "%s(%d): bad value '%s'",
				slot_file, lineno, s);
	else
		ret = atoi(s);

	s = unwhite();

	return ret;
}

Bstream parse_hex(char*& s)
{
	Bstream ret;

	next_white(s);

	if (strlen(s) < 3 || *s != '0' || *(s+1) != 'x') {
		skip_log(SKIP_ERROR, "%s(%d): bad hex string '%s'",
				slot_file, lineno, s);
		return ret;
	}

	Bstream_atob(ret, s+2);

	s = unwhite();

	return ret;
}

void parse_cdp_server(char*& s, LL<char*>& m)
{
	char *save;

	next_white(s);

	save = (char*) malloc(strlen(s) + 1);
	if (save == 0)
		skip_log(SKIP_ERROR, "parse_cdp_server: out of memory");
	else {
		strcpy(save, s);
		m.append(save);
	}

	s = unwhite();
}

void SlotParse::load_cert(char *filename)
{
	SkipCert *my_cert = 0;
	char root[256];
	char certfnam[256];
	char *s;
	Bstream certstr;

	strcpy(root, filename);

	// back up to first . in filename

	for (s = root; *s; s++)
		;
	while (s >= root && *s != '.' && *s != '/')
		s--;
	if (*s == '.')
		*s = '\0';

	if (!my_cert) {
		sprintf(certfnam, "%s.x509", root);

		certstr = File_to_Bstr(certfnam);
		if (certstr.getlength() > 0)
			my_cert = new X509SkipCert;
	}

	if (!my_cert) {
		sprintf(certfnam, "%s.dhpub", root);
		certstr = File_to_Bstr(certfnam);
		if (certstr.getlength() > 0)
			my_cert = new HashCert;
	}

	if (!my_cert)
		return;

	if (my_cert->decode(certstr)) {
		skip_log(SKIP_ERROR, "can't parse certificate in %s",
							certfnam);
		delete my_cert;
		return;
	}

	if (my_cert->skip_notvalidafter() < NTPNOW) {
		skip_log(SKIP_ERROR, "expired certificate %s", certfnam);
		return;
	}

	if (my_cert->skip_params(g, p)) {
		skip_log(SKIP_ERROR, "can't extract (g,p) from certificate %s",
							certfnam);
		return;
	}

	if (mkid.getlength() == 0)
		mkid = my_cert->skip_name();
	else if (my_cert->skip_name() != mkid) {
		char buf[256];
	
		sprintf(buf, "%s", mkid.get_info_str());

		skip_log(SKIP_ERROR,
		"stated name '%s' does not match certificate name '%s'",
				buf, my_cert->skip_name().get_info_str());

		return;
	}

	char *reason;
	if (reason = my_cert->isValid(auth))
		skip_log(SKIP_NOTICE,
			"warning: verification failed for %s, reason=%s",
						certfnam, reason);

	cert = my_cert;
}

void SlotParse::load_secret(char *filename)
{
	char root[256];
	char fnam[256];
	char *s;

	strcpy(root, filename);

	// back up to first . in filename

	for (s = root; *s; s++)
		;
	while (s >= root && *s != '.' && *s != '/')
		s--;
	if (*s == '.')
		*s = '\0';

	sprintf(fnam, "%s.secret", root);

	Bstream privkeystr = File_to_Bstr(fnam);

	if (privkeystr.getlength() > 0)
		secret = Bigint(privkeystr);

}

int SlotParse::sanity_check()
{

	if (type != SLOT_SKIP_CONF) {
		if (mkid.getlength() == 0) {
			skip_log(SKIP_ERROR, "%s: no name (mkid) set",
							slot_file);
			return FALSE;
		}

		if (nsid == 0) {
			skip_log(SKIP_ERROR, "%s: no nsid set", slot_file);
			return FALSE;
		}

		int expected_len = 0;
		switch (nsid) {
		case SKIP_NSID_IPV4:
			expected_len = 4;
			break;

		case SKIP_NSID_MD5_DH_PUB:
			expected_len = 16;
			break;
		}

		if (expected_len && mkid.getlength() != expected_len) {
			skip_log(SKIP_ERROR,
			"%s: name length for nsid %d expected %d, actual %d",
					slot_file,
					nsid,
					expected_len,
					mkid.getlength());
			return FALSE;
		}
	}

	switch (type) {
	case SLOT_SKIP_CONF:
		break;

	case SLOT_SOFT:
		if (secret == Bigint()) {
			skip_log(SKIP_ERROR, "%s: no secret set",
							slot_file);
			return FALSE;
		}

		if (cert == 0) {
			skip_log(SKIP_ERROR, "%s: no certificate",
							slot_file);
			return FALSE;
		}
		break;

	case SLOT_CARD:
	default:
		skip_log(SKIP_ERROR, "%s: unknown type", slot_file);
		return FALSE;
	}

	return TRUE;
}

int SlotParse::load(char *filename, int default_type)
{
	FILE *fp;
	char buf[1024];
	char *s;

	clear();

	lineno = 0;
	slot_file = filename;
	type = default_type;

	fp = fopen(filename, "r");
	if (fp == NULL) {
		skip_log(SKIP_ERROR, "can't read %s: %s", filename,
				strerror(errno));
		return FALSE;
	}

	while (fgets(buf, 1024, fp) != NULL)
	{
		lineno++;

// kill \n at eol, also do # comments 

		for (s = buf; *s && *s != '\n' && *s != '#'; s++)
			;
		*s = '\0';
		s = buf;

		int cmd = parse_top(s);

		switch (cmd) {
		case CMD_NONE:
			break;

		case CMD_TYPE:
			type = parse_type(s);
			break;

		case CMD_NAME:
			mkid = parse_hex(s);
			break;

		case CMD_NSID:
			nsid = parse_nsid(s);
			break;

		case CMD_ENABLE_CDP:
			enable_cdp = parse_int(s);
			break;

		case CMD_ENABLE_CERTD:
			enable_certd = parse_int(s);
			break;

		case CMD_REQUIRE_CDP_COOKIE:
			require_cdp_cookie = parse_int(s);
			break;

		case CMD_RNG_DEV_AUDIO:
			rng_dev_audio = parse_int(s);
			break;

		case CMD_CDP_RETRY_TIME:
			cdp_retry_time = parse_int(s);
			break;

		case CMD_CDP_MAX_RETRY:
			cdp_max_retry = parse_int(s);
			break;

		case CMD_CDP_SERVER:
			parse_cdp_server(s, cdp_resolvers);
			break;

		case CMD_MAX_CDP_QUEUE:
			max_cdp_queue = parse_int(s);
			break;

		case CMD_SERVE_ONLY_LOCAL:
			serve_only_local = parse_int(s);
			break;

		case CMD_DISALLOW_CDP_PUT:
			disallow_cdp_put = parse_int(s);
			break;

		case CMD_MAX_CERTDB_SIZE:
			max_certdb_size = parse_int(s);
			break;

		default:
			skip_log(SKIP_ERROR, "%s(%d): internal error",
				slot_file, lineno);
		}
	}

	fclose(fp);

	load_cert(filename);
	load_secret(filename);

	if (!sanity_check())
		return FALSE;

	return TRUE;
}

void
SlotParse::print(int verbosity, int machprt)
{
	switch(type){
		case SLOT_SOFT:
			if (machprt)
				printf(" type=soft ");
			else
				printf("\tType: Software Slot\n");
			break;
		case SLOT_CARD:
		default:
			if (machprt)
				printf(" type=unknown");
			else
				printf("\tType: Unknown \n");
	}
	if (machprt)
		printf("nsid=%d ", nsid);
	else
		printf("\tNSID: %d ", nsid);

	if (machprt)
		printf("mkid=%s ",mkid.get_info_str());
	else
		printf("MKID (name): %s\n",mkid.get_info_str());
	if (!verbosity) {
		if (machprt) {
			printf("invalidbefore=%lu ",
					cert->skip_notvalidbefore());
			printf("invalidafter=%lu ",
					cert->skip_notvalidafter());
		} else {
			long t;
			t = NTP_TO_UNIX(cert->skip_notvalidbefore());
			printf("\tNot Valid Before: %s", ctime(&t));
			t = NTP_TO_UNIX(cert->skip_notvalidafter());
			printf("\tNot Valid After: %s", ctime(&t));
		}
	} else 
		cert->print();

	int bits = 0;
	if (cert) {
		Bigint g,  p;
	
		cert->skip_params(g, p);
		bits = p.bits();
	}

	if (machprt)
		printf("modulus_size=%d ", bits);
	else
		printf("\tModulus size: %d bits\n", bits);
}

#define RAW_R_NSID	1
#define	RAW_R_NAME	2
#define RAW_L_NSID	3
#define	RAW_L_NAME	4
#define	RAW_E_KP	5
#define	RAW_A_KP	6

LL<raw_key> raw_keys;

raw_key*
find_raw_key(u_char r_nsid, const Bstream& r_keyid,
			u_char l_nsid, const Bstream& l_keyid)
{
	raw_key *p;

	for (p = raw_keys.first(); p; p = raw_keys.next()) {
		if (p->r_nsid == r_nsid &&
		    p->r_keyid == r_keyid &&
		    p->l_nsid == l_nsid &&
		    p->l_keyid == l_keyid)
			return p;
	}

	return NULL;
}

int parse_raw_key(char*& s)
{
	int ret = 0;

	next_white_equals(s);

	if (*s == '\0')
		ret = 0;
	else if (strcmp(s, "r_nsid") == 0)
		ret = RAW_R_NSID;
	else if (strcmp(s, "l_nsid") == 0)
		ret = RAW_L_NSID;
	else if (strcmp(s, "r_name") == 0 || strcmp(s, "r_mkid") == 0)
		ret = RAW_R_NAME;
	else if (strcmp(s, "l_name") == 0 || strcmp(s, "l_mkid") == 0)
		ret = RAW_L_NAME;
	else if (strcmp(s, "e_kp") == 0)
		ret = RAW_E_KP;
	else if (strcmp(s, "a_kp") == 0)
		ret = RAW_A_KP;
	else
		skip_log(SKIP_ERROR, "%s(%d): unknown raw keyword '%s'",
				slot_file, lineno, s);
	
	s = unwhite_equals();

	return ret;
}

void parse_raw(char*& s)
{
	raw_key entry;
	int cmd;

	while (1) {
		cmd = parse_raw_key(s);

		switch (cmd) {
		case 0:
			raw_keys.append(entry);
			return;

		case RAW_R_NSID:
			entry.r_nsid = parse_nsid(s);
			break;

		case RAW_L_NSID:
			entry.l_nsid = parse_nsid(s);
			break;

		case RAW_R_NAME:
			entry.r_keyid = parse_hex(s);
			break;

		case RAW_L_NAME:
			entry.l_keyid = parse_hex(s);
			break;

		case RAW_E_KP:
			entry.e_kp = parse_hex(s);
			break;

		case RAW_A_KP:
			entry.a_kp = parse_hex(s);
			break;
		}
	}
}

void load_raw(char *filename)
{
	FILE *fp;
	char buf[1024];
	char *s;

	lineno = 0;
	slot_file = filename;

	fp = fopen(filename, "r");
	if (fp == NULL)
		return;

	while (fgets(buf, 1024, fp) != NULL)
	{
		lineno++;

		// kill \n at eol, also do # comments 

		for (s = buf; *s && *s != '\n' && *s != '#'; s++)
			;
		*s = '\0';
		s = buf;

		if (*s)
			parse_raw(s);
	}

	fclose(fp);
}

