/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_safercbc.c	1.5 96/08/07"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include "skip_crypt.h"

#include "safer.h"

struct safer_cbc_state {
	safer_key_t keysched;
	int pad;
};

#define MODULE_NAME	    "safer_128sk_cbc"

char safer_128sk_cbc_name[] = MODULE_NAME " v1.5";

STATIC
unsigned int safer_cbc_outlen(unsigned int);

STATIC
void *safer_cbc_open(char *, unsigned int , char *, unsigned int );

STATIC
void *
safer_cbc_open_nopad(char *, unsigned int , char *, unsigned int );

STATIC
void
desafer_cbc(void *, char *, unsigned int *, char *, char *, unsigned int , 
		unsigned char *);

STATIC
void 
ensafer_cbc(void *, char *, unsigned int *, char *, char *, unsigned int, 
	    	unsigned char *);

STATIC
void safer_cbc_close(void *);

/******************* Module Data **********************************************/

/******************* Functions ************************************************/

/******************************************************************************/

const skip_cryptor_t	    safer_128sk_cbc_module = {
	SKIP_CRYPTOR_V1,	/* binary version */
	MODULE_NAME,		/* name */
	safer_cbc_open,		/* open */
	safer_cbc_close,	/* close */
	ensafer_cbc,		/* encrypt */
	desafer_cbc,		/* decrypt */
	safer_cbc_outlen,	/* getoutputlen */
	16,			/* kp_len  */
	16,			/* ekp_len */
	8,			/* blocksize */
	8,			/* mid_len */
	SKIP_BLOCK_CIPHER,	/* flags */
	0,			/* Kij alg */
	SKIP_CRYPT_SAFER_128SK_CBC,	/* kp alg */
};

const skip_cryptor_t	    safer_128sk_cbc_nopad_module = {
	SKIP_CRYPTOR_V1,	/* binary version */
	MODULE_NAME,		/* name */
	safer_cbc_open_nopad,	/* open */
	safer_cbc_close,	/* close */
	ensafer_cbc,		/* encrypt */
	desafer_cbc,		/* decrypt */
	safer_cbc_outlen,	/* getoutputlen */
	16,			/* kp_len  */
	16,			/* ekp_len */
	8,			/* blocksize */
	8,			/* mid_len */
	SKIP_BLOCK_CIPHER,	/* flags */
	SKIP_KIJ_SAFER_128SK_CBC, /* kij alg */
	0,     			/* kp alg */
};

int safer_128sk_cbc_busy;

extern
void Safer_Expand_Userkey( safer_block_t, safer_block_t, unsigned int , 
			   int , safer_key_t );
extern
void Safer_Encrypt_Block(safer_block_t , safer_key_t , safer_block_t );

extern 
void Safer_Decrypt_Block(safer_block_t , safer_key_t , safer_block_t );

STATIC
unsigned int
safer_cbc_outlen(unsigned int inlen)
{       
	/* Worst-case padding, rounded down to a multiple of 8 bytes */
	return (inlen + 9) & (unsigned)-8;
}

/******************************************************************************/

STATIC
void *
safer_cbc_open1(char  *keyp, unsigned int keylen, char *ivp, unsigned int ivlen,
		int pad)
{
    struct safer_cbc_state *cs;

    if (keylen != 16)	/* We only allow 128 bit keys for safer */
	return NULL;

    cs = (struct safer_cbc_state *) 
			SYSMEM_ALLOC(sizeof(struct safer_cbc_state));
    if (cs == (void *)0)
	     return cs;

    safer_128sk_cbc_busy++;
    cs->pad=pad;

/* We use the default number of rounds and the strengthed key schedule */
    Safer_Expand_Userkey(keyp, keyp+8, SAFER_K128_DEFAULT_NOF_ROUNDS,  
			 1, cs->keysched);

    return cs;
}

STATIC
void *
safer_cbc_open(char *keyp, unsigned int keylen, char *ivp, unsigned int ivlen)
{
	return safer_cbc_open1(keyp, keylen, ivp, ivlen, 1);
}

STATIC
void *
safer_cbc_open_nopad(char *keyp, unsigned int keylen, char *ivp, 
		     unsigned int ivlen)
{
	return safer_cbc_open1(keyp, keylen, ivp, ivlen, 0);
}

STATIC
void 
ensafer_cbc(	void *csp, char *in, unsigned int *lenp, char *out, 
		char *iv, unsigned int ivlen, unsigned char *payload)

{   
	int len;
	struct safer_cbc_state *cs;
	char *ivn;
	int i;

	char ecbio[8];

	cs=(struct safer_cbc_state *)csp;

	len = *lenp;

	/* If we're in padding mode, then pad using KarN style padding with
	** payload stored as last byte.  Otherwise, we just encrypt a block
	*/
	if (cs->pad) {
		while ((len & 0x07 ) != 6 ) {	/* actually mod 8 */
	       		 in[len]=len;
	       		 len++;
		}

		in[len]= len - *lenp;     /* pad length */
		len++;

		in[len] = *payload;      /* payload type */
		len++;
		*lenp = len;
	} else {
		if (len % 8 ) {			/* mod 8 */
			printf("safer:Error, unpadded mode with"
			       "bad block size");
			*lenp = 0;
			return;
		}
	}

	ivn = iv;
	/* Now do the Actual SAFER encryption using a CBC mode */
	for(i = 0; i < len ; i += 8) {
		*(unsigned int *)ecbio = *(u_int *)(ivn) ^ (*(u_int *)&in[i]);

		*(unsigned int *)(ecbio + 4) = *(u_int *)(ivn + 4) ^
					       (*(u_int *)&in[i+4]);

		Safer_Encrypt_Block(ecbio, cs->keysched, out+i);
		ivn = out + i;
	}
	BCOPY(ivn, iv, 8); 	/* save the last block as the IV */
}

STATIC
void
desafer_cbc(	void *csp, char *in, unsigned int *lenp, 
		char *out, char *iv, unsigned int ivlen, 
		unsigned char *payload)
{
	int i;
	u_char padlen;
	struct safer_cbc_state *cs;
	int len = *lenp;
	char *ivn;
	char ecbio[8];

	if (len == 0)
		return;
	cs=(struct safer_cbc_state *)csp;

	/* CBC decryption always expects block-size multiple. */
	if (len & 0x07 ){ 		/* Technically mod 8 */
		*lenp = 0;
		return;
	}

	ivn = iv;

	/*
	** Perform CBC decryption.  Decrypt the block then xor in the 
	** previous encrypted block.  
	*/
	for(i = 0; i< len ; i += 8) {
		Safer_Decrypt_Block(&in[i], cs->keysched, ecbio);

		*(unsigned int *)&out[i] = *(unsigned int *)ivn ^
					*(unsigned int *)ecbio;
		*(unsigned int *)&out[i+4] = *(unsigned int *)(ivn + 4) ^
					*(unsigned int *)(ecbio + 4);
		
		ivn = &in[i];  
	}
	/* Copy the final block to the IV so we can keep chaining */
	BCOPY(ivn, iv, 8);
	
	if (cs->pad) {
		padlen = (int)(unsigned char)out[len - 2];
		if ( padlen > 0x08 ){
			printf("safer_cbc: error in pad block, pad > 07\n");
			*lenp = 0;
			return;
		}
		*payload = out[len-1];

		/*
	 	* Adjust the length of the output
	 	* for padding. 
	 	*/

		*lenp = len - padlen - 2; /* strip off payload and pad bytes */
	} 
	return;
}

STATIC  
void    
safer_cbc_close(void *cs)
{ 
	BZERO(cs, sizeof(struct safer_cbc_state));
	SYSMEM_FREE(cs, sizeof(struct safer_cbc_state));  
	safer_128sk_cbc_busy--;
	return;
}

