/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)passwd_subr.C	1.3 95/11/19 Sun Microsystems"

//
// This is an implementation of RSA DSI's PKCS #5
// (Password based Encryption Standard).
// For both MD5 and MD2 with DES-CBC.
//

#include <stdio.h>
#include <stdlib.h>
#include "Bstream.h"
#include "Bigint.h"
#include "ObjId.h"
#include "Passwd.h"
#include "md2.h"
#include "md5.h"
#include "des_user.h"

//
// Generates a salt given a null terminated password, and a
// plain text string.
//
static void
make_salt(const char *passwd, const Bstream &plainstr,
		const ObjId& md_oid, Bstream *salt)
{
	unsigned char digest[DIGESTSIZE];

	if (md_oid == md5) { 
		xMD5Context context;

		xMD5Init(&context);
		xMD5Update(&context, (unsigned char *)passwd, strlen(passwd));
		xMD5Update(&context, plainstr.getdatap(), plainstr.getlength());
		xMD5Final(digest, &context);
	} else if (md_oid == md2) {
		xMD2Context context;

		xMD2Init(&context);
		xMD2Update(&context, (unsigned char *)passwd, strlen(passwd));
		xMD2Update(&context, plainstr.getdatap(), plainstr.getlength());
		xMD2Final(&context, digest);
	} else {
		fprintf(stderr, "Unsupported message digest type.\n");
		return;
	}
	*salt = Bstream(SALTSIZE, digest);
}

//
// Generates a DES key and an initializing vector given a null
// terminated password, the salt and an iteration count.
//
static void
make_deskey(const char *passwd, const Bstream &salt, int iter_cnt,
		const ObjId& md_oid, unsigned char *deskey, unsigned char *ivec)
{
	unsigned char digest[DIGESTSIZE], *p, test;

	if (md_oid == md5) { 
		xMD5Context context;

		xMD5Init(&context);
		xMD5Update(&context, (unsigned char *)passwd, strlen(passwd));
		xMD5Update(&context, salt.getdatap(), SALTSIZE);
		xMD5Final(digest, &context);
	} else if (md_oid == md2) {
		xMD2Context context;

		xMD2Init(&context);
		xMD2Update(&context, (unsigned char *)passwd, strlen(passwd));
		xMD2Update(&context, salt.getdatap(), SALTSIZE);
		xMD2Final(&context, digest);
	} else {
		fprintf(stderr, "Unsupported message digest type\n");
		return;
	}


	while (--iter_cnt > 0) {
		if (md_oid == md5) {
			xMD5Context context;

			xMD5Init(&context);
			xMD5Update(&context, digest, DIGESTSIZE);
			xMD5Final(digest, &context);
		} else if (md_oid == md2) {
			xMD2Context context;

			xMD2Init(&context);
			xMD2Update(&context, digest, DIGESTSIZE);
			xMD2Final(&context, digest);
		}
	}

	bcopy(digest, deskey, 8);
	// add odd parity to low bit of 8 byte key
	p = deskey;
	int n, x;
	for (int i = 0; i < 8; i++) {
		test = (1<<7);
		x = 0;
		for (n=0; n < 8; n++) {
			if (*p & test) x++;
			test = (test >> 1);	
		}
		if ((x%2) == 0) 		//even parity
			*p = (*p & 0x01) ? (*p & 0xfe) : (*p | 0x01);
		p++;
	}
	bcopy(digest+8, ivec, 8);
}

//
// Do a DES-CBC encryption given a password and a plain text string.
//
int
cbc_encrypt(const char *passwd, const Bstream &plainstr,
		const ObjId& md_oid, Bstream *encryptedstr)
{
	unsigned char deskey[8], ivec[8], padblock[8];
        unsigned char *encstr = NULL, notused;
        int  padlen, i;
        unsigned int len;
        Bstream salt, encblock, nullbstr;
        void *cs;

	make_salt(passwd, plainstr, md_oid, &salt);
	if (salt == nullbstr)
		return(-1);

	make_deskey(passwd, (const Bstream &)salt, ITERATION_COUNT, md_oid,
		deskey, ivec);

	//
	// pad length = 8 - (length_of_plaintext mod 8)
	// pad character = value of pad length
	// e.g. EncryptionBlock = plaintext | 01
	//	-- if length_of_plaintext mod 8 = 7
	//
	padlen = (8 - (plainstr.getlength() % 8));

	for (i = 0; i < padlen; i++)
		padblock[i] = padlen;

	encblock = plainstr + Bstream(padlen, padblock);

	len = encblock.getlength();
	if ((encstr = (unsigned char *)malloc(len)) == NULL)
		return(-1);

	cs = des_cbc_open_nopad(deskey, 8, ivec, 8);
        (void) endes_cbc_v1(cs, encblock.getdatap(), &len, encstr, ivec,
                        8, &notused);
        (void) des_cbc_close(cs);

	//
	// need the salt for decryption.
	// XXX - for now iteration count is hard-coded
	//
	*encryptedstr = salt + Bstream(len, encstr);
        free((char *)encstr);
	return (0);
}

//
// carries out and MD{2,5}-CBC decryption given a password and
// an encrypted string.
//
int
cbc_decrypt(const char *passwd, const Bstream &encryptedstr,
		const ObjId& md_oid, Bstream *plainstr)
{
	unsigned char deskey[8], ivec[8], *p, *decstr=NULL, notused;
        int padlen, i, decblocklen;
        unsigned int len;
        Bstream salt, decblock;
        void *cs;

	// get the salt
	salt = Bstream(SALTSIZE, encryptedstr.getdatap());

	decblocklen = encryptedstr.getlength() - SALTSIZE;
	// verify remaining str is a multiple of eight
	if ((decblocklen & 0x7) != 0) {
		fprintf(stderr,
			"cbc_decrypt: corrupted encrypted string given\n");
		return (-1);
	}

	make_deskey(passwd, salt, ITERATION_COUNT, md_oid, deskey, ivec);

	p = encryptedstr.getdatap();
	decblock = Bstream(decblocklen, p+SALTSIZE);

	len = decblock.getlength();
	if ((decstr = (unsigned char *)malloc(len)) == NULL)
		return(-1);

	cs = des_cbc_open_nopad(deskey, 8, ivec, 8);
        (void) dedes_cbc_v1(cs, decblock.getdatap(), &len, decstr, ivec,
                        8, &notused);
        (void) des_cbc_close(cs);

        p = decstr;
        p += (len - 1);         // point to a pad character
	padlen = *p;
	if (padlen <= 0 || padlen > 8) {
		fprintf(stderr, "cbc_decrypt: wrong pad length.\n");
		return (-1);
	}
	for (i = 0; i < padlen; i++)
		if (*p-- != padlen) {
			fprintf(stderr, "cbc_decrypt: incorrect padding.\n");
			return (-1);
		}

	*plainstr = Bstream(len - padlen, decstr);
        free((char *)decstr);
	return (0);
}
