/* 
  Copyright (C) 1989 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */

#include "krb_locl.h"

RCSID("$Id: get_krbrlm.c,v 1.20 1998/03/18 13:46:51 bg Exp $");

/*
 * krb_get_lrealm takes a pointer to a string, and a number, n.  It fills
 * in the string, r, with the name of the nth realm specified on the
 * first line of the kerberos config file (KRB_CONF, defined in "krb.h").
 * It returns 0 (KSUCCESS) on success, and KFAILURE on failure.  If the
 * config file does not exist, and if n=1, a successful return will occur
 * with r = KRB_REALM (also defined in "krb.h").
 *
 * NOTE: for archaic & compatibility reasons, this routine will only return
 * valid results when n = 1.
 *
 * For the format of the KRB_CONF file, see comments describing the routine
 * krb_get_krbhst().
 */

static int
krb_get_lrealm_f(char *r, int n, const char *fname)
{
    FILE *f;
    int ret = KFAILURE;
    f = fopen(fname, "r");
    if(f){
	char buf[REALM_SZ];
	if(fgets(buf, sizeof(buf), f)){
	    char *p = buf + strspn(buf, " \t");
	    p[strcspn(p, " \t\r\n")] = 0;
	    p[REALM_SZ - 1] = 0;
	    strcpy(r, p);
	    ret = KSUCCESS;
	}
	fclose(f);
    }
    return ret;
}

static const char *no_default_realm = "NO.DEFAULT.REALM";

int
krb_get_lrealm(char *r, int n)
{
    int i;
    char file[MaxPathLen];

    if (n > 1)
	return(KFAILURE);		/* Temporary restriction */

    for (i = 0; krb_get_krbconf(i, file, sizeof(file)) == 0; i++)
	if (krb_get_lrealm_f(r, n, file) == KSUCCESS)
	    return KSUCCESS;

    /* When nothing else works try default realm */
    if (n == 1) {
      char *t = krb_get_default_realm();

      if (strcmp(t, no_default_realm) == 0)
	return KFAILURE;	/* Can't figure out default realm */

      strcpy(r, t);
      return KSUCCESS;
    }
    else
	return(KFAILURE);
}

/* Returns local realm if that can be figured out else NO.DEFAULT.REALM */
char *
krb_get_default_realm(void)
{
  static char local_realm[REALM_SZ]; /* Local kerberos realm */

  if (local_realm[0] == 0)
    {
      char *t, hostname[MaxHostNameLen];

      strcpy(local_realm, no_default_realm); /* Provide default */

      gethostname(hostname, sizeof(hostname));
      t = krb_realmofhost(hostname);
      if (t && strcmp(t, no_default_realm) != 0)
	strncpy(local_realm, t, REALM_SZ);
    }
  return local_realm;
}
