#include "krb_locl.h"

RCSID("$Id: rd_safe.c,v 1.13 1996/05/15 03:39:54 joda Exp $");

/* application include files */
#include "lsb_addr_comp.h"

/*
 * krb_rd_safe() checks the integrity of an AUTH_MSG_SAFE message.
 * Given the message received, "in", the length of that message,
 * "in_length", the "key" to compute the checksum with, and the
 * network addresses of the "sender" and "receiver" of the message,
 * krb_rd_safe() returns RD_AP_OK if message is okay, otherwise
 * some error code.
 *
 * The message data retrieved from "in" is returned in the structure
 * "m_data".  The pointer to the application data (m_data->app_data)
 * refers back to the appropriate place in "in".
 *
 * See the file "mk_safe.c" for the format of the AUTH_MSG_SAFE
 * message.  The structure containing the extracted message
 * information, MSG_DAT, is defined in "krb.h".
 */

int32_t
krb_rd_safe(void *in, u_int32_t in_length, des_cblock *key, 
	    struct sockaddr_in *sender, struct sockaddr_in *receiver, 
	    MSG_DAT *m_data)
{
    unsigned char *p = (unsigned char*)in, *start;

    unsigned char pvno, type;
    int little_endian;
    struct timeval tv;
    u_int32_t src_addr;
    des_cblock checksum[2];
    int delta_t;
    

    pvno = *p++;
    if(pvno != KRB_PROT_VERSION)
	return RD_AP_VERSION;
    
    type = *p++;
    little_endian = type & 1;
    type &= ~1;
    if(type != AUTH_MSG_SAFE)
	return RD_AP_MSG_TYPE;

    start = p;
    
    p += get_int(p, &m_data->app_length, 4, little_endian);
    
    if(m_data->app_length + 31 > in_length)
	return RD_AP_MODIFIED;
    
    m_data->app_data = p;

    p += m_data->app_length;

    m_data->time_5ms = *p++;

    p += get_int(p, &src_addr, 4, 0); 

    /* addresses are always sent in network byteorder, but since
       get_int makes right, we get this wrong on little endian
       machines, so we must make right again */

    src_addr = htonl(src_addr);

    if (!krb_equiv(src_addr, sender->sin_addr.s_addr))
        return RD_AP_MODIFIED;

    p += get_int(p, &m_data->time_sec, 4, little_endian);
    m_data->time_sec = lsb_time(m_data->time_sec, sender, receiver);
    
    gettimeofday(&tv, NULL);

    delta_t = abs((int)((long) tv.tv_sec - m_data->time_sec));
    if (delta_t > CLOCK_SKEW) return RD_AP_TIME;

    /*
     * caller must check timestamps for proper order and replays, since
     * server might have multiple clients each with its own timestamps
     * and we don't assume tightly synchronized clocks.
     */

    des_quad_cksum((des_cblock*)start, checksum, p - start, 2, key);

    if(HOST_BYTE_ORDER != little_endian)
	swap_u_16(checksum);
    
    if(memcmp(checksum, p, sizeof(checksum)))
	return RD_AP_MODIFIED;

    return KSUCCESS;
}
