#include "krb_locl.h"

RCSID("$Id: mk_safe.c,v 1.11 1996/05/15 03:39:48 joda Exp $");

/* application include files */
#include "lsb_addr_comp.h"



/*
 * krb_mk_safe() constructs an AUTH_MSG_SAFE message.  It takes some
 * user data "in" of "length" bytes and creates a packet in "out"
 * consisting of the user data, a timestamp, and the sender's network
 * address, followed by a checksum computed on the above, using the
 * given "key".  The length of the resulting packet is returned.
 *
 * The "out" packet consists of:
 *
 * Size			Variable		Field
 * ----			--------		-----
 *
 * 1 byte		KRB_PROT_VERSION	protocol version number
 * 1 byte		AUTH_MSG_SAFE |		message type plus local
 *			HOST_BYTE_ORDER		byte order in low bit
 *
 * ===================== begin checksum ================================
 * 
 * 4 bytes		length			length of user data
 * length		in			user data
 * 1 byte		msg_time_5ms		timestamp milliseconds
 * 4 bytes		sender->sin.addr.s_addr	sender's IP address
 *
 * 4 bytes		msg_time_sec or		timestamp seconds with
 *			-msg_time_sec		direction in sign bit
 *
 * ======================= end checksum ================================
 *
 * 16 bytes		big_cksum		quadratic checksum of
 *						above using "key"
 */

int32_t
krb_mk_safe(void *in, void *out, u_int32_t length, des_cblock *key, 
	    struct sockaddr_in *sender, struct sockaddr_in *receiver)
{
    unsigned char * p = (unsigned char*)out;
    struct timeval tv;
    unsigned char *start;
    u_int32_t src_addr;

    p += put_int(KRB_PROT_VERSION, p, 1);
    p += put_int(AUTH_MSG_SAFE, p, 1);
    
    start = p;

    p += put_int(length, p, 4);

    memcpy(p, in, length);
    p += length;
    
    gettimeofday(&tv, NULL);

    *p++ = tv.tv_usec/5000; /* 5ms */
    
    /* see comment in rd_safe */
    src_addr = htonl(sender->sin_addr.s_addr);
    p += put_int(src_addr, p, 4);

    p += put_int(lsb_time(tv.tv_sec, sender, receiver), p, 4);

    des_quad_cksum((des_cblock*)start, (des_cblock*)p, p - start, 2, key);
    if(HOST_BYTE_ORDER)
	swap_u_16(p);
    p += 16;

    return p - (unsigned char*)out;
}
