/*
 * This software may now be redistributed outside the US.
 *
 */

/* 
  Copyright (C) 1989 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */

#include "krb_locl.h"

RCSID("$Id: get_admhst.c,v 1.7 1996/03/25 13:09:15 bg Exp $");

/*
 * Given a Kerberos realm, find a host on which the Kerberos database
 * administration server can be found.
 *
 * krb_get_admhst takes a pointer to be filled in, a pointer to the name
 * of the realm for which a server is desired, and an integer n, and
 * returns (in h) the nth administrative host entry from the configuration
 * file (KRB_CONF, defined in "krb.h") associated with the specified realm.
 *
 * On error, get_admhst returns KFAILURE. If all goes well, the routine
 * returns KSUCCESS.
 *
 * For the format of the KRB_CONF file, see comments describing the routine
 * krb_get_krbhst().
 *
 * This is a temporary hack to allow us to find the nearest system running
 * a Kerberos admin server.  In the long run, this functionality will be
 * provided by a nameserver.
 */

static int
krb_get_admhst_f(char *h, char *r, int n, const char *fname)
{
  FILE *cnffile;
  char tr[REALM_SZ];
  char linebuf[BUFSIZ];
  char scratch[64];
  register int i;

  if ((cnffile = fopen(fname,"r")) == NULL)
    return(KFAILURE);

  if (fgets(linebuf, BUFSIZ, cnffile) == NULL) {
    /* error reading */
    (void) fclose(cnffile);
    return(KFAILURE);
  }
  if (!strchr(linebuf, '\n')) {
    /* didn't all fit into buffer, punt */
    (void) fclose(cnffile);
    return(KFAILURE);
  }
  for (i = 0; i < n; ) {
    /* run through the file, looking for admin host */
    if (fgets(linebuf, BUFSIZ, cnffile) == NULL) {
      (void) fclose(cnffile);
      return(KFAILURE);
    }
    /* need to scan for a token after 'admin' to make sure that
       admin matched correctly */
    if (sscanf(linebuf, "%s %s admin %s", tr, h, scratch) != 3)
      continue;
    if (!strcmp(tr,r))
      i++;
  }
  (void) fclose(cnffile);
  return(KSUCCESS);
}

int
krb_get_admhst(char *h, char *r, int n)
{
  static const char *const files[] = KRB_CNF_FILES;
  int i;
  
  const char *dir = getenv("KRBCONFDIR");

  /* First try user specified file */
  if (dir != 0)
    {
      char fname[MaxPathLen];
      strncpy(fname, dir, sizeof(fname));
      strncat(fname, "/krb.conf", sizeof(fname));
      if (krb_get_admhst_f(h, r, n, fname) == KSUCCESS)
	return KSUCCESS;
    }

  for (i = 0; files[i] != 0; i++)
    if (krb_get_admhst_f(h, r, n, files[i]) == KSUCCESS)
      return KSUCCESS;

  /* If nothing else works try kerberos.REALMNAME if it exists */
  strncpy(h, "kerberos.", MaxHostNameLen);
  strncat(h, r, MaxHostNameLen);
  if (gethostbyname(h) != 0)
    return KSUCCESS;
  else
    return KFAILURE;
}
