/*
 * Copyright (c) 1995, 1996 Kungliga Tekniska Hgskolan (Royal Institute
 * of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "krb_locl.h"
#include <klog.h>

RCSID("$Id: logging.c,v 1.9 1996/11/17 20:47:46 joda Exp $");

struct krb_log_facility{
    char filename[MaxPathLen]; 
    FILE *file; 
    krb_log_func_t func;
};


static int
log_with_timestamp_and_nl(FILE *file, const char *format, va_list args)
{
    time_t now;
    if(file == NULL)
	return KFAILURE;
    time(&now);
    fputs(krb_stime(&now), file);
    fputs(": ", file);
    vfprintf(file, format, args);
    fputs("\n", file);
    fflush(file);
    return KSUCCESS;
}

static int
log_tty(FILE *f, const char *format,  va_list args)
{
    if (f != NULL && isatty(fileno(f)))
	vfprintf(f, format, args);
    return KSUCCESS;
}

static int
log_not(FILE *f, const char *format,  va_list args)
{
    return KSUCCESS;
}

int
krb_vlogger(struct krb_log_facility *f, const char *format, va_list args)
{
    FILE *file = NULL;
    int ret;

    if (f->file != NULL)
	file = f->file;
    else if (f->filename && f->filename[0])
	file = fopen(f->filename, "a");

    ret = f->func(file, format, args);

    if (file != f->file)
	fclose(file);
    return ret;
}

int
krb_logger(struct krb_log_facility *f, const char *format, ...)
{
    va_list args;
    int ret;
    va_start(args, format);
    ret = krb_vlogger(f, format, args);
    va_end(args);
    return ret;
}

/*
 * If FILE * is given log to it, otherwise, log to filename. When
 * given a file name the file is opened and closed for each log
 * record.
 */
int
krb_openlog(struct krb_log_facility *f,
	    char *filename,
	    FILE *file,
	    krb_log_func_t func)
{
    strcpy(f->filename, filename);
    f->file = file;
    f->func = func;
    return KSUCCESS;
}

/* ------------------------------------------------------------
   Compatibility functions from log.c
   ------------------------------------------------------------ */

/* static facilities for krb_log() and klog() */
static struct krb_log_facility file_log[2];

static int KRBLOG_inited[2] = { 0, 0 };	/* Yuk! */

/*
 * set_logfile() changes the name of the file to which
 * messages are logged. If set_logfile() is not called,
 * the logfile defaults to KRBLOG, defined in "krb.h".
 */

void
set_logfile(char *filename)
{
    krb_openlog(&file_log[0], filename, NULL, log_with_timestamp_and_nl);
    KRBLOG_inited[0] = 1;
}

/*
 * krb_log() is used to add entries to the logfile (see set_logfile()).
 *
 * The log entry consists of a timestamp and the given arguments
 * printed according to the given "format".
 *
 * The log file is opened and closed for each log entry.
 */

void
krb_log(const char *format, ...)
{
    va_list args;

    if (!KRBLOG_inited[0])
        set_logfile(KRBLOG);	/* Horrible default! */

    va_start(args, format);
    krb_vlogger(&file_log[0], format, args);
    va_end(args);
}

/* ------------------------------------------------------------
   Compatibility functions from klog.c
   ------------------------------------------------------------ */

/*
 * klog() is used to add entries to the logfile (see kset_logfile()
 * below).
 *
 * The log entry consists of a timestamp and the given arguments
 * printed according to the given "format" string.
 *
 * The log file is opened and closed for each log entry.
 *
 * CHANGE: the type is always ignored
 *
 * If the given log type "type" is unknown, or if the log file
 * cannot be opened, no entry is made to the log file.
 *
 * The return value is always a pointer to the formatted log
 * text string "logtxt".
 */

static char logtxt[1024];

char *
klog(int type, const char *format, ...)
{
    va_list ap;

    va_start(ap, format);
    vsprintf(logtxt, format, ap);
    va_end(ap);
    
    if(!KRBLOG_inited[1])
	kset_logfile(KRBLOG);
    krb_logger(&file_log[1], "%s", logtxt);
    
    return logtxt;
}

void
kset_logfile(char *filename)
{
    krb_openlog(&file_log[1], filename, NULL, log_with_timestamp_and_nl);
    KRBLOG_inited[1] = 1;
}


/* ------------------------------------------------------------
   Compatibility functions from warning.c
   ------------------------------------------------------------ */

static struct krb_log_facility std_log = { "/dev/tty", stderr, log_not };

/*
 *
 */
void
krb_set_warnfn (krb_warnfn_t newfunc)
{
    std_log.func =  newfunc;
}

/*
 *
 */
krb_warnfn_t
krb_get_warnfn (void)
{
    return std_log.func;
}

/*
 * Log warnings to stderr if it's a tty.
 */
void
krb_warning (const char *format, ...)
{
    va_list args;
    
    va_start(args, format);
    krb_vlogger(&std_log, format, args);
    va_end(args);
}
