/*
 * This software may now be redistributed outside the US.
 *
 */

/* 
  Copyright (C) 1989 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */

#include "krb_locl.h"

RCSID("$Header: /afs/nada.kth.se/src/packages/kth-krb/SourceRepository/krb4/lib/krb/get_krbhst.c,v 1.4 1995/10/24 10:14:21 bg Exp $");

/*
 * Given a Kerberos realm, find a host on which the Kerberos authenti-
 * cation server can be found.
 *
 * krb_get_krbhst takes a pointer to be filled in, a pointer to the name
 * of the realm for which a server is desired, and an integer, n, and
 * returns (in h) the nth entry from the configuration file (KRB_CONF,
 * defined in "krb.h") associated with the specified realm.
 *
 * On end-of-file, krb_get_krbhst returns KFAILURE.  If n=1 and the
 * configuration file does not exist, krb_get_krbhst will return KRB_HOST
 * (also defined in "krb.h").  If all goes well, the routine returnes
 * KSUCCESS.
 *
 * The KRB_CONF file contains the name of the local realm in the first
 * line (not used by this routine), followed by lines indicating realm/host
 * entries.  The words "admin server" following the hostname indicate that 
 * the host provides an administrative database server.
 *
 * For example:
 *
 *	ATHENA.MIT.EDU
 *	ATHENA.MIT.EDU kerberos-1.mit.edu admin server
 *	ATHENA.MIT.EDU kerberos-2.mit.edu
 *	LCS.MIT.EDU kerberos.lcs.mit.edu admin server
 *
 * This is a temporary hack to allow us to find the nearest system running
 * kerberos.  In the long run, this functionality will be provided by a
 * nameserver.
 */

static int
krb_get_krbhst_f(char *h, char *r, int n, const char *fname)
{
  FILE *cnffile;
  char tr[REALM_SZ];
  char linebuf[BUFSIZ];
  register int i;

  if ((cnffile = fopen(fname,"r")) == NULL)
    return(KFAILURE);

  if (fscanf(cnffile,"%s",tr) == EOF)
    {
      (void) fclose(cnffile);
      return(KFAILURE);
    }

  /* run through the file, looking for the nth server for this realm */
  for (i = 1; i <= n;)
    {
      if (fgets(linebuf, BUFSIZ, cnffile) == NULL) {
	(void) fclose(cnffile);
	return(KFAILURE);
      }
      if (sscanf(linebuf, "%s %s", tr, h) != 2)
	continue;
      if (!strcmp(tr,r))
	i++;
    }
  (void) fclose(cnffile);
  return(KSUCCESS);
}

int
krb_get_krbhst(char *h, char *r, int n)
{
  static const char *const files[] = KRB_CNF_FILES;
  int i;
  
  const char *dir = getenv("KRBCONFDIR");

  /* First try user specified file */
  if (dir != 0)
    {
      char fname[MaxPathLen];
      strncpy(fname, dir, sizeof(fname));
      strncat(fname, "/krb.conf", sizeof(fname));
      if (krb_get_krbhst_f(h, r, n, fname) == KSUCCESS)
	return KSUCCESS;
    }

  for (i = 0; files[i] != 0; i++)
    if (krb_get_krbhst_f(h, r, n, files[i]) == KSUCCESS)
      return KSUCCESS;

  /* If nothing else works try kerberos.REALMNAME if it exists */
  strncpy(h, "kerberos.", MaxHostNameLen);
  strncat(h, r, MaxHostNameLen);
  if (gethostbyname(h) != 0)
    return KSUCCESS;
  else
    return KFAILURE;
}
