/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  Module Initialization
**  _________________________________________________________________
*/

/*
 *  Per-module initialization
 */
void ssl_init_Module(server_rec *s, pool *p)
{
    server_rec *s2;
    SSLSrvConfigRec *pConfig;

    ssl_ModConfig->nInitCount++;

    ssl_ModConfig->rCtx.pServ = s;
    ssl_ModConfig->rCtx.pPool = p;

    /* 
     *  try to fix the configuration and open the dedicated SSL
     *  logfile as early as possible 
     */
    for (s2 = s; s2 != NULL; s2 = s2->next) {
        pConfig = mySrvConfig(s2);

        /* Fix up stuff that may not have been set */
        if (pConfig->bEnabled == UNSET)
            pConfig->bEnabled = FALSE;
        if (pConfig->nVerifyClient == VERIFY_UNSET)
            pConfig->nVerifyClient = VERIFY_NONE;
        if (pConfig->bFakeBasicAuth == UNSET)
            pConfig->bFakeBasicAuth = FALSE;
        if (pConfig->nPassPhraseDialogType == SSL_PPTYPE_UNSET)
            pConfig->nPassPhraseDialogType = SSL_PPTYPE_BUILTIN;

        /* Open the dedicated SSL logfile */
        ssl_log_open(s2, p);
    }

    if (ssl_ModConfig->nInitCount == 1)
        ssl_log(s, SSL_LOG_INFO, "Init: 1st startup round (still not detached)");
    else if (ssl_ModConfig->nInitCount == 2)
        ssl_log(s, SSL_LOG_INFO, "Init: 2nd startup round (already detached)");
    else
        ssl_log(s, SSL_LOG_INFO, "Init: %d%s restart round (already detached)",
                ssl_ModConfig->nInitCount-2, (ssl_ModConfig->nInitCount-2) == 1 ? "st" : "nd");

    /*  
     *  The Apache API does two initializations of modules.  While we want to
     *  initialize only in the second one we have to do the SSLeay
     *  initialization and the Pass Phrase dialog in the first one.  Because
     *  between the first and the second round Apache detaches from the
     *  terminal.
     */
    if (ssl_ModConfig->nInitCount == 1) {
        /*
         *  initialize SSLeay (already needed for the pass phrase dialog)
         */
        ssl_log(s, SSL_LOG_INFO, "Init: Initializing SSLeay library");
        __SSLeay SSL_load_error_strings();
        __SSLeay SSLeay_add_ssl_algorithms();

        /*
         * Handle the pass phrases
         */
        ssl_pphrase_Handle(s, p);

        /* 
         * immediately return because we perform the real
         * initialization in the 2nd startup round only.
         */
        return;
    }

    /*
     * Warn the user that he should use the session cache.
     * But we can operate without it, of course.
     */
    if (!ssl_ModConfig->nSessionCacheMode == SSL_SCMODE_UNSET) {
        ssl_log(s, SSL_LOG_WARN, 
                "Init: Session Cache is not configured [hint: SSLSessionCache]");
        ssl_ModConfig->nSessionCacheMode = SSL_SCMODE_NONE;
    }

    /*
     *  initialize the mutex handling and session caching
     */
    ssl_mutex_init(s, p);
    ssl_scache_init(s, p);

    /*
     *  pre-generate the temporary RSA key
     */
    if (ssl_ModConfig->pRSATmpKey == NULL) {
        ssl_log(s, SSL_LOG_INFO, "Init: Generating temporary (512 bit) RSA key");
#if SSLEAY_VERSION_NUMBER >= 0x0900
        ssl_ModConfig->pRSATmpKey = __SSLeay RSA_generate_key(512, RSA_F4, NULL, NULL);
#else
        ssl_ModConfig->pRSATmpKey = __SSLeay RSA_generate_key(512, RSA_F4, NULL);
#endif
        if (ssl_ModConfig->pRSATmpKey == NULL) {
            ssl_log(s, SSL_LOG_ERROR, "Init: Failed to generate temporary (512 bit) RSA key");
            ssl_die();
        }
    }

    /*
     *  initialize servers
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Initializing servers for SSL");
    for (; s != NULL; s = s->next) {
        pConfig = mySrvConfig(s);

        if (pConfig->bEnabled)
            ssl_log(s, SSL_LOG_INFO, 
                    "Init: Configuring server %s:%d for SSL protocol",
                    s->server_hostname, s->port);

        /* If we are disabled skip this server but
           make sure the port is initialized correctly */
        if (!pConfig->bEnabled) {
            if (!s->port)
                s->port = DEFAULT_HTTP_PORT;
            continue;
        }
        else {
            if (!s->port)
                s->port = DEFAULT_HTTPS_PORT;
        }

        /* 
         * Now check for important parameters and the
         * possibility that the user forgot to set them.
         */
        if (pConfig->nSessionCacheTimeout == 0) {
            ssl_log(s, SSL_LOG_ERROR,
                    "Init: No SSL Session Cache Timeout set [hint: SSLSessionCacheTimeout]");
            ssl_die();
        }
        if (!pConfig->szCertificateFile) {
            ssl_log(s, SSL_LOG_ERROR,
                    "Init: No SSL Certificate set for server %s:%d [hint: SSLCertificateFile]",
                    s->server_hostname, s->port);
            ssl_die();
        }

        /*
         * Read the server certificate and key
         */
        ssl_init_GetCertAndKey(s, p, pConfig);
    }

    /*
     * Optionally wipe out the PassPhrase if
     * `SSLPassPhraseCaching off' is active
     */
#if 0 /* XXX */
    pConfig = mySrvConfig(ssl_ModConfig->rCtx.pServ);
    if (ssl_ModConfig->bPassPhrase) {
        if (   ssl_ModConfig->szPassPhrase != NULL
            && pConfig->bPassPhraseCaching == FALSE) {
            memset(buf, 0, (unsigned int)strlen(ssl_ModConfig->szPassPhrase));
            free(ssl_ModConfig->szPassPhrase);
            ssl_ModConfig->szPassPhrase = NULL;
            ssl_log(ssl_ModConfig->rCtx.pServ, SSL_LOG_INFO, "Init: Wiping out pass phrase from memory");
        }
        else {
            ssl_log(ssl_ModConfig->rCtx.pServ, SSL_LOG_INFO, "Init: Caching pass phrase in memory");
        }
    }
#endif

    /*
     *  Announce mod_ssl and SSLeay in HTTP Server field
     *  as ``mod_ssl/X.X.X SSLeay/X.X.X''
     */
    ap_add_version_component(ssl_var_lookup(p, NULL, NULL, NULL, "SSL_VERSION_INTERFACE"));
    ap_add_version_component(ssl_var_lookup(p, NULL, NULL, NULL, "SSL_VERSION_LIBRARY"));

    return;
}

void ssl_init_Child(server_rec *s, pool *p)
{
     /* open the mutex lockfile */
     ssl_mutex_open(s, p);
     return;
}

/*
 * Read the SSL Server Certificate and Key
 */
void ssl_init_GetCertAndKey(server_rec *s, pool *p, SSLSrvConfigRec *pConfig)
{
    int nVerify;
    char *cpServerName;
    RSA **ppRSA;
    X509 **ppX509;

    cpServerName = ap_psprintf(p, "%s:%d", s->server_hostname, s->port);

    /*  
     *  Check for problematic re-initializations
     */
    if (pConfig->px509Certificate) {
        ssl_log(s, SSL_LOG_ERROR, 
                "Init: Illegal attempt to re-initialise SSL for server %s",
                cpServerName);
        ssl_die();
    }

    /*
     *  Calculate SSLeay verify type
     */
    nVerify = SSL_VERIFY_NONE;
    switch (pConfig->nVerifyClient) {
        case VERIFY_REQUIRE:
            nVerify |= SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
            break;
        case VERIFY_OPTIONAL:
        case VERIFY_OPTIONAL_NO_CA:
            nVerify |= SSL_VERIFY_PEER;
            break;
        default:
            break;
    }

    /*
     *  Create new SSL context and configure callbacks
     */
    ssl_log(s, SSL_LOG_INFO, 
            "Init: Creating new SSL context for server %s", cpServerName);
    pConfig->pSSLCtx = __SSLeay SSL_CTX_new(__SSLeay SSLv23_server_method());
    __SSLeay SSL_CTX_set_verify(pConfig->pSSLCtx, nVerify, 
                 ssl_callback_SSLVerify);
    __SSLeay SSL_CTX_sess_set_new_cb(pConfig->pSSLCtx, 
                 ssl_callback_NewSessionCacheEntry);
    __SSLeay SSL_CTX_sess_set_get_cb(pConfig->pSSLCtx, 
                 ssl_callback_GetSessionCacheEntry);
    __SSLeay SSL_CTX_sess_set_remove_cb(pConfig->pSSLCtx, 
                 ssl_callback_DelSessionCacheEntry);
    __SSLeay SSL_CTX_set_tmp_rsa_callback(pConfig->pSSLCtx, 
                 ssl_callback_TmpRSA);

    /*
     *  Configure required SSL Ciphers 
     */
    ssl_log(s, SSL_LOG_INFO, 
            "Init: Configuring permitted SSL ciphers for server %s",
            cpServerName);
    if (pConfig->szCipherSuite != NULL) {
        if (!__SSLeay SSL_CTX_set_cipher_list(pConfig->pSSLCtx, 
                                              pConfig->szCipherSuite)) {
            ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                    "Unable to configure permitted SSL ciphers");
            ssl_die();
        }
    }

    /*
     * Configure SSL Client verification
     */
    ssl_log(s, SSL_LOG_INFO, 
            "Init: Configuring client verification paths for server %s", cpServerName);
    if (   ( (   pConfig->szCACertificateFile 
              || pConfig->szCACertificatePath)
           && !__SSLeay SSL_CTX_load_verify_locations(pConfig->pSSLCtx,
                                                      pConfig->szCACertificateFile,
                                                      pConfig->szCACertificatePath) )
        || !__SSLeay SSL_CTX_set_default_verify_paths(pConfig->pSSLCtx) ) {
        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                "Unable to configure SSL verify locations");
        ssl_die();
    }

    /* 
     *  Additionally load certificates which will be sent to the
     *  client on `SSLv3 write certificate request A'. This is
     *  optionally used by the clients to speedup the server
     *  authentication when SSLv3 certificate chaining is ised.
     *  Here the client loads intermediate certificates in the
     *  chain from the server. This defaults to
     *  SSLCACertificateFile. 
     */
    if (   pConfig->szCACertificateReqFile != NULL
        || pConfig->szCACertificateFile    != NULL)
        ssl_log(s, SSL_LOG_INFO, "Init: Reading CA certification file for server %s",
                cpServerName);
    if (pConfig->szCACertificateReqFile != NULL)
        __SSLeay SSL_CTX_set_client_CA_list(pConfig->pSSLCtx,
            __SSLeay SSL_load_client_CA_file(pConfig->szCACertificateReqFile));
    else
        if (pConfig->szCACertificateFile != NULL)
            __SSLeay SSL_CTX_set_client_CA_list(pConfig->pSSLCtx,
                __SSLeay SSL_load_client_CA_file(pConfig->szCACertificateFile));

    /*
     *  Configure server certificate
     */
    ssl_log(s, SSL_LOG_INFO, 
            "Init: Configuring certificate for server %s", cpServerName);
    if ((ppX509 = (X509 **)ssl_ds_table_get(ssl_ModConfig->tPublicCert, cpServerName)) == NULL) {
        ssl_log(s, SSL_LOG_ERROR, 
                "Ops, can't find SSL certificate for server %s", cpServerName);
        ssl_die();
    }
    pConfig->px509Certificate = *ppX509;

    /*
     *  Configure server private key 
     */
    ssl_log(s, SSL_LOG_INFO, 
            "Init: Configuring private key for server %s", cpServerName);
    if ((ppRSA = (RSA **)ssl_ds_table_get(ssl_ModConfig->tPrivateKey, cpServerName)) == NULL) {
        ssl_log(s, SSL_LOG_ERROR, 
                "Ops, can't find SSL certificate for server %s", cpServerName);
        ssl_die();
    }
    pConfig->prsaKey = *ppRSA;

    return;
}


