/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/* Once again (see getrusage.c), AT&T doesn't believe in resource statistics.
   By using the weird waitid() call I can at least get at the childs
   cpu usage information, but the rest is currently not available, sigh.. */

/* using SVR4 headers here!! */
#include <sys/types.h>
#include <sys/time.h>
#include <wait.h>

/* from bsd sys/resource.h file */

/*
 * Resource utilization information.
 */

#define	RUSAGE_SELF	0
#define	RUSAGE_CHILDREN	-1

struct	rusage {
	struct timeval ru_utime;	/* user time used */
	struct timeval ru_stime;	/* system time used */
	long	ru_maxrss;		/* max resident set size */
#define	ru_first	ru_ixrss
	long	ru_ixrss;		/* integral shared memory size */
	long	ru_idrss;		/* integral unshared data " */
	long	ru_isrss;		/* integral unshared stack " */
	long	ru_minflt;		/* page reclaims */
	long	ru_majflt;		/* page faults */
	long	ru_nswap;		/* swaps */
	long	ru_inblock;		/* block input operations */
	long	ru_oublock;		/* block output operations */
	long	ru_msgsnd;		/* messages sent */
	long	ru_msgrcv;		/* messages received */
	long	ru_nsignals;		/* signals received */
	long	ru_nvcsw;		/* voluntary context switches */
	long	ru_nivcsw;		/* involuntary " */
#define	ru_last		ru_nivcsw
};

int
wait4 (int pid, int *stat, int wopt, struct rusage *ru)
{
  int rc;
  idtype_t id;
  int arg;
  int options;
  siginfo_t si;

  if (pid == -1)
    {
      id = P_ALL;
      arg = 0;
    }
  else if (pid == 0)
    {
      id = P_PGID;
      arg = getpgrp ();
    }
  else if (pid > 0)
    {
      id = P_PID;
      arg = pid;
    }
  else if (pid < 0)
    {
      id = P_PGID;
      arg = -pid;
    }

  options = WEXITED | WTRAPPED;
  if (wopt & WUNTRACED)
    options |= WSTOPPED;
  if (wopt & WNOHANG)
    options |= WNOHANG;

  rc = waitid (id, arg, &si, options);
  
  if (rc < 0)
    return rc;

  if (stat)
    *stat = si.si_status;

  if (ru)
    {
      int t;

      bzero (ru, sizeof (*ru));
      t = si.si_utime;
      ru->ru_utime.tv_usec = (t % CLK_TCK) * (1000000 / CLK_TCK);
      ru->ru_utime.tv_sec  = t / CLK_TCK;
      t = si.si_stime;
      ru->ru_stime.tv_usec = (t % CLK_TCK) * (1000000 / CLK_TCK);
      ru->ru_stime.tv_sec  = t / CLK_TCK;
    }

  return si.si_pid;
}

#ifdef TEST
main()
{
  int st;
  int rc;
  struct rusage ru;

  if (fork() == 0)
    {
      int i;
      for (i = 0; i < 10000000; i++) ;
      printf ("%d terminating.\n", getpid());
      exit (111);
    }

  if (fork() == 0)
    {
      int i;
      for (i = 0; i < 50000000; i++) ;
      printf ("%d terminating.\n", getpid());
      exit (222);
    }

  rc = wait4 (-1, &st, WNOHANG | WUNTRACED, &ru);

  printf ("rc = %d, st = %d\n", rc, st);

  do
    {
      rc = wait4 (-1, &st, WUNTRACED, &ru);
      printf ("rc = %d, st = %d\n", rc, st);
      printf ("ru.u_utime = %d.%06d\n", ru.ru_utime.tv_sec, ru.ru_utime.tv_usec);
      printf ("ru.u_stime = %d.%06d\n", ru.ru_stime.tv_sec, ru.ru_stime.tv_usec);
    }
  while (rc > 0);
}

#endif
