/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/* This maps AT&T xstat structures into BSD stat structures. There's a
   little kludge that unconditionally set the blksize field of a 
   ufs filesystem to 8k. The (2.03c) kernel returns 1k blocksizes in that
   case, and this causes gross performance penalties (for example for
   stdio which queries the blksize field to achieve optimal throughput). */

#include <sys/types.h>
#include <sys/time.h>

#define STAT_VERS 2
#define _ST_FSTYPSZ 16

struct	att_stat {
	dev_t	st_dev;
	long	st_pad1[3];	/* reserved for network id */
	ino_t	st_ino;
	mode_t	st_mode;
	nlink_t st_nlink;
	uid_t 	st_uid;
	gid_t 	st_gid;
	dev_t	st_rdev;
	long	st_pad2[2];
	off_t	st_size;
	long	st_pad3;	/* future off_t expansion */
	timestruc_t st_atim;	
	timestruc_t st_mtim;	
	timestruc_t st_ctim;	
	long	st_blksize;
	long	st_blocks;
	char	st_fstype[_ST_FSTYPSZ];
	long	st_pad4[8];	/* expansion area */
};

struct bsd_stat
{
	dev_t	st_dev;			/* inode's device */
	ino_t	st_ino;			/* inode's number */
	mode_t	st_mode;		/* inode protection mode */
	nlink_t	st_nlink;		/* number of hard links */
	uid_t	st_uid;			/* user ID of the file's owner */
	gid_t	st_gid;			/* group ID of the file's group */
	dev_t	st_rdev;		/* device type */
	off_t	st_size;		/* file size, in bytes */
	time_t	st_atime;		/* time of last access */
	long	st_spare1;
	time_t	st_mtime;		/* time of last data modification */
	long	st_spare2;
	time_t	st_ctime;		/* time of last file status change */
	long	st_spare3;
	long	st_blksize;		/* optimal blocksize for I/O */
	long	st_blocks;		/* blocks allocated for file */
	u_long	st_flags;		/* user defined flags for file */
	u_long	st_gen;			/* file generation number */
};

int
stat (const char *name, struct bsd_stat *stb)
{
  struct att_stat at;

  if (_xstat (STAT_VERS, name, &at) < 0)
    return -1;

  stb->st_dev = at.st_dev;
  stb->st_ino = at.st_ino;
  stb->st_mode = at.st_mode;
  stb->st_nlink = at.st_nlink;
  stb->st_uid = at.st_uid;
  stb->st_gid = at.st_gid;
  stb->st_rdev = at.st_rdev;
  stb->st_size = at.st_size;
  stb->st_atime = at.st_atim.tv_sec;
  stb->st_mtime = at.st_mtim.tv_sec;
  stb->st_ctime = at.st_ctim.tv_sec;
  stb->st_blksize = !strcmp (at.st_fstype, "ufs") ? 8192 : at.st_size;
  stb->st_blocks = at.st_blocks;
  stb->st_flags = 0;
  stb->st_gen = 0;

  return 0;
}

int
fstat (int fd, struct bsd_stat *stb)
{
  struct att_stat at;

  if (_fxstat (STAT_VERS, fd, &at) < 0)
    return -1;

  stb->st_dev = at.st_dev;
  stb->st_ino = at.st_ino;
  stb->st_mode = at.st_mode;
  stb->st_nlink = at.st_nlink;
  stb->st_uid = at.st_uid;
  stb->st_gid = at.st_gid;
  stb->st_rdev = at.st_rdev;
  stb->st_size = at.st_size;
  stb->st_atime = at.st_atim.tv_sec;
  stb->st_mtime = at.st_mtim.tv_sec;
  stb->st_ctime = at.st_ctim.tv_sec;
  stb->st_blksize = !strcmp (at.st_fstype, "ufs") ? 8192 : at.st_size;
  stb->st_blocks = at.st_blocks;
  stb->st_flags = 0;
  stb->st_gen = 0;

  return 0;
}

int
lstat (const char *name, struct bsd_stat *stb)
{
  struct att_stat at;

  if (_lxstat (STAT_VERS, name, &at) < 0)
    return -1;

  stb->st_dev = at.st_dev;
  stb->st_ino = at.st_ino;
  stb->st_mode = at.st_mode;
  stb->st_nlink = at.st_nlink;
  stb->st_uid = at.st_uid;
  stb->st_gid = at.st_gid;
  stb->st_rdev = at.st_rdev;
  stb->st_size = at.st_size;
  stb->st_atime = at.st_atim.tv_sec;
  stb->st_mtime = at.st_mtim.tv_sec;
  stb->st_ctime = at.st_ctim.tv_sec;
  stb->st_blksize = !strcmp (at.st_fstype, "ufs") ? 8192 : at.st_size;
  stb->st_blocks = at.st_blocks;
  stb->st_flags = 0;
  stb->st_gen = 0;

  return 0;
}
