/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/* this implementation uses the priocntl() function. Correct usage of
   this function is not easy to understand (at least it wasn't for me),
   so I just hope set/getpriority work as they should... */

#include <sys/types.h>
#include <sys/priocntl.h>
#include <sys/procset.h>
#include <sys/rtpriocntl.h>
#include <sys/tspriocntl.h>
#include <sys/resource.h>
#include <errno.h>

/*
 * Process priority specifications to get/setpriority.
 */
#define	PRIO_MIN	-20
#define	PRIO_MAX	20

#define	PRIO_PROCESS	0
#define	PRIO_PGRP	1
#define	PRIO_USER	2


static ts_cid = -1;

int
setpriority (int which, int who, int pri)
{
  idtype_t id;
  pcparms_t pcp;
  tsparms_t *tsp;
  tsp = (tsparms_t *) &pcp.pc_clparms[0];

  /* BSD: lower priority is `better', SVR4: higher priority is `better'.
     So just negate the priority argument to get the desired change */
  pri = -pri;

  switch (which)
    {
    case PRIO_PROCESS:
      id = P_PID;
      if (! who)
	who = getpid ();
      break;

    case PRIO_PGRP:
      id = P_PGID;
      if (! who)
	who = getpgid (0);
      break;

    case PRIO_USER:
      id = P_UID;
      if (! who)
	who = getuid ();
      break;

    default:
      errno = EINVAL;
      return -1;
    }

  if (ts_cid == -1)
    {
      pcinfo_t pci;

      /* Note: this will fail if you try to setpriority() a realtime
	       process. But since BSD doesn't offer them, I don't bother
	       supporting them ;-)) */
      strcpy (pci.pc_clname, "TS");
      priocntl (0, 0, PC_GETCID, &pci);
      ts_cid = pci.pc_cid;
    }

  pcp.pc_cid = ts_cid;

  if (priocntl (id, who, PC_GETPARMS, &pcp) != -1)
    {
      /* BSD is more strict in allowing/disallowing setpriority()
	 to succeed. In particular, you cannot raise your priority (ie.
	 lower the pri value in BSD) if you're not root. SVR4 would
	 allow the operation, but silently enforce the maxpri value */
      if (tsp->ts_uprilim < pri && getuid())
	{
	  errno = EPERM;
	  return -1;
	}

      /* by setting both, the limit and the maximum, we get BSD
	 semantics (sort of;-)) */
      tsp->ts_uprilim = 
	tsp->ts_upri = pri;

      return priocntl (id, who, PC_SETPARMS, &pcp);
    }
  else
    return -1;
}

int
getpriority (int which, int who)
{
  idtype_t id;
  pcparms_t pcp;
  tsparms_t *tsp;
  int rc;

  tsp = (tsparms_t *) &pcp.pc_clparms[0];
  
  switch (which)
    {
    case PRIO_PROCESS:
      id = P_PID;
      if (! who)
	who = getpid();
      break;

    case PRIO_PGRP:
      id = P_PGID;
      if (! who)
	who = getpgid (0);
      break;

    case PRIO_USER:
      id = P_UID;
      if (! who)
	who = getuid ();
      break;

    default:
      errno = EINVAL;
      return -1;
    }

  if (ts_cid == -1)
    {
      pcinfo_t pci;

      strcpy (pci.pc_clname, "TS");
      priocntl (0, 0, PC_GETCID, &pci);
      ts_cid = pci.pc_cid;
    }

  pcp.pc_cid = ts_cid;
  errno = 0;
  rc = priocntl (id, who, PC_GETPARMS, &pcp);
  if (rc == -1)
    return -1;
  else
    /* negate the value to get BSD semantics */
    return - tsp->ts_upri;
}

