/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/poll.h>
/* don't forget, inside sys/ (here;-)) we're using AT&T headers, so the
   select macros are *not* defined in sys/types.h */
#include <sys/select.h>
#include <errno.h>

int
select (int nfds, fd_set *in, fd_set *out, fd_set *exc, struct timeval *tv)
{
  struct pollfd *pp, *p;
  int mili;
  int rc;
  int i;

  if (tv)
    mili = (tv->tv_usec + 500) / 1000 + tv->tv_sec * 1000;
  else
    mili = -1;	/* infinit timeout */

  pp = p = 0;

  if (nfds)
    {
      pp = (struct pollfd *) malloc (nfds * sizeof (struct pollfd));
      if (! pp)
	{
	  errno = ENOMEM;
	  return -1;
	}

      for (i = 0, p = pp; i < nfds; i++)
	{
	  p->fd = i;
	  p->events = p->revents = 0;

	  if (in && FD_ISSET (i, in))
	    {
	      p->events |= POLLRDNORM;
	      FD_CLR (i, in);
	    }
	  if (out && FD_ISSET (i, out))
	    {
	      p->events |= POLLWRNORM;
	      FD_CLR (i, out);
	    }
	  if (exc && FD_ISSET (i, exc))
	    {
	      p->events |= POLLRDBAND;
	      FD_CLR (i, exc);
	    }

	  /* if the fd has been used, get next poll slot */
	  if (p->events)
	    p++;
	}
    }

  rc = poll (pp, p - pp, mili);

  /* error or timeout condition */
  if (rc <= 0)
    {
      if (pp)
	free (pp);
      return rc;
    }

  /* ok.. at least one fd is ready for the requested operation. Go thru
     the array again then */
  
  while (--p >= pp)
    {
      if (!(p->revents & (POLLERR | POLLHUP | POLLNVAL)))
	{
	  if (in)
	    {
	      if (p->revents & (POLLIN | POLLRDNORM))
		FD_SET (p->fd, in);
	    }
	  if (out)
	    {
	      if (p->revents & (POLLOUT | POLLWRNORM))
		FD_SET (p->fd, out);
	    }
	  if (exc)
	    {
	      if (p->revents & (POLLRDBAND | POLLWRBAND))
		FD_SET (p->fd, exc);
	    }
	}
    }

  free (pp);

  return rc;
}
