/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/* this implements getdirentries() using the SVR4 getdents() function.
   It's debatable which system call approach is `better', but providing
   this emulation, we can use the straight opendir/readdir,etc functions
   in libc/gen, and this is worth the bit-fiddling done here ;-) */

#include <sys/types.h>
#include <unistd.h>

struct att_dirent {
	u_long		d_ino;		/* "inode number" of entry */
	off_t		d_off;		/* offset of disk directory entry */
	unsigned short	d_reclen;	/* length of this record */
	char		d_name[0];	/* name of file */
};

struct bsd_dirent {
	u_long	d_fileno;		/* file number of entry */
	u_short	d_reclen;		/* length of this record */
	u_short	d_namlen;		/* length of string in d_name */
	char	d_name[0];
};

int
getdirentries (int fd, void *buf, int len, int offset)
{
  int n, arec;
  struct att_dirent *ad;
  struct bsd_dirent *bd;

  lseek (fd, offset, SEEK_SET);
  n = getdents (fd, buf, len);

  if (n < 0)
    return n;

  /* AT&T dirent is larger than BSD, so the given `len' is always sufficient
     to convert in place */
  for (ad = buf, bd = buf, arec = ad->d_reclen; 
       n > 0; 
       n -= arec,
         ad = (struct att_dirent *) ((int)ad + arec),
           bd = (struct bsd_dirent *) ((int)bd + bd->d_reclen),
             arec = ad->d_reclen)
    {
      bd->d_fileno = ad->d_ino;
      bd->d_namlen = strlen (ad->d_name);
      bcopy (ad->d_name, bd->d_name, bd->d_namlen + 1);
      bd->d_reclen = sizeof (struct bsd_dirent) + bd->d_namlen + 1;
      /* readdir() expects long aligned directory structures. If we don't
	 comply, it considers our directories to be corrupt.. */
      if (bd->d_reclen & 03)
	bd->d_reclen += 4-(bd->d_reclen & 03);
    }

  return (int)bd - (int)buf;
}


