/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

/* this implementation uses the F_FREESP command documented in the
   fcntl() man page. Seems SVR4 finally documents a previously
   undocumented command... */

int
ftruncate (int fd, off_t len)
{
  struct flock f;

  f.l_whence = SEEK_SET;	/* absolute positioning */
  f.l_start = len;		/* start from the given offset */
  f.l_len = 0;			/* 0 = until EOF */
  /* is this necessary? The man page doesn't document it, but I don't
     like unknown states... */
  f.l_type = F_WRLCK;		/* we require write privileges for this */

  return fcntl (fd, F_FREESP, &f) < 0 ? -1 : 0;
}


/* this is certainly cheaper in BSD requiring only one system call
   instead of three. But I guess we should be lucky AT&T finally
   grants its customers the privilege to cut files at all... */

int
truncate (const char *path, off_t len)
{
  int fd, rc, err;

  if ((fd = open (path, O_RDWR)) >= 0)
    {
      err = errno;
      rc = ftruncate (fd, len);
      close (fd);
      /* we don't want the errno from close().. */
      errno = err;
    }
  else
    rc = -1;

  return -1;
}
