/* Copyright 1992 by Markus M. Wild.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER ``AS IS'' AND ANY
   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
   BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
   WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
   OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
   IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

#include <sys/types.h>
#include <unistd.h>
#include <sys/fcntl.h>
#include <errno.h>

/* lock operations for flock(2) */
#define	LOCK_SH		0x01		/* shared file lock */
#define	LOCK_EX		0x02		/* exclusive file lock */
#define	LOCK_NB		0x04		/* don't block when locking */
#define	LOCK_UN		0x08		/* unlock file */

/* this is not really a BSD style flock() function, but quite close. The
   POSIX semantics are slightly different than flock() would require, the
   difference is mostly what happens to a lock when a file referring to a
   locked file is closed. POSIX says that then all locks are cleared, BSD
   clears locks only when the *last* file referring to the lock is closed. 

   Another (probably more severe) difference is that you're able to
   get an exclusive (LOCK_EX) lock on files you only opened for reading
   in BSD. This is not possible with fcntl-style locking, exclusive locks
   are only granted on files opened for writing. This will probably create
   some problems if you just compile BSD code and hope it will work... */   

int
flock (int fd, int lock)
{
  struct flock fl;
  int cmd;

  switch (lock)
    {
    case LOCK_SH:
      fl.l_type = F_RDLCK;
      cmd = F_SETLKW;
      break;

    case LOCK_EX:
      fl.l_type = F_WRLCK;
      cmd = F_SETLKW;
      break;

    case LOCK_SH | LOCK_NB:
      fl.l_type = F_RDLCK;
      cmd = F_SETLK;
      break;

    case LOCK_EX | LOCK_NB:
      fl.l_type = F_WRLCK;
      cmd = F_SETLK;
      break;

    case LOCK_UN:
    case LOCK_UN | LOCK_NB:
      fl.l_type = F_UNLCK;
      cmd = F_SETLK;
      break;

    default:
      errno = EINVAL;
      return -1;
    }

  fl.l_whence = SEEK_SET;
  fl.l_start = 0;
  fl.l_len = 0;			/* until eof */
  fl.l_sysid = 0;
  fl.l_pid = 0;

  return fcntl (fd, cmd, &fl);
}

