/* 
 * HISTORY
 * $Log:	mach_os.c,v $
 * Revision 2.2  92/02/15  19:38:49  rpd
 * 	Initial checkin.
 * 	[92/02/15  19:04:07  rpd]
 * 
 *
 * Revision 1.1  1992/01/25  04:31:22  jtv
 * Made it work on mach 3.0 without libmach3_vus and ptrace.
 * Fixed run and quit commands and several other deficiencies.
 * This file is no longer included in other machine dependent files.
 *
 * Revision 1.2  91/01/15  11:15:38  jms
 * 	JMS fixes
 * 	[91/01/15  11:02:06  jms]
 * 
 * Revision 1.1.1.1  91/01/15  11:02:06  jms
 * 	JMS fixes
 * 
 * This code is based on the mach_os.c version
 * that used to run on Mach 2.5 version by
 *
 * Revision 1.1  90/09/07  16:55:51  roy
 * Revision 3.2  90/05/25  16:32:56  kupfer
 */

/* Print and select threads for GDB, the GNU debugger, on Mach 3.0 systems.
   (Most) Mach 3.0 related routines live in this file.
   Copyright (C) 1986, 1987 Free Software Foundation, Inc.
This file is part of GDB.

GDB is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GDB is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GDB; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <errno.h>

#include <mach.h>
#include <servers/netname.h>
#include <mach/message.h>
#include <mach/notify.h>
#include <mach_error.h>
#include <servers/machid.h>
#include <mig_errors.h>
#include <mach/exception.h>

#include <obstack.h>

#include "defs.h"
#include "param.h"
#include "frame.h"
#include "inferior.h"
#include "symtab.h"
#include "value.h"
#include "wait.h"

extern int errno;

WAITTYPE wait_status;

/*
 * This is the MIG ID number of the emulator/server bsd_execve() RPC call.
 *
 * It SHOULD not change, but if it does, gdb `run'
 * command won't work until you fix this define.
 * 
 */
#define MIG_EXEC_SYSCALL_ID		101000


/* If you define this, intercepted bsd server calls will be
 * dumped while waiting the inferior to EXEC the correct
 * program
 */
/* #define DUMP_SYSCALL		/* debugging interceptor */


/* debug() printfs if nonzero.
 * If > 1, DUMP_SYSCALL will dump message contents.
 */
int debug_level = 0;

/* Error handler for mach calls */
#define CHK(str,ret)	\
  do if (ret != KERN_SUCCESS) \
       error ("Gdb %s [%d] %s : %s\n",__FILE__,__LINE__,str,mach_error_string(ret));while(0)

/* Portable ptrace */
int
ptrace (a,b,c,d)
int a,b,c,d;
{
  error ("Loose, Loose! Somebody called ptrace\n");
}

/* "Temporary" debug stuff */
void
debug (fmt, a,b,c)
char *fmt;
int a,b,c;
{
  if (debug_level)
    message (fmt, a, b, c);
}

/* Exported globals */

boolean_t inferior_is_stopped;	/* Has the inferior task been suspended? */
extern  mach_port_t  name_server_port;

/* interface with catch_exception_raise: */
int stop_exception, stop_code, stop_subcode;
int stopped_in_exception;

char *hostname = "";

task_t inferior_task;
thread_t current_thread;

mach_port_t inferior_exception_port, inferior_old_exception_port;
/* 
 * When single-stepping, we wait for an exception from the 
 * single-stepped thread.  Otherwise, we do an unrestricted wait.  
 * Rather than creating and destroying a port each time we do a 
 * single-step, we cache a port here and reuse it as necessary. 
 */
mach_port_t cached_exception_port = MACH_PORT_NULL;
mach_port_t current_wait_port;
mach_port_t our_notify_port = MACH_PORT_NULL;

boolean_t other_threads_suspended;

mach_port_t all_enabled_ports;

/* Chain of ports */
struct port_chain {
  struct port_chain *next;
  mach_port_t	     port;
};
typedef struct port_chain *port_chain_t;

/* Room for chain nodes comes from pchain_obstack */
struct obstack pchain_obstack;
struct obstack *port_chain_obstack = &pchain_obstack;

/* the list of notified ports */
port_chain_t notify_chain = (port_chain_t) NULL;

port_chain_t
port_chain_append (list, elem)
     port_chain_t list;
     mach_port_t elem;
{
  port_chain_t new;

  if (! MACH_PORT_VALID (elem))
    return list;
  
  new = (port_chain_t) obstack_alloc (port_chain_obstack,
				      sizeof (struct port_chain));

  new->next = (port_chain_t) NULL;
  new->port = elem;

  if (! list)
    list = new;
  else
    {
      port_chain_t scan = list;
      while (scan->next)
	scan = scan->next;
      scan->next = new;
    }
  return list;
}

port_chain_t
port_chain_delete (list, elem)
     port_chain_t list;
     mach_port_t elem;
{
  if (list)
    if (list->port == elem)
      list = list->next;
    else
      while (list->next)
	{
	  if (list->next->port == elem)
	    list->next = list->next->next; /* GCd with obstack_free() */
	  else
	    list = list->next;
	}
  return list;
}

void
port_chain_destroy (ostack)
     struct obstack *ostack;
{
  obstack_free (ostack, 0);
  obstack_init (ostack);
}

int
port_chain_member_p (list, elem)
     port_chain_t list;
     mach_port_t elem;
{
  while (list)
    {
      if (list->port == elem)
	return 1;
      list = list->next;
    }
  return 0;
}

kill_inferior ()
{
  kill_inferior_fast ();
}

/* This is used when GDB is exiting.  It gives less chance of error.*/
kill_inferior_fast ()
{
  if (remote_debugging || inferior_pid == 0 || inferior_pid == 1 ||
      ! MACH_PORT_VALID (inferior_task))
    return;

  kill (inferior_pid, 9); /* Massacre */

  (void) task_terminate(inferior_task);
  wait_for_inferior ();
}

void
set_exception_port(thread)
     thread_t thread;
{
  mach_port_t exc_port;
  kern_return_t ret;
  
  if (MACH_PORT_VALID (thread))
    {
      exc_port = cached_exception_port;
      if (! MACH_PORT_VALID(exc_port))
	{
	  ret =  mach_port_allocate(mach_task_self(), 
				    MACH_PORT_RIGHT_RECEIVE,
				    &exc_port);
	  if (ret != KERN_SUCCESS)
	    fatal("set_exc_port: mach_port_allocate [%d] %s\n",
		  ret, mach_error_string(ret));
	  
	  ret = mach_port_insert_right(mach_task_self(), exc_port,
				       exc_port, MACH_MSG_TYPE_MAKE_SEND);
	  if (ret != KERN_SUCCESS)
	    fatal("set_exc_port: mach_port_allocate [%d] %s\n",
		  ret, mach_error_string(ret));
	  
	  cached_exception_port = exc_port;
	}
      
      ret = thread_set_special_port(thread, THREAD_EXCEPTION_PORT, exc_port);
      if (ret != KERN_SUCCESS)
	fatal("set_exc_port: thread_set_special_port [%d] %s\n",
	      ret, mach_error_string(ret));
    }
  current_wait_port = exc_port;	
}

void
clear_exception_port(thread)
     thread_t thread;
{
  mach_port_t exc_port;
  kern_return_t ret;
  
  if (MACH_PORT_VALID (thread))
    {
      ret = thread_get_special_port(thread, THREAD_EXCEPTION_PORT, &exc_port);
      if (ret != KERN_SUCCESS)
	fatal("clear_exc_port: thread_get_special_port [%d] %s\n",
	      ret, mach_error_string(ret));

      if (! MACH_PORT_VALID(exc_port))
	message ("clear_exc_port: wrong thread!");

      ret = thread_set_special_port(thread,
				    THREAD_EXCEPTION_PORT, MACH_PORT_NULL);
      if (ret != KERN_SUCCESS)
	fatal("clear_exc_port: thread_set_special_port [%d] %s\n",
	      ret, mach_error_string(ret));
    }
  current_wait_port = (mach_port_t) all_enabled_ports;
}

static request_notify(name, variant)
     mach_port_t	name;
     mach_msg_id_t	variant;
{
  kern_return_t ret;
  mach_port_t	previous_port_dummy = MACH_PORT_NULL;
  
  if (! MACH_PORT_VALID (name))
    return;
  
  if (port_chain_member_p (notify_chain, name))
    return;

  ret = mach_port_request_notification(mach_task_self(),
				       name,
				       variant,
				       1,
				       our_notify_port,
				       MACH_MSG_TYPE_MAKE_SEND_ONCE,
				       &previous_port_dummy);
  if (ret != KERN_SUCCESS)
    fatal ("request_notify %s", mach_error_string (ret));

  (void) mach_port_deallocate (mach_task_self (),
			       previous_port_dummy);

  notify_chain = port_chain_append (notify_chain, name);
}

reverse_msg_bits(msgp, type)
     mach_msg_header_t	*msgp;
     int type;
{
  int		rbits,lbits;
  rbits = MACH_MSGH_BITS_REMOTE(msgp->msgh_bits);
  lbits = type;
  msgp->msgh_bits =
    (msgp->msgh_bits & ~MACH_MSGH_BITS_PORTS_MASK) |
      MACH_MSGH_BITS(lbits,rbits);
}

/* @@@ get rid of this */
int wait_inferior_startup_flag = 0;

/*
 * Intercept system calls to Unix server.
 * After EXEC_COUNTER calls to exec(), return.
 *
 * Pre-assertion:  Child is suspended. (NOT VERIFIED)
 * Post-condition: Child is suspended after EXEC_COUNTER exec() calls.
 */

/* On the third day He said:

   	Let this be global
	and then it was global.

   When creating the inferior fork, the
   child code in inflow.c sets the name of the
   bootstrap_port in its address space to this
   variable.

   The name is transferred to our address space
   with read_inferior_memory().

   Thou shalt not do this with
   task_get_bootstrap_port() in this task, since
   the name in the inferior task is different than
   the one we get.

   For blessed are the meek, as they shall inherit
   the address space.
 */
mach_port_t original_server_port_name = MACH_PORT_NULL;

void
intercept_exec_calls (exec_counter)
     int exec_counter;
{
  struct syscall_msg_t {
    mach_msg_header_t	header;
    mach_msg_type_t 	type;
    char room[ 2000 ];	/* Enuff space */
  };

  struct syscall_msg_t syscall_in, syscall_out;

  mach_port_t fake_server;
  mach_port_t original_server_send;
  mach_port_t original_exec_reply;
  mach_port_t exec_reply;
  mach_port_t exec_reply_send;
  mach_msg_type_name_t acquired;
  mach_port_t emulator_server_port_name;
  struct task_basic_info info;
  mach_msg_type_number_t info_count;

  kern_return_t ret;

  if (exec_counter <= 0)
    return;		/* We are already set up in the correct program */

  ret = mach_port_allocate(mach_task_self(), 
			   MACH_PORT_RIGHT_RECEIVE,
			   &fake_server);
  CHK("create inferior_fake_server port failed", ret);
  
  /* Wait for inferior_task to suspend itself */
  while(1)
    {
      info_count = sizeof (info);
      ret = task_info (inferior_task,
		       TASK_BASIC_INFO,
		       &info,
		       &info_count);
      CHK ("Task info", ret);

      if (info.suspend_count)
	break;
      (void) swtch_pri (42); /* Universal Priority Value */
    }

  /* Read the inferior's bootstrap port name */
  if (read_inferior_memory (&original_server_port_name,
			    &original_server_port_name,
			    sizeof (original_server_port_name)))
    error ("Can't read inferior task bootstrap port name");

  /* Steal the original bsd server send right from inferior */
  ret = mach_port_extract_right (inferior_task,
				 original_server_port_name,
				 MACH_MSG_TYPE_MOVE_SEND,
				 &original_server_send,
				 &acquired);
  CHK("mach_port_extract_right (bsd server send)",ret);
  
  if (acquired != MACH_MSG_TYPE_PORT_SEND)
    error("Incorrect right extracted, send right to bsd server excpected");

  ret = mach_port_insert_right (inferior_task,
				original_server_port_name,
				fake_server,
				MACH_MSG_TYPE_MAKE_SEND);
  CHK("mach_port_insert_right (fake server send)",ret);

  debug ("inferior task bsd server ports set up \nfs %x, ospn %x, oss %x\n",
	 fake_server,
	 original_server_port_name, original_server_send);

  /* A receive right to the reply generated by unix server exec() request */
  ret = mach_port_allocate(mach_task_self(), 
			   MACH_PORT_RIGHT_RECEIVE,
			   &exec_reply);
  CHK("create intercepted_reply_port port failed", ret);
    
  /* Pass this send right to Unix server so it replies to us after exec() */
  ret = mach_port_extract_right (mach_task_self (),
				 exec_reply,
				 MACH_MSG_TYPE_MAKE_SEND_ONCE,
				 &exec_reply_send,
				 &acquired);
  CHK("mach_port_extract_right (exec_reply)",ret);

  if (acquired != MACH_MSG_TYPE_PORT_SEND_ONCE)
    error("Incorrect right extracted, send once excpected for exec reply");

  ret = mach_port_move_member(mach_task_self(), 
			      fake_server,
			      all_enabled_ports);
  CHK ("Moving fake syscall port to all_enabled_ports", ret);

  debug ("syscall fake server set up, resuming inferior\n");
  
  ret = task_resume (inferior_task);
  CHK("task_resume (startup)", ret);
	
  /* Read requests from the inferior.
     Pass directly through everything else except exec() calls.
   */
  while(exec_counter > 0)
    {
      ret = mach_msg (&syscall_in.header,	/* header */
		      MACH_RCV_MSG,		/* options */
		      0,			/* send size */
		      sizeof (struct syscall_msg_t), /* receive size */
		      all_enabled_ports,        /* receive_name */
		      MACH_MSG_TIMEOUT_NONE,
		      MACH_PORT_NULL);
      CHK("mach_msg (intercepted sycall)", ret);
	    
#ifdef DUMP_SYSCALL
      print_msg (&syscall_in.header);
#endif

      /* ASSERT : msgh_local_port == fake_server */

      if (notify_server (&syscall_in.header, &syscall_out.header))
	error ("received a notify while intercepting syscalls");

      if (syscall_in.header.msgh_id == MIG_EXEC_SYSCALL_ID)
	{
	  debug ("Received EXEC SYSCALL, counter = %d\n", exec_counter);
	  if (exec_counter == 1)
	    {
	      original_exec_reply = syscall_in.header.msgh_remote_port;
	      syscall_in.header.msgh_remote_port = exec_reply_send;
	    }
	  exec_counter--;
	}
	    
      syscall_in.header.msgh_local_port  = syscall_in.header.msgh_remote_port;
      syscall_in.header.msgh_remote_port = original_server_send;

      reverse_msg_bits(&syscall_in.header, MACH_MSG_TYPE_COPY_SEND);

      ret = mach_msg_send (&syscall_in.header);
      CHK ("Forwarded syscall", ret);
    }
	
  ret = mach_port_move_member(mach_task_self(), 
			      fake_server,
			      MACH_PORT_NULL);
  CHK ("Moving fake syscall out of all_enabled_ports", ret);

  ret = mach_port_move_member(mach_task_self(), 
			      exec_reply,
			      all_enabled_ports);
  CHK ("Moving exec_reply to all_enabled_ports", ret);

  ret = mach_msg (&syscall_in.header,	/* header */
		  MACH_RCV_MSG,		/* options */
		  0,			/* send size */
		  sizeof (struct syscall_msg_t),	/* receive size */
		  all_enabled_ports,		/* receive_name */
		  MACH_MSG_TIMEOUT_NONE,
		  MACH_PORT_NULL);
  CHK("mach_msg (exec reply)", ret);

  ret = task_suspend (inferior_task);
  CHK ("Suspending inferior after last exec", ret);

  debug ("Received exec reply from bsd server, suspended inferior task\n");

#ifdef DUMP_SYSCALL
      print_msg (&syscall_in.header);
#endif

  /* Message should appear as if it came from the unix server */
  syscall_in.header.msgh_local_port = MACH_PORT_NULL;

  /*  and go to the inferior task original reply port */
  syscall_in.header.msgh_remote_port = original_exec_reply;

  reverse_msg_bits(&syscall_in.header, MACH_MSG_TYPE_MOVE_SEND_ONCE);

  ret = mach_msg_send (&syscall_in.header);
  CHK ("Forwarding exec reply to inferior", ret);

  /* Garbage collect */
  ret = mach_port_deallocate (inferior_task,
			      original_server_port_name);
  CHK ("deallocating fake server send right", ret);

  ret = mach_port_insert_right (inferior_task,
				original_server_port_name,
				original_server_send,
				MACH_MSG_TYPE_MOVE_SEND);
  CHK ("Restoring the original bsd server send right", ret);

  ret = mach_port_destroy (mach_task_self (),
			   fake_server);
  fake_server = MACH_PORT_DEAD;
  CHK("mach_port_destroy (fake_server)", ret);

  ret = mach_port_destroy (mach_task_self (),
			   exec_reply);
  exec_reply = MACH_PORT_DEAD;
  CHK("mach_port_destroy (exec_reply)", ret);

  wait_inferior_startup_flag = 1;
}

int
select_thread (task, index)
     mach_port_t task;
     int index;
{
  thread_array_t thread_list;
  int thread_count;
  kern_return_t ret;

  if (index < 0)
    error ("Can't select negative threads");

  ret = task_threads (task, &thread_list, &thread_count);
  if (ret != KERN_SUCCESS)
    {
      message ("Can not select a thread from a dead task");
      current_thread = MACH_PORT_NULL;
      mach_really_wait (&wait_status);
      return -1;
    }

  if (thread_count == 0)
    {
      message ("Task (0x%x) has no threads", task);
      current_thread = MACH_PORT_NULL;
      return -1;
    }

  if (index >= thread_count)
    {
      message ("Task has only %d threads. Selecting thread 0",
	       thread_count);
      index = 0;
    }

  current_thread = thread_list[index];
  
  /* Notify when the selected thread dies */
  request_notify(current_thread, MACH_NOTIFY_DEAD_NAME);
  
  ret = vm_deallocate(mach_task_self(),
		      thread_list,
		      (thread_count * sizeof(mach_port_t)));
  CHK ("vm_deallocate", ret);
  
  return index;
}

mach_create_inferior_hook(pid)
     int pid;
{
  kern_return_t ret;
  char *err = 0;

#define CHECK(s) if (ret != KERN_SUCCESS) {err = s; goto error_exit;}

  errno = 0;

  inferior_task = task_by_pid (pid);

  if (! MACH_PORT_VALID(inferior_task))
    {
      err = "task_by_pid";
      ret = unix_err (errno);
      goto error_exit;
    }

  /* When notification appears, the inferior task died */
  request_notify(inferior_task, MACH_NOTIFY_DEAD_NAME);

  /* By default, select the first thread */
  /* If task has no threads, gives a warning */
  select_thread (inferior_task, 0);

  inferior_exception_port = MACH_PORT_NULL;

  setup_exception_port ();

  debug("Now the debugged task is created\n");
  
  return;
error_exit:
  mach_error(err, ret);
  error("in create_inferior_hook");  
}

setup_exception_port ()
{
  kern_return_t ret;

  ret =  mach_port_allocate(mach_task_self(), 
			    MACH_PORT_RIGHT_RECEIVE,
			    &inferior_exception_port);
  CHK("mach_port_allocate",ret);

  /* add send right */
  ret = mach_port_insert_right (mach_task_self (),
				inferior_exception_port,
				inferior_exception_port,
				MACH_MSG_TYPE_MAKE_SEND);
  CHK("mach_port_insert_right",ret);

  ret = mach_port_move_member(mach_task_self(), 
			      inferior_exception_port,
			      all_enabled_ports);
  CHK("mach_port_move_member",ret);

  ret = task_get_special_port (inferior_task, 
			       TASK_EXCEPTION_PORT,
			       &inferior_old_exception_port);
  CHK ("task_get_special_port(old exc)",ret);

  ret = task_set_special_port(inferior_task,
			      TASK_EXCEPTION_PORT, 
			      inferior_exception_port);
  CHK("task_set_special_port",ret);

  ret = mach_port_deallocate (mach_task_self (),
			      inferior_exception_port);
  CHK("mack_port_deallocate",ret);

#if 0
  /* When notify appears, the inferior_task's exception
   * port has been destroyed.
   *
   * Not used, since the notification appears when task dies.
   */
  request_notify(inferior_exception_port, MACH_NOTIFY_NO_SENDERS);
#endif
}


/* Wait for some action by the inferior.  
   
   If the inferior gets an exception, then we return 0.  If the
   inferior dies, then inferior_task will be null.  Otherwise, the
   exception will be stored in stop_exception et al.
   
   Otherwise, we return non-zero (meaning that the inferior was
   stopped by something other than an exception, e.g., a signal), and
   the status from wait() is stored in "w". */
int
mach_really_wait (w)
     WAITTYPE *w;
{
  int pid;
  kern_return_t ret;

  struct msg {
    mach_msg_header_t    header;
    mach_msg_type_t foo;
    int             data[8000];
  } in_msg, out_msg;

  /* Here's where we do the waiting part. */
  stop_exception = stop_code = stop_subcode = -1;

  stopped_in_exception = 0; /* Either notify (death) or exception or startup */

  /* When starting, return at once */
  if (wait_inferior_startup_flag)
    {
      wait_inferior_startup_flag = 0;
      wait3(w, WNOHANG, 0); /* Get something to w */
      return -1;
    }

  while (1)
    {
      QUIT;
      
      ret = mach_msg (&in_msg.header,		/* header */
		      MACH_RCV_MSG,		/* options */
		      0,			/* send size */
		      sizeof (struct msg),	/* receive size */
		      current_wait_port,	/* receive_name */
		      MACH_MSG_TIMEOUT_NONE,
		      MACH_PORT_NULL);
      CHK("mach_msg (receive)", ret);

      /* Check if we received a notify of the childs' death */
      if (notify_server (&in_msg.header, &out_msg.header))
	{
	  /* If inferior_task is null then the inferior has
	     gone away and we want to return to command level.
	     Otherwise it was just an informative message and we
	     need to look to see if there are any more. */
	  if (inferior_task == MACH_PORT_NULL)
	    return inferior_pid;
	  else
	    continue;
	}

      /* Not a notify message.  The server will decide if the user
	 needs to be involved and either suspend the task or just
	 get it ready to go. */
      if (! exc_server (&in_msg.header, &out_msg.header))
	error ("Unrecognized message received in mach_really_wait");
      
      /* He will try and run, but he is suspended, so all he will
	 do is be ready to leave the rpc.  We would prefer to do
	 this in resume(), but this keeps all of the message stuff
	 localized. */
      
      ret = mach_msg_send (&out_msg.header);
      CHK ("mach_msg_send (exc reply)", ret);
      
      if (inferior_is_stopped)
	{
	  /* Get the Unix state of the inferior */
	  wait3(w, WNOHANG, 0);
	  return inferior_pid;
	}
    }
}

/* this stuff here is an upcall via libmach/excServer.c 
   and mach_really_wait which does the actual upcall.  */

kern_return_t
catch_exception_raise (port, thread, task, exception, code, subcode)
     mach_port_t port;
     thread_t thread;
     task_t task;
     int exception, code, subcode;
{
  kern_return_t ret;
  extern running_in_shell;
  int fwd_exc_p();			/* do we forward this exception? */

  if (! MACH_PORT_VALID (thread))
    {
      /* If the exception was sent and thread dies before we
	 receive it, the thread will be MACH_PORT_DEAD
       */

      current_thread = thread = MACH_PORT_NULL;
      message ("Received exception from nonexistent thread");
    }
  else if (current_thread != thread)
    {
      current_thread = thread;
      request_notify(current_thread, MACH_NOTIFY_DEAD_NAME);
    }
  
  stop_exception = exception;
  stop_code      = code;
  stop_subcode   = subcode;  
  
  /* Decide whether to pass the exception on to the old exception
     handler.  There are three reasons for doing this.

     (1) If the exception isn't from our direct child, pass it on.
     This is generally useful when debugging gdb.

     (2) If the user has flagged this exception, pass it on.  This is
     normally used to pass exceptions to the default exception server,
     which maps the exception into a Unix signal.  The user might want
     to do this in case the inferior has a signal handler written to
     deal with the exception (e.g., segmentation violation or floating
     point exception).  If you're debugging gdb, then the target gdb's
     old exception handler is the top gdb (which will pass the
     exception on via rule (1).  

     (3) If the shell hasn't exec'd the debuggee yet and the exception is
     an address fault, pass it on.  This is to accomodate sh's strange
     memory management techniques.
   */

  /* @@@ mach programs? */
  if (inferior_pid == 0)
    error("got an exception, but inferior_pid is 0.");

  if (! MACH_PORT_VALID(inferior_task))
    error("got an exception, but inferior_task is null or dead");

  if (task != inferior_task
      || fwd_exc_p (exception, code)
      || (running_in_shell && exception == EXC_BAD_ACCESS))
    {
      /* Send the exception to the original handler */
      ret = exception_raise (inferior_old_exception_port,
			     thread, 
			     task,
			     exception,
			     code,
			     subcode);

      inferior_is_stopped = FALSE;
      return ret;
    }
  
  /* Otherwise, handle the exception directly.
     Suspend the inferior so the other threads don't run away.  */
  if ((ret = task_suspend (task)) != KERN_SUCCESS)
    {
      mach_error ("task_suspend", ret);
      error ("Error suspending victim after exception.\n");
    }

  inferior_is_stopped  = 1;
  stopped_in_exception = 1;

  /* If the exception is from a single-step command, we need to put
     things back.
   */
  if (other_threads_suspended == 1)
    {
      if (MACH_PORT_VALID (current_thread))
	{
	  if ((ret = thread_suspend (current_thread)) != KERN_SUCCESS) {
	    mach_error ("thread_suspend", ret);
	    error("in catch_exception_raise");
	  }
	}
      clear_exception_port(current_thread);
      clear_trace_bit (current_thread);
      resume_all_threads ();			/* Task is still suspended! */
      other_threads_suspended = FALSE;
    }

  return KERN_SUCCESS;
}



/* Resume execution of the inferior process.
   If STEP is nonzero, single-step it.
   If SIGNAL is nonzero, give it that signal.  */


resume (step, signal)
     int step;
     int signal;
{
  kern_return_t	ret;
  extern boolean_t vm_read_cache_valid;

  errno = 0;
  vm_read_cache_valid = FALSE;

  if (signal && inferior_pid)
    kill (inferior_pid, signal);

  if (step)
    {
      suspend_all_threads ();
      set_trace_bit (current_thread);
      set_exception_port (current_thread);
      if ((ret = thread_resume (current_thread)) != KERN_SUCCESS)
	{
	  mach_error ("thread_resume", ret);
	  error("in resume.");
	}
      other_threads_suspended = TRUE;
    }
  else
    other_threads_suspended = FALSE;

  if ((ret = task_resume (inferior_task)) != KERN_SUCCESS)
    {
      mach_error("task_resume", ret);
      error("in resume.");
    }

  /* HACK HACK This is needed by the multiserver system HACK HACK */
  while ((ret = task_resume(inferior_task)) == KERN_SUCCESS)
    /* make sure it really runs */;
  /* HACK HACK This is needed by the multiserver system HACK HACK */
}


#ifdef ATTACH_DETACH

extern int attach_flag;

static mach_port_t mid_server;
static mach_port_t auth = MACH_PORT_NULL;


/* Start debugging the process with the given task */
task_t
task_attach(tid)
  task_t tid;
{
  void fetch_inferior_registers();
  kern_return_t ret;

  inferior_task = tid;
  ret = task_suspend (inferior_task);
  if (ret != KERN_SUCCESS)
    error ("task_suspend", mach_error_string (ret));

  request_notify(inferior_task, MACH_NOTIFY_DEAD_NAME);

  select_thread (inferior_task, 0);

  setup_exception_port ();
  
  fetch_inferior_registers();
  stop_pc = read_pc();
  set_current_frame (create_new_frame (read_register (FP_REGNUM), stop_pc));
  stop_frame_address = FRAME_FP (get_current_frame ());

  /* The task is attached allright: the following
     makes happy have_inferior_p() */

  attach_flag = 1;
  return inferior_task;
}

mid_attach(mid)
    int	mid;
{
    kern_return_t kr;

    if (MACH_PORT_NULL == auth) {
	kr = netname_look_up(name_server_port, hostname, "MachID", &mid_server);
	if (kr != KERN_SUCCESS)
	    error("mid_attach: netname_lookup_up(MachID): %s\n",
		mach_error_string(kr));

	auth = mach_host_priv_self();
	if (auth == MACH_PORT_NULL) {
	    auth = mach_task_self();
	}

	kr = machid_mach_port(mid_server, auth, mid, &inferior_task);
	if (kr != KERN_SUCCESS)
	error("mid_attach: machid_mach_port: %s\n",
	     mach_error_string(kr));

    }
    task_attach(inferior_task);
    inferior_pid = (int)(-(mid));
    return mid;
}

/* 
 * Start debugging the process whose unix process-id is PID.
 * A negative "pid" value is legal and signifies a mach_id not a unix pid.
 *
 * Prevent (possible unwanted) dangerous operations by enabled users
 * like "atta 0" or "atta foo" (equal to the previous :-) and
 * "atta pidself". Anyway, these are allowed by specifying a MID.
 */
attach (pid)
     int pid;
{
  kern_return_t ret;

  if (pid == 0) {
    error("If you really want to debug the UNIX server, use mid");
  }

  if (pid == getpid())
    error ("I will debug myself only by mid");

  if (pid < 0) {
	mid_attach(-(pid));
	return(inferior_pid);
  }

  errno = 0;
  inferior_task = task_by_pid (pid);
  if (! MACH_PORT_VALID (inferior_task))
    error("task_by_pid", unix_err(errno));

  task_attach(inferior_task);

  /* The task is attached allright: the following
     makes happy have_inferior_p() */

  if (0 <= pid) inferior_pid = pid;
  return pid;
}

/* Stop debugging the process whose number is PID
   and continue it with signal number SIGNAL.
   SIGNAL = 0 means just continue it.  */

void
detach (signal)
     int signal;
{
  kern_return_t ret;

  ret = task_set_special_port (inferior_task, TASK_EXCEPTION_PORT, 
			       inferior_old_exception_port);
  CHK ("task_set_special_port", ret);

  attach_flag = 0;

  if (signal)
    kill (inferior_pid,signal);

  /* Now, on a multi the task might be dead by now */
  (void) task_resume (inferior_task);
}
#endif /* ATTACH_DETACH */

/* Copy LEN bytes from inferior's memory starting at MEMADDR
   to debugger memory starting at MYADDR. 
   On failure (cannot read from inferior, usually because address is out
   of bounds) returns the value of errno. */

boolean_t vm_read_cache_valid = FALSE;

int
read_inferior_memory (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  vm_offset_t vm_addr;
  vm_size_t vm_size;
  kern_return_t ret;
  static pointer_t buff = 0;
  static int buff_size = 0;
  static vm_offset_t last_vm_addr = 0;
  static vm_size_t last_vm_size = 0;
  static boolean_t vm_cache_in_use = FALSE;

  /* compute page aligned numbers */
  vm_addr = trunc_page(memaddr);
  vm_size = round_page(len);
  
  if (((vm_offset_t) ((vm_offset_t) memaddr + (vm_size_t) len)) > 
      (vm_addr + vm_size)) 
    {
      vm_size += vm_page_size;
    }

  /* Do we really need to do a vm_read? */
   if ((vm_read_cache_valid == FALSE)
      || (vm_cache_in_use == FALSE)
      || (vm_addr != last_vm_addr)
      || (vm_size != last_vm_size)) 
    {

      /* 
       * deallocate the old cache if it is being used
       */
      if (vm_cache_in_use)
	{
	  vm_cache_in_use = FALSE;
	  ret = vm_deallocate(mach_task_self(), (vm_offset_t) buff,
			      (vm_size_t) buff_size);
	  if (ret != KERN_SUCCESS)
	    {
	      mach_error("vm_deallocate", ret);
	      error("Error handling gdb's memory cache.");
	    }
	}
      
      ret = vm_read (inferior_task, vm_addr, vm_size, &buff, &buff_size);
      if (ret != KERN_SUCCESS)
	{
	  char msg_buf[1024];

	  vm_read_cache_valid = FALSE;
	  sprintf(msg_buf, "Can't read inferior's memory at 0x%x:", vm_addr);
	  mach_error (msg_buf, ret);
	  *myaddr = 0;
	  return EFAULT;	/* do more careful mapping to errno value? */
	}
    }
  
  /* copy the relevant bytes to our caller */
  bcopy((char *)(buff + (memaddr % vm_page_size)), myaddr, len);
  last_vm_addr = vm_addr;
  last_vm_size = vm_size;
  vm_cache_in_use = TRUE;
  vm_read_cache_valid = TRUE;
  return (0);
}



vm_get_prot(task, where, how_much, start, size, protection, max_protection)
	task_t		task;
	vm_offset_t where;
	vm_size_t how_much;
	vm_offset_t *start;
	vm_size_t *size;
	vm_prot_t	*protection;
	vm_prot_t	*max_protection;
{
	vm_inherit_t	inheritance;
	boolean_t	shared;
	mach_port_t	object_name;
	vm_offset_t	offset;

	*start = where;
	*size  = how_much;
	(void) vm_region(task, start, size,
				protection, max_protection,
				&inheritance, &shared,
				&object_name, &offset);
}

/* Copy LEN bytes of data from debugger memory at MYADDR
   to inferior's memory at MEMADDR.
   On failure (cannot write the inferior)
   returns EFAULT (the expected bad value of errno).  */

int
write_inferior_memory (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  vm_offset_t vm_addr;
  vm_size_t vm_size;
  pointer_t buff;
  int buff_size;
  vm_prot_t prot;
  kern_return_t ret;
  char *err = 0;

  if (remote_debugging)
    {
      register int i;
      /* Round starting address down to longword boundary.  */
      register CORE_ADDR addr = memaddr & - sizeof (int);
      /* Round ending address up; get number of longwords that makes.  */
      register int count
        = (((memaddr + len) - addr) + sizeof (int) - 1) / sizeof (int);
      /* Allocate buffer of that many longwords.  */
      register int *buffer = (int *) xmalloc (count * sizeof (int));

      /* Fill start and end extra bytes of buffer with existing memory data.  */

      buffer[0] = remote_fetch_word (addr);
      if (count > 1)
	buffer[count - 1]
	  = remote_fetch_word (addr + (count - 1) * sizeof (int));

      /* Copy data to be written over corresponding part of buffer */

      bcopy (myaddr, (char *) buffer + (memaddr & (sizeof (int) - 1)), len);

      /* Write the entire buffer.  */

      errno = 0;
      for (i = 0; i < count; i++, addr += sizeof (int))
        {
	  remote_store_word (addr, buffer[i]);
          if (errno)
	    break;
        }

      free(buffer);
      return errno;
    }

  /* compute page-aligned numbers */
  vm_addr = trunc_page(memaddr);
  vm_size = round_page(len);

  /* read the whole thing */
  ret = vm_read (inferior_task, vm_addr, vm_size, &buff, &buff_size);
  if (ret != KERN_SUCCESS) {
    mach_error ("write_inferior_memory: vm_read", ret);
    return EFAULT;
  }

  /* diddle just what we want */
  bcopy(myaddr, (char *) buff + (memaddr %vm_page_size), len);

  /* Write the whole thing back.
   * If we get a protection failure, up the protection so that
   * we CAN write, do the write, and put it back.
   */
  
  ret = vm_write (inferior_task, vm_addr, buff, vm_size);
  if (ret == KERN_PROTECTION_FAILURE) {    
    vm_offset_t start;
    vm_size_t size, how_much;
    vm_prot_t maxp;

again:
    vm_get_prot(inferior_task, vm_addr, vm_size, &start, &size, &prot, &maxp);
    ret = vm_protect (inferior_task, start, size, TRUE,  maxp | VM_PROT_WRITE);
    ret = vm_protect (inferior_task, start, size, FALSE, prot | VM_PROT_WRITE);
    if (ret != KERN_SUCCESS) {
      err = "write_inferior_memory: vm_protect";
      goto out;
    }
    /* Find out how much this time through */
    how_much = vm_size;
    if (start + size < vm_addr + vm_size)
      how_much = (start + size) - vm_addr;

    ret = vm_write (inferior_task, vm_addr, buff, how_much);
    if (ret != KERN_SUCCESS) {
      err = "write_inferior_memory: vm_write";
      goto out;
    }

    ret = vm_protect (inferior_task, start, size, TRUE,  maxp);
    ret = vm_protect (inferior_task, start, size, FALSE, prot);
    if (ret != KERN_SUCCESS) {
      err = "write_inferior_memory: vm_protect";
      goto out;
    }
    /* All done ? */
    if (how_much != vm_size)
      {
	vm_size -= how_much;
	vm_addr += how_much;
        goto again;
      }
  } else if (ret != KERN_SUCCESS) {
    err = "write_inferior_memory: vm_write";
    goto out;
  }

out:
  if (err)
    mach_error(err, ret);

  /* deallocate that buffer we needed.. */
  (void) vm_deallocate(mach_task_self(), (vm_offset_t) buff, (vm_size_t) buff_size);

  /* just in case.. */
  vm_read_cache_valid = FALSE;

  return (err) ? EFAULT : 0;
}


static char
translate_state(state)
int	state;
{
  switch (state) {
  case TH_STATE_RUNNING:	return('R');
  case TH_STATE_STOPPED:	return('S');
  case TH_STATE_WAITING:	return('W');
  case TH_STATE_UNINTERRUPTIBLE: return('U');
  case TH_STATE_HALTED:		return('H');
  default:			return('?');
  }
}

void
thread_list_command()
{
    int i, current_thread_index;
    thread_array_t thread_table;
    int thread_count;
    thread_basic_info_data_t ths;
    int size;
    kern_return_t ret;

    ERROR_NO_INFERIOR;

    ret = task_threads(inferior_task, &thread_table, &thread_count);
    if (ret != KERN_SUCCESS)
      {
	message ("Error getting inferior's thread list.");
	mach_error("task_threads", ret);	

	/* Inferior has died, a notification is pending */
	mach_really_wait (&wait_status); 
	return;
      }
    printf("There are %d threads.\n", thread_count);
    printf("Thread\tState\tSuspend\tPort\n");
    for (i = 0; i < thread_count; i++) {
	if (thread_table[i] == current_thread)
	    current_thread_index = i;

	size = THREAD_BASIC_INFO_COUNT;

	ret = thread_info(thread_table[i],
			  THREAD_BASIC_INFO,
			  (thread_info_t)&ths,
			  &size);

	if (ret != KERN_SUCCESS)
	  {
	    message ("Unable to get info on thread 0x%x.",
		     thread_table[i]);
	    mach_error("thread_statistics", ret);
	    continue;
	  }

	printf ("%d\t%c\t%d\t0x%x\n", i,
		translate_state (ths.run_state), ths.suspend_count,
		thread_table[i]);
    }

    printf("\nThe current thread is thread %d.\n", current_thread_index);

    ret = vm_deallocate(mach_task_self(), (vm_address_t)thread_table, 
			(thread_count * sizeof(int)));
    if (ret != KERN_SUCCESS) {
	message ("Error trying to deallocate thread list.");
	mach_error("vm_deallocate", ret);
    }
}


void
thread_select_command(args, from_tty)
     char *args;
     int from_tty;
{
  int thread_index;
  thread_array_t thread_list;
  int thread_count;
  kern_return_t ret;

  ERROR_NO_INFERIOR;

  if (!args)
    error_no_arg ("thread to select");

  thread_index = atoi(args);

  if (select_thread (inferior_task, thread_index) == -1)
    return;

  fetch_inferior_registers ();
  stop_pc = read_pc ();
  set_current_frame (create_new_frame (read_register (FP_REGNUM), stop_pc));
  stop_frame_address = FRAME_FP (get_current_frame ());
}

set_trace_bit (thread)
{
  int			flavor = TBIT_FLAVOR;
  unsigned int		stateCnt = TBIT_FLAVOR_SIZE;
  kern_return_t		ret;
  thread_state_data_t	state;

  if (! MACH_PORT_VALID (thread))
    {
      message ("You can not trace a nonexistent thread");
      return;
    }

  ret = thread_get_state(thread, flavor, state, &stateCnt);
  if (ret != KERN_SUCCESS) {
      if (ret==MIG_ARRAY_TOO_LARGE)
	      ret = KERN_SUCCESS;
      else {
	      message ("set_trace_bit: error reading status register");
	      mach_error("thread_get_state", ret);
	      exit (1);
      }
  }

  TBIT_SET(thread,state);

  ret = thread_set_state(thread, flavor, state, stateCnt);
  if (ret != KERN_SUCCESS) {
    message ("set_trace_bit: error writing status register");
    mach_error("thread_set_state", ret);
    exit (1);
  }  
}

clear_trace_bit (thread)
{
  int			flavor = TBIT_FLAVOR;
  unsigned int		stateCnt = TBIT_FLAVOR_SIZE;
  kern_return_t		ret;
  thread_state_data_t	state;

  if (! MACH_PORT_VALID (thread))
    return;

  ret = thread_get_state(thread, flavor, state, &stateCnt);
  if (ret != KERN_SUCCESS) {
      if (ret==MIG_ARRAY_TOO_LARGE)
	      ret = KERN_SUCCESS;
      else {
	    message ("clear_trace_bit: error reading status register");
	      mach_error("thread_get_state", ret);
	      exit (1);	
      }
  }

  TBIT_CLEAR(thread,state);

  ret = thread_set_state(thread, flavor, state, stateCnt);
  if (ret != KERN_SUCCESS) {
    message ("set_trace_bit: error writing status register");
    mach_error("thread_set_state", ret);
    exit (1);
  }  

}

#ifdef	FLUSH_INFERIOR_CACHE

/* When over-writing code on some machines the I-Cache must be flushed
   explicitly, because it is not kept coherent by the lazy hardware.
   This definitely includes breakpoints, for instance, or else we
   endup looping in misterious Bpt traps */

#ifndef MATTR_CACHE	/* fornow */
#include <mach/vm_attributes.h>
#endif

flush_inferior_icache(pc,amount)
	CORE_ADDR pc;
{
	vm_machine_attribute_val_t flush = MATTR_VAL_ICACHE_FLUSH;
	kern_return_t   ret;

	ret = vm_machine_attribute(inferior_task, pc, amount,
				   MATTR_CACHE, &flush);
	if (ret != KERN_SUCCESS) {
		mach_error("vm_machine_attribute", ret);
		message ("Error flushing inferior's cache.");
	}
}

#endif	FLUSH_INFERIOR_CACHE


suspend_command (args, from_tty)
     char *args;
     int from_tty;
{
  int thread_index;
  mach_port_t saved_thread;
  kern_return_t ret;

  if (!args)
    error_no_arg ("thread to suspend or ALL");

  if (strcmp(args, "all") == 0) {
    suspend_all_threads();
    return;
  }

  saved_thread = current_thread;

  thread_index = atoi(args);

  if (select_thread (inferior_task, thread_index) != thread_index)
    return_to_top_level ();

  ret = thread_suspend (current_thread);
  if (ret != KERN_SUCCESS) {
      mach_error ("thread_suspend", ret);
  }
  
  current_thread = saved_thread;
}

suspend_all_threads ()
{
    thread_array_t	thread_list;
    int			thread_count;
    kern_return_t	ret;

    ret = task_threads(inferior_task, &thread_list, &thread_count);
    if (ret != KERN_SUCCESS)
      {
	message ("Could not suspend inferior threads.");
	wait_for_inferior (); /* Get dead port notification, set state */
	return_to_top_level ();
      }
    while (thread_count > 0) {
	ret = thread_suspend(thread_list[--thread_count]);
	if (ret != KERN_SUCCESS) {
	    message ("Error trying to suspend thread 0x%x.", 
		    thread_list[--thread_count]);
	    mach_error("thread_suspend", ret);
	}
    }
    ret = vm_deallocate(mach_task_self(), (vm_address_t)thread_list, 
			(thread_count * sizeof(int)));
    if (ret != KERN_SUCCESS) {
	message ("Error trying to deallocate thread list.");
	mach_error("vm_deallocate", ret);
    }
}

resume_command (args, from_tty)
     char *args;
     int from_tty;
{
  int thread_index;
  mach_port_t saved_thread;
  kern_return_t ret;

  if (!args)
    error_no_arg ("thread to resume or ALL");

  if (strcmp(args, "all") == 0) {
    resume_all_threads();
    return;
  }

  saved_thread = current_thread;

  thread_index = atoi(args);

  if (select_thread (inferior_task, thread_index) != thread_index)
    return_to_top_level ();

  ret = thread_resume (current_thread);
  if (ret != KERN_SUCCESS)
      mach_error ("thread_resume", ret);

  current_thread = saved_thread;
}


resume_all_threads ()
{
    thread_array_t	thread_list;
    int			thread_count;
    kern_return_t	ret;

    ret = task_threads(inferior_task, &thread_list, &thread_count);
    if (ret != KERN_SUCCESS)
	error("task_threads", mach_error_string( ret));

    while (thread_count > 0) {
	ret = thread_resume(thread_list[--thread_count]);
	if (ret != KERN_SUCCESS)
	    error("thread_resume", mach_error_string( ret));
    }
    ret = vm_deallocate(mach_task_self(),
			(vm_address_t)thread_list, 
			(thread_count * sizeof(int)));
}

void
_initialize_mach_os ()
{
  kern_return_t ret;

  ret = mach_port_allocate(mach_task_self(), 
			   MACH_PORT_RIGHT_PORT_SET,
			   &all_enabled_ports);
  if (ret != KERN_SUCCESS)
    fatal("initial port set %s",mach_error_string(ret));

  ret = mach_port_allocate (mach_task_self(),
			    MACH_PORT_RIGHT_RECEIVE,
			    &our_notify_port);
  if (ret != KERN_SUCCESS)
    fatal("Creating notify port %s", mach_error_string(ret));

  ret = mach_port_move_member(mach_task_self(), 
			      our_notify_port,
			      all_enabled_ports);
  if (ret != KERN_SUCCESS)
    fatal("initial move member %s",mach_error_string(ret));

  current_wait_port = (mach_port_t) all_enabled_ports;

  obstack_init (port_chain_obstack);

  add_com ("thread-select", class_stack, thread_select_command,
	   "Select and print a thread.");
  add_com_alias ("ts", "thread-select", class_stack, 0);

  add_com ("thread-list", class_stack, thread_list_command,
	   "List all of threads.");
  add_com_alias ("tl", "thread-list", class_stack, 0);

  add_com ("suspend", class_run, suspend_command,
	   "Suspend one or all of the threads in the inferior.");
  add_com ("resume", class_run, resume_command,
	   "Resume one or all of the threads in the inferior.");
}

kern_return_t
do_mach_notify_dead_name (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  kern_return_t kr = KERN_SUCCESS;
  
  extern int terminal_is_ours;
  int myterm = terminal_is_ours;

  if (! myterm)
    {
      terminal_ours ();		/* Should be ok even if no inf.  */
      fflush (stdout);
    }

  /* Take name of from unreceived dead name notification list */
  notify_chain = port_chain_delete (notify_chain, name);

  /* did the whole task sink?  */
  if (name == inferior_task)
    {
      int pid;
      WAITTYPE w;

      message ("Inferior task died  (0x%x)",
	       inferior_task);

      mach_port_destroy (mach_task_self(), name);
      inferior_task = MACH_PORT_NULL;

      inferior_died ();

      /* Collect Unix exit status */
      pid = wait3(&w, WNOHANG, 0);
      if (WTERMSIG (w))
	printf ("Program received signal %d\n",
		WTERMSIG (w));
      else
	if (WRETCODE (w))
	  printf ("Program (%d) exited with code %d\n",
		  pid, WRETCODE (w));
	else
	  printf ("Program (%d) exited normally\n", pid);
      fflush (stdout);

      /* @@@@ Here would be a good place to wait for
	 unreceived notifications ??? */

      if (notify_chain)
	message ("There are still unreceived dead_name_notifications???");

      /* release notification chain */
      notify_chain = (port_chain_t) NULL;
      port_chain_destroy (port_chain_obstack);
    }
  else /* how about one of the threads?  */
    if (name == current_thread)
      {
	message ("\nCurrent thread died (0x%x)",
		 current_thread);
	
	mach_port_destroy (mach_task_self(), name);
	current_thread = MACH_PORT_NULL;

#if 0
	select_thread (inferior_task, 0); /* @@@ Should we? */
#endif
	if (! myterm)
	  terminal_inferior();
      }
    else
      {
	message ("Something died, probably a thread (0x%x)",
		 name);
	message ("Btw, why don't you fix this?");
	fflush (stderr);

	if (! myterm)
	  terminal_inferior();
      }
  
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_msg_accepted (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  message ("do_mach_notify_msg_accepted : notify %x, name %x",
	   notify, name);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_no_senders (notify, mscount)
     mach_port_t notify;
     mach_port_mscount_t mscount;
{
  message ("do_mach_notify_no_senders : notify %x, mscount %x",
	   notify, mscount);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_port_deleted (notify, name)
     mach_port_t notify;
     mach_port_t name;
{
  message ("do_mach_notify_port_deleted : notify %x, name %x",
	   notify, name);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_port_destroyed (notify, rights)
     mach_port_t notify;
     mach_port_t rights;
{
  message ("do_mach_notify_port_destroyed : notify %x, rights %x",
	   notify, rights);
  return KERN_SUCCESS;
}

kern_return_t
do_mach_notify_send_once (notify)
     mach_port_t notify;
{
#ifdef DUMP_SYSCALL
  /* MANY of these are generated. */
  message ("do_mach_notify_send_once : notify %x",
	   notify);
#endif
  return KERN_SUCCESS;
}


#ifdef DUMP_SYSCALL
/* Following msg dumper written by alo@hut.fi */
#ifdef __STDC__
#define STR(x) #x
#else
#define STR(x) "x"
#endif

char	*bsd1_names[] = {
  "execve",
  "fork",
  "take_signal",
  "sigreturn",
  "getrusage",
  "chdir",
  "chroot",
  "open",
  "creat",
  "mknod",
  "link",
  "symlink",
  "unlink",
  "access",
  "stat",
  "readlink",
  "chmod",
  "chown",
  "utimes",
  "truncate",
  "rename",
  "mkdir",
  "rmdir",
  "xutimes",
  "mount",
  "umount",
  "acct",
  "setquota",
  "write_short",
  "write_long",
  "send_short",
  "send_long",
  "sendto_short",
  "sendto_long",
  "select",
  "task_by_pid",
  "recvfrom_short",
  "recvfrom_long",
  "setgroups",
  "setrlimit",
  "sigvec",
  "sigstack",
  "settimeofday",
  "adjtime",
  "setitimer",
  "sethostname",
  "bind",
  "accept",
  "connect",
  "setsockopt",
  "getsockopt",
  "getsockname",
  "getpeername",
  "init_process",
  "table_set",
  "table_get",
  "pioctl",
  "emulator_error",
  "readwrite",
  "share_wakeup",
  0,
  "maprw_request_it",
  "maprw_release_it",
  "maprw_remap",
  "pid_by_task",
};

int	bsd1_nnames = sizeof(bsd1_names)/sizeof(bsd1_names[0]);

char*
name_str(name,buf)

int	name;
char	*buf;

{
  switch (name) {
  case MACH_MSG_TYPE_BOOLEAN:
    return "boolean";
  case MACH_MSG_TYPE_INTEGER_16:
    return "short";
  case MACH_MSG_TYPE_INTEGER_32:
    return "long";
  case MACH_MSG_TYPE_CHAR:
    return "char";
  case MACH_MSG_TYPE_BYTE:
    return "byte";
  case MACH_MSG_TYPE_REAL:
    return "real";
  case MACH_MSG_TYPE_STRING:
    return "string";
  default:
    sprintf(buf,"%d",name);
    return buf;
  }
}

char *
id_str(id,buf)

int	id;
char	*buf;

{
  char	*p;
  if (id >= 101000 && id < 101000+bsd1_nnames) {
    if (p = bsd1_names[id-101000])
      return p;
  }
  if (id == 102000)
    return "psignal_retry";
  if (id == 100000)
    return "syscall";
  sprintf(buf,"%d",id);
  return buf;
}

print_msg(mp)
mach_msg_header_t	*mp;
{
  char	*fmt_x = "%20s : 0x%08x\n";
  char	*fmt_d = "%20s : %10d\n";
  char	*fmt_s = "%20s : %s\n";
  char	buf[100];

  printf("\n");
#define pr(fmt,h,x) printf(fmt,STR(x),(h).x)
  pr(fmt_x,(*mp),msgh_bits);
  pr(fmt_d,(*mp),msgh_size);
  pr(fmt_x,(*mp),msgh_remote_port);
  pr(fmt_x,(*mp),msgh_local_port);
  pr(fmt_d,(*mp),msgh_kind);
  printf(fmt_s,STR(msgh_id),id_str(mp->msgh_id,buf));
  
  if (debug_level > 1)
  {
    char	*p,*ep,*dp;
    int		plen;
    p = (char*)mp;
    ep = p+mp->msgh_size;
    p += sizeof(*mp);
    for(; p < ep; p += plen) {
      mach_msg_type_t	*tp;
      mach_msg_type_long_t	*tlp;
      int	name,size,number;
      tp = (mach_msg_type_t*)p;
      if (tp->msgt_longform) {
	tlp = (mach_msg_type_long_t*)tp;
	name = tlp->msgtl_name;
	size = tlp->msgtl_size;
	number = tlp->msgtl_number;
	plen = sizeof(*tlp);
      } else {
	name = tp->msgt_name;
	size = tp->msgt_size;
	number = tp->msgt_number;
	plen = sizeof(*tp);
      }
      printf("name=%-16s size=%2d number=%7d inline=%d long=%d deal=%d\n",
	     name_str(name,buf),size,number,tp->msgt_inline,tp->msgt_longform,
	     tp->msgt_deallocate);
      dp = p+plen;
      if (tp->msgt_inline) {
	int	l;
	l = size*number/8;
	l = (l+sizeof(long)-1)&~((sizeof(long))-1);
	plen += l;
	print_data(dp,size,number);
      } else {
	plen += sizeof(int*);
      }
      printf("plen=%d\n",plen);
    }
  }
}

print_data(p,size,number)

char	*p;

{
  int	*ip;
  short	*sp;
  int	i;

  switch (size) {
  case 8:
    for(i = 0; i < number; i++) {
      printf(" %02x",p[i]);
    }
    break;
  case 16:
    sp = (short*)p;
    for(i = 0; i < number; i++) {
      printf(" %04x",sp[i]);
    }
    break;
  case 32:
    ip = (int*)p;
    for(i = 0; i < number; i++) {
      printf(" %08x",ip[i]);
    }
    break;
  }
  printf("\n");
}
#endif  DUMP_SYSCALL
