/*
 window.c: Program initialization, display hardware recocnition and view
           window initialization routines. 

 $Header: window.c,v 2.17 93/09/03 14:45:56 wet Exp $
 */
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xatom.h>
#include <Xm/Xm.h>
#include <Xm/Protocols.h>
#include <Xm/AtomMgr.h>
#include <Xm/RowColumn.h>
#include <Xm/Form.h>
#include <Xm/BulletinB.h>
#include <Xm/ScrolledW.h>
#include <Xm/ScrollBar.h>
#include <Xm/DrawingA.h>
#include <Xm/Frame.h>
#include <Xm/PushB.h>
#include <Xm/Label.h>
#include <Xm/Separator.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawnB.h>
#include <stdio.h>
#include <stdlib.h>
#include "callb.h"
#include "tools1.h"
#include "dialogw.h"
#include "menu.h"
#include "list.h"
#include "window.h"
#include "dialog.h"
#include "event.h"
#include "X_misc.h"
#include "bitmaps.h"
#include "convert.h"

/* DESCRIPTION:    Checks the available visuals, chooses the best and sets the
                   view_window resources accordingly; creates toolbox and
                   enters input loop.
   PARAMETERS:
   int *ac         Program arguments.
   char *av[]      --------""--------  
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 12th August 1993.
*/   
void init_phoenix(int *ac, char *av[])
{
  extern char *visual;
  extern int depth;
  extern int force_gray;
  Arg wargs[10];
  XtAppContext app;
  Status result;
  XVisualInfo info;
  Display *disp;
  Window win;
  XSizeHints hints;
  int scr, i = 0, j = 0, best_vis, best_depth, count;
  XStandardColormap *std_cmap;
  Colormap cmap;
  unsigned long mask;
  XSetWindowAttributes attr;

  struct visuals {
    char *name;
    int class;
  };

 struct visuals vis[] = { /* visuals in order of prefence */
   {"TrueColor", TrueColor},
   {"DirectColor", DirectColor},
   {"StaticColor", StaticColor},
   {"PseudoColor", PseudoColor},
   {"StaticGray", StaticGray},
   {"GrayScale", GrayScale}
 };

  int depths[] = {24, 16, 12, 8, 4, 1}; /* visuals in same order */

  /* Initialize the toolkit and open the display */
  XtToolkitInitialize();
  app = XtCreateApplicationContext();
  disp = XtOpenDisplay(app, "", av[0], "Phoenix", NULL, 0, ac, av);
  if (disp==NULL)
    {
      fprintf(stderr, "Unable to open display connection.\n");
      exit(-1);
    }
  scr = DefaultScreen(disp);

  /* Initializing */
  arg_count = 0;
  grab_flag = False;
  first_time = True;
  need_mask = False;

  /* Find appropriate visual, depth and colormap for the view windows */
  result = 0;
  if (depth!=0 && visual!=NULL) /* Depth and visual as arguments */
    {                                 
      while (strcmp(visual, vis[i].name) && i<XtNumber(vis))
	i++;
      if (i==XtNumber(vis))
	{
	  fprintf(stderr, "Invalid visual: %s.\n", visual);
	  exit(-1);
	}

      while (depth!=depths[j] && j<XtNumber(depths))
	j++;
      if (j==XtNumber(depths))
	{
	  fprintf(stderr, "Invalid depth: %d.\n", depth);
	  exit(-1);
	}
      printf("Matching visual %s, depth %d...\n", vis[i].name, depth);
      result = XMatchVisualInfo(disp, scr, depth, vis[i].class, &info);
      if (result)
	{
	  printf("Visual match found, visual %s, depth %d.\n", 
		 vis[i].name, depth);
	  best_depth = depth;
	  best_vis = vis[i].class;
	}
      else
	{
	  fprintf(stderr, "Visual matching failed, visual %s, depth %d.\n", 
		  vis[i].name, depth);
	  exit(-1);
	}
    }
  else if (depth!=0 && visual==NULL) /* Depth given, decide visual */
    {
      while (depth!=depths[j] && j<XtNumber(depths))
	j++;
      if (j==XtNumber(depths))
	{
	  fprintf(stderr, "Invalid depth: %d.\n", depth);
	  exit(-1);
	}
      while (i<XtNumber(vis) && !result)
	{
	  printf("Matching visual %s, depth %d ...\n", vis[i].name, depth);
      	  result = XMatchVisualInfo(disp, scr, depth, vis[i].class, &info);
	  i++;
	}
      if (result)
	{
	  printf("Visual match found, visual %s, depth %d.\n", 
		 vis[i-1].name, depth);
	  best_depth = depth;
	  best_vis = vis[i-1].class;
	}
      else
	{
	  fprintf(stderr, "No visual of depth %d found.\n", depth);
	  exit(-1);
	}
    }
  else if (depth==0 && visual!=NULL) /* Visual given, decide depth */
    {
      while (strcmp(visual, vis[i].name) && i<XtNumber(vis))
	i++;
      if (i==XtNumber(vis))
	{
	  fprintf(stderr, "Invalid visual: %s.\n", visual);
	  exit(-1);
	}
      
      while (j<XtNumber(depths) && !result)
	{
	  printf("Matching visual %s, depth %d...\n", vis[i].name,
		 depths[j]);
      	  result = XMatchVisualInfo(disp, scr, depths[j], vis[i].class,
				    &info);
	  j++;
	}
      if (result)
	{
	  printf("Visual match found, visual %s, depth %d.\n", 
		 vis[i].name, depths[j-1]);
	  best_depth = depths[j-1];
	  best_vis = vis[i].class;
	}
      else
	{
	  fprintf(stderr, "No visual %s found.\n", vis[i].name);
	  exit(-1);
	}
    }
  else /* No arguments, choose the best available depth and visual */
    {
      while (!result)
	{
	  for (j = 0; j<XtNumber(depths); j++)
	    {
	      printf("Matching visual %s, depth %d...\n", vis[i].name,
		     depths[j]);
	      result = XMatchVisualInfo(disp, scr, depths[j], vis[i].class,
					&info);
	      if (result)
		{
		  best_depth = depths[j];
		  best_vis = vis[i].class;
		  break;
		}
	    }
	  i++;
	}
      if (result)
	printf("Visual match found, visual %s, depth %d.\n", 
	       vis[i-1].name, depths[j]);
      else
	{
	  fprintf(stderr, "No visual match found!\n");
	  exit(-1);
	}
    }

  /* Set view window depth and visual */
  XtSetArg(view_args[arg_count], XtNvisual, info.visual); arg_count++;
  XtSetArg(view_args[arg_count], XtNdepth, best_depth); arg_count++;
  depth = best_depth;

  view_vis = info.visual;
  /* Set bitmap pad used by XCreateImage */
  if (depth<=8)
    pad = 8;
  else if (depth<=16)
    pad = 16;
  else
    pad = 32;

  /* Get the appropriate colormap and set the converter */
  if (best_vis==DirectColor||best_vis==PseudoColor)
    {
      if (force_gray)
	{
	  result = XGetRGBColormaps(disp, DefaultRootWindow(disp), 
				    &std_cmap, &count, XA_RGB_GRAY_MAP);
	  switch (depth)
	    {
	    case 8:
	      set_block_to_screen(block_to_8gs);
	      break;
	    case 4:
	      init_4gs();
	      set_block_to_screen(block_to_4gs);
	      break;
	    case 1:
	      init_1gs();
	      set_block_to_screen(block_to_1gs);
	      break;
	    }
	}
      else
	{
	  result = XGetRGBColormaps(disp, DefaultRootWindow(disp), 
				    &std_cmap, &count, XA_RGB_BEST_MAP);
/*
	  printf("base pixel: %u\nr-max : %u, r-mul: %u\ng-max : %u, g-mul: %u\nb-max : %u, b-mul: %u\n", std_cmap->base_pixel, 
		 std_cmap->red_max, std_cmap->red_mult,
		 std_cmap->green_max, std_cmap->green_mult,
		 std_cmap->blue_max, std_cmap->blue_mult);
	  printf("r-mask: %#x, g-mask: %#x, b-mask: %#x\n", 
		 std_cmap->base_pixel+std_cmap->red_max*std_cmap->red_mult,
		 std_cmap->base_pixel+std_cmap->green_max*std_cmap->green_mult,
		 std_cmap->base_pixel+std_cmap->blue_max*std_cmap->blue_mult);
*/		 
	  switch (depth)
	    {
	    case 24:
	      set_block_to_screen(block_to_24bit);
	      break;
	    case 12:
	      init_12c();
	      set_block_to_screen(block_to_12c);
	      break;
	    case 8:
	      init_8c();
	      set_block_to_screen(block_to_8c);
	      break;
	    default:
	      fprintf(stderr, "No converter for this visual\n");
	      exit(-1);
	      break;
	    }
	}
      if (result)
	{ 
	  if (force_gray)
	    printf("RGB gray map found on server.\n");
	  else
	    printf("RGB best map found on server.\n");
	  XtSetArg(view_args[arg_count], XtNcolormap, std_cmap->colormap);
	  arg_count++;
	  view_map = std_cmap->colormap;
	}
      else
	{
	  fprintf(stderr, "No standard colormap available.\n");
	  exit(-1);
	}
    }
  else if (best_vis==GrayScale)
    {
      switch (depth)
	{
	case 8:
	  set_block_to_screen(block_to_8gs);
	  break;
	case 4:
	  init_4gs();
	  set_block_to_screen(block_to_4gs);
	  break;
	case 1:
	  init_1gs();
	  set_block_to_screen(block_to_1gs);
	  break;
	}
      result = XGetRGBColormaps(disp, DefaultRootWindow(disp), 
				&std_cmap, &count, XA_RGB_GRAY_MAP);
      if (result)
	{ 
	  printf("RGB gray map found on server.\n");
	  XtSetArg(view_args[arg_count], XtNcolormap, std_cmap->colormap);
	  arg_count++;
	  view_map = std_cmap->colormap;
	}
      else
	{
	  fprintf(stderr, "No standard colormap available.\n");
	  exit(-1);
	}
    }
  else if (best_vis==StaticGray) 
    {
      switch (depth)
	{
	case 8:
	  set_block_to_screen(block_to_8gs);
	  break;
	case 4:
	  init_4gs();
	  set_block_to_screen(block_to_4gs);
	  break;
	case 1:
	  init_1gs();
	  set_block_to_screen(block_to_1gs);
	  break;
	}
      cmap = XCreateColormap(disp, DefaultRootWindow(disp), info.visual,
			     AllocNone);
      if (cmap!=(Colormap)NULL)
	{ 
	  printf("Colormap created succesfully.\n");
	  XtSetArg(view_args[arg_count], XtNcolormap, cmap);
	  arg_count++;
	  view_map = cmap;
	}
      else
	{
	  fprintf(stderr, "Colormap creation failure.\n");
	  exit(-1);
	}
    }
  else /* StaticColor or TrueColor */
    {
      need_mask = True;
      cmap = XCreateColormap(disp, DefaultRootWindow(disp), info.visual, 
			     AllocNone);
      switch (depth)
	{
	case 24:
	  set_block_to_screen(block_to_24bit);
	  break;
	case 12:
	  init_12c();
	  set_block_to_screen(block_to_12c);
	  break;
	case 8:
	  init_8c();
	  set_block_to_screen(block_to_8c);
	  break;
	default:
	  fprintf(stderr, "No converter for this visual\n");
	  exit(-1);
	  break;
	}
      if (cmap!=(Colormap)NULL)
	{ 
	  printf("Colormap created succesfully.\n");
	  XtSetArg(view_args[arg_count], XtNcolormap, cmap);
	  arg_count++;
	  view_map = cmap;
	}
      else
	{
	  fprintf(stderr, "Colormap creation failure.\n");
	  exit(-1);
	}
    }
  /* Atom for window deletion handling */
  wm_delete_window = XmInternAtom(disp, "WM_DELETE_WINDOW", TRUE);

  XtSetArg(wargs[0], XmNdeleteResponse, XmDO_NOTHING);
  toplevel = XtAppCreateShell(NULL, "Phoenix", applicationShellWidgetClass,
			      disp, wargs, 1);
  if (toplevel==NULL)
    {
      fprintf(stderr, "Toplevel shell creation error.\n");
      exit(-1);
    }
  XmAddWMProtocolCallback(toplevel, wm_delete_window,
			  (XtCallbackProc)delete_toolbox, (XtPointer)NULL);

  create_cursors(toplevel);
  create_dialogs(toplevel);
  make_toolbox(toplevel);

  XtRealizeWidget(toplevel);

  win = XtWindow(toplevel);
  disp = XtDisplay(toplevel);
  scr = DefaultScreen(disp);

  /* Deny toplevel resize */
  hints.max_width = TOP_WIDTH;
  hints.max_height = TOP_HEIGHT;
  hints.min_width = TOP_WIDTH;
  hints.min_height = TOP_HEIGHT;
  hints.flags = PMaxSize|PMinSize;
  XSetWMNormalHints(disp, win, &hints);

  /* This is from comp.windows.x faq, create a window for the busy cursor */
  mask = CWDontPropagate|CWCursor;
  attr.do_not_propagate_mask = (KeyPressMask|KeyReleaseMask|ButtonPressMask|
				ButtonReleaseMask|PointerMotionMask);
  attr.cursor = hourglass_c;
  busy_top = XCreateWindow(disp, win, 0, 0, DisplayWidth(disp, scr),
			   DisplayHeight(disp, scr), (unsigned int) 0, 0,
			   InputOnly, CopyFromParent, mask, &attr);

  update_menu();

  XtAppMainLoop(app);
}


/* FUNCTION:       make_toolbox(Widget toplevel)
   DESCRIPTION:    Creates the toolbox with menus and buttons.
   PARAMETERS:
   Widget toplevel Parent widget.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 15th August 1993.
*/   
void make_toolbox(Widget toplevel)
{
  Widget toolbox, menubar, toolbuttons;
  Arg wargs[20];
  Pixel fg, bg, ac, ts, bs;
  int i, j, c;
  Cardinal n;
  Dimension h;
  char name[3];

  n = 0;
  XtSetArg(wargs[n], XmNheight, TOP_HEIGHT); n++;
  XtSetArg(wargs[n], XmNwidth, TOP_WIDTH); n++;
  toolbox = XtCreateManagedWidget("Toolbox", xmFormWidgetClass, toplevel,
				  wargs, n);

  n = 0;
  XtSetArg(wargs[n], XmNspacing, 7); n++;
  XtSetArg(wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
  menubar = XmCreateMenuBar(toolbox, "Menubar", wargs, n);
  XtManageChild(menubar);

  menu = (menu_widgets *)NULL;
  xs_create_menu_buttons(NULL, menubar, PulldownData, menu, 
			 XtNumber(PulldownData));

  /* Create a label for messages from the main module */
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
  tool_label = XtCreateManagedWidget(" ", xmLabelWidgetClass, toolbox, wargs,
				      n);
  print_tool(); /* print the initial program state */

  XtSetArg(wargs[0], XmNheight, &h);
  XtGetValues(tool_label, wargs, 1);
  label_h = h;
  XtAddEventHandler(tool_label, ButtonPressMask, False, 
		    (XtEventHandler)lower_toolbox, (caddr_t)NULL);

  /* Set toolbutton flags */
  tool_set = TB_POINT;
  maskmode_set = TB_EVERYWHERE;
  drawmode_set = TB_REPLACE;
  perimmode_set = TB_PERIM;
  showedge_set = True;
  clone_set = False;

  /* create the toolbutton holding frame */
  n = 0;
  XtSetArg(wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNtopWidget, menubar); n++;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNbottomWidget, tool_label); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
  toolbuttons = XtCreateManagedWidget("Toolbuttons", xmFormWidgetClass,
				      toolbox, wargs, n);
  XtAddEventHandler(toolbuttons, ButtonPressMask, False, 
		    (XtEventHandler)lower_toolbox, (caddr_t)NULL);
  
  /* create a dummy button and get the color resources for pixmap creation */
  tool_btns[0] = XmCreatePushButton(toolbuttons, "dummy", NULL, 0);
  n = 0;
  XtSetArg(wargs[n], XmNforeground, &fg); n++;
  XtSetArg(wargs[n], XmNbackground, &bg); n++;
  XtSetArg(wargs[n], XmNarmColor, &ac); n++;
  XtSetArg(wargs[n], XmNtopShadowColor, &ts); n++;
  XtSetArg(wargs[n], XmNbottomShadowColor, &bs); n++;
  XtGetValues(tool_btns[0], wargs, n);
  XtDestroyWidget(tool_btns[0]);

  init_pixmaps(toolbuttons, fg, bg, ac);

  for (i = 0; i<NO_BUTTONS/(NO_BUTTONS/3); i++)
    {
      for (j = 0; j<NO_BUTTONS/3; j++)      
	{
	  c = i*NO_BUTTONS/3+j;
	  n = 0;
	  XtSetArg(wargs[n], XmNtopAttachment, XmATTACH_POSITION); n++;
	  XtSetArg(wargs[n], XmNtopPosition, 33*i); n++;
	  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
	  XtSetArg(wargs[n], XmNbottomPosition, 33*i+33); n++;
	  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	  XtSetArg(wargs[n], XmNleftPosition, 10*j); n++;
	  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	  XtSetArg(wargs[n], XmNrightPosition, 10*j+10); n++;
	  XtSetArg(wargs[n], XmNlabelType, XmPIXMAP); n++;
	  if (c==TB_POINT||c==TB_PERIM||c==TB_SHOWMASK||
	      c==TB_EVERYWHERE||c==TB_REPLACE)
	    {
	      
	      XtSetArg(wargs[n], XmNlabelPixmap, tool_bmps[c+NO_PIXMAPS]); n++;
	      XtSetArg(wargs[n], XmNtopShadowColor, bs); n++;
	      XtSetArg(wargs[n], XmNbottomShadowColor, ts); n++;
	    }
	  else
	    {
	      XtSetArg(wargs[n], XmNlabelPixmap, tool_bmps[c]); n++;
	    }
	  sprintf(name, "%d", c);
	  tool_btns[c] =
	    XtCreateManagedWidget(name, xmPushButtonWidgetClass, toolbuttons,
				  wargs, n);
	  XtAddCallback(tool_btns[c], XmNarmCallback, 
			(XtCallbackProc)button_arm_callback, (caddr_t)c);
	  XtAddCallback(tool_btns[c], XmNdisarmCallback, 
			(XtCallbackProc)button_disarm_callback, (caddr_t)c);
	  XtAddEventHandler(tool_btns[c], ButtonPressMask, False, 
			    (XtEventHandler)lower_toolbox, (caddr_t)NULL);
	}
    }
}


/* FUNCTION:       make_view_window(struct view *view_p)
   DESCRIPTION:    Creates a view window: a scrolled DrawingArea and some
                   Labels.
   PARAMETERS:
   struct view *view_p Pointer to view structure binding the window to a
                       picture in memory.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void make_view_window(struct view *view_p)
{
  extern int depth;
  Widget window, scrolld_w, prim_label, prim_frame, 
  second_label, second_frame;
  Arg wargs[15];
  Cardinal n;
  Display *disp;
  Window win;
  XSizeHints hints;
  XmString zoom_xs;
  char zoom[10];
  int scr, scroll_h, scroll_w;
  unsigned long mask;
  XSetWindowAttributes attr;
  XImage *image;

  /* create a shell with the apropriate resources */
  XtSetArg(view_args[arg_count], XmNdeleteResponse, XmDO_NOTHING); arg_count++;
  view_p->main = XtAppCreateShell(NULL, "Phoenix", topLevelShellWidgetClass,
				  XtDisplay(toplevel), view_args, arg_count);
  XtAddEventHandler(view_p->main, StructureNotifyMask, False, 
		    (XtEventHandler)view_map_handler, view_p);
  XmAddWMProtocolCallback(view_p->main, wm_delete_window,
			  (XtCallbackProc)delete_canvas, (XtPointer)view_p);

  disp = XtDisplay(view_p->main);
  scr = DefaultScreen(disp);

  /* Initializing */
  if (first_time)
    {
      if (need_mask)
	{
	  image = XCreateImage(disp, view_vis, depth, ZPixmap, 0,
			       None, 10, 10, pad, 0);
/*	  printf("r-mask: %#x, g-mask: %#x, b-mask: %#x\n", image->red_mask,
		 image->green_mask, image->blue_mask);*/
	  XFree(image);
	}
      get_global_bounds(toplevel, &decor_w, &decor_h);
      decor_w = decor_w-TOP_WIDTH;
      decor_h = decor_h-TOP_HEIGHT;
      ind_h = label_h-4;
      ind_w = label_h;
      bar_t = 17;
      space = 4;
    }

  window = XtCreateManagedWidget("ViewWindow", xmFormWidgetClass, view_p->main,
				 NULL, 0);

  /* Make the labels and indicators in the bottom of the window */
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
  view_p->coord_ind = XtCreateManagedWidget("(    ,    )",
					    xmLabelWidgetClass, window,
					    wargs, n);
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  second_frame = XtCreateManagedWidget("Frame", xmFrameWidgetClass, window,
				       wargs, n);
  n = 0;
  XtSetArg(wargs[n], XmNheight, ind_h); n++;
  XtSetArg(wargs[n], XmNwidth, ind_w); n++;
  view_p->second_ind = XtCreateManagedWidget("SecondaryColor", 
					     xmDrawingAreaWidgetClass, 
					     second_frame, wargs, n);
  XtAddCallback(view_p->second_ind, XmNexposeCallback,
		(XtCallbackProc)expose_ind, view_p);
  XtAddEventHandler(view_p->second_ind, ButtonPressMask, False, 
		    (XtEventHandler)color_select_callback, (caddr_t)SEC);
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNrightWidget, second_frame); n++;
  second_label = XtCreateManagedWidget("2:", xmLabelWidgetClass, window, wargs,
				       n);
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNrightWidget, second_label); n++;
  prim_frame = XtCreateManagedWidget("Frame", xmFrameWidgetClass, window,
				     wargs, n);
  n = 0;
  XtSetArg(wargs[n], XmNheight, ind_h); n++;
  XtSetArg(wargs[n], XmNwidth, ind_w); n++;
  view_p->prim_ind = XtCreateManagedWidget("PrimaryColor", 
					   xmDrawingAreaWidgetClass, 
					   prim_frame, wargs, n);
  XtAddCallback(view_p->prim_ind, XmNexposeCallback,
		(XtCallbackProc)expose_ind, view_p);
  XtAddEventHandler(view_p->prim_ind, ButtonPressMask, False, 
		    (XtEventHandler)color_select_callback, (caddr_t)PRIM);
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNrightWidget, prim_frame); n++;
  prim_label = XtCreateManagedWidget("1:", xmLabelWidgetClass, window, wargs,
				       n);
  sprintf(zoom, " %d00%% ( )", view_p->scale);
  zoom_xs = XmStringLtoRCreate(zoom, XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNrightWidget, prim_label); n++;
  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNleftWidget, view_p->coord_ind); n++;
  XtSetArg(wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
  XtSetArg(wargs[n], XmNlabelString, zoom_xs); n++;
  view_p->zoom = XtCreateManagedWidget("Label", xmLabelWidgetClass,
				       window, wargs, n);
  XmStringFree(zoom_xs);

  /* If the picture doesn't fit to the screen, make the window smaller */
  if (view_p->scale*view_p->pic_p->width+decor_w+bar_t+space
      <XDisplayWidth(disp, scr))
    scroll_w = view_p->scale*view_p->pic_p->width+bar_t+space;
  else
    scroll_w = XDisplayWidth(disp, scr)-decor_w;
  if (view_p->scale*view_p->pic_p->height+decor_h+label_h+bar_t+space
      <XDisplayHeight(disp, scr)) 
    scroll_h = view_p->scale*view_p->pic_p->height+bar_t+space;
  else
    scroll_h = XDisplayHeight(disp, scr)-decor_h-label_h;

  /* Create the scrolled window, no automatic scrolling */
  n = 0;
  XtSetArg(wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNbottomWidget, view_p->zoom); n++;  
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNscrollingPolicy, XmAPPLICATION_DEFINED); n++;
  XtSetArg(wargs[n], XmNspacing, space); n++;
  XtSetArg(wargs[n], XmNshadowThickness, 0); n++;
  XtSetArg(wargs[n], XmNheight, scroll_h); n++;
  XtSetArg(wargs[n], XmNwidth, scroll_w); n++;
  scrolld_w = XtCreateManagedWidget("ScrolledWindow",
				    xmScrolledWindowWidgetClass, window,
				    wargs, n);

  /* Create the horizontal scrollbar and add the callbacks */
  n = 0;
  XtSetArg(wargs[n], XmNminimum, 0); n++;
  XtSetArg(wargs[n], XmNmaximum, view_p->pic_p->width-1); n++;
  XtSetArg(wargs[n], XmNsliderSize,
	   (scroll_w-bar_t-space)/view_p->scale-1); n++;
  XtSetArg(wargs[n], XmNpageIncrement,
	   (scroll_w-bar_t-space)/view_p->scale); n++;
  XtSetArg(wargs[n], XmNincrement, view_p->scale); n++;
  XtSetArg(wargs[n], XmNorientation, XmHORIZONTAL); n++;
  XtSetArg(wargs[n], XmNheight, bar_t); n++;
  view_p->h_scroll = XtCreateManagedWidget("ScrollBar",
				    xmScrollBarWidgetClass, scrolld_w,
				    wargs, n);
  XtAddCallback(view_p->h_scroll, XmNdragCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);
  XtAddCallback(view_p->h_scroll, XmNincrementCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);
  XtAddCallback(view_p->h_scroll, XmNdecrementCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);
  XtAddCallback(view_p->h_scroll, XmNpageIncrementCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);
  XtAddCallback(view_p->h_scroll, XmNpageDecrementCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);
  XtAddCallback(view_p->h_scroll, XmNtoTopCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);
  XtAddCallback(view_p->h_scroll, XmNtoBottomCallback,
		(XtCallbackProc)h_scroll_canvas, view_p);

  /* Create the vertical scrollbar and add the callbacks*/
  n = 0;
  XtSetArg(wargs[n], XmNminimum, 0); n++;
  XtSetArg(wargs[n], XmNmaximum, view_p->pic_p->height-1); n++;
  XtSetArg(wargs[n], XmNsliderSize,
	   (scroll_h-bar_t-space)/view_p->scale-1); n++;
  XtSetArg(wargs[n], XmNpageIncrement,
	   (scroll_h-bar_t-space)/view_p->scale); n++;
  XtSetArg(wargs[n], XmNincrement, view_p->scale); n++;
  XtSetArg(wargs[n], XmNorientation, XmVERTICAL); n++;
  XtSetArg(wargs[n], XmNwidth, bar_t); n++;
  view_p->v_scroll = XtCreateManagedWidget("ScrollBar",
				    xmScrollBarWidgetClass, scrolld_w,
				    wargs, n);
  XtAddCallback(view_p->v_scroll, XmNdragCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  XtAddCallback(view_p->v_scroll, XmNincrementCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  XtAddCallback(view_p->v_scroll, XmNdecrementCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  XtAddCallback(view_p->v_scroll, XmNpageIncrementCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  XtAddCallback(view_p->v_scroll, XmNpageDecrementCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  XtAddCallback(view_p->v_scroll, XmNtoTopCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  XtAddCallback(view_p->v_scroll, XmNtoBottomCallback,
		(XtCallbackProc)v_scroll_canvas, view_p);
  
  /* Create the DrawingArea for viewing the actual image */
  n = 0;
  XtSetArg(wargs[n], XmNwidth, XDisplayWidth(disp, scr)); n++;
  XtSetArg(wargs[n], XmNheight, XDisplayHeight(disp, scr)); n++;
  view_p->draw_area = XtCreateManagedWidget("Canvas",
					    xmDrawingAreaWidgetClass, 
					    scrolld_w,
					    wargs, n);
  XtAddCallback(view_p->draw_area, XmNresizeCallback,
		(XtCallbackProc)resize_canvas, view_p);
  XtAddEventHandler(view_p->draw_area, ExposureMask, False,
		    (XtEventHandler)canvas_exposure_handler, view_p);
  XtAddEventHandler(view_p->draw_area, ButtonReleaseMask, False, 
		    (XtEventHandler)canvas_release_handler, view_p);
  XtAddEventHandler(view_p->draw_area, ButtonPressMask, False, 
		    (XtEventHandler)canvas_press_handler, view_p);
  XtAddEventHandler(view_p->draw_area, PointerMotionMask, False, 
		    (XtEventHandler)canvas_motion_handler, view_p);
  XtAddEventHandler(view_p->draw_area, LeaveWindowMask, False, 
		    (XtEventHandler)canvas_leave_handler, view_p);
  XtAddEventHandler(view_p->draw_area, KeyPressMask, False, 
		    (XtEventHandler)canvas_keypress_handler, view_p);

  XmScrolledWindowSetAreas(scrolld_w, view_p->h_scroll, view_p->v_scroll,
			   view_p->draw_area);
  XtRealizeWidget(view_p->main);
  win = XtWindow(view_p->main);

  /* Create GC to receive graphicsexpose events */
  view_p->gc = XCreateGC(disp, win, (XtGCMask)NULL, NULL);
  XSetGraphicsExposures(disp, view_p->gc, TRUE);

  /* This is from comp.windows.x faq, create a window for the busy cursor */
  mask = CWDontPropagate|CWCursor;
  attr.do_not_propagate_mask = (KeyPressMask|KeyReleaseMask|ButtonPressMask|
				ButtonReleaseMask|PointerMotionMask);
  attr.cursor = hourglass_c;
  view_p->busy_view =
    XCreateWindow(disp, win, 0, 0, DisplayWidth(disp, scr),
		  DisplayHeight(disp, scr), (unsigned int) 0, 0, InputOnly, 
		  CopyFromParent, mask, &attr);

  /* map the hourglass for this window too */
  all_busy();

  /* Deny larger window than the image or the screen */
  hints.max_width =  scroll_w;
  hints.max_height = scroll_h+label_h;
  hints.flags = PMaxSize;
  XSetWMNormalHints(disp, win, &hints);

  /* name is the name of the being edited */
  XStoreName(disp, win, view_p->pic_p->name);
}


/* FUNCTION:       xs_create_menu_buttons(char *title, Widget cascade,
                                          xs_menu_struct *menulist, 
					  menu_widgets *menu_p, int nitems)
   DESCRIPTION:    Creates a menu system with data defined in window.h
   PARAMETERS:
   char *title              Title for the menu.
   Widget cascade           Menubar's or CascadeButton's widget_id.
   xs_menu_struct *menulist Menu data structure.
   menu_widgets *menu_p     Pointer to menu_widget struct.
   int nitems               Items in a menu or a sub-menu.
   RETURNS:        Nothing.

   Written by Douglas A. Young, last modifications (by Vesa T|rm{nen)
   28th June 1993.
*/
void xs_create_menu_buttons(char *title, Widget cascade, 
			    xs_menu_struct *menulist, menu_widgets *menu_p, 
			    int nitems)
{
  Arg wargs[10];
  int i;
  Cardinal n;
  int separators = 0;
  XmString label, acc;
  XFontStruct *menu_font;
  XmFontList menu_fontlist;
  Display *display;
  
  display = XtDisplay(cascade);
  menu_font = XLoadQueryFont(display, "-b&h-lucida-bold-r-normal-sans-14-100-100-100-p-80-iso8859-1");
  menu_fontlist = XmFontListCreate(menu_font, "menu_chset");

  /* allocate memory for a new _menu_w struct */
  if (menu_p==NULL)
    {
      menu = alloc_menu();
      menu_p = menu;
    }
  else
    {
      while (menu_p->next!=NULL)
	menu_p = menu_p->next;
      menu_p->next = alloc_menu();
      menu_p = menu_p->next;
    }
  /*
   * Allocate a widget list to hold all 
   * button widgets.
   */
  menu_p->menu_buttons = (WidgetList) XtMalloc(nitems * sizeof(Widget));
  if (menu_p->menu_buttons==NULL)
    show_errormsg("xs_create_menu_buttons(): Memory allocation error.", 
		  "OK", NULL);
  menu_p->button_labels = (char (*)[])calloc(nitems, MAX_LABEL*sizeof(char));
  if (menu_p->button_labels==NULL)
    show_errormsg("xs_create_menu_buttons(): Memory allocation error.", 
		  "OK", NULL);
  /*
   * If a title is given, create Label and Separator widgets.
   */
  if (title)
    {
      label = XmStringCreateLtoR(title, "menu_chset");
      n = 0;
      XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
      XtSetArg(wargs[n], XmNlabelString, label); n++;
      XtCreateManagedWidget(title, xmLabelWidgetClass, cascade,
			    wargs, n);      
      XmStringFree(label);

      n = 0;
      XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
      XtCreateManagedWidget("separator", xmSeparatorWidgetClass,
			    cascade, wargs, n);
    }
  /*
   * Create an entry for each item in the menu.
   */
  for (i = 0; i<nitems; i++)
    {
   /* 
    * A NULL name represents a separator.
    */
      if (menulist[i].name==NULL)
	{ 
	  n = 0;
	  XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
	  XtCreateManagedWidget("separator",
				xmSeparatorWidgetClass,
				cascade, wargs, n);
	  separators++; /* Count how many entries aren't menu_buttons */
	}
      /*
       * If there is a name and a callback, create a "normal"
       * menu entry and register the callback function.
       */
      else if (menulist[i].func)
	{
	  if (menulist[i].flag_toggle)
	    {
	      label = XmStringCreateLtoR(menulist[i].name, "menu_chset");
	      n = 0;
	      if (menulist[i].flag_toggle==RADIO)
		{
		  XtSetArg(wargs[n], XmNindicatorType, XmONE_OF_MANY); n++;
		}
	      if (menulist[i].state==INSENSITIVE)
		{
		  XtSetArg(wargs[n], XmNsensitive, FALSE); n++;
		}
	      else if (menulist[i].state==SENSITIVE_SET)
		{
		  XtSetArg(wargs[n], XmNset, TRUE); n++;
		}
	      XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
	      XtSetArg(wargs[n], XmNlabelString, label); n++;
	      XtSetArg(wargs[n], XmNvisibleWhenOff, TRUE); n++;
	      XtSetArg(wargs[n], XmNindicatorOn, TRUE); n++;
	      XtSetArg(wargs[n], XmNfillOnSelect, TRUE); n++;
	      strcpy(menu_p->button_labels[i-separators], menulist[i].name);
	      menu_p->menu_buttons[i-separators] =
		XtCreateWidget(menulist[i].name, xmToggleButtonGadgetClass,
			       cascade, wargs, n);
	      XtAddCallback(menu_p->menu_buttons[i-separators],
			    XmNvalueChangedCallback,
			    (XtCallbackProc)menulist[i].func,
			    (caddr_t)menulist[i].data);
	      XmStringFree(label);
	    }
	  else
	    {
	      label = XmStringCreateLtoR(menulist[i].name, "menu_chset");
	      acc = XmStringCreateLtoR(menulist[i].acctext, "menu_chset");
	      n = 0;
	      if (menulist[i].state==INSENSITIVE)
		{
		  XtSetArg(wargs[n], XmNsensitive, FALSE); n++;
		}
	      XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
	      XtSetArg(wargs[n], XmNlabelString, label); n++;
	      XtSetArg(wargs[n], XmNaccelerator, menulist[i].accelerator); n++;
	      XtSetArg(wargs[n], XmNacceleratorText, acc); n++;
	      strcpy(menu_p->button_labels[i-separators], menulist[i].name);
	      menu_p->menu_buttons[i-separators] =
		XtCreateWidget(menulist[i].name, xmPushButtonWidgetClass,
			       cascade, wargs, n);
	      XtAddCallback(menu_p->menu_buttons[i-separators],
			    XmNactivateCallback,
			    (XtCallbackProc)menulist[i].func,
			    (caddr_t)menulist[i].data);
	      XmStringFree(label);
	    }
	}
      /*
       * If there is a name, but no callback function, the entry
       * must be a label, unless there is a submenu.
       */
      else if (!menulist[i].sub_menu)
	{
	  label = XmStringCreateLtoR(menulist[i].name, "menu_chset");
	  n = 0;
	  XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
	  XtSetArg(wargs[n], XmNlabelString, label); n++;
	  strcpy(menu_p->button_labels[i-separators], menulist[i].name);
	  menu_p->menu_buttons[i-separators] = XtCreateWidget(menulist[i].name,
							    xmLabelWidgetClass,
							    cascade, wargs, n);
	  XmStringFree(label);
	}
      /*
       * If we got here, the entry must be a submenu.
       * Create a pulldown menu pane and an XmCascadeButton
       * widget. Attach the menu pane and make a recursive call
       * to create the entries in the submenu.
       */
      else
	{  
	  Widget sub_menu;
	  sub_menu = XmCreatePulldownMenu(cascade, menulist[i].sub_menu_title,
					  NULL, 0);
	  label = XmStringCreateLtoR(menulist[i].name, "menu_chset");
	  n = 0;
	  XtSetArg(wargs[n], XmNsubMenuId, sub_menu); n++;
	  XtSetArg(wargs[n], XmNfontList, menu_fontlist); n++;      
	  XtSetArg(wargs[n], XmNlabelString, label); n++;
	  if (menulist[i].state==INSENSITIVE)
	    {
	      XtSetArg(wargs[n], XmNsensitive, FALSE); n++;
	    }
	  strcpy(menu_p->button_labels[i-separators], menulist[i].name);
	  menu_p->menu_buttons[i-separators] =
	    XtCreateWidget(menulist[i].name, xmCascadeButtonWidgetClass,
			   cascade, wargs, n);
	  XmStringFree(label);
	  if (menulist[i].name=="Help")
	    {
	      XtSetArg(wargs[0], XmNmenuHelpWidget,
		       menu_p->menu_buttons[i-separators]);
	      XtSetValues(cascade, wargs, 1);
	    }
	  xs_create_menu_buttons(menulist[i].sub_menu_title, sub_menu,
				 menulist[i].sub_menu, menu_p,
				 menulist[i].n_sub_items);
	}
    } 
  /*
   * Manage all button widgets. Menu panes are not managed.
   */
  XtManageChildren(menu_p->menu_buttons, nitems - separators);
  menu_p->items = nitems-separators;
}


/* FUNCTION:       get_menu_widget(char *entry)
   DESCRIPTION:    Returns the widget id of the menu button with label entry,
                   NULL if no match was found.
   PARAMETERS:
   char *entry     Label of the menu button.
   RETURNS:        Widget.

   Written by Vesa T|rm{nen, last modifications 17th July 1993.
*/   
Widget get_menu_widget(char *entry)
{
  int i;
  menu_widgets *menu_p = menu;
  
  while (menu_p!=NULL)
    {
      for (i = 0; i<menu_p->items; i++)
	if (!strcmp(entry, menu_p->button_labels[i]))
	    return menu_p->menu_buttons[i];
      menu_p = menu_p->next;
    }
  return (Widget)NULL;
}


/* FUNCTION:       alloc_menu(void)
   DESCRIPTION:    Allocates a new menu_widget struct
   PARAMETERS:     
   void            Nothing.
   RETURNS:        Pointer to the allocated struct.

   Written by Vesa T|rm{nen, last modifications 17th July 1993.
*/   
menu_widgets *alloc_menu(void)
{
  menu_widgets *menu_p = (menu_widgets *)malloc(sizeof(struct _menu_w));
  if (menu_p==NULL)
    {
      show_errormsg("alloc_menu(): Memory allocation error.", "OK", NULL);
      exit(-1);
    }
  menu_p->next = (menu_widgets *)NULL;
  return menu_p;
}
