/* FILE: filter.c

   Filter routines are here.

   Written by Petri Kuittinen, last modifications 12th August 1993.
*/ 

#include <stdlib.h>
#include "misc.h"
#include "pixel.h"
#include "filter.h"


/* FUNCTION:     void invert_block# (struct picture *pic_p)
   DESCRIPTION:  inverts every pixel
   #=1           inside screen (no mask check)
   #=2           inside masked area. 
   #=3           outside masked area.
   PARAMETERS:
   struct picture *pic_p  pointer to a block to be inverted.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 5th August 1993
*/
void invert_block1 (struct picture *pic_p)
{
  register int i;
  register pixel *p;
  
  i = pic_p->height*pic_p->width;
  p = pic_p->buf;

  while (i--)
    {
      *p ^= CMASK;
      p++;
    }
}

void invert_block2 (struct picture *pic_p)
{
  register int w, h, x_offset;
  register pixel *p;
  
  w = pic_p->max_x-pic_p->min_x+1;
  h = pic_p->max_y-pic_p->min_y+1;
  x_offset = pic_p->width-w;
  p = pic_p->buf+pic_p->min_x+pic_p->width*pic_p->min_y;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*p))
	    *p ^= CMASK;
	  p++;
	}
      w = pic_p->max_x-pic_p->min_x+1;
      p += x_offset;
    }
}

void invert_block3 (struct picture *pic_p)
{
  register int i;
  register pixel *p;
  
  i = pic_p->height*pic_p->width;
  p = pic_p->buf;

  while (i--)
    {
      if (!(IS_MASK(*p)))
	*p ^= CMASK;
      p++;
    }
}


/* FUNCTION:     void rgb_adjust_block1 (struct picture *pic_p,
   double r_scale, double g_scale, double b_scale, int r_add, int g_add,
   int b_add
   DESCRIPTION:  does RGB adjusting to every pixel
   #=1           inside screen (no mask check)
   #=2           inside masked area. 
   #=3           outside masked area.
   PARAMETERS:
   struct picture *pic_p  pointer to a block to be operated.
   double r_scale  value to multiply r-component
   double g_scale  value to multiply g-component
   double b_scale  value to multiply b-component
   int r_add     value to add to r-component (after scaling)
   int g_add     value to add to r-component (-""-)
   int b_add     value to add to r-component (-""-)
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 5th August 1993
*/
void rgb_adjust_block1 (struct picture *pic_p, double r_scale, double g_scale,
			double b_scale, int r_add, int g_add, int b_add)
{
  register int i;
  register pixel *p, r, g, b;
  
  i = pic_p->height*pic_p->width;
  p = pic_p->buf;

  while (i--)
    {
      r = (pixel)(GET_RED(*p)*r_scale)+r_add; LIMIT (r);
      g = (pixel)(GET_GREEN(*p)*g_scale)+g_add; LIMIT (g);
      b = (pixel)(GET_BLUE(*p)*b_scale)+b_add; LIMIT (b);
      *p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
      p++;
    }
}

void rgb_adjust_block2 (struct picture *pic_p, double r_scale, double g_scale,
			double b_scale, int r_add, int g_add, int b_add)
{
  register int w, h, x_offset;
  register pixel *p, r, g, b;
  
  w = pic_p->max_x-pic_p->min_x+1;
  h = pic_p->max_y-pic_p->min_y+1;
  x_offset = pic_p->width-w;
  p = pic_p->buf+pic_p->min_x+pic_p->width*pic_p->min_y;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*p))
	    {
	      r = (pixel)(GET_RED(*p)*r_scale)+r_add; LIMIT (r);
	      g = (pixel)(GET_GREEN(*p)*g_scale)+g_add; LIMIT (g);
	      b = (pixel)(GET_BLUE(*p)*b_scale)+b_add; LIMIT (b);
	      *p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  p++;
	}
      w = pic_p->max_x-pic_p->min_x+1;
      p += x_offset;
    }
}

void rgb_adjust_block3 (struct picture *pic_p, double r_scale, double g_scale,
			double b_scale, int r_add, int g_add, int b_add)
{
  register int i;
  register pixel *p, r, g, b;
  
  i = pic_p->height*pic_p->width;
  p = pic_p->buf;

  while (i--)
    {
      if (!(IS_MASK(*p)))
	{
	  r = (pixel)(GET_RED(*p)*r_scale)+r_add; LIMIT (r);
	  g = (pixel)(GET_GREEN(*p)*g_scale)+g_add; LIMIT (g);
	  b = (pixel)(GET_BLUE(*p)*b_scale)+b_add; LIMIT (b);
	  *p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
      p++;
    }
}

/* Some macros used in do_5x5matrix_block to reduce the amount of code. */
#define INIT \
area = tar = src1 = calloc(5*(pic_p->width+4), sizeof(pixel)); \
       if (src1==NULL) return; \
       src1 += 2*(pic_p->width+4); \
       p = pic_p->buf; \
       h = pic_p->height; \
       tar += 5*2+pic_p->width*2; \
       w = pic_p->width; \
       while (w--) *tar++ = *p++; \
       if ((--h)==0) goto done; \
       tar += 4; w = pic_p->width; \
       while (w--) *tar++ = *p++; \
       if ((--h)==0) goto done; \
       tar += 4; w = pic_p->width; \
       while (w--) *tar++ = *p++; \
       if ((--h)==0) goto done; \
     done: tar = pic_p->buf;

/* Lot's of space for optimisations here, how ever I have no time that... */
#define MAT(X_OFF,Y_OFF) \
  src2 = src1+(X_OFF)+(Y_OFF)*(pic_p->width+4); \
  val = GET_RED(*src2); r += (*m)*val; \
  val = GET_GREEN(*src2); g += (*m)*val; \
  val = GET_BLUE(*src2); b += (*m)*val; \
  m++;

#define DO_5X5 \
r = g = b = 0; m = mat; \
  MAT(-2, -2); MAT(-1, -2); MAT(0, -2); MAT(1, -2); MAT(2, -2); \
  MAT(-2, -1); MAT(-1, 1); MAT(0, -1); MAT(1, -1); MAT(2, -1); \
  MAT(-2, 0); MAT(-1, 0); MAT(0, 0); MAT(1, 0); MAT(2, 0); \
  MAT(-2, 1); MAT(-1, 1); MAT(0, 1); MAT(1, 1); MAT(2, 1); \
  MAT(-2, 2); MAT(-1, 2); MAT(0, 2); MAT(1, 2); MAT(2, 2); \
  r = r*muldiv; LIMIT (r); \
  g = g*muldiv; LIMIT (g); \
  b = b*muldiv; LIMIT (b); \
  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);

#define SCROLL_DOWN \
src1 -= pic_p->width*2+4; \
  src2 = src1-pic_p->width-4; \
  w = pic_p->width; \
  while (w--) *src2++ = *src1++; \
  src1 += 4; src2 += 4 ; w = pic_p->width; \
  while (w--) *src2++ = *src1++; \
  src1 += 4; src2 += 4 ; w = pic_p->width; \
  while (w--) *src2++ = *src1++; \
  src1 += 4; src2 += 4 ; w = pic_p->width; \
  while (w--) *src2++ = *src1++; \
  src2 += 4; w = pic_p->width; \
  if (h>2) \
  while (w--) *src2++ = *p++; \
  else \
  while (w--) *src2++ = 0; \
  src1 -= pic_p->width*3+8;

/* FUNCTION:     void do_5x5_matrix_block1 (struct picture *pic_p, int mul,
   int div, int *mat);
   DESCRIPTION:  do 5x5 matrix filtering with every pixel
   #=1           inside screen (no mask check)
   #=2           inside masked area. 
   #=3           outside masked area.
   PARAMETERS:
   struct picture *pic_p
   double mul    multiplier value.
   double div    divider value.
   double *mat   pointer to 5x5 int array;
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 10th August 1993
*/
void do_5x5_matrix_block1 (struct picture *pic_p, double mul, double div,
			   double *mat)
{
  register int h, w;
  register double *m, muldiv;
  register pixel val, r, g, b, *p, *src1, *src2, *tar;
  pixel *area;

  INIT;

  h = pic_p->height;
  muldiv = mul/div;

  while (h--)
    {
      w = pic_p->width;

      while (w--)
	{
	  DO_5X5;
	  src1++; tar++;
	}
      SCROLL_DOWN;
    }
  free (area);
}

void do_5x5_matrix_block2 (struct picture *pic_p, double mul, double div,
			   double *mat)
{
  register int h, w;
  register double *m, muldiv;
  register pixel val, r, g, b, *p, *src1, *src2, *tar;
  pixel *area;

  INIT;

  h = pic_p->height;
  muldiv = mul/div;

  while (h--)
    {
      w = pic_p->width;

      while (w--)
	{
	  if (IS_MASK(*tar))
	    {
	      DO_5X5;
	    }
	  src1++; tar++;
	}
      SCROLL_DOWN;
    }
  free (area);
}

void do_5x5_matrix_block3 (struct picture *pic_p, double mul, double div,
			   double *mat)
{
  register int h, w;
  register double *m, muldiv;
  register pixel val, r, g, b, *p, *src1, *src2, *tar;
  pixel *area;

  INIT;

  h = pic_p->height;
  muldiv = mul/div;

  while (h--)
    {
      w = pic_p->width;

      while (w--)
	{
	  if (!(IS_MASK(*tar)))
	    {
	      DO_5X5;
	    }
	  src1++; tar++;
	}
      SCROLL_DOWN;
    }
  free (area);
}


/* FUNCTION:     void add_noise_block# (struct picture *pic_p, int range)
   DESCRIPTION:  add noise to every pixel
   #=1           inside screen (no mask check)
   #=2           inside masked area. 
   #=3           outside masked area.
   PARAMETERS:
   struct picture *pic_p
   int range     max value to add or sub from pixel value.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 9th August 1993
*/
void add_noise_block1 (struct picture *pic_p, int range)
{
  register int i, mod;
  register pixel r, g, b, *p;
  
  i = pic_p->height*pic_p->width;
  mod = range*2;
  p = pic_p->buf;

  while (i--)
    {
      r = GET_RED(*p)+(rnd_int()%mod)-range; LIMIT (r);
      g = GET_GREEN(*p)+(rnd_int()%mod)-range; LIMIT (g);
      b = GET_BLUE(*p)+(rnd_int()%mod)-range; LIMIT (b);
      *p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
      p++;
    }
}

void add_noise_block2 (struct picture *pic_p, int range)
{
  register int mod, w, h, x_offset;
  register pixel r, g, b, *p;
  
  mod = range*2;
  w = pic_p->max_x-pic_p->min_x+1;
  h = pic_p->max_y-pic_p->min_y+1;
  x_offset = pic_p->width-w;
  p = pic_p->buf+pic_p->min_x+pic_p->width*pic_p->min_y;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*p))
	    {
	      r = GET_RED(*p)+(rnd_int()%mod)-range; LIMIT (r);
	      g = GET_GREEN(*p)+(rnd_int()%mod)-range; LIMIT (g);
	      b = GET_BLUE(*p)+(rnd_int()%mod)-range; LIMIT (b);
	      *p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  p++;
	}
      w = pic_p->max_x-pic_p->min_x+1;
      p += x_offset;
    }
}

void add_noise_block3 (struct picture *pic_p, int range)
{
  register int i, mod;
  register pixel r, g, b, *p;
  
  i = pic_p->height*pic_p->width;
  mod = range*2;
  p = pic_p->buf;

  while (i--)
    {
      if (!(IS_MASK(*p)))
	{
	  r = GET_RED(*p)+(rnd_int()%mod)-range; LIMIT (r);
	  g = GET_GREEN(*p)+(rnd_int()%mod)-range; LIMIT (g);
	  b = GET_BLUE(*p)+(rnd_int()%mod)-range; LIMIT (b);
	  *p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
      p++;
    }
}


/* A large macro to reduce lenght of code */
#define DO_HSV_ADJUST \
r = GET_RED(*p); g = GET_GREEN(*p); b = GET_BLUE(*p); \
/* RGB to HSV */ \
t1 = MAX(r, g); t1 = MAX(t1, b); \
t2 = MIN(r, g); t2 = MIN(t2, b); \
v = t1/255.0; \
if (t1!=0.0) \
 s = (t1-t2)/t1; \
else \
 s = 0.0; \
if (s==0.0) \
 h = -1; /* Saturation is zero, so hue cannot be determined */ \
else \
{ \
 t3 = t1-t2; \
 if (r==t1) \
  h = (g-b)/t3; \
 else if (g==t1) \
  h = 2+(b-r)/t3; \
 else \
  h = 4+(r-g)/t3; \
 /* h *= 60; */\
 if (h<0) h += 6; \
} \
/* HSV mod */ \
if (h>=0) \
{ h += h_add; if (h<0) h += 6.0; else if (h>6.0) h -= 6.0; } \
s += s_add; if (s<0) s = 0; else if (s>1.0) s = 1.0; \
v += v_add; if (v<0) v = 0; else if (v>1.0) v = 1.0; \
\
/* HSV back to RGB */ \
if (h<0) \
 r = g = b = 255.0*v; \
else \
{ \
 /* if (h==360) h=0; \
 h /= 60; */ \
 j = h; \
 t1 = v*(1-s); \
 t2 = v*(1-s*(h-j)); \
 t3 = v*(1-(s*(1-(h-j)))); \
 switch(j) \
 { \
  case 0: \
   r = 255.0*v; g = 255.0*t3; b = 255.0*t1; break; \
  case 1: \
   r = 255.0*t2; g = 255.0*v; b = 255.0*t1; break; \
  case 2: \
   r = 255.0*t1; g = 255.0*v; b = 255.0*t3; break; \
  case 3: \
   r = 255.0*t1; g = 255.0*t2; b = 255.0*v; break; \
  case 4: \
   r = 255.0*t3; g = 255.0*t1; b = 255.0*v; break; \
  default: \
   r = 255.0*v; g = 255.0*t1; b = 255.0*t2; break; \
 } \
} \
*p = GET_UP_BITS (*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);

/* FUNCTION:     void hsv_adjust_block1 (struct picture *pic_p, double h_add,
   double s_add, double v_add)
   DESCRIPTION:  adjusts HSV components of every pixel
   #=1           inside screen (no mask check)
   #=2           inside masked area. 
   #=3           outside masked area.
   PARAMETERS:
   struct picture *pic_p
   double h_add  value to add to h (hue) ]-360, 360 [.
   double s_add  value to add to s (saturation) [-1, 1].
   double v_add  value to add to v (value) [-1, 1].
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 12th August 1993
*/
void hsv_adjust_block1 (struct picture *pic_p, double h_add, double s_add,
			double v_add)
{
  register int i, j;
  register pixel r, g, b, *p;
  register double h, s, v, t1, t2, t3;
  
  i = pic_p->height*pic_p->width;
  p = pic_p->buf;
  h_add /= 60;

  while (i--)
    {
      DO_HSV_ADJUST;
      p++;
    }
}

void hsv_adjust_block2 (struct picture *pic_p, double h_add, double s_add,
			double v_add)
{
  register int j, w, i, x_offset;

  register pixel r, g, b, *p;
  register double h, s, v, t1, t2, t3;
  
  w = pic_p->max_x-pic_p->min_x+1;
  i = pic_p->max_y-pic_p->min_y+1;
  x_offset = pic_p->width-w;
  p = pic_p->buf+pic_p->min_x+pic_p->width*pic_p->min_y;
  h_add /= 60;

  while (i--)
    {
      while (w--)
	{
	  if (IS_MASK(*p))
	    {
	      DO_HSV_ADJUST;
	    }
	  p++;
	}
      w = pic_p->max_x-pic_p->min_x+1;
      p += x_offset;
    }
}

void hsv_adjust_block3 (struct picture *pic_p, double h_add, double s_add,
			double v_add)
{
  register int i, j;
  register pixel r, g, b, *p;
  register double h, s, v, t1, t2, t3;
  
  i = pic_p->height*pic_p->width;
  p = pic_p->buf;
  h_add /= 60;

  while (i--)
    {
      if (!(IS_MASK(*p)))
	{
	  DO_HSV_ADJUST;
	}
      p++;
    }
}
