/* FILE: block.c

   Routines for creating, deleting, extracting and copying blocks (=pictures).
   Routines to resize, flip and rotate blocks.

   Written by Petri Kuittinen, last modifications 9th August 1993.
*/ 

#include <math.h>
#include <stdlib.h>
#include "misc.h"
#include "pixel.h"
#include "block.h"


/* FUNCTION:     struct picture *create_block (char *name, int width,
                                               int height)
   DESCRIPTION:  allocate space for picture struct and picture data.
   Creates a blank picture.
   PARAMETERS:
   char *name    pointer to name string.
   int width     dimensions of picture.
   int height
   RETURNS:      pointer to picture struct, holding the data. NULL if
   allocating memory for block fails.
   
   Written by Petri Kuittinen, last modifications 12nd July 1993
*/
struct picture *create_block (char *name, int width, int height)
{
  struct picture *new_block;

  new_block = malloc (sizeof(struct picture));
  if (new_block==NULL) return NULL;

  if (name!=NULL)
    {
      new_block->name = duplicate_string (name);
      if (new_block->name==NULL)
	{
	  free (new_block); return NULL;
	}
    }

  new_block->buf = calloc (width*height, sizeof(pixel));
  if (new_block->buf==NULL)
    {
      free (new_block);
      return NULL;
    }

  new_block->name = name;
  new_block->width = width;
  new_block->height = height;
  new_block->mask_edge = NULL;
  new_block->n_edges = 0;
  new_block->flags = 0;

  return new_block;
}


/* FUNCTION:     struct picture *duplicate_block (struct picture *pic_p)
   DESCRIPTION:  duplicate a block.
   PARAMETERS:
   struct picture *pic_p
   RETURNS:      pointer to duplicated block, NULL if allocating memory fails.
   
   Written by Petri Kuittinen, last modifications 19th July 1993
*/
struct picture *duplicate_block (struct picture *pic_p)
{
  register int i;
  register pixel *p1, *p2;
  register struct coord *f1, *f2;
  register struct picture *new_block;

  new_block = malloc (sizeof(struct picture));
  if (new_block==NULL) return NULL;
  
  if (pic_p->name!=NULL)
    {
      new_block->name = duplicate_string (pic_p->name);
      if (new_block->name==NULL)
	{
	  free (new_block); return NULL;
	}
    }

  new_block->buf = calloc (pic_p->width*pic_p->height, sizeof(pixel));
  if (new_block->buf==NULL)
    {
      free (new_block); return NULL;
    }

  i = pic_p->width*pic_p->height;
  p1 = pic_p->buf;
  p2 = new_block->buf;
  while (i--) *p2++ = *p1++;
  
  if (pic_p->n_edges)
    {
      new_block->mask_edge = malloc (pic_p->n_edges*sizeof(struct coord));
      if (new_block->mask_edge==NULL)
	{
	  free (new_block); return NULL;
	}
      i = pic_p->n_edges;
      f1 = pic_p->mask_edge;
      f2 = new_block->mask_edge;
      while (i--)
	{
	  f2->x = f1->x;
	  f2->y = f1->y;
	  f2++; f1++;
	}
    }
  else
    new_block->mask_edge = NULL;

  new_block->min_x = pic_p->min_x;
  new_block->max_x = pic_p->max_x;
  new_block->min_y = pic_p->min_y;
  new_block->max_y = pic_p->max_y;
  new_block->width = pic_p->width;
  new_block->height = pic_p->height;
  new_block->n_edges = pic_p->n_edges;
  new_block->flags = 0;

  return new_block;
}


/* FUNCTION:     struct picture *delete_block (struct picture *pic_p)
   DESCRIPTION:  free space previously allocated for picture struct and
   picture data.
   PARAMETERS:
   struct picture *pic_p
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 12nd July 1993
*/
void delete_block (struct picture *pic_p)
{
  if (pic_p->name!=NULL)
    free (pic_p->name);
  if (pic_p->n_edges)
    free (pic_p->mask_edge);
  free (pic_p->buf);
  free (pic_p);
}


/* FUNCTION:     void swap_blocks (struct picture *pic_p1,
   struct picture *pic_p2, int x, int y)
   DESCRIPTION:  swap a part block1 with block2. Swaps only picture data.
   PARAMETERS:
   struct picture *pic_p1 pointer to block1
   struct picture *pic_p2 pointer to block2
   int x         upper left hand coordinates of location in block1
   int y         (where blocks are swapped)
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 19th July 1993
*/

void swap_blocks (struct picture *pic_p1, struct picture *pic_p2, int x, int y)
{
  register int w, h, x_offset;
  register pixel *buf1, *buf2, d1, d2;

  if ((x+pic_p2->width)>pic_p1->width ||
      (y+pic_p2->height)>pic_p1->height) return; 

  w = pic_p2->width;
  h = pic_p2->height;
  
  buf1 = pic_p1->buf+x+y*pic_p1->width;
  buf2 = pic_p2->buf;

  x_offset = pic_p1->width-pic_p2->width;

  while (h--)
    {
      while (w--)
        {
	  d1 = *buf1;
	  d2 = *buf2;
	  SWAP (d1, d2);
	  *buf1++ = d1;
	  *buf2++ = d2;
	}
      w = pic_p2->width; buf1 += x_offset;
    }
}


/* FUNCTION:     struct picture *copy_block (struct picture *pic_p,
   int x1, int y1, int x2, int y2)
   DESCRIPTION:  copy a rectangle shaped aread from the picture.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the
   rectangle.
   PARAMETERS:
   struct picture *pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   RETURNS:      pointer to picture struct, holding the data. NULL if
   allocating memory for block fails.
   
   Written by Petri Kuittinen, last modifications 16th July 1993
*/
struct picture *copy_block (struct picture *pic_p, int x1, int y1, int x2,
			    int y2)
{
  char *new_name;
  register int w, h, x_offset;
  register pixel *src, *tar;
  register struct picture *new_block;

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;

  w = x2-x1+1;
  h = y2-y1+1;
  
  new_name = duplicate_string (pic_p->name);
  if (new_name==NULL) return NULL;

  new_block = create_block (new_name, w, h);
  if (new_block==NULL) return NULL;

  src = pic_p->buf+x1+y1*pic_p->width;
  tar = new_block->buf;
  x_offset = pic_p->width-w;

  while (h--)
    {
      while (w--)
        {
          *tar++ = *src++;
        }
      w = x2-x1+1; src += x_offset;
    }

  return new_block;
}


/* FUNCTION:     struct picture *extract_block# (struct picture *pic_p,
   int x1, int y1, int x2, int y2, pixel bits)
   DESCRIPTION:  extract a rectangle shaped aread from picture.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the
   rectangle. Extracts pixels
   #=1           inside screen (no mask bit check)
   #=2           inside mask
   #=3           outside mask
   PARAMETERS:
   struct picture *pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   bits          a bit mask that defines which bits are copied (if bit=1 then
                 that bit is copied).
   RETURNS:      pointer to picture struct, holding the data. NULL if
   allocating memory for block fails.
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
struct picture *extract_block1 (struct picture *pic_p, int x1, int y1, int x2,
				int y2, pixel bits)
{
  char *new_name;
  register int w, h, x_offset;
  register pixel *src, *tar;
  register struct picture *new_block;

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;

  w = x2-x1+1;
  h = y2-y1+1;
  
  new_name = duplicate_string (pic_p->name);
  if (new_name==NULL) return NULL;

  new_block = create_block (new_name, w, h);
  if (new_block==NULL) return NULL;

  src = pic_p->buf+x1+y1*pic_p->width;
  tar = new_block->buf;
  x_offset = pic_p->width-w;

  while (h--)
    {
      while (w--)
        {
	  *tar = bits&(*src++);
          tar++;
        }
      w = x2-x1+1; src += x_offset;
    }

  return new_block;
}

struct picture *extract_block2 (struct picture *pic_p, int x1, int y1, int x2,
				int y2, pixel bits)
{
  char *new_name;
  register int w, h, x_offset;
  register pixel *src, *tar, data;
  register struct picture *new_block;

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;

  w = x2-x1+1;
  h = y2-y1+1;
  
  new_name = duplicate_string (pic_p->name);
  if (new_name==NULL) return NULL;

  new_block = create_block (new_name, w, h);
  if (new_block==NULL) return NULL;

  src = pic_p->buf+x1+y1*pic_p->width;
  tar = new_block->buf;
  x_offset = pic_p->width-w;

  while (h--)
    {
      while (w--)
        {
	  data = *src++; 
          if (IS_MASK(data))
            {
              *tar = bits&data;
            }
          tar++;
        }
      w = x2-x1+1; src += x_offset;
    }

  return new_block;
}

struct picture *extract_block3 (struct picture *pic_p, int x1, int y1, int x2,
				int y2, pixel bits)
{
  char *new_name;
  register int w, h, x_offset;
  register pixel *src, *tar, data;
  register struct picture *new_block;

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;

  w = x2-x1+1;
  h = y2-y1+1;
  
  new_name = duplicate_string (pic_p->name);
  if (new_name==NULL) return NULL;

  new_block = create_block (new_name, w, h);
  if (new_block==NULL) return NULL;

  src = pic_p->buf+x1+y1*pic_p->width;
  tar = new_block->buf;
  x_offset = pic_p->width-w;

  while (h--)
    {
      while (w--)
        {
	  data = *src++; 
          if (!(IS_MASK(data)))
            {
              *tar = bits&data;
            }
          tar++;
        }
      w = x2-x1+1; src += x_offset;
    }

  return new_block;
}


/* FUNCTION:     void vflip_block (struct picture *pic_p)
   DESCRIPTION:  flip a block in vertical direction (the upper edge of the
   block becomes the lower edge and vice versa).
   PARAMETERS:
   struct picture *pic_p  pointer to a block to be vflipped.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 4th August 1993
*/
void vflip_block (struct picture *pic_p)
{
  register int w, h;
  register pixel *p1, *p2, d1, d2;
  register struct picture *save;

  h = pic_p->height/2;
  p1 = pic_p->buf;
  p2 = pic_p->buf+pic_p->width*(pic_p->height-1);

  while (h--)
    {
      w = pic_p->width;
      while (w--)
	{
	  d1 = *p1;
	  d2 = *p2;
	  SWAP (d1,d2);
	  *p1++ = d1;
	  *p2++ = d2;
	}
      p2 -= 2*pic_p->width;
    }

  /* flip mask edge */
  h = pic_p->min_y;
  pic_p->min_y = pic_p->height-pic_p->max_y-1;
  pic_p->max_y = pic_p->height-h-1;

  save = cur_pic; /* temporarily save cur_pic */
  cur_pic = pic_p;
  minimize_mask_edge();
  cur_pic = save;
}


/* FUNCTION:     void hflip_block (struct picture *pic_p)
   DESCRIPTION:  flip a block in horizontical direction (the left edge of the
   block becomes the right edge and vice versa).
   PARAMETERS:
   struct picture *pic_p  pointer to a block to be hflipped.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 4th August 1993
*/
void hflip_block (struct picture *pic_p)
{
  register int w, wc, h;
  register pixel *p1, *p2, d1, d2;
  register struct picture *save;

  h = pic_p->height;
  w = pic_p->width/2;
  p1 = pic_p->buf;
  p2 = p1+pic_p->width-1;

  while (h--)
    {
      wc = w;
      while (wc--)
	{
	  d1 = *p1;
	  d2 = *p2;
	  SWAP (d1,d2);
	  *p1++ = d1;
	  *p2-- = d2;
	}
      p1 += pic_p->width-w;
      p2 += pic_p->width+w;
    }

  /* flip mask edge */
  w = pic_p->min_x;
  pic_p->min_x = pic_p->width-pic_p->max_x-1;
  pic_p->max_x = pic_p->width-w-1;

  save = cur_pic; /* temporarily save cur_pic */
  cur_pic = pic_p;
  minimize_mask_edge();
  cur_pic = save;
}


/* FUNCTION:     struct picture *resize_block (struct picture *pic_p,
   int new_w, int new_h)
   DESCRIPTION:  change size of a block using an algorithm by the idiot
   himself (integet math modified from Bresenham's line drawing algo).
   PARAMETERS:
   struct picture *pic_p  pointer to a block to be resized.
   int new_w     new width
   int new_h     new height
   RETURNS:      pointer to resized block or NULL if memory allocation fails.
   
   Written by Petri Kuittinen, last modifications 9th August 1993
*/
struct picture *resize_block (struct picture *pic_p, int new_w, int new_h)
{
  register int w, h, dx, dy;
  char *new_name;
  register pixel *src1, *src2, *tar;
  register struct picture *new_block, *save;

  new_name = duplicate_string(pic_p->name);
  if (new_name==NULL) return NULL;

  new_block = create_block (new_name, new_w, new_h);
  if (new_block==NULL) return NULL;

  src1 =  pic_p->buf;
  tar = new_block->buf;

  dy = pic_p->height-new_h;

  h = new_h;

  while (h--)
    {
      src2 = src1;
      w = new_w;
      dx = pic_p->width-new_w;

      while (w--)
	{
	  *tar++ = *src2;
	  while(dx>=0)
	    {
	      src2++;
	      dx -= new_w;
	    }
	  dx += pic_p->width;
	}
      while (dy>=0)
	{
	  src1 += pic_p->width;
	  dy -= new_h;
	}
      dy += pic_p->height;
    }

  /* strech mask edge */
  new_block->min_x = 0;
  new_block->max_x = new_block->width-1;
  new_block->min_y = 0;
  new_block->max_y = new_block->height-1;

  save = cur_pic;
  cur_pic = new_block;
  minimize_mask_edge();
  cur_pic = save;

  return new_block;
}


/* Some macros used in rotate block */
#define MINMAX_X(X) \
if ((X)<min_x) min_x = (X); \
if ((X)>max_x) max_x = (X);

#define MINMAX_Y(Y) \
if ((Y)<min_y) min_y = (Y); \
if ((Y)>max_y) max_y = (Y);

/* FUNCTION:     struct picture *rotate_block (struct picture *pic_p,
   double angle)
   DESCRIPTION:  rotate a block by an angle in clockwise direction around
   the block centre.
   PARAMETERS:
   struct picture *pic_p  pointer to a block to rotated.
   double angle  angle in radians [0, 2*pi].
   RETURNS:      pointer to resized block or NULL if memory allocation fails.
   
   Written by Petri Kuittinen, last modifications 10th August 1993
*/
struct picture *rotate_block (struct picture *pic_p, double angle)
{
  register int x1, y1, x2, y2, xc1, yc1, xc2, yc2;
  register double c, s;
  register pixel *src, *tar;

  int min_x, max_x, min_y, max_y, w, h;
  char *new_name;
  struct picture *new_block, *save;

  c = cos(-angle);
  s = sin(-angle);
  xc1 = pic_p->width/2;  /* center of old block */
  yc1 = pic_p->height/2;

  /* Find the max and min coordinates of the rotated block */
  min_x = max_x = x2 = (0-xc1)*c-(0-yc1)*s;
  min_y = max_y = y2 = (0-xc1)*s+(0-yc1)*c;

  x2 = (pic_p->width-xc1)*c-(0-yc1)*s; MINMAX_X(x2);
  y2 = (pic_p->width-xc1)*s+(0-yc1)*c; MINMAX_Y(y2);

  x2 = (0-xc1)*c-(pic_p->height-yc1)*s; MINMAX_X(x2);
  y2 = (0-xc1)*s+(pic_p->height-yc1)*c; MINMAX_Y(y2);

  x2 = (pic_p->width-xc1)*c-(pic_p->height-yc1)*s; MINMAX_X(x2);
  y2 = (pic_p->width-xc1)*s+(pic_p->height-yc1)*c; MINMAX_Y(y2);

  w = max_x-min_x+1; /* now we finally have size of the new block */
  h = max_y-min_y+1;

  new_name = duplicate_string(pic_p->name);
  if (new_name==NULL) return NULL;

  new_block = create_block (new_name, w, h);
  if (new_block==NULL) return NULL;

  xc2 = new_block->width/2;  /* center of new block */
  yc2 = new_block->height/2;

  src =  pic_p->buf;
  tar = new_block->buf;

  /* The actual rotation of the picture happens here */
  for (y2 = -yc2; y2<h-yc2; y2++)
    for (x2 = -xc2; x2<w-xc2; x2++)
      {
	/* Find the correct pixels map to the original image using
	   inverse rotation */
	x1 = x2*c+y2*s+xc1;
	y1 = -x2*s+y2*c+yc1;
	
	if (x1<0 || y1<0 || x1>=pic_p->width || y1>=pic_p->height)
	  *tar++ = 0;  /* If outside source picture, then insert black */
	else
	  *tar++ = *(src+x1+pic_p->width*y1);
      }

  /* rotate mask edge */
  new_block->min_x = 0;
  new_block->max_x = new_block->width-1;
  new_block->min_y = 0;
  new_block->max_y = new_block->height-1;

  save = cur_pic;
  cur_pic = new_block;
  minimize_mask_edge();
  cur_pic = save;

  return new_block;
}
