/*
   File: synthesis_filter.cc
*/

#include <stdio.h>
#include <stdlib.h>
#include <fstream.h>
#ifdef IRIX
#include <dmedia/audio.h>
#endif
#ifdef SOLARIS
#include <sys/audioio.h>
#endif

#include "athread.hh"

#include "error.hh"
#include "debug.hh"
#include "util.hh"

#include "sync.hh"
#include "mpeg2const.hh"
#include "mpeg2buff.hh"

#include "astream.hh"
#include "crc.hh"
#include "header.hh"
#include "obuffer.hh"
#include "synthesis_filter.hh"


const real SynthesisFilter::n[32][31] = {
  {  0.707106781,  0.671558955,  0.634393284,  0.595699304,
     0.555570233,  0.514102744,  0.471396737,  0.427555093,
     0.382683432,  0.336889853,  0.290284677,  0.242980180,
     0.195090322,  0.146730474,  0.098017140,  0.049067674,
    -0.740951126, -0.773010454, -0.803207532, -0.831469612,
    -0.857728610, -0.881921265, -0.903989293, -0.923879533,
    -0.941544065, -0.956940336, -0.970031253, -0.980785280,
    -0.989176510, -0.995184727, -0.998795456 },
  { -0.707106781, -0.803207532, -0.881921265, -0.941544065,
    -0.980785280, -0.998795456, -0.995184727, -0.970031253,
    -0.923879532, -0.857728610, -0.773010453, -0.671558954,
    -0.555570233, -0.427555093, -0.290284677, -0.146730474,
     0.595699304,  0.471396737,  0.336889853,  0.195090322,
     0.049067674, -0.098017141, -0.242980180, -0.382683433,
    -0.514102744, -0.634393284, -0.740951126, -0.831469612,
    -0.903989293, -0.956940336, -0.989176510 },
  { -0.707106781, -0.514102744, -0.290284677, -0.049067674,
     0.195090323,  0.427555094,  0.634393285,  0.803207532,
     0.923879533,  0.989176510,  0.995184727,  0.941544065,
     0.831469612,  0.671558955,  0.471396737,  0.242980180,
     0.857728610,  0.956940336,  0.998795456,  0.980785280,
     0.903989293,  0.773010453,  0.595699304,  0.382683432,
     0.146730474, -0.098017141, -0.336889854, -0.555570233,
    -0.740951126, -0.881921265, -0.970031253 },
  {  0.707106782,  0.903989293,  0.995184727,  0.970031253,
     0.831469612,  0.595699304,  0.290284677, -0.049067675,
    -0.382683433, -0.671558955, -0.881921265, -0.989176510,
    -0.980785280, -0.857728610, -0.634393284, -0.336889853,
    -0.427555093, -0.098017140,  0.242980181,  0.555570234,
     0.803207532,  0.956940336,  0.998795456,  0.923879532,
     0.740951125,  0.471396737,  0.146730474, -0.195090322,
    -0.514102744, -0.773010454, -0.941544065 },
  {  0.707106781,  0.336889853, -0.098017141, -0.514102744,
    -0.831469612, -0.989176510, -0.956940336, -0.740951125,
    -0.382683432,  0.049067675,  0.471396737,  0.803207532,
     0.980785281,  0.970031253,  0.773010453,  0.427555093,
    -0.941544065, -0.995184727, -0.857728610, -0.555570233,
    -0.146730474,  0.290284678,  0.671558955,  0.923879533,
     0.998795456,  0.881921264,  0.595699304,  0.195090322,
    -0.242980180, -0.634393284, -0.903989293 },
  { -0.707106781, -0.970031253, -0.956940336, -0.671558954,
    -0.195090321,  0.336889854,  0.773010454,  0.989176510,
     0.923879532,  0.595699304,  0.098017140, -0.427555094,
    -0.831469612, -0.998795456, -0.881921264, -0.514102744,
     0.242980180, -0.290284677, -0.740951126, -0.980785280,
    -0.941544065, -0.634393284, -0.146730474,  0.382683433,
     0.803207532,  0.995184727,  0.903989293,  0.555570233,
     0.049067674, -0.471396737, -0.857728610 },
  { -0.707106781, -0.146730474,  0.471396737,  0.903989293,
     0.980785280,  0.671558955,  0.098017140, -0.514102744,
    -0.923879533, -0.970031253, -0.634393284, -0.049067674,
     0.555570234,  0.941544065,  0.956940336,  0.595699304,
     0.989176510,  0.881921264,  0.427555093, -0.195090322,
    -0.740951126, -0.995184727, -0.857728610, -0.382683432,
     0.242980181,  0.773010454,  0.998795456,  0.831469612,
     0.336889853, -0.290284677, -0.803207532 },
  {  0.707106782,  0.998795456,  0.773010453,  0.146730474,
    -0.555570233, -0.970031253, -0.881921264, -0.336889853,
     0.382683433,  0.903989293,  0.956940336,  0.514102744,
    -0.195090322, -0.803207532, -0.995184727, -0.671558954,
    -0.049067674,  0.634393285,  0.989176510,  0.831469612,
     0.242980180, -0.471396737, -0.941544065, -0.923879532,
    -0.427555093,  0.290284678,  0.857728610,  0.980785280,
     0.595699304, -0.098017141, -0.740951126 },
  {  0.707106781, -0.049067675, -0.773010454, -0.989176510,
    -0.555570233,  0.242980181,  0.881921265,  0.941544065,
     0.382683432, -0.427555094, -0.956940336, -0.857728610,
    -0.195090321,  0.595699305,  0.995184727,  0.740951125,
    -0.998795456, -0.634393284,  0.146730475,  0.831469613,
     0.970031253,  0.471396737, -0.336889854, -0.923879533,
    -0.903989293, -0.290284677,  0.514102745,  0.980785281,
     0.803207531,  0.098017140, -0.671558955 },
  { -0.707106781, -0.989176510, -0.471396736,  0.427555094,
     0.980785281,  0.740951125, -0.098017141, -0.857728610,
    -0.923879532, -0.242980179,  0.634393285,  0.998795456,
     0.555570233, -0.336889854, -0.956940336, -0.803207531,
    -0.146730475, -0.881921265, -0.903989293, -0.195090321,
     0.671558955,  0.995184727,  0.514102744, -0.382683433,
    -0.970031253, -0.773010453,  0.049067675,  0.831469613,
     0.941544065,  0.290284677, -0.595699305 },
  { -0.707106781,  0.242980181,  0.956940336,  0.740951125,
    -0.195090322, -0.941544065, -0.773010453,  0.146730475,
     0.923879533,  0.803207531, -0.098017141, -0.903989293,
    -0.831469612,  0.049067675,  0.881921265,  0.857728610,
     0.970031253,  0.290284677, -0.671558955, -0.980785280,
    -0.336889853,  0.634393285,  0.989176510,  0.382683432,
    -0.595699305, -0.995184727, -0.427555093,  0.555570234,
     0.998795456,  0.471396737, -0.514102744 },
  {  0.707106782,  0.941544065,  0.098017140, -0.857728610,
    -0.831469612,  0.146730475,  0.956940336,  0.671558955,
    -0.382683433, -0.998795456, -0.471396736,  0.595699305,
     0.980785280,  0.242980180, -0.773010454, -0.903989293,
     0.336889854,  0.995184727,  0.514102744, -0.555570233,
    -0.989176510, -0.290284677,  0.740951126,  0.923879532,
     0.049067674, -0.881921265, -0.803207531,  0.195090323,
     0.970031253,  0.634393284, -0.427555094 },
  {  0.707106781, -0.427555094, -0.995184727, -0.242980179,
     0.831469613,  0.803207531, -0.290284677, -0.998795456,
    -0.382683432,  0.740951126,  0.881921264, -0.146730475,
    -0.980785280, -0.514102744,  0.634393285,  0.941544065,
    -0.903989293,  0.098017141,  0.970031253,  0.555570233,
    -0.595699305, -0.956940336, -0.049067674,  0.923879533,
     0.671558955, -0.471396737, -0.989176510, -0.195090321,
     0.857728610,  0.773010453, -0.336889854 },
  { -0.707106781, -0.857728610,  0.290284678,  0.998795456,
     0.195090322, -0.903989293, -0.634393284,  0.595699305,
     0.923879532, -0.146730475, -0.995184727, -0.336889853,
     0.831469613,  0.740951125, -0.471396737, -0.970031253,
    -0.514102744, -0.956940336,  0.049067675,  0.980785281,
     0.427555093, -0.773010454, -0.803207531,  0.382683433,
     0.989176510,  0.098017140, -0.941544065, -0.555570233,
     0.671558955,  0.881921264, -0.242980180 },
  { -0.707106781,  0.595699305,  0.881921264, -0.336889854,
    -0.980785280,  0.049067675,  0.995184727,  0.242980180,
    -0.923879533, -0.514102744,  0.773010454,  0.740951125,
    -0.555570233, -0.903989293,  0.290284678,  0.989176510,
     0.803207531, -0.471396737, -0.941544065,  0.195090323,
     0.998795456,  0.098017140, -0.970031253, -0.382683432,
     0.857728610,  0.634393284, -0.671558955, -0.831469612,
     0.427555094,  0.956940336, -0.146730475 },
  {  0.707106782,  0.740951125, -0.634393284, -0.803207531,
     0.555570234,  0.857728610, -0.471396737, -0.903989293,
     0.382683433,  0.941544065, -0.290284677, -0.970031253,
     0.195090323,  0.989176510, -0.098017141, -0.998795456,
     0.671558955,  0.773010453, -0.595699305, -0.831469612,
     0.514102745,  0.881921264, -0.427555094, -0.923879532,
     0.336889854,  0.956940336, -0.242980180, -0.980785280,
     0.146730475,  0.995184727, -0.049067675 },
  {  0.707106781, -0.740951126, -0.634393284,  0.803207532,
     0.555570233, -0.857728610, -0.471396736,  0.903989293,
     0.382683432, -0.941544065, -0.290284677,  0.970031253,
     0.195090322, -0.989176510, -0.098017140,  0.998795456,
    -0.671558954,  0.773010454,  0.595699304, -0.831469612,
    -0.514102744,  0.881921265,  0.427555093, -0.923879533,
    -0.336889853,  0.956940336,  0.242980180, -0.980785280,
    -0.146730474,  0.995184727,  0.049067674 },
  { -0.707106781, -0.595699304,  0.881921265,  0.336889853,
    -0.980785280, -0.049067674,  0.995184727, -0.242980180,
    -0.923879532,  0.514102745,  0.773010453, -0.740951126,
    -0.555570233,  0.903989293,  0.290284677, -0.989176510,
    -0.803207532, -0.471396736,  0.941544065,  0.195090322,
    -0.998795456,  0.098017141,  0.970031253, -0.382683433,
    -0.857728610,  0.634393285,  0.671558955, -0.831469612,
    -0.427555093,  0.956940336,  0.146730474 },
  { -0.707106781,  0.857728610,  0.290284677, -0.998795456,
     0.195090323,  0.903989293, -0.634393284, -0.595699304,
     0.923879533,  0.146730474, -0.995184727,  0.336889854,
     0.831469612, -0.740951126, -0.471396736,  0.970031253,
     0.514102744, -0.956940336, -0.049067674,  0.980785280,
    -0.427555094, -0.773010453,  0.803207532,  0.382683432,
    -0.989176510,  0.098017141,  0.941544065, -0.555570233,
    -0.671558954,  0.881921265,  0.242980180 },
  {  0.707106782,  0.427555093, -0.995184727,  0.242980181,
     0.831469612, -0.803207532, -0.290284677,  0.998795456,
    -0.382683433, -0.740951125,  0.881921265,  0.146730474,
    -0.980785280,  0.514102745,  0.634393284, -0.941544065,
     0.903989293,  0.098017140, -0.970031253,  0.555570234,
     0.595699304, -0.956940336,  0.049067675,  0.923879532,
    -0.671558955, -0.471396736,  0.989176510, -0.195090322,
    -0.857728610,  0.773010454,  0.336889853 },
  {  0.707106781, -0.941544065,  0.098017141,  0.857728610,
    -0.831469612, -0.146730474,  0.956940336, -0.671558955,
    -0.382683432,  0.998795456, -0.471396737, -0.595699304,
     0.980785281, -0.242980180, -0.773010453,  0.903989293,
    -0.336889853,  0.995184727, -0.514102744, -0.555570233,
     0.989176510, -0.290284677, -0.740951125,  0.923879533,
    -0.049067675, -0.881921264,  0.803207532,  0.195090322,
    -0.970031253,  0.634393285,  0.427555093 },
  { -0.707106781, -0.242980179,  0.956940336, -0.740951126,
    -0.195090321,  0.941544065, -0.773010454, -0.146730474,
     0.923879532, -0.803207532, -0.098017140,  0.903989293,
    -0.831469612, -0.049067674,  0.881921264, -0.857728610,
    -0.970031253,  0.290284678,  0.671558955, -0.980785280,
     0.336889854,  0.634393284, -0.989176510,  0.382683433,
     0.595699304, -0.995184727,  0.427555094,  0.555570233,
    -0.998795456,  0.471396737,  0.514102744 },
  { -0.707106781,  0.989176510, -0.471396737, -0.427555093,
     0.980785280, -0.740951126, -0.098017140,  0.857728610,
    -0.923879533,  0.242980181,  0.634393284, -0.998795456,
     0.555570234,  0.336889853, -0.956940336,  0.803207532,
     0.146730474, -0.881921264,  0.903989293, -0.195090322,
    -0.671558954,  0.995184727, -0.514102744, -0.382683432,
     0.970031253, -0.773010454, -0.049067674,  0.831469612,
    -0.941544065,  0.290284678,  0.595699304 },
  {  0.707106782,  0.049067674, -0.773010453,  0.989176510,
    -0.555570233, -0.242980179,  0.881921264, -0.941544065,
     0.382683433,  0.427555093, -0.956940336,  0.857728610,
    -0.195090322, -0.595699304,  0.995184727, -0.740951126,
     0.998795456, -0.634393284, -0.146730474,  0.831469612,
    -0.970031253,  0.471396737,  0.336889853, -0.923879532,
     0.903989293, -0.290284677, -0.514102744,  0.980785280,
    -0.803207532,  0.098017141,  0.671558955 },
  {  0.707106781, -0.998795456,  0.773010454, -0.146730475,
    -0.555570233,  0.970031253, -0.881921265,  0.336889854,
     0.382683432, -0.903989293,  0.956940336, -0.514102744,
    -0.195090321,  0.803207531, -0.995184727,  0.671558955,
     0.049067675,  0.634393284, -0.989176510,  0.831469613,
    -0.242980180, -0.471396736,  0.941544065, -0.923879533,
     0.427555094,  0.290284677, -0.857728610,  0.980785281,
    -0.595699305, -0.098017140,  0.740951125 },
  { -0.707106781,  0.146730475,  0.471396737, -0.903989293,
     0.980785281, -0.671558955,  0.098017141,  0.514102744,
    -0.923879532,  0.970031253, -0.634393284,  0.049067675,
     0.555570233, -0.941544065,  0.956940336, -0.595699305,
    -0.989176510,  0.881921265, -0.427555094, -0.195090321,
     0.740951125, -0.995184727,  0.857728610, -0.382683433,
    -0.242980179,  0.773010453, -0.998795456,  0.831469613,
    -0.336889854, -0.290284677,  0.803207531 },
  { -0.707106781,  0.970031253, -0.956940336,  0.671558955,
    -0.195090322, -0.336889853,  0.773010453, -0.989176510,
     0.923879533, -0.595699305,  0.098017141,  0.427555093,
    -0.831469612,  0.998795456, -0.881921265,  0.514102745,
    -0.242980180, -0.290284677,  0.740951125, -0.980785280,
     0.941544065, -0.634393284,  0.146730475,  0.382683432,
    -0.803207531,  0.995184727, -0.903989293,  0.555570234,
    -0.049067675, -0.471396736,  0.857728610 },
  {  0.707106782, -0.336889854, -0.098017140,  0.514102744,
    -0.831469612,  0.989176510, -0.956940336,  0.740951126,
    -0.382683433, -0.049067674,  0.471396737, -0.803207531,
     0.980785280, -0.970031253,  0.773010454, -0.427555094,
     0.941544065, -0.995184727,  0.857728610, -0.555570233,
     0.146730475,  0.290284677, -0.671558954,  0.923879532,
    -0.998795456,  0.881921265, -0.595699305,  0.195090323,
     0.242980180, -0.634393284,  0.903989293 },
  {  0.707106781, -0.903989293,  0.995184727, -0.970031253,
     0.831469613, -0.595699305,  0.290284678,  0.049067674,
    -0.382683432,  0.671558955, -0.881921264,  0.989176510,
    -0.980785280,  0.857728610, -0.634393284,  0.336889854,
     0.427555094, -0.098017141, -0.242980179,  0.555570233,
    -0.803207531,  0.956940336, -0.998795456,  0.923879533,
    -0.740951126,  0.471396737, -0.146730475, -0.195090321,
     0.514102744, -0.773010453,  0.941544065 },
  { -0.707106781,  0.514102745, -0.290284677,  0.049067675,
     0.195090322, -0.427555093,  0.634393284, -0.803207531,
     0.923879532, -0.989176510,  0.995184727, -0.941544065,
     0.831469613, -0.671558955,  0.471396737, -0.242980180,
    -0.857728610,  0.956940336, -0.998795456,  0.980785281,
    -0.903989293,  0.773010454, -0.595699305,  0.382683433,
    -0.146730475, -0.098017140,  0.336889853, -0.555570233,
     0.740951125, -0.881921264,  0.970031253 },
  { -0.707106781,  0.803207531, -0.881921264,  0.941544065,
    -0.980785280,  0.998795456, -0.995184727,  0.970031253,
    -0.923879533,  0.857728610, -0.773010454,  0.671558955,
    -0.555570233,  0.427555094, -0.290284677,  0.146730475,
    -0.595699305,  0.471396737, -0.336889854,  0.195090323,
    -0.049067675, -0.098017140,  0.242980180, -0.382683432,
     0.514102744, -0.634393284,  0.740951125, -0.831469612,
     0.903989293, -0.956940336,  0.989176510 },
  {  0.707106782, -0.671558955,  0.634393285, -0.595699305,
     0.555570234, -0.514102744,  0.471396737, -0.427555094,
     0.382683433, -0.336889854,  0.290284678, -0.242980180,
     0.195090323, -0.146730475,  0.098017141, -0.049067675,
     0.740951125, -0.773010453,  0.803207531, -0.831469612,
     0.857728610, -0.881921264,  0.903989293, -0.923879532,
     0.941544065, -0.956940336,  0.970031253, -0.980785280,
     0.989176510, -0.995184727,  0.998795456 }
};


const real SynthesisFilter::d[512] = {
  // Warning: These values are not in the same order
  // as in Annex 3-B.3 of the ISO/IEC DIS 11172-3
   0.000000000, -0.000442505,  0.003250122, -0.007003784,
   0.031082153, -0.078628540,  0.100311279, -0.572036743,
   1.144989014,  0.572036743,  0.100311279,  0.078628540,
   0.031082153,  0.007003784,  0.003250122,  0.000442505,
  -0.000015259, -0.000473022,  0.003326416, -0.007919312,
   0.030517578, -0.084182739,  0.090927124, -0.600219727,
   1.144287109,  0.543823242,  0.108856201,  0.073059082,
   0.031478882,  0.006118774,  0.003173828,  0.000396729,
  -0.000015259, -0.000534058,  0.003387451, -0.008865356,
   0.029785156, -0.089706421,  0.080688477, -0.628295898,
   1.142211914,  0.515609741,  0.116577148,  0.067520142,
   0.031738281,  0.005294800,  0.003082275,  0.000366211,
  -0.000015259, -0.000579834,  0.003433228, -0.009841919,
   0.028884888, -0.095169067,  0.069595337, -0.656219482,
   1.138763428,  0.487472534,  0.123474121,  0.061996460,
   0.031845093,  0.004486084,  0.002990723,  0.000320435,
  -0.000015259, -0.000625610,  0.003463745, -0.010848999,
   0.027801514, -0.100540161,  0.057617188, -0.683914185,
   1.133926392,  0.459472656,  0.129577637,  0.056533813,
   0.031814575,  0.003723145,  0.002899170,  0.000289917,
  -0.000015259, -0.000686646,  0.003479004, -0.011886597,
   0.026535034, -0.105819702,  0.044784546, -0.711318970,
   1.127746582,  0.431655884,  0.134887695,  0.051132202,
   0.031661987,  0.003005981,  0.002792358,  0.000259399,
  -0.000015259, -0.000747681,  0.003479004, -0.012939453,
   0.025085449, -0.110946655,  0.031082153, -0.738372803,
   1.120223999,  0.404083252,  0.139450073,  0.045837402,
   0.031387329,  0.002334595,  0.002685547,  0.000244141,
  -0.000030518, -0.000808716,  0.003463745, -0.014022827,
   0.023422241, -0.115921021,  0.016510010, -0.765029907,
   1.111373901,  0.376800537,  0.143264771,  0.040634155,
   0.031005859,  0.001693726,  0.002578735,  0.000213623,
  -0.000030518, -0.000885010,  0.003417969, -0.015121460,
   0.021575928, -0.120697021,  0.001068115, -0.791213989,
   1.101211548,  0.349868774,  0.146362305,  0.035552979,
   0.030532837,  0.001098633,  0.002456665,  0.000198364,
  -0.000030518, -0.000961304,  0.003372192, -0.016235352,
   0.019531250, -0.125259399, -0.015228271, -0.816864014,
   1.089782715,  0.323318481,  0.148773193,  0.030609131,
   0.029937744,  0.000549316,  0.002349854,  0.000167847,
  -0.000030518, -0.001037598,  0.003280640, -0.017349243,
   0.017257690, -0.129562378, -0.032379150, -0.841949463,
   1.077117920,  0.297210693,  0.150497437,  0.025817871,
   0.029281616,  0.000030518,  0.002243042,  0.000152588,
  -0.000045776, -0.001113892,  0.003173828, -0.018463135,
   0.014801025, -0.133590698, -0.050354004, -0.866363525,
   1.063217163,  0.271591187,  0.151596069,  0.021179199,
   0.028533936, -0.000442505,  0.002120972,  0.000137329,
  -0.000045776, -0.001205444,  0.003051758, -0.019577026,
   0.012115479, -0.137298584, -0.069168091, -0.890090942,
   1.048156738,  0.246505737,  0.152069092,  0.016708374,
   0.027725220, -0.000869751,  0.002014160,  0.000122070,
  -0.000061035, -0.001296997,  0.002883911, -0.020690918,
   0.009231567, -0.140670776, -0.088775635, -0.913055420,
   1.031936646,  0.221984863,  0.151962280,  0.012420654,
   0.026840210, -0.001266479,  0.001907349,  0.000106812,
  -0.000061035, -0.001388550,  0.002700806, -0.021789551,
   0.006134033, -0.143676758, -0.109161377, -0.935195923,
   1.014617920,  0.198059082,  0.151306152,  0.008316040,
   0.025909424, -0.001617432,  0.001785278,  0.000106812,
  -0.000076294, -0.001480103,  0.002487183, -0.022857666,
   0.002822876, -0.146255493, -0.130310059, -0.956481934,
   0.996246338,  0.174789429,  0.150115967,  0.004394531,
   0.024932861, -0.001937866,  0.001693726,  0.000091553,
  -0.000076294, -0.001586914,  0.002227783, -0.023910522,
  -0.000686646, -0.148422241, -0.152206421, -0.976852417,
   0.976852417,  0.152206421,  0.148422241,  0.000686646,
   0.023910522, -0.002227783,  0.001586914,  0.000076294,
  -0.000091553, -0.001693726,  0.001937866, -0.024932861,
  -0.004394531, -0.150115967, -0.174789429, -0.996246338,
   0.956481934,  0.130310059,  0.146255493, -0.002822876,
   0.022857666, -0.002487183,  0.001480103,  0.000076294,
  -0.000106812, -0.001785278,  0.001617432, -0.025909424,
  -0.008316040, -0.151306152, -0.198059082, -1.014617920,
   0.935195923,  0.109161377,  0.143676758, -0.006134033,
   0.021789551, -0.002700806,  0.001388550,  0.000061035,
  -0.000106812, -0.001907349,  0.001266479, -0.026840210,
  -0.012420654, -0.151962280, -0.221984863, -1.031936646,
   0.913055420,  0.088775635,  0.140670776, -0.009231567,
   0.020690918, -0.002883911,  0.001296997,  0.000061035,
  -0.000122070, -0.002014160,  0.000869751, -0.027725220,
  -0.016708374, -0.152069092, -0.246505737, -1.048156738,
   0.890090942,  0.069168091,  0.137298584, -0.012115479,
   0.019577026, -0.003051758,  0.001205444,  0.000045776,
  -0.000137329, -0.002120972,  0.000442505, -0.028533936,
  -0.021179199, -0.151596069, -0.271591187, -1.063217163,
   0.866363525,  0.050354004,  0.133590698, -0.014801025,
   0.018463135, -0.003173828,  0.001113892,  0.000045776,
  -0.000152588, -0.002243042, -0.000030518, -0.029281616,
  -0.025817871, -0.150497437, -0.297210693, -1.077117920,
   0.841949463,  0.032379150,  0.129562378, -0.017257690,
   0.017349243, -0.003280640,  0.001037598,  0.000030518,
  -0.000167847, -0.002349854, -0.000549316, -0.029937744,
  -0.030609131, -0.148773193, -0.323318481, -1.089782715,
   0.816864014,  0.015228271,  0.125259399, -0.019531250,
   0.016235352, -0.003372192,  0.000961304,  0.000030518,
  -0.000198364, -0.002456665, -0.001098633, -0.030532837,
  -0.035552979, -0.146362305, -0.349868774, -1.101211548,
   0.791213989, -0.001068115,  0.120697021, -0.021575928,
   0.015121460, -0.003417969,  0.000885010,  0.000030518,
  -0.000213623, -0.002578735, -0.001693726, -0.031005859,
  -0.040634155, -0.143264771, -0.376800537, -1.111373901,
   0.765029907, -0.016510010,  0.115921021, -0.023422241,
   0.014022827, -0.003463745,  0.000808716,  0.000030518,
  -0.000244141, -0.002685547, -0.002334595, -0.031387329,
  -0.045837402, -0.139450073, -0.404083252, -1.120223999,
   0.738372803, -0.031082153,  0.110946655, -0.025085449,
   0.012939453, -0.003479004,  0.000747681,  0.000015259,
  -0.000259399, -0.002792358, -0.003005981, -0.031661987,
  -0.051132202, -0.134887695, -0.431655884, -1.127746582,
   0.711318970, -0.044784546,  0.105819702, -0.026535034,
   0.011886597, -0.003479004,  0.000686646,  0.000015259,
  -0.000289917, -0.002899170, -0.003723145, -0.031814575,
  -0.056533813, -0.129577637, -0.459472656, -1.133926392,
   0.683914185, -0.057617188,  0.100540161, -0.027801514,
   0.010848999, -0.003463745,  0.000625610,  0.000015259,
  -0.000320435, -0.002990723, -0.004486084, -0.031845093,
  -0.061996460, -0.123474121, -0.487472534, -1.138763428,
   0.656219482, -0.069595337,  0.095169067, -0.028884888,
   0.009841919, -0.003433228,  0.000579834,  0.000015259,
  -0.000366211, -0.003082275, -0.005294800, -0.031738281,
  -0.067520142, -0.116577148, -0.515609741, -1.142211914,
   0.628295898, -0.080688477,  0.089706421, -0.029785156,
   0.008865356, -0.003387451,  0.000534058,  0.000015259,
  -0.000396729, -0.003173828, -0.006118774, -0.031478882,
  -0.073059082, -0.108856201, -0.543823242, -1.144287109,
   0.600219727, -0.090927124,  0.084182739, -0.030517578,
   0.007919312, -0.003326416,  0.000473022,  0.000015259
};



SynthesisFilter::SynthesisFilter (uint32 channelnumber, real factor){
  register real *floatp, *floatp2;

  // initialize v1[] and v2[]:
  for (floatp = v1 + 512, floatp2 = v2 + 512; floatp > v1; )
    *--floatp = *--floatp2 = 0.0;

  channel = channelnumber;
  number_of_samples = range_violations = written_samples = 0;
  actual_v = v1;
  actual_write_pos = 15;
  scalefactor = factor;
}


void SynthesisFilter::compute_new_v(real *new_v){
  {
    unsigned int i;
    register real sample = 0.0, *v = new_v;
    register const real *cooefs;

    // initialize new_v[]:
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    v = new_v + 33;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;
    *v++ = sample; *v++ = sample; *v++ = sample; *v++ = sample;

    // compute new values:
    for (i = 0; i < number_of_samples; ++i){
      sample = samples[i];
      cooefs = n[subbandnumbers[i]];
      v = new_v;
      // add to new_v[0]-new_v[15]: (new_v[16] remains always 0.0)
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      v = new_v + 33;
      // add to new_v[33]-new_v[47]:
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++; *v++ += sample * *cooefs++;
      *v++ += sample * *cooefs++;
      *v   -= sample;		// new_v[48]
    }
  }

  // fold new values to there boundaries: (a few streams need this)
  {
    register real *v;
    for (v = new_v + 15; v >= new_v; --v)
      if (*v < -1.0)
	*v = -1.0;
      else if (*v > 1.0)
	*v = 1.0;
    for (v = new_v + 48; v >= new_v + 33; --v)
      if (*v < -1.0)
	*v = -1.0;
      else if (*v > 1.0)
	*v = 1.0;
  }

  {
    // copy values -new_v[0-15] to new_v[32-17]:
    register real *v1 = new_v, *v2 = new_v + 33;
    *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++;
    *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++;
    *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++;
    *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++; *--v2 = -*v1++;

    // copy values new_v[33-47] to new_v[63-49]:
    v1 = new_v + 33;
    v2 = new_v + 64;
    *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++;
    *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++;
    *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++;
    *--v2 = *v1++; *--v2 = *v1++; *--v2 = *v1++;
  }

  // insert new values into v1[] and v2[]:
  {
    register int i;
    register real *nnv, *nv;

    // insert new_v[0-31] into actual v:
    nnv = new_v;
    nv = actual_v + actual_write_pos;
    for (i = 0; i < 32; ++i, nv += 16)
      *nv = *nnv++;

    // insert new_v[32-63] into other v:
    nv = (actual_v == v1 ? v2 : v1) + actual_write_pos;
    for (; i < 64; ++i, nv += 16)
      *nv = *nnv++;
  }
}


void SynthesisFilter::compute_pcm_samples (Obuffer *buffer){
  int i;
  register real floatreg, *lower_bound, *vp;
  register const real *dp;
  int pcm_sample;

  dp = d;
  lower_bound = actual_v;
  if (actual_write_pos != 15)
    for (i = 0; i < 32; ++i)
    {
      floatreg = 0;
      for (vp = lower_bound + actual_write_pos + 1; vp > lower_bound; )
        floatreg += *--vp * *dp++;
      for (vp += 16; vp > lower_bound + actual_write_pos + 1; )
        floatreg += *--vp * *dp++;

      pcm_sample = (int)(floatreg * scalefactor);
      if (pcm_sample > 32767)
        {
          ++range_violations;
          if (floatreg > max_violation)
            max_violation = floatreg;
          pcm_sample = 32767;
        }
      else if (pcm_sample < -32768)
        {
          ++range_violations;
          if (-floatreg > max_violation)
            max_violation = -floatreg;
          pcm_sample = -32768;
        }
      buffer->append (channel, (int16)pcm_sample);
      lower_bound += 16;
    }
  else
    for (i = 0; i < 32; ++i)
      {
        floatreg = 0;
        for (vp = lower_bound + 16; vp > lower_bound; )
          floatreg += *--vp * *dp++;
        
        pcm_sample = (int)(floatreg * scalefactor);
        if (pcm_sample > 32767)
          {
            ++range_violations;
            if (floatreg > max_violation)
              max_violation = floatreg;
            pcm_sample = 32767;
          }
        else if (pcm_sample < -32768)
          {
            ++range_violations;
            if (-floatreg > max_violation)
              max_violation = -floatreg;
            pcm_sample = -32768;
          }
        buffer->append (channel, (int16)pcm_sample);
        lower_bound += 16;
      }
}


void SynthesisFilter::calculate_pcm_samples (Obuffer *buffer){
  static real new_v[64];

  compute_new_v (new_v);
  compute_pcm_samples (buffer);

  written_samples += 32;
  if (actual_write_pos < 15)
    ++actual_write_pos;
  else
    actual_write_pos = 0;
  actual_v = (actual_v == v1 ? v2 : v1);
  number_of_samples = 0;
}
