/* ----------------------------- MNI Header -----------------------------------
@NAME       : display.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Functions dealing with full-colour display of MPEG movies
              using GL:
	        CalcZoomFromWindow
		CalcWindowFromZoom
		CalcImageLocation
	        InitColorDisplay
		DisplayFrame
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 6/94, Greg Ward.
@MODIFIED   : 
---------------------------------------------------------------------------- */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <gl.h>
#include "mpeg.h"
#include "gl_mpeg.h"


long ScreenX = -1;
long ScreenY = -1;



/* ----------------------------- MNI Header -----------------------------------
@NAME       : CalcZoomFromWindow
@INPUT      : WinInfo - window size
              ImageInfo - image size
	      KeepAspect - if TRUE, the x and y zoom factors are 
	         guaranteed to be the same on exit
@OUTPUT     : WinInfo - image zoom factors 
@RETURNS    : (void)
@DESCRIPTION: Calculates the amount by which images (of geometry described
              by ImageInfo) must be zoomed to fit best in a window of
	      size described by WinInfo.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/2?, Greg Ward
@MODIFIED   : 94/7/6, GW: Added KeepAspect argument
---------------------------------------------------------------------------- */
void CalcZoomFromWindow (WindowDesc *WinInfo, ImageDesc *ImageInfo,
			 Boolean KeepAspect)
{
   WinInfo->ZoomX = (float) WinInfo->Width / ImageInfo->Width;
   WinInfo->ZoomY = (float) WinInfo->Height / ImageInfo->Height;

   if (KeepAspect)
   {
      if (WinInfo->ZoomX < WinInfo->ZoomY)
      {
	 WinInfo->ZoomY = WinInfo->ZoomX;
      }
      else if (WinInfo->ZoomY < WinInfo->ZoomX)
      {
	 WinInfo->ZoomX = WinInfo->ZoomY;
      }
   }      
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : CalcWindowFromZoom
@INPUT      : WinInfo - window zoom factors
              ImageInfo - image size
@OUTPUT     : WinInfo - window size
@RETURNS    : (void)
@DESCRIPTION: Calculates the size a window must be to hold images (of
              size described by ImageInfo) to be zoomed by factors
	      specified in WinInfo.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/2?, Greg Ward
@MODIFIED   : 
@COMMENTS   : This is one of the functions where the restriction to 
              integer zooms occurs.
---------------------------------------------------------------------------- */
void CalcWindowFromZoom (WindowDesc *WinInfo, ImageDesc *ImageInfo)
{
   WinInfo->Width = (int) (WinInfo->ZoomX * ImageInfo->Width);
   WinInfo->Height = (int) (WinInfo->ZoomY * ImageInfo->Height);
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : CheckZoom
@INPUT      : WinInfo->{ZoomX,ZoomY}
              ImageInfo->{Width,Height}
	      ScreenX, ScreenY - dimensions of the screen
	      KeepAspect - if TRUE, zoom factors will be the same on exit
@OUTPUT     : WinInfo->{ZoomX,ZoomY}
@RETURNS    : TRUE if neither zoom factor is too big
              FALSE if either zoom factor was too big, and was therefore
	         decremented
@DESCRIPTION: Checks to see if both zoom factors are small enough for 
              the image to fit on screen.  If not, decrements zoom factors
	      (independently of each other, unless KeepAspect is TRUE) 
	      until they are small enough.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/29, Greg Ward
@MODIFIED   : 
@COMMENTS   : This is one of the functions where the restriction to 
              integer zooms occurs.
---------------------------------------------------------------------------- */
Boolean CheckZoom (WindowDesc *WinInfo, ImageDesc *ImageInfo,
		   long ScreenX, long ScreenY, Boolean KeepAspect)
{
   Boolean   retval = TRUE;

   while (WinInfo->ZoomX * ImageInfo->Width > ScreenX)
   {
      WinInfo->ZoomX = ceilf (WinInfo->ZoomX) - 1;
      retval = FALSE;
   }

   while (WinInfo->ZoomY * ImageInfo->Height > ScreenY)
   {	
      WinInfo->ZoomY = ceilf (WinInfo->ZoomY) - 1;
      retval = FALSE;
   }

   if (KeepAspect && !retval)
   {
      WinInfo->ZoomX = WinInfo->ZoomY = min (WinInfo->ZoomX, WinInfo->ZoomY);
   }
   return (retval);
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : CheckWindowSize
@INPUT      : WinInfo - window size (checked for validity)
              ImageInfo - image size (only used if window size is corrected)
	      WinSize - one of the window's sizes (either width or height)
@OUTPUT     : WinInfo->{Width,Height} - only changed if they were too big
@RETURNS    : TRUE if both window dimensions were OK and no changes made;
              FALSE otherwise
@DESCRIPTION: Checks to see if both window dimensions are valid, i.e.
              small enough to fit in the screen.  If either is too big,
	      it is set to the corresponding screen dimension.
@METHOD     : 
@GLOBALS    : 
@CALLS      : CalcZoomFromWindow () [only if height or width changed]
@CREATED    : 94/6/?, Greg Ward
@MODIFIED   : 94/7/29, GW: now checks both width and height at the same time
---------------------------------------------------------------------------- */
Boolean CheckWindowSize (WindowDesc *WinInfo, ImageDesc *ImageInfo,
			 long ScreenX, long ScreenY)
{
   Boolean  retval = TRUE;

   if (WinInfo->Width > ScreenX)
   {
      WinInfo->Width = ScreenX;
      retval = FALSE;
   }

   if (WinInfo->Height > ScreenY)
   {
      WinInfo->Height = ScreenY;
      retval = FALSE;
   }

   if (!retval)
   {
      CalcZoomFromWindow (WinInfo, ImageInfo, TRUE);
   }
   return (retval);
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : CalcImageLocation
@INPUT      : WinInfo - window size, image zoom factors
              ImageInfo - actual image size (unzoomed)
@OUTPUT     : WinInfo - where to put the image in the window so that it's
                 centred, taking zooming and whatnot into account
@RETURNS    : (void)
@DESCRIPTION: Calculates where the image should go in the window (taking
              the image zoom into account; note that the zoom factors
	      are truncated to integers) so that it is centred.
@METHOD     : 
@GLOBALS    : 
@CALLS      : CalcZoomFromWindow
@CREATED    : 94/6/?, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void CalcImageLocation (WindowDesc *WinInfo, ImageDesc *ImageInfo)
{
   int  ZoomWidth, ZoomHeight;	/* size of image after zooming */

   /* Truncate the zoom factors (because most SGI's don't support
    * fractional zoom)
    */

   WinInfo->ZoomX = floorf (WinInfo->ZoomX);
   WinInfo->ZoomY = floorf (WinInfo->ZoomY);

   /* Calculate where to put the image so that it's centred in the window */
   
   ZoomWidth = (int) (WinInfo->ZoomX * ImageInfo->Width);
   ZoomHeight = (int) (WinInfo->ZoomY * ImageInfo->Height);
   WinInfo->ImageX = (WinInfo->Width - ZoomWidth) / 2;
   WinInfo->ImageY = (WinInfo->Height - ZoomHeight) / 2;
}




/* ----------------------------- MNI Header -----------------------------------
@NAME       : FullScreenMode
@INPUT      : Movie->Image - gets size of image to determine zoom factors
@OUTPUT     : Movie->Window - sets size of window (from screen size) and
                 zoom factors (from window size and image size)
@RETURNS    : (void)
@DESCRIPTION: Sets the size of the window to the size of the screen,
              calculates the required image zoom factors to best fit
	      the image in this window, and calls noborder() and 
	      prefposition() to setup the window appropriately.  The
	      caller should take care of actually opening the window.
@METHOD     : 
@GLOBALS    : ScreenX, ScreenY
@CALLS      : CalcZoomFromWindow, CalcImageLocation, GL stuff
@CREATED    : 94/7/5, Greg Ward (copied from code in InitColorDisplay()).
@MODIFIED   : 
---------------------------------------------------------------------------- */
void FullScreenMode (MovieState *Movie)
{
   WindowDesc   *Window = &Movie->Window;

   Window->Width = ScreenX;
   Window->Height = ScreenY;
   CalcZoomFromWindow (Window, &Movie->Image, TRUE);
   CalcImageLocation (Window, &Movie->Image);
   
   dprintf ("  Going fullscreen: %dx%d, zoom %gx%g\n", 
	    Window->Width, Window->Height, Window->ZoomX, Window->ZoomY);
   
   prefposition (0, ScreenX, 0, ScreenY); /* take up the whole screen */
   noborder ();
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : WindowMode
@INPUT      : Movie->WinInfo - uses either the height and width or the 
                 zoom factors; see description of InitColorDisplay() 
		 for details
	      PrefX, PrefY - the preferred coordinates for the lower
	         left-hand corner of the window (passed to 
		 prefposition()).  If either one is negative, 
		 prefsize() is called so the user positions the window.
@OUTPUT     : Movie->WinInfo - sets either the zoom factors or the 
                 height and width - again, see InitColorDisplay()
@RETURNS    : (void)
@DESCRIPTION: Calculates either the image zoom factor from the window
              and image size, or the window size from the zoom factor
	      and image size.  Makes sure that the window size is
	      within the bounds of the screen.
@METHOD     : 
@GLOBALS    : ScreenX, ScreenY
@CALLS      : CalcWindowFromZoom, CalcZoomFromWindow
              CheckWindowSize
	      CalcImageLocation
	      GL stuff
@CREATED    : 94/7/5, Greg Ward (from code in InitColorDisplay())
@MODIFIED   : 
---------------------------------------------------------------------------- */
void WindowMode (MovieState *Movie, long PrefX, long PrefY)
{
   WindowDesc   *Window = &Movie->Window;
   ImageDesc    *Image = &Movie->Image;

   /* If both Width and Height are zero, use the zoom factors */
   
   if (Window->Width == 0 && Window->Height == 0)
   {	
      CheckZoom (Window, Image, ScreenX, ScreenY, TRUE);
      CalcWindowFromZoom (Window, Image);
      dprintf ("  Zooming by (%g,%g) to give (Width,Height) = (%ld,%ld)\n", 
	       Window->ZoomX, Window->ZoomY, 
	       Window->Width, Window->Height);
   }
   
   /* Otherwise, calculate the zoom from the supplied width and height */
   
   else
   {
      CheckWindowSize (Window, Image, ScreenX, ScreenY);
      CalcZoomFromWindow (Window, Image, TRUE);
      dprintf ("  %dx%d window results in (ZoomX,ZoomY) = (%g,%g)\n", 
	       Window->Width, Window->Height, 
	       Window->ZoomX, Window->ZoomY);
   }

/*   
   CheckWindowSize (Window, Image, &Window->Width, ScreenX, "wide");
   CheckWindowSize (Window, Image, &Window->Height, ScreenY, "high");
*/
   CalcImageLocation (Window, Image);
   
   if (PrefX < 0 || PrefY < 0)
   {
      prefsize (Window->Width, Window->Height);
   }
   else
   {
      prefposition (PrefX, PrefX+Window->Width-1, 
		    PrefY, PrefY+Window->Height-1);
   }
}     /* WindowMode () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : OpenWindow
@INPUT      : Movie->Window - needed for zoom factors
              Movie->Options - if DoubleBuff set, enables double buffering 
	         and clears both buffers
@OUTPUT     : Movie->Window - sets handle
@RETURNS    : FALSE if winopen() fails
              TRUE otherwise
@DESCRIPTION: Calls winopen() to create window (size or position should
              have already been specified by a call to either 
	      FullScreenMode() or WindowMode()), minsize() to set minimum
	      size, winconstraints() to make it resizeable, rectzoom()
	      to set the zoom factors, gconfig() to make it all work,
	      clears the window, and optionally turns on double buffering.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/5, Greg Ward (from code in InitColorDisplay())
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean OpenWindow (MovieState *Movie)
{
   char	         winname [100];
   char          IconName [100];
   WindowDesc   *Window = &Movie->Window;
   ImageDesc    *Image = &Movie->Image;

   /* Create the window title */

   dprintf ("InitColorDisplay:\n");
   sprintf (winname, "MPEG Player: %s", 
	    Movie->Filename ? Movie->Filename : "<stdin>");

   if (Movie->Filename == NULL)
   {
      strcpy (IconName, "MPEG");
   }
   else
   {
      char *ptr;
      ptr = strrchr (Movie->Filename, '/');
      if (ptr == NULL)
      {
	 strcpy (IconName, Movie->Filename);
      }
      else
      {
	 strcpy (IconName, ptr+1);
      }
   }

   iconsize (85, 67);

   if ((Window->Handle = winopen (winname)) == -1)
   {
      fprintf (stderr, "Unable to open window");
      return (FALSE);
   }
   minsize ((long) Image->Width, (long) Image->Height);
   winconstraints ();

   icontitle (IconName);

   rectzoom (Window->ZoomX, Window->ZoomY);
   RGBmode ();
   if (Movie->Options.DoubleBuff) doublebuffer ();

   gconfig ();
   clear ();
   if (Movie->Options.DoubleBuff) 
   {
      swapbuffers ();
      clear ();
   }
   return (TRUE);
}     /* OpenWindow () */


/* ----------------------------- MNI Header -----------------------------------
@NAME       : InitColorDisplay
@INPUT      : Movie:
                the omnibus descriptor for all things movie-related.
                Fields of interest here are Filename (for the window
                title), Image and Window (to size the window), and
                Options.FullScreen.  The following rules govern how
                Movie->Image and Movie->Window interact to determine
                the size of the window and the zoom factor(s) used
                to display the movie:

		   - if Window.(Width,Height) are both zero, then
		     the window size will be determined from 
		     Image.(Width,Height), multiplied by 
		     Window.(ZoomX,ZoomY) (this is why ZoomX and ZoomY
		     default to 1.0 -- see ParseCmdLine()).
		   - if Window.(Width,Height) are not both zero, then
		     the zoom values specified in ZoomX and ZoomY will
		     be ignored, and ZoomX and ZoomY will be recalculated
		     based on the image width/height and the window
		     width/height specified in Window.  
                   - if FullScreen is TRUE, then Window.(Width,Height)
		     will be set to the dimensions of the screen, and
		     (ZoomX,ZoomY) will be calculated as described above

		Note that in all three cases, fractional zoom is possible.
		However, since most current SGI hardware doesn't support
		fractional zoom, InitColorDisplay() *always* truncates
		the zoom factor before calculating the location of the
		image within the window.  This is because the image will
		only be zoomed by an integer amount regardless of what
		is passed to rectzoom() (again, on most current SGI 
		hardware), and we want an accurate idea of what the REAL
		zoom will be so that the image can be properly centred
		in the window.  The results won't be screwed up if run
		on hardware that supports fractional zoom; we simply won't
		be taking full advantage of the hardware.  Unfortunately,
		it doesn't look as though there's a way to find out if
		fractional zoom is supported on the current hardware.

@OUTPUT     : Movie->Window:
                 one of the pairs (Width,Height) or (ZoomX,ZoomY) will
                 be calculated from the other, depending on
                 whether (Width,Height) is (0,0) or not.  The caller
                 really doesn't need to worry about the contents of
                 Movie->Window, except for initially setting either
                 (Width,Height) or (ZoomX,ZoomY).  After that, just
		 leave it alone in the MovieState struct and don't
		 worry about it.  (Unless, of course, the window
		 is resized.  In that case, you'll want to call 
		 CalcZoomFromWindow() and CalcImageLocation() given 
		 the new window size.)

@RETURNS    : FALSE if call to dglopen() fails
              FALSE if call to winopen() fails
	      (prints error message to stderr in either case)
              TRUE otherwise
@DESCRIPTION: Determines the desired window size, sets up various GL
              things, and opens the window.  WinInfo provides both
	      input (either desired window size or desired zoom factors)
	      and output (whichever of those two is not supplied) which
	      determine later behaviour when displaying images.
@METHOD     : 
@GLOBALS    : DisplayName
@CALLS      : GL stuff
@CREATED    : 94/6/??, Greg Ward.
@MODIFIED   : 94/6/27, GW: replaced long list of arguments with Movie,
                           and updated code appropriately;
			   made double-buffering stuff optional
              94/7/5, GW: replaced some chunks of code with calls
                          to FullScreenMode(), WindowMode(), OpenWindow()
@COMMENTS   : a bit of this code is reproduced in function RedrawWindow(),
              in interface.c.  In the interestes of modularity, there
	      really ought to be a separate function for this... (along




	      the lines of FullScreenMode())
---------------------------------------------------------------------------- */
Boolean InitColorDisplay(MovieState *Movie)
{
   WindowDesc *WinInfo;		/* 'cause I'm a lazy typist ... */
   ImageDesc  *ImageInfo;
   int         result;

   WinInfo = &Movie->Window;
   ImageInfo = &Movie->Image;

   /* If the user supplied an alternate display (eg. with -display), open
    * a DGL connection to it
    */

   if (DisplayName != NULL)
   {
      result = dglopen (DisplayName, DGLTSOCKET);
      if (result < 0)
      {
	 fprintf (stderr, "Could not open DGL connection to %s: %s\n", 
		  DisplayName, strerror(result));
	 return (FALSE);
      }
   }

   /* Get the screen size */

   ScreenX = getgdesc (GD_XPMAX);
   ScreenY = getgdesc (GD_YPMAX);

   /* Calculate the window width/height and/or zoom factors, and set
    * the position/size/border preferences in preparation for winopen()
    */

   if (Movie->Options.FullScreen)
   {
      FullScreenMode (Movie);
   }
   else
   {
      WindowMode (Movie, -1, -1);
   }

   /* And finally, open the bloody window */

   foreground ();
   if (!OpenWindow (Movie))
      return (FALSE);

/* OpenHelpWindow (Movie); */

   InitializeQueues ();
   return (TRUE);

}     /* InitColorDisplay () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : DisplayFrame
@INPUT      : Movie - contains everything we could possibly want
                      to know about a movie; in this case, the location
		      of images within the window, and the size of images
	      Frame - the actual image data for the current frame, in
	              the format expected by lrectwrite() (i.e., four bytes
		      per pixel: red, green, blue, alpha)
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Displays the current frame using lrectwrite(), and swaps
              buffers (assuming double buffering is active!).
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/??, Greg Ward
@MODIFIED   : 94/6/27, GW: replaced WinInfo and ImgInfo arguments with Movie
                           made swapbuffers() optional
              94/6/28, GW: took out Frame arguments because of new 
                           CurrentImag field in MovieState
---------------------------------------------------------------------------- */
void DisplayFrame (MovieState *Movie)
{
   WindowDesc *WinInfo = &Movie->Window;
   ImageDesc  *ImgInfo = &Movie->Image;

   dprintf ("Displaying frame %d\n", Movie->CurFrame);

   lrectwrite (WinInfo->ImageX, WinInfo->ImageY, 
	       WinInfo->ImageX+ImgInfo->Width-1, 
	       WinInfo->ImageY+ImgInfo->Height-1, 
	       (unsigned long *) Movie->CurrentImage);

   if (Movie->Options.DoubleBuff)
   {
      swapbuffers ();
   }

}
