/* interp.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: interp.c,v 1.6 1995/06/12 15:43:57 brianp Exp $

$Log: interp.c,v $
 * Revision 1.6  1995/06/12  15:43:57  brianp
 * separate GLint and GLubyte interpolation functions
 *
 * Revision 1.5  1995/06/02  13:58:36  brianp
 * faster gl_interpolate(), tried gl_interpolate_rgba()
 *
 * Revision 1.4  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.3  1995/03/17  19:15:21  brianp
 * tried to improve gl_interp_texcoords, not much luck
 *
 * Revision 1.2  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.1  1995/02/27  22:20:13  brianp
 * Initial revision
 *
 */


#include "context.h"
#include "macros.h"


/*#define TEST*/


/*
 * Linear integer interpolation:
 * Iterpolate n integer values between y0 and y1 and put into yspan.
 * When finished, yspan[0] = y0, yspan[n-1] = y1, and the rest of
 * yspan[] is filled with interpolated values.
 */
void gl_interpolate_i( GLint n, GLint y0, GLint y1, GLint yspan[] )
{
   switch (n) {
      case 1:
         yspan[0] = y0;
	 return;
      case 2:
         yspan[0] = y0;
         yspan[1] = y1;
	 return;
      case 3:
         yspan[0] = y0;
	 yspan[1] = (y0+y1) >> 1;
         yspan[2] = y1;
	 return;
      default:
	 if (y0==y1) {
	    register GLint i;
	    for (i=0;i<n;i++) {
	       yspan[i] = y0;
	    }
	 }
	 else {
	    register GLint i;
	    register GLint dx, dy;
	    register GLint a, b, d;
	    register GLint y;
	    register GLint qa, qb;
   	    dx = n-1;
	    dy = y1 - y0;
	    qa = dy / dx;
	    dy = dy % dx;
	    if (dy<0) {
	       dy = -dy;
	       qb = qa - 1;
	    }
	    else {
	       qb = qa + 1;
	    }
	    a = dy+dy;   d = a-dx;   b = d-dx;
	    y = y0;
	    for (i=0;i<n;i++) {
	       yspan[i] = y;
	       if (d<0) {
		  d += a;
		  y += qa;
	       }
	       else {
		  d += b;
		  y += qb;
	       }
	    }
	 }
   }
}


/*
 * Same as above but store results in array of GLubytes.
 */
void gl_interpolate_ub( GLint n, GLint y0, GLint y1, GLubyte yspan[] )
{
   switch (n) {
      case 1:
         yspan[0] = y0;
	 return;
      case 2:
         yspan[0] = y0;
         yspan[1] = y1;
	 return;
      case 3:
         yspan[0] = y0;
	 yspan[1] = (y0+y1) >> 1;
         yspan[2] = y1;
	 return;
      default:
	 if (y0==y1) {
	    register GLint i;
	    for (i=0;i<n;i++) {
	       yspan[i] = y0;
	    }
	 }
	 else {
	    register GLint i;
	    register GLint dx, dy;
	    register GLint a, b, d;
	    register GLint y;
	    register GLint qa, qb;
   	    dx = n-1;
	    dy = y1 - y0;
	    qa = dy / dx;
	    dy = dy % dx;
	    if (dy<0) {
	       dy = -dy;
	       qb = qa - 1;
	    }
	    else {
	       qb = qa + 1;
	    }
	    a = dy+dy;   d = a-dx;   b = d-dx;
	    y = y0;
	    for (i=0;i<n;i++) {
	       yspan[i] = y;
	       if (d<0) {
		  d += a;
		  y += qa;
	       }
	       else {
		  d += b;
		  y += qb;
	       }
	    }
	 }
   }
}



/*
 * RGBA color interpolation.
 * Input:  n - number of values to produce
 *         r0, r1 - first and last red values
 *         g0, g1 - first and last green values
 *         b0, b1 - first and last blue values
 *         a0, a1 - first and last alpha values
 * Output:  ir, ig, ib, ia - arrays of [n] interpolated color components
 */
void gl_interpolate_rgba( GLint n, GLint r0, GLint r1, GLint *ir,
				   GLint g0, GLint g1, GLint *ig,
				   GLint b0, GLint b1, GLint *ib,
				   GLint a0, GLint a1, GLint *ia )
{
   register GLint i;
   register GLint dr, dg, db, da;  /* deltas */
   register GLint er, eg, eb, ea;  /* error term */
   register GLint era, ega, eba, eaa;  /* error adjuster a */
   register GLint erb, egb, ebb, eab;  /* error adjuster b */
   register GLint qr, qg, qb, qa;  /* quotient */
   register GLint pr, pg, pb, pa;  /* remainder */

   switch (n) {
      case 1:
         ir[0] = r0;   ig[0] = g0;   ib[0] = b0;   ia[0] = a0;
	 return;
      case 2:
	 ir[0] = r0;   ig[0] = g0;   ib[0] = b0;   ia[0] = a0;
	 ir[1] = r1;   ig[1] = g1;   ib[1] = b1;   ia[1] = a1;
	 return;
      case 3:
	 ir[0] = r0;   ig[0] = g0;   ib[0] = b0;   ia[0] = a0;
	 ir[1] = (r0+r1)>>1;   ig[1] = (g0+g1)>>1;
	 ib[1] = (b0+b1)>>1;   ia[1] = (a0+a1)>>1;
	 ir[2] = r1;   ig[2] = g1;   ib[2] = b1;   ia[2] = a1;
	 return;
      default:
	 n--;
	 /* red           green         blue          alpha */
	 dr = r1-r0;   dg = g1-g0;   db = b1-b0;   da = a1-a0;
	 qr = dr/n;    qg = dg/n;    qb = db/n;    qa = da/n; 
	 dr = dr%n;    dg = dg%n;    db = db%n;    da = da%n; 
	 if (dr<0) { dr = -dr;   pr = qr-1; } else { pr = qr+1; }  /* red */
	 if (dg<0) { dg = -dg;   pg = qg-1; } else { pg = qg+1; }  /* green */
	 if (db<0) { db = -db;   pb = qb-1; } else { pb = qb+1; }  /* blue */
	 if (da<0) { da = -da;   pa = qa-1; } else { pa = qa+1; }  /* alpha */

	 /*  red            green          blue           alpha */
	 era = dr+dr;   ega = dg+dg;   eba = db+db;   eaa = da+da;
	 er = era-n;    eg = ega-n;    eb = eba-n;    ea = eaa-n; 
	 erb = er-n;    egb = eg-n;    ebb = eb-n;    eab = ea-n; 

	 for (i=0;i<=n;i++) {
	    ir[i] = r0;   ig[i] = g0;   ib[i] = b0;   ia[i] = a0;
	    if (er<0) { er += era;  r0 += qr; } else { er += erb;  r0 += pr; }
	    if (eg<0) { eg += ega;  g0 += qg; } else { eg += egb;  g0 += pg; }
	    if (eb<0) { eb += eba;  b0 += qb; } else { eb += ebb;  b0 += pb; }
	    if (ea<0) { ea += eaa;  a0 += qa; } else { ea += eab;  a0 += pa; }
	 }
   }
}


#ifndef TEST

/*
 * Perform texture coordinate interpolation along a line in window coord-
 * inate space.  Depending on the perspective correction hint we'll either
 * just do simple linear interpolation or interpolation with perspective
 * correction.
 * Input:  n - number of texture coords to produce
 *         eyez0, eyez1 - z coordinate of end points in eye coords
 *         winz0  winz1 - z coordinate of end points in window coords
 *         s0, s1 - S-component of texture coords at end points
 *         t0, t1 - T-component of texture coords at end points
 * Output:  s, t - resulting arrays of S and T texture coordinates
 *          z - interpolated Z component of eye coordinates along the line
 *              (note that z can be a NULL pointer if z isn't needed)
 */
void gl_interp_texcoords( GLuint n,
			  GLfloat eyez0, GLfloat eyez1,
			  GLfloat winz0, GLfloat winz1,
			  GLfloat s0, GLfloat s1,
			  GLfloat t0, GLfloat t1,
			  GLfloat s[], GLfloat t[], GLfloat *z )
{
   /* TODO: the main loop in this function could be optimized with */
   /* forward differences, etc.  However, since texture mapping is */
   /* so expensive, optimizing this might not make a big difference. */

   /* TODO: this function has a terrible problem with numerical error */
   /* under certain conditions, namely when interpolating along a vector */
   /* which is _nearly_ parallel to the viewplane. */

   GLfloat d = CC.ProjectionMatrix[14];
   GLfloat c = CC.ProjectionMatrix[10];
   GLfloat delta_eyez = eyez1 - eyez0;
   GLfloat delta_winz = winz1 - winz0;
   GLfloat delta_s = s1 - s0;
   GLfloat delta_t = t1 - t0;
   GLuint i;

   if (n==1) {
      s[0] = s0;
      t[0] = t0;
      if (z)  z[0] = eyez0;
      return;
   }

   if (CC.Hint.PerspectiveCorrection==GL_NICEST && d!=0.0
       && ABS(delta_eyez)>0.001) {
      /* interpolate with perspective compensation */
      GLfloat q, r, winz, ndcz, eyez;

      for (i=0;i<n;i++) {
	 q = (GLfloat) i / (n-1);
	 winz = (winz0 + q * delta_winz) / (GLfloat) MAX_DEPTH;

	 ndcz = (winz - CC.Viewport.Tz) / CC.Viewport.Sz;

	 eyez = -d / (c+ndcz);

	 r = (eyez - eyez0) / delta_eyez;
	 r = CLAMP( r, 0.0, 1.0 );    /* to catch over/underflow */

	 s[i] = s0 + r * delta_s;
	 t[i] = t0 + r * delta_t;
	 if (z) {
	    z[i] = eyez;
	 }
      }
   }
   else {
      /* simple linear interpolation */
      if (z) {
	 for (i=0;i<n;i++) {
	    GLfloat q = (GLfloat) i / (n-1);
	    s[i] = s0 + q * delta_s;
	    t[i] = t0 + q * delta_t;
	    z[i] = eyez0 + q * delta_eyez;
	 }
      }
      else {
	 for (i=0;i<n;i++) {
	    GLfloat q = (GLfloat) i / (n-1);
	    s[i] = s0 + q * delta_s;
	    t[i] = t0 + q * delta_t;
	 }
      }
   }
}

#endif


#ifdef TEST
static void foo( int n, int red[], int green[], int blue[], int alpha[] )
{

}



#define ITER 700
#define MAX 500

#include "interp.h"

main()
{
   int i, j, n, jj;

   for (i=0;i<ITER;i++) {
      int red[MAX], green[MAX], blue[MAX], alpha[MAX];
      jj = MAX2( 2, i );
      for (j=0;j<jj;j++) {
	 n = j/4+1;

	 GL_INTERPOLATE( n, j, n, red );
	 GL_INTERPOLATE( n, j+3, j*8, green );
	 GL_INTERPOLATE( n, i, j+i*2, blue );
	 GL_INTERPOLATE( n, i, i, alpha );
/*	 gl_interpolate_rgba( n, j, n, red,
	                         j+3, j*8, green,
                                 i, j+i*2, blue,
                                 i, i, alpha );
*/
      }
      foo( n, red, green, blue, alpha );
   }

}

#endif
