/* nurbssrf.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: nurbssrf.c,v 1.3 1995/08/02 15:12:29 brianp Exp $

$Log: nurbssrf.c,v $
 * Revision 1.3  1995/08/02  15:12:29  brianp
 * use MEMCPY macro instead of memcpy
 *
 * Revision 1.2  1995/07/28  21:37:30  brianp
 * updates from Bogdan on July 28
 *
 * Revision 1.1  1995/07/28  14:45:01  brianp
 * Initial revision
 *
 */


/*
 * NURBS implementation written by Bogdan Sikorski (gstbs@io.coi.pw.edu.pl)
 * See README-nurbs for more info.
 */


#include <stdlib.h>
#include <string.h> /* prototype of memcpy() */
#include <math.h>
#include "nurbs.h"
#include "gluP.h"



static GLenum
test_nurbs_surface(GLUnurbsObj *nobj)
{
	nurbs_surface surface=nobj->surface;
	GLint tmp_int;

	if(surface.sorder < 0)
	{
		call_user_error(nobj,GLU_INVALID_VALUE);
		return GLU_ERROR;
	}
	if(surface.torder < 0)
	{
		call_user_error(nobj,GLU_INVALID_VALUE);
		return GLU_ERROR;
	}
	glGetIntegerv(GL_MAX_EVAL_ORDER,&tmp_int);
	if(surface.sorder > tmp_int || surface.sorder < 2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR1);
		return GLU_ERROR;
	}
	if(surface.torder > tmp_int || surface.torder < 2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR1);
		return GLU_ERROR;
	}
	if(surface.sknot_count < surface.sorder +2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR2);
		return GLU_ERROR;
	}
	if(surface.tknot_count < surface.torder +2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR2);
		return GLU_ERROR;
	}
	if(surface.s_stride < 0 || surface.t_stride < 0)
	{
		call_user_error(nobj,GLU_NURBS_ERROR34);
		return GLU_ERROR;
	}
	if(surface.type!=GL_MAP2_VERTEX_3 && surface.type!=GL_MAP2_VERTEX_4)
	{
		call_user_error(nobj,GLU_INVALID_ENUM);
		return GLU_ERROR;
	}
	if(surface.sknot==NULL || surface.tknot==NULL || surface.ctrlarray==NULL)
	{
		call_user_error(nobj,GLU_NURBS_ERROR8);
		return GLU_ERROR;
	}
	return GLU_NO_ERROR;
}

static GLenum
convert_surf( GLUnurbsObj *nobj, GLint dim, GLfloat **new_ctrl,
	GLint *s_n_ctrl, GLint *t_n_ctrl)
{
	knot_str_type s_knot,t_knot;
	nurbs_surface surface;
	GLfloat **tmp_ctrl;
	GLfloat *ctrl_offset;
	GLint tmp_n_control;
	GLint i,j,t_cnt,s_cnt;
	GLint tmp_stride;
	GLenum err;

	surface=nobj->surface;
	t_cnt=surface.tknot_count-surface.torder;
	s_cnt=surface.sknot_count-surface.sorder;
	if((tmp_ctrl=(GLfloat **)malloc(sizeof(GLfloat *)*t_cnt))==NULL)
	{
		call_user_error(nobj,GLU_OUT_OF_MEMORY);
		return GLU_ERROR;
	}
	s_knot.knot=surface.sknot;
	s_knot.order=surface.sorder;
	s_knot.nknots=surface.sknot_count;
	if((err=explode_knot(&s_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		call_user_error(nobj,err);
		return GLU_ERROR;
	}
	if((err=calc_alphas(&s_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		free(s_knot.new_knot);
		call_user_error(nobj,err);
		return GLU_ERROR;
	}
	free(s_knot.new_knot);
	ctrl_offset=surface.ctrlarray;
	for(i=0;i<t_cnt;i++)
	{
		if((err=calc_new_ctrl_pts(ctrl_offset,surface.s_stride,&s_knot,
			dim,&(tmp_ctrl[i]),&tmp_n_control))!=GLU_NO_ERROR)
		{
			for(--i;i<=0;i--)
				free(tmp_ctrl[i]);
			free(tmp_ctrl);
			free(s_knot.alpha);
			call_user_error(nobj,err);
			return GLU_ERROR;
		}
		ctrl_offset+=surface.t_stride;
	}
	free(s_knot.alpha);
	tmp_stride=dim*tmp_n_control;
	if((*new_ctrl=(GLfloat *)malloc(sizeof(GLfloat)*tmp_stride*t_cnt))
		==NULL)
	{
		for(i=0;i<t_cnt;i++)
			free(tmp_ctrl[i]);
		free(tmp_ctrl);
		call_user_error(nobj,GLU_OUT_OF_MEMORY);
		return GLU_ERROR;
	}
	for(i=0;i<tmp_n_control;i++)
		for(j=0;j<t_cnt;j++)
			MEMCPY(*new_ctrl+j*dim+i*dim*t_cnt,tmp_ctrl[j]+dim*i,
				sizeof(GLfloat)*dim);
	for(i=0;i<t_cnt;i++)
		free(tmp_ctrl[i]);
	free(tmp_ctrl);
	*s_n_ctrl=tmp_n_control;
	
	if((tmp_ctrl=(GLfloat **)malloc(sizeof(GLfloat *)*(*s_n_ctrl)))==NULL)
	{
		call_user_error(nobj,GLU_OUT_OF_MEMORY);
		return GLU_ERROR;
	}
	t_knot.knot=surface.tknot;
	t_knot.order=surface.torder;
	t_knot.nknots=surface.tknot_count;
	if((err=explode_knot(&t_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		call_user_error(nobj,err);
		return GLU_ERROR;
	}
	if((err=calc_alphas(&t_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		free(t_knot.new_knot);
		call_user_error(nobj,err);
		return GLU_ERROR;
	}
	free(t_knot.new_knot);
	ctrl_offset=*new_ctrl;
	for(i=0;i<(*s_n_ctrl);i++)
	{
		if((err=calc_new_ctrl_pts(ctrl_offset,dim,&t_knot,
			dim,&(tmp_ctrl[i]),&tmp_n_control))!=GLU_NO_ERROR)
		{
			for(--i;i<=0;i--)
				free(tmp_ctrl[i]);
			free(tmp_ctrl);
			free(t_knot.alpha);
			call_user_error(nobj,err);
			return GLU_ERROR;
		}
		ctrl_offset+=dim*t_cnt;
	}
	free(t_knot.alpha);
	free(*new_ctrl);
	tmp_stride=dim*tmp_n_control;
	if((*new_ctrl=(GLfloat *)malloc(sizeof(GLfloat)*tmp_stride*(*s_n_ctrl)))
		==NULL)
	{
		for(i=0;i<(*s_n_ctrl);i++)
			free(tmp_ctrl[i]);
		free(tmp_ctrl);
		call_user_error(nobj,GLU_OUT_OF_MEMORY);
		return GLU_ERROR;
	}
	for(i=0;i<(*s_n_ctrl);i++)
	{
		MEMCPY(*new_ctrl+i*tmp_stride,tmp_ctrl[i],sizeof(GLfloat)*tmp_stride);
		free(tmp_ctrl[i]);
	}
	free(tmp_ctrl);
	*t_n_ctrl=tmp_n_control;
	return GLU_NO_ERROR;
}

void
do_nurbs_surface( GLUnurbsObj *nobj )
{
	GLint s_n_ctrl,t_n_ctrl;
	GLfloat *new_ctrl;
	GLint	*sfactors,*tfactors;
	GLint i,j,s_indx,t_indx;
	GLenum type;
	GLenum display_mode;
	GLint sorder,torder;
	GLint dim;

	if(test_nurbs_surface(nobj)!=GLU_NO_ERROR)
		return;
	type=nobj->surface.type;
	sorder=nobj->surface.sorder;
	torder=nobj->surface.torder;
	switch(type)
	{
		case GL_MAP2_VERTEX_3:
			dim=3;
			break;
		case GL_MAP2_VERTEX_4:
			dim=4;
			break;
		default: /* can't be here, just to keep gcc happy */
			return;
	}
/*	if(culling_test_3D(nobj,dim))
		return;*/
	if(convert_surf(nobj,dim,&new_ctrl,&s_n_ctrl,&t_n_ctrl)!=GLU_NO_ERROR)
		return;
	if(glu_do_sampling_3D(nobj,new_ctrl,s_n_ctrl,t_n_ctrl,dim,&sfactors,
		&tfactors)!=GLU_NO_ERROR)
	{
		free(new_ctrl);
		return;
	}
	switch(nobj->display_mode)
	{
		case GLU_FILL:
			display_mode=GL_FILL;
			break;
		case GLU_OUTLINE_POLYGON:
			display_mode=GL_LINE;
			break;
		case GLU_OUTLINE_PATCH:
			display_mode=GL_LINE;
			break;
		default: /* to keep gcc happy */
			display_mode=GL_LINE;
	}
	glEnable(type);
	for(j=0 , s_indx=0;j<s_n_ctrl;j+=sorder , s_indx++)
		for(i=0 , t_indx=0;i<t_n_ctrl;i+=torder , t_indx++)
		{
			if(fine_culling_test_3D(nobj,new_ctrl+i*dim+j*dim*t_n_ctrl,
				sorder,torder,dim*t_n_ctrl,dim,dim))
				continue;
			glMap2f(type,0.0,1.0,dim*t_n_ctrl,sorder,
				0.0,1.0,dim,torder,
				new_ctrl+i*dim+j*dim*t_n_ctrl);
			glMapGrid2f(sfactors[s_indx],0.0,1.0,
				tfactors[t_indx],0.0,1.0);
			glEvalMesh2(display_mode,0,sfactors[s_indx],
				0,tfactors[t_indx]);
		}
	free(new_ctrl);
	free(sfactors);
	free(tfactors);
}

