/*
    pixtolff - convert .pix files to .lff (LucasFilm Format) files
    Usage: pixtolff [--] [-t rows cols] [-rle] [-dump] pixfile lucasfilm-file
*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "pixlib.h"

#ifdef sys5
#define rindex strrchr
#endif

char Usage[] = "Usage: %s [--] [-t rows cols] [-rle] [-dump] pixfile lucasfilm-file\n";
char *Example[] = {
    "\tpixtolff converts a .pix format image into a Lucas-Film Format\n",
    "\timage. The first argument should be the name of an input\n",
    "\t.pix file, the second the lucas-film file to be created.\n",
    "\tIf the -t argument is specified, it must be followed with 2 numbers\n",
    "\tindicating the number of vertical and horizontal tiles to divide the\n",
    "\tpicture into.\n",
    "\tIf the -rle or -dump flags are specified, the generated image\n",
    "\twill be forced to run-length encoded or dumped formats respectively\n",
    "\n\tExample: pixtolff tracer.pix tracer.lff\n",
    NullPtr(char)
};

PIX_HDR Header; /* Header of the input .pix file */
RGBPIXEL *Pix;	/* Pointer to the loaded picture */

main(argc,argv)
int argc;
char *argv[];
{
    char *pixfile = NullPtr(char), *lff_file = NullPtr(char);
    int i, rle, dump, status;
    int tflag, vtiles, htiles;
    int getpixfun(),   /* Function to return pixels from .pix file */
	pixtilefun();
    int getformat();/* Convert .pix format code into .lff format code */
    int storage_flag = STORAGE_DUMP;

    rle = dump = FALSE;
    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i],"--"))
	    userhelp(argv[0],Usage,Example);
	else if (!strcmp(argv[i],"-rle"))
	    storage_flag = STORAGE_RLE, rle = TRUE;
	else if (!strcmp(argv[i],"-dump"))
	    storage_flag = STORAGE_DUMP, dump = TRUE;
	else if (!strcmp(argv[i],"-t")) {
	    tflag = TRUE;
	    if (!sscanf(argv[++i],"%d",&vtiles) ||
		!sscanf(argv[++i],"%d",&htiles))
		gr_bomb("-t flag must be followed by two numbers\n");
	} else if (!pixfile)
	    pixfile = argv[i];
	else
	    lff_file = argv[i];
    }

    if (!pixfile)
	userhelp(argv[0],Usage,Example);

    if (!lff_file) {
	char *tailptr, *rindex();

	lff_file = malloc(strlen(pixfile)+4);
	strcpy(lff_file,pixfile);
	tailptr = rindex(lff_file,'.');

	if (tailptr)
	    strcpy(tailptr+1,"lff");
	else
	    strcat(lff_file,".lff");

	fprintf(stderr,"No output file specified - creating %s\n",lff_file);
    }

    /* Open up input and output files */
    status = open_pix(pixfile,&Header);
    if (status != PIX_OK) {
	fprintf(stderr,"%s: %s\n",pixfile,pixerrmess(status));
	exit(-1);
    }

    /*
	Verify the .lff file does not exist
    */
    {
	struct stat buf;
	if (stat(lff_file,&buf) == 0)
	    gr_bomb("Output file %s already exists\n",lff_file);
    }

    /* If the picture is one tile, we can optimize a bit. Maybe we can
	optimize, that is!
    */
    if (!tflag || (vtiles == 1 && htiles == 1))
	status = write_lff(
	    lff_file,
	    Header.height,
	    Header.width,
	    1,1,
	    getformat(Header.mask),
	    storage_flag,
	    0,0,
	    LFF_VERSION,
	    0,
	    getpixfun);
    else {
	/* First, allocate enough memory to read the entire damned
	    pix file in; then write the lff file
	*/
	int y;
	RGBPIXEL *pix, *ptr;

	pix = (RGBPIXEL *)malloc(Header.height * Header.width *
				 sizeof(RGBPIXEL));
	if (pix == (RGBPIXEL *)NULL)
	    gr_bomb("Not enough memory for tile conversion\n");

	/* Now, read each line of the .pix file in */
	for (ptr = pix, y = 0; y < Header.height; y++,ptr += Header.width) {
	    int pixy;

	    status = pix_read_line(ptr,&pixy);
	    if (status != PIX_OK)
		gr_bomb("Error loading .pix file on line %s: %s\n",
		    y,pixerrmess(status));
	}

	/* Now set up the lff pixel reading function so that it has access
	    to the globally stored .pix file.
	*/
	Pix = pix;

	status = write_lff(
	    lff_file,
	    Header.height,
	    Header.width,
	    vtiles,htiles,
	    getformat(Header.mask),
	    storage_flag,
	    0,
	    0,
	    pixtilefun);
    }
    if (status != LFF_OK) {
	fprintf(stderr,"%s: %s\n",lff_file,pixerrmess(status));
	exit(-1);
    }
}

/*
    Function called by write_lff, which returns the next line from the
	.pix file
*/
int getpixfun(row,col,length,pixels)
int row, col, length;
RGBPIXEL pixels[];
{
    int status, y;

    status = pix_read_line(pixels,&y);
    if (status != PIX_OK)
	return 0;
    else
	return length;
}

/*
    Function called by write_lff, which returns the next line from the
	.pix file. This version for tiled pictures.
*/
int pixtilefun(row,col,length,pixels)
int row, col, length;
RGBPIXEL pixels[];
{
    RGBPIXEL *pixsrc;

    pixsrc = Pix + row * Header.width + col;

    bcopy( (char *)pixsrc,(char *)pixels,length * sizeof(RGBPIXEL) );
    return length;
}

int getformat(mask)
int mask;
{
    int format = 0;

    if (mask & BW)		/* This will make the .lff file 3 times too big! */
	return FORMAT_RGB;

    if (mask & RED)
	format |= FORMAT_R;
    if (mask & GRN)
	format |= FORMAT_G;
    if (mask & BLU)
	format |= FORMAT_B;

    return format;
}
