#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "pixlib.h"

LFF_HDR Imagehdr;

char Usage[] =
"%s [--] input-file output-file [-rle] [-dump] [-tiles vt ht]\n\
   [-title 'name'] [-xoff x] [-yoff y] [-vers v] [-alpha a] [-force]\n";

char *Example[] = {
    "\tlffxform transforms an input lucasfilm file to an output file, performing",
    "\tconversions as specified:",
    "\t\t-rle  : run-length encode output\n",
    "\t\t-dump : dump output\n",
    "\t\t-tiles: output specified number of vertical & horizontal tiles\n",
    "\t\t-title: place specified title in image header\n",
    "\t\t-xoff : place specified x offset in header\n",
    "\t\t-yoff : place specified y offset in header\n",
    "\t\t-vers : place specified version number in header\n",
    "\t\t-alpha: place specified alpha mode flag (0 or 1) in header\n",
    "\t\t-force: force writing of output file even if it exists already\n",
    "\n\tExample: lffxform encoded.lff -dump -title 'Dumped' dumped.lff\n",
    NullPtr(char)
};

main (argc, argv)
int	argc;
char   *argv[];
{
    int  i, getpixfun(), status;
    char *lffname = NULL, *outname = NULL;
    RGBPIXEL *image;

    int rleflag, dumpflag, tileflag, titleflag, xflag, yflag;
    int verflag, aflag, forceflag;
    int vtiles, htiles, xoff, yoff, version, alpha;
    char *title;

    rleflag = dumpflag = tileflag = titleflag = xflag = yflag =
	      verflag = aflag = forceflag = FALSE;

    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i],"--"))
	    userhelp(argv[0],Usage,Example);
	else if (!strcmp(argv[i],"-rle"))
	    rleflag = TRUE;
	else if (!strcmp(argv[i],"-dump"))
	    dumpflag = TRUE;
	else if (!strcmp(argv[i],"-tiles")) {
	    if (!sscanf(argv[++i],"%d",&vtiles) ||
		!sscanf(argv[++i],"%d",&htiles))
		gr_bomb("Error scanning number of vertical tiles (%s)\n\
\tor horizontal tiles (%s)\n",argv[i-1],argv[i]);
	    tileflag = TRUE;
	} else if (!strcmp(argv[i],"-title")) {
	    title = argv[i];
	    titleflag = TRUE;
	} else if (!strcmp(argv[i],"-xflag")) {
	    if (!sscanf(argv[++i],"%d",&xoff))
		gr_bomb("Error scanning x offset '%s'\n",argv[i]);
	    xflag = TRUE;
	} else if (!strcmp(argv[i],"-yflag")) {
	    if (!sscanf(argv[++i],"%d",&yoff))
		gr_bomb("Error scanning y offset '%s'\n",argv[i]);
	    yflag = TRUE;
	} else if (!strcmp(argv[i],"-vers")) {
	    if (!sscanf(argv[++i],"%d",&version))
		gr_bomb("Error scanning version '%s'\n",argv[i]);
	    verflag = TRUE;
	} else if (!strcmp(argv[i],"-alpha")) {
	    if (!sscanf(argv[++i],"%d",&alpha))
		gr_bomb("Error scanning alpha mode '%s'\n",argv[i]);
	    aflag = TRUE;
	} else if (!strcmp(argv[i],"-force"))
	    forceflag = TRUE;
	else {
	    if (lffname == NULL)
		lffname = argv[i];
	    else if (outname == NULL)
		outname = argv[i];
	    else {
		fprintf(stderr,"Unknown switch %s\n",argv[i]);
		userhelp(argv[0],Usage,Example);
	    }
	}
    }

    /* Verify such of the parameters as we can before reading the input file */
    if (rleflag && dumpflag)
	gr_bomb("Output cannot be both encoded and dumped\n");
    if (tileflag && (vtiles < 1 || htiles < 1))
	gr_bomb("Number of tiles must both be positive (vert = %d, horiz = %d)\n",
	    vtiles,htiles);
    if (titleflag && strlen(title) > 245) {
	gr_error("Warning: title is too long, and will be chopped to 245 chars\n");
	title[245] = '\0';
    }

    /*
	Verify the .lff file does not exist
    */
    if (!forceflag) {
	struct stat buf;
	if (stat(outname,&buf) == 0)
	    gr_bomb("Output file %s already exists\n",outname);
    }

    printf("\nReading input file %s\n",lffname);

    status = open_lff(lffname,&Imagehdr);
    if (status != LFF_OK)
	gr_bomb("Can't open '%s', open_lff returns: %s\n",lffname,pixerrmess(status));

    /* Allocate memory for the image in core */
    i = Imagehdr.hdr_height * Imagehdr.hdr_width * sizeof(RGBPIXEL);
    image = (RGBPIXEL *)malloc(i);
    if (image == (RGBPIXEL *)NULL)
	gr_bomb("Failed to allocate %d bytes for incore buffer\n",i);

    /* Scan each tile in order and save in the incore buffer */
    {
	int j, vtiles,htiles;

	vtiles = Imagehdr.hdr_height / Imagehdr.hdr_tileheight;
	htiles = Imagehdr.hdr_width / Imagehdr.hdr_tilewidth;
	for (i = 0; i < vtiles; i++) {
	    int yoff = i * Imagehdr.hdr_tileheight;

	    for (j = 0; j < htiles; j++) {
		int y, xoff = j * Imagehdr.hdr_tilewidth;

		if (lff_seek_tile(i,j) != LFF_OK)
		    gr_bomb("Error reading tile %d, row %d, column %d\n",
			i*htiles+j,i,j);
		for (y = yoff; y < yoff+Imagehdr.hdr_tileheight; y++) {
		    int newy;
		    RGBPIXEL *ptr = &image[Imagehdr.hdr_width * y + xoff];

		    status = lff_read_line(ptr,&newy);
		    if (status != LFF_OK)
			gr_bomb("lff_read_line(y=%d) returns: %s\n",
			    newy,pixerrmess(status));
		}
	    }
	}
    }

    /* Now write the image out to the new file */

    pixinit(image,Imagehdr.hdr_height,Imagehdr.hdr_width);
    status = write_lff(
	outname,
	Imagehdr.hdr_height,
	Imagehdr.hdr_width,
	tileflag ? vtiles :
		   Imagehdr.hdr_height / Imagehdr.hdr_tileheight,
	tileflag ? htiles :
		   Imagehdr.hdr_width  / Imagehdr.hdr_tilewidth,
	Imagehdr.hdr_format,
	rleflag ? STORAGE_RLE :
		 (dumpflag ? STORAGE_DUMP :
			     Imagehdr.hdr_storage),
	xflag ? xoff :
		Imagehdr.hdr_xoffset,
	yflag ? yoff :
		Imagehdr.hdr_yoffset,
	verflag ? version :
		  Imagehdr.hdr_version,
	aflag ? alpha :
		Imagehdr.hdr_alphamode,
	getpixfun);
    if (status != LFF_OK)
	gr_bomb("Error writing output file %s: %s\n",outname,pixerrmess(status));
}

static RGBPIXEL *Image;
static int Height, Width;

pixinit(image,height,width)
RGBPIXEL *image;
int height, width;
{
    Image = image;
    Height = height;
    Width = width;
}

/*
    Function called by write_lff, which returns the next line from the
	input lucasfilm file.
*/
int getpixfun(row,col,length,pixels)
int row, col, length;
RGBPIXEL pixels[];
{
    RGBPIXEL *pixsrc;

    pixsrc = &Image[row * Width + col];

    bcopy( (char *)pixsrc,(char *)pixels,length * sizeof(RGBPIXEL) );
    return length;
}

