#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#ifdef sys5
#include <fcntl.h>
#endif
#include "pixlib.h"

LFF_HDR Imagehdr;

char Usage[] =
"%s [--] image-file [-patch] [-format f] [-rle] [-dump] [-size v h] [-tsize vt ht]\n\
   [-title 'name'] [-xoff x] [-yoff y] [-vers v] [-alpha a]\n";

char *Example[] = {
    "\tfixlffhdr changes the header block of a lucasfilm image file\n",
    "\t    as specified:",
    "\t\t-patch : force the magic number to be correct\n",
    "\t\t-size  : place specified height and width in header\n",
    "\t\t-dump  : force storage mode to dumped\n",
    "\t\t-rle   : force storage mode to run-length encoded\n",
    "\t\t-format: place specified format code in header\n",
    "\t\t-tsize: place specified tile height and width in header\n",
    "\t\t-title: place specified title in header\n",
    "\t\t-xoff : place specified x offset in header\n",
    "\t\t-yoff : place specified y offset in header\n",
    "\t\t-vers : place specified version number in header\n",
    "\t\t-alpha: place specified alpha mode flag (0 or 1) in header\n",
    "\n\tExample: fixlffhdr -title 'Untitled' -xoff 10 -yoff -10 image.lff\n",
    NullPtr(char)
};

main (argc, argv)
int	argc;
char   *argv[];
{
    int  i, bytes, fd;
    char *lffname = NULL;

    int patchflag;
    int titleflag, xflag, yflag, formatflag;
    int verflag, aflag, sizeflag, tsizeflag, dumpflag, rleflag;
    int xoff, yoff, height, width, theight, twidth, version, alpha, format;
    char *title;
    patchflag = dumpflag = rleflag = formatflag =
    titleflag = sizeflag = tsizeflag =
	xflag = yflag = verflag = aflag = FALSE;

    for (i = 1; i < argc; i++) {
	if (!strcmp(argv[i],"--"))
	    userhelp(argv[0],Usage,Example);
	else if (!strcmp(argv[i], "-patch"))
	    patchflag = TRUE;
	else if (!strcmp(argv[i], "-dump"))
	    dumpflag  = TRUE;
	else if (!strcmp(argv[i], "-rle"))
	    rleflag   = TRUE;
	else if (!strcmp(argv[i],"-size")) {
	    if (!sscanf(argv[++i],"%d",&height) ||
		!sscanf(argv[++i],"%d",&width))
		gr_bomb("Error scanning height '%s' or width '%s'\n",
		    argv[i-1],argv[i]);
	    sizeflag = TRUE;
	} else if (!strcmp(argv[i],"-tsize")) {
	    if (!sscanf(argv[++i],"%d",&theight) ||
		!sscanf(argv[++i],"%d",&twidth))
		gr_bomb("Error scanning tile height '%s' or width '%s'\n",
		    argv[i-1],argv[i]);
	    tsizeflag = TRUE;
	} else if (!strcmp(argv[i],"-title")) {
	    title = argv[++i];
	    titleflag = TRUE;
	} else if (!strcmp(argv[i],"-xoff")) {
	    if (!sscanf(argv[++i],"%d",&xoff))
		gr_bomb("Error scanning x offset '%s'\n",argv[i]);
	    xflag = TRUE;
	} else if (!strcmp(argv[i],"-yoff")) {
	    if (!sscanf(argv[++i],"%d",&yoff))
		gr_bomb("Error scanning y offset '%s'\n",argv[i]);
	    yflag = TRUE;
	} else if (!strcmp(argv[i], "-format")) {
	    if (!sscanf(argv[++i], "%d", &format))
		gr_bomb("Error scanning format '%s'\n", argv[i]);
	    formatflag = TRUE;
	} else if (!strcmp(argv[i],"-vers")) {
	    if (!sscanf(argv[++i],"%d",&version))
		gr_bomb("Error scanning version '%s'\n",argv[i]);
	    verflag = TRUE;
	} else if (!strcmp(argv[i],"-alpha")) {
	    if (!sscanf(argv[++i],"%d",&alpha))
		gr_bomb("Error scanning alpha mode '%s'\n",argv[i]);
	    aflag = TRUE;
	} else {
	    if (lffname == NULL)
		lffname = argv[i];
	    else {
		fprintf(stderr,"Unknown switch %s\n",argv[i]);
		userhelp(argv[0],Usage,Example);
	    }
	}
    }

    /* Verify such of the parameters as we can before reading the input file */
    if (titleflag && strlen(title) > 245) {
	gr_error("Warning: title is too long, and will be chopped to 245 chars\n");
	title[245] = '\0';
    }
    if (lffname == NULL)
	gr_bomb("No file specified\n");

    fd = open(lffname,O_RDWR);
    if (fd == -1)
	gr_bomb("Can't open %s for read/write\n",lffname);

    /* Read the header */
    lseek(fd,0L,0);
    if (read(fd,(char *)&Imagehdr,sizeof(Imagehdr)) != sizeof(Imagehdr))
	gr_bomb("Can't read header block from %s\n",lffname);
    hdr_to_host(&Imagehdr);

    /* Patch the header */
    if (patchflag)
	Imagehdr.hdr_magic = MAGIC_LFF;
    /* Oink */

    if (dumpflag)
	Imagehdr.hdr_storage = STORAGE_DUMP;
    else if (rleflag)
	Imagehdr.hdr_storage = STORAGE_RLE;

    if (formatflag)
	Imagehdr.hdr_format = format;

    if (sizeflag) {
	Imagehdr.hdr_height = height;
	Imagehdr.hdr_width = width;
    }
    if (tsizeflag) {
	Imagehdr.hdr_tileheight = theight;
	Imagehdr.hdr_tilewidth = twidth;
    }
    if (titleflag)
	strcpy(Imagehdr.hdr_label,title);
    if (xflag)
	Imagehdr.hdr_xoffset = xoff;
    if (yflag)
	Imagehdr.hdr_yoffset = yoff;
    if (verflag)
	Imagehdr.hdr_version = version;
    if (alpha)
	Imagehdr.hdr_alphamode = alpha;

    /* Write the header */
    hdr_to_vax(&Imagehdr);
    lseek(fd,0L,0);
    bytes = write(fd,(char *)&Imagehdr,sizeof(Imagehdr));
    if (bytes != sizeof(Imagehdr))
	gr_bomb("Error writing header block (wrote %d, tried %d): %s may be corrupted!\n",
	    bytes,sizeof(Imagehdr),lffname);
    close(fd);
}
