/*
    Routines to read a .lff format file into memory, one scan line at a time

    int open_lff(char *name, LFF_HDR *hdr)

	The specified .lff file 'name' is opened and the header  block
	is read into the passed structure.  If successful, 0  (LFF_OK)
	is returned; otherwise, one of the following error codes:

	    LFF_BADMAGIC    Bad magic number
	    LFF_BADSTORAGE  Bad file storage class
	    LFF_BADFORMAT   Bad pixel format
	    LFF_OPENERR     Can't open file (examine errno)
	    LFF_EOF	    Premature end of file
	    LFF_REOPEN	    A .lff file is currently open

    int open_lff_fd(FD fd, LFF_HDR *hdr, off_t offset)

	Like  open_lff,  but  fd  must	be  an	already  opened   file
	descriptor from which to read the image. Offset is a  'virtual
	image base' within the file; the image	is  assumed  to  begin
	there.

    int lff_seek_tile(int row, col)

	The file is positioned so that lff_read_line will read from
	the specified tile. If successful, LFF_OK is returned, otherwise,
	one of the following error codes:

	    LFF_BADTILE     A bad tile was specified (row or col < 0 or too big)

    int lff_read_line(RGBPIXEL line[], int *y)

	The next  sequential  scanline	from  the  current  tile  (see
	lff_seek_tile above) is read and placed in the	array  line[],
	which must be long enough.  Y is set to  the  y-coordinate  of
	the  scan  line  which	was  read  in  within  the  tile.   If
	successful,  LFF_OK  is  returned;  otherwise,	one   of   the
	following error code(s):

	    LFF_EOF	    End of file (or end of tile)

    int lff_read_packed(RPIXEL line[], int *y)

	Just like  lff_read_line,  except  that  the  pixels  are  NOT
	unpacked from the file into 32-bit RGBA format;  instead,  the
	buffer as read off disk is placed in the  byte	array  RPIXEL.
	Normally this will just consist of RGB RGB RGB	...   suitable
	for  feeding  on  to  the  frame  buffer.  For	binary	images
	(FORMAT_BINARY), bits are packed with the LSB of a byte  being
	the leftmost.
*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include "pixlib.h"
#include "lffbufio.h"

/* Globally accessible symbols */
int  open_lff(), open_lff_fd();
int  lff_seek_tile();
int  lff_read_line();
int  lff_read_packed();
void close_lff();

/* Private variables */

/* Pointer to currently open file */
static FD Lff_desc = BAD_FD;

static LFF_HDR Lffhdr;			/* Header block of current file */
static TILEPTR *Tile_table = NullPtr(TILEPTR);	/* Tile pointer table */
static int     Vtile, Htile;		/* Position of tile currently being read in */
static int     Num_vert_tiles,		/* Number of tiles vertically */
	       Num_horiz_tiles;		/*   and horizontally */
static int     Format;			/* Image mask of file */
static int     Pixsize;			/* Pixel size, in bytes */
static int     Y_current;		/* Current scanline coordinate */
static off_t   Fpos;			/* Current position pointer in file */

/* Private functions */
static int   setup_lff_read();
static int   lff_read();
static void  unpack_pixels();
static int   lff_rle_read();

int open_lff(name, hdr)
char *name;
LFF_HDR *hdr;
{
    if (Lff_desc != BAD_FD)
	return LFF_REOPEN;

    Lff_desc = _open_lff(name, LFF_READ, (off_t)0);
    if (Lff_desc == BAD_FD)
	return LFF_OPENERR;

    return setup_lff_read(hdr);
}

int open_lff_fd(fd, hdr, offset)
FD fd;
LFF_HDR *hdr;
off_t offset;
{
    if (Lff_desc != BAD_FD)
	return LFF_REOPEN;

    Lff_desc = _open_lff_fd(fd, LFF_READ, offset);
    if (Lff_desc == BAD_FD)
	return LFF_OPENERR;

    return setup_lff_read(hdr);
}

static int setup_lff_read(hdr)
LFF_HDR *hdr;
{
    int needed,
	status = LFF_OK;

    if (_read_lff(Lff_desc, (char *)hdr, sizeof(LFF_HDR)) == -1) {
	status = LFF_EOF;
	goto done;
    }

    hdr_to_host(hdr);	       /* Fix up byte order of header */

    /* Verify magic number */
    if (hdr->hdr_magic != MAGIC_LFF) {
	status = LFF_BADMAGIC;
	goto done;
    }

    /* Verify storage code */
    if (hdr->hdr_storage != STORAGE_DUMP &&
	hdr->hdr_storage != STORAGE_RLE) {
	    status = LFF_BADSTORAGE;
	    goto done;
    }

    Format = hdr->hdr_format;
    Pixsize = pixel_length(Format);
#ifdef DEBUG
    printf("setup_lff_read: Format = %d, Pixsize = %d\n",Format,Pixsize);
#endif

    Num_vert_tiles = hdr->hdr_height / hdr->hdr_tileheight;
    Num_horiz_tiles = hdr->hdr_width / hdr->hdr_tilewidth;

    /* Allocate space for and read the tile pointer table */
    needed = Num_vert_tiles * Num_horiz_tiles * sizeof(TILEPTR);
    if (Tile_table != NULL)
	free(Tile_table);
    if ((Tile_table = (TILEPTR *)malloc(needed)) == NULL) {
	status = LFF_ENOMEM;
	goto done;
    }

    if (_read_lff(Lff_desc, (char *)Tile_table, needed) == -1) {
	status = LFF_EOF;
	goto done;
    }

    tile_table_to_host(Tile_table,Num_vert_tiles * Num_horiz_tiles);

    if (Format != FORMAT_BINARY && (Pixsize < 1 || Pixsize > 4)) {
	status = LFF_BADFORMAT;
	goto done;
    }

    /* Set up so we will read from first tile */
    (void)lff_seek_tile(0,0);

    /* Copy header block to static block for reading file */
    Lffhdr = *hdr;

done:
    if (status != LFF_OK)
	close_lff();
    return status;
}

int lff_seek_tile(vtile,htile)
int vtile, htile;
{
    int tilenum;
    long ptr;

    if (vtile < 0 || vtile >= Num_vert_tiles ||
	htile < 0 || htile >= Num_horiz_tiles)
	    return LFF_BADTILE;

    tilenum = vtile * Num_horiz_tiles + htile;
    Vtile = vtile;
    Htile = htile;

    /* Look up in tile pointer table and seek to tile */
    ptr = Tile_table[tilenum].tileptr;
#ifdef DEBUG
    printf("lff_seek_tile(%d,%d): Fpos is %d, tile ptr is %d ",
	vtile,htile,Fpos,ptr);
#endif
    Fpos = _seek_lff(Lff_desc, ptr, L_SET);
#ifdef DEBUG
    printf(" -> fpos returns %d\n",Fpos);
#endif

    if (Fpos != ptr)	    /* This should never happen */
	return LFF_BADTILE;

    Y_current = 0;	    /* Position to first line of tile */
    return LFF_OK;
}

/*
    lff_read_line(line,y)
	Read the next scanline into  the  passed  buffer  line[].  The
	pixels are unpacked to 32-bit RGBA format and the row read  is
	returned in y.
*/
int lff_read_line(line,y)
RGBPIXEL line[];
int *y;
{
    LINE_PTR ptr;
    int format;

    if (Lff_desc == BAD_FD)
	return LFF_OPENERR;

    /* Cannot read beyond number of lines in the file or tile */
    if (Y_current > Lffhdr.hdr_tileheight)
	return LFF_EOF;

    ptr.rgbpixel = line;
    *y = Y_current++;
    switch (Lffhdr.hdr_storage) {
	case STORAGE_DUMP:  return lff_read(ptr,UNPACKED);
	case STORAGE_RLE:   return lff_rle_read(ptr,UNPACKED,&format);
    }
}

/*
    lff_read_packed(line,y,format)
	Read the next scanline into the  passed  buffer  line[].   The
	pixels are packed and may be run-length encoded.  If  encoded,
	format is set to STORAGE_RLE else STORAGE_DUMP. The  row  read
	is returned in y.
*/
int lff_read_packed(line,y,format)
RPIXEL line[];
int *y, *format;
{
    LINE_PTR ptr;
    int status;

    if (Lff_desc == BAD_FD)
	return LFF_OPENERR;

    /* Cannot read beyond number of lines in the file or tile */
    if (Y_current > Lffhdr.hdr_tileheight)
	return LFF_EOF;

    *y = Y_current++;
    ptr.rpixel = line;
    switch (Lffhdr.hdr_storage) {
	case STORAGE_DUMP:
	    *format = STORAGE_DUMP;
	    status = lff_read(ptr,PACKED);
	    break;
	case STORAGE_RLE:
	    status = lff_rle_read(ptr,PACKED,format);
	    break;
    }
    return status;
}

/*
    Read the next scanline from the .lff file (in DUMPED format)  into
	the passed array. Return  an  error  code  to  lff_read_line()
	above.
    If mode is PACKED, don't unpack the bytes into 32 bit RGBA	pixels
	and assume line is a RPIXEL (byte) array; otherwise unpack the
	pixels.
*/
static int lff_read(line,mode)
LINE_PTR line;
int mode;
{
    int size;

    if (Format != FORMAT_BINARY)
	size = Pixsize * Lffhdr.hdr_tilewidth;
    else
	size = Lffhdr.hdr_tilepad;

    if (mode == PACKED) {
	if (_read_lff(Lff_desc, (char *)line.rpixel, size) == -1)
	    return LFF_EOF;
    } else {
	RPIXEL buf[MAXLINE * 4];

	if (_read_lff(Lff_desc, (char *)buf, size) == -1)
	    return LFF_EOF;

	/* Now decode into the passed array */
	unpack_pixels(buf,line.rgbpixel);
    }
    /* We should not need to update the pointer with _seek_lff since the
	pixels are contiguous in DUMPED format.
    */
    Fpos += size;

    return LFF_OK;
}

/*
    Unpack the pixels in buf (packed as  determined  by  Format)  into
	line. For the moment, I will assume  the  only	valid  formats
	are FORMAT_R, FORMAT_RGB, FORMAT_RGBA, and FORMAT_BINARY.
*/
static void unpack_pixels(buf,line)
register RPIXEL *buf;
register RGBPIXEL *line;
{
    register int i;

    if (Lffhdr.hdr_format == FORMAT_BINARY) {
	register int mask = 0x1;

	for (i = 0; i < Lffhdr.hdr_tilewidth; i++) {

	    /* 'On' bits turn into white, 'Off' into black */
	    if (*buf & mask)
		*line++ = RGB8(0xFF,0xFF,0xFF);
	    else
		*line++ = RGB8(0,0,0);

	    if ((mask <<= 1) > 0x80) {
		buf++;
		mask = 0x1;
	    }
	}
    } else {
	switch (Pixsize) {
	    case 4:
		for (i = 0; i < Lffhdr.hdr_tilewidth; i++) {
		    *line++ = RGBA8(buf[0],buf[1],buf[2],buf[3]);
		    buf += sizeof(RGBPIXEL);
		}
		break;
	    case 3:
		for (i = 0; i < Lffhdr.hdr_tilewidth; i++) {
		    *line++ = RGB8(buf[0],buf[1],buf[2]);
		    buf += 3;
		}
		break;
	    case 2:
		for (i = 0; i < Lffhdr.hdr_tilewidth; i++) {
		    *line++ = RGB8(buf[0],buf[1],0);
		    buf += 2;
		}
		break;
	    case 1:
		for (i = 0; i < Lffhdr.hdr_tilewidth; i++)
		    *line++ = (RGBPIXEL)(*buf++);
		break;
	}
    }
}

/*
    Read the next scanline from the .lff file  (in  RLE  format)  into
	the passed array. Return  an  error  code  to  lff_read_line()
	above.

    If mode is PACKED, don't unpack the bytes into 32 bit RGBA	pixels
	and assume line is a RPIXEL  (byte)  array  (unlike  lff_read,
	there is a count-value retained, so the data is  not  strictly
	pixels), otherwise unpack and run-length  decode  the  pixels.
	Since the Lucas format allows mixing  run-lengths  and	dumped
	data in the same image, the passed variable format is  set  to
	STORAGE_DUMP or STORAGE_RLE as appropriate (a single line  may
	be dumped or encoded but not both).

    Binary images are not handled correctly since it is  assumed  they
	are only stored as DUMPed images.
*/
static int lff_rle_read(line,mode,format)
LINE_PTR line;
int mode, *format;
{
    PACKET_HDR packet;
    LINE_PTR temp;
    int pix_needed = Lffhdr.hdr_tilewidth;

#ifdef DEBUG
    fprintf(stderr, "lff_rle_read: pix_needed = %d\n", pix_needed );
#endif

    /* Loop reading packets until the buffer is filled up */
    temp = line;
    while (pix_needed > 0) {
	RPIXEL buf[MAXLINE * 5];
	register int bytes_needed, count, i;

#ifdef DEBUG
	fprintf(stderr, "    scanloop: pix_needed = %d\n", pix_needed );
#endif
	if (_read_lff(Lff_desc, (char *)&packet, PACKET_SIZE) == -1)
	    return LFF_EOF;
	packet_to_host(&packet);
	switch (packet.flag) {
	    case FLAG_PAD:		/* Seek to next block */
		count = _tell_lff(Lff_desc);
#ifdef DEBUG
		fprintf(stderr, "\tpadding packet @%d", count );
#endif
		if (count % LFF_BLOCKSIZE) {
		    count = (count + LFF_BLOCKSIZE) / LFF_BLOCKSIZE;
		    count *= LFF_BLOCKSIZE;
		    _seek_lff(Lff_desc, count, L_SET);
		}
#ifdef DEBUG
		fprintf(stderr, ", seeking to %d\n", _tell_lff(Lff_desc));
#endif
		break;
	    case FLAG_DUMP:		/* Read a dumped scanline */
		count = packet.count + 1;		/* Number of pixels */
		pix_needed -= count;			/* Adjust count */
		bytes_needed = count * Pixsize;		/* # bytes to read */
		*format = STORAGE_DUMP;
#ifdef DEBUG
		fprintf(stderr, "\tdumped packet, count = %d\n", count );
#endif
		if (mode == PACKED) {
		    if (_read_lff(Lff_desc, (char *)temp.rpixel, bytes_needed)
			== -1)
			return LFF_EOF;
		    temp.rpixel += bytes_needed;
		} else {
		    if (_read_lff(Lff_desc, (char *)buf, bytes_needed) == -1)
			return LFF_EOF;
		    /* Now unpack the buffer */
		    unpack_pixels(buf,temp.rgbpixel);
		    temp.rgbpixel += count;
		}
		break;
	    case FLAG_RLE:
		count = packet.count + 1;		/* Number of runs */
		bytes_needed = count * (Pixsize + 1);	/* # bytes to read */
#ifdef DEBUG
		fprintf(stderr, "\tencoded packet, count = %d, bytes_needed = %d\n",
		    count, bytes_needed
		);
#endif
		if (mode == PACKED) {
		    RPIXEL *ptr;

		    *format = STORAGE_RLE;
		    if (_read_lff(Lff_desc, (char *)temp.rpixel, bytes_needed)
			== -1) {
#ifdef DEBUG
			fprintf(stderr, "\tpremature EOF\n" );
#endif
			return LFF_EOF;
		    }
		    ptr = temp.rpixel;
		    temp.rpixel += bytes_needed;

		    /* Add up the run lengths to keep pix_needed correct */
		    for (i = 0; i < count; i++, ptr += (Pixsize + 1)) {
#ifdef DEBUG
			fprintf(stderr, "\t\tlambda = %3d, value = 0x%08x\n",
			     1 + *ptr, getpix(ptr+1) );
#endif
			pix_needed -= 1 + *ptr;
		    }
		} else {
		    RPIXEL *bufptr = buf;
		    RGBPIXEL pixel;

		    *format = STORAGE_DUMP;
		    if (_read_lff(Lff_desc, (char *)buf, bytes_needed) == -1) {
#ifdef DEBUG
			fprintf(stderr, "\tpremature EOF\n" );
#endif
			return LFF_EOF;
		    }
		    /* Decode the run-lengths into pixels. Repeat  for
			each run in the packet.
		    */
		    for (i = 0; i < count; i++, bufptr += (1 + Pixsize)) {
			register int
			    lambda = 1 + *bufptr;   /* Lambda count of run */
			pix_needed -= lambda;
#ifdef DEBUG

			fprintf(stderr, "\t\tlambda = %3d\n", lambda );
#endif
			pixel = 0;
			switch (Pixsize) {
			    case 4:
				((RPIXEL *)&pixel)[OFF_R] = bufptr[1];
				((RPIXEL *)&pixel)[OFF_G] = bufptr[2];
				((RPIXEL *)&pixel)[OFF_B] = bufptr[3];
				((RPIXEL *)&pixel)[OFF_A] = bufptr[4];
				break;
			    case 3:
				((RPIXEL *)&pixel)[OFF_R] = bufptr[1];
				((RPIXEL *)&pixel)[OFF_G] = bufptr[2];
				((RPIXEL *)&pixel)[OFF_B] = bufptr[3];
				break;
			    case 2:
				((RPIXEL *)&pixel)[OFF_R] = bufptr[1];
				((RPIXEL *)&pixel)[OFF_G] = bufptr[2];
				break;
			    case 1:
				pixel = bufptr[1];
				break;
			}
			while (lambda--)
			    *temp.rgbpixel++ = pixel;
		    }
		}
		break;
	} /* End of switch (packet.flag) */
    } /* End of while (pix_needed > 0) */

    Fpos = _tell_lff(Lff_desc);
    return LFF_OK;
}

void close_lff()
{
    if (Lff_desc != BAD_FD) {
	_close_lff(Lff_desc);
	Lff_desc = BAD_FD;
    }

    if (Tile_table != NullPtr(TILEPTR)) {
	free(Tile_table);
	Tile_table = NullPtr(TILEPTR);
    }
}

getpix(bufptr)
u_char *bufptr;
{
    int pixel;

    switch (Pixsize) {
	case 4:
	    pixel = bufptr[0];
	    pixel |= (unsigned int)bufptr[1] << 8;
	    pixel |= (unsigned int)bufptr[2] << 16;
	    pixel |= (unsigned int)bufptr[3] << 24;
	    break;
	case 3:
	    pixel = bufptr[0];
	    pixel |= (unsigned int)bufptr[1] << 8;
	    pixel |= (unsigned int)bufptr[2] << 16;
	    break;
	case 2:
	    pixel = bufptr[0];
	    pixel |= (unsigned int)bufptr[1] << 8;
	    break;
	case 1:
	    pixel = bufptr[0];
	    break;
    }
    return pixel;
}
