/****************************************************************************
*                                                                           *
*                                                                           *
*                                                                           *
*                  CONSTRUCTION TRIDIMENSIONNELLE DE SCENES                 *
*                            PAR UNE TECHNIQUE K2D                          *
*                                                                           *
*                                ----------------                           *
*                                   LIBRAIRIE                               *
*                   APPROXIMATION AU SENS DES MOINDRES CARRES               *
*                                  D'UNE DROITE                             *
*                        SOURCE : NUMERICAL RECIPES IN C                    *
*                                ----------------                           *
*                                                                           *
*               - Fichier                 :         c_fit.c     -           *
*               - Date de creation        :         28.01.91    -           *
*               - Date de mise a jour     :         09.03.92    -           *
*                                                                           *
*                                                                           *
*   Ecrit par M.VIALA, JP GUERIN          ( CEA / DLETI / DEIN / SIR )      *
*                                                                           *
*****************************************************************************/


#include <math.h>
#include <stdio.h>

#define SQR(a) ((a)*(a))

#define ITMAX 100
#define EPS 3.0e-7

/** Proc : gammln() **********************************************************/

double gammln(xx)
double xx;
{
	double x,tmp,ser;
	static double cof[6]={76.18009173,-86.50532033,24.01409822,
		-1.231739516,0.120858003e-2,-0.536382e-5};
	int j;

	x=xx-1.0;
	tmp=x+5.5;
	tmp -= (x+0.5)*log(tmp);
	ser=1.0;
	for (j=0;j<=5;j++) {
		x += 1.0;
		ser += cof[j]/x;
	}
	return (-tmp+log(2.50662827465*ser));
}


/** Proc : gcf() *************************************************************/

void gcf(gammcf,a,x,gln)
double a,x,*gammcf,*gln;
{
	int n;
	double gold=0.0,g,fac=1.0,b1=1.0;
	double b0=0.0,anf,ana,an,a1,a0=1.0;
	double gammln();
	void Error();

	*gln=gammln(a);
	a1=x;
	for (n=1;n<=ITMAX;n++) {
		an=(double) n;
		ana=an-a;
		a0=(a1+a0*ana)*fac;
		b0=(b1+b0*ana)*fac;
		anf=an*fac;
		a1=x*a0+anf*a1;
		b1=x*b0+anf*b1;
		if (a1) {
			fac=1.0/a1;
			g=b1*fac;
			if (fabs((g-gold)/g) < EPS) {
				*gammcf=exp(-x+a*log(x)-(*gln))*g;
				return;
			}
			gold=g;
		}
	}
	Error("a too large, ITMAX too small in routine GCF");
}


/** Proc : gser() ************************************************************/

void gser(gamser,a,x,gln)
double a,x,*gamser,*gln;
{
	int n;
	double sum,del,ap;
	double gammln();
	void Error();

	*gln=gammln(a);
	if (x <= 0.0) {
		if (x < 0.0) Error("x less than 0 in routine GSER");
		*gamser=0.0;
		return;
	} else {
		ap=a;
		del=sum=1.0/a;
		for (n=1;n<=ITMAX;n++) {
			ap += 1.0;
			del *= x/ap;
			sum += del;
			if (fabs(del) < fabs(sum)*EPS) {
				*gamser=sum*exp(-x+a*log(x)-(*gln));
				return;
			}
		}
		Error("a too large, ITMAX too small in routine GSER");
		return;
	}
}

/** Proc : gammq() ***********************************************************/

double gammq(a,x)
double a,x;
{
	double gamser,gammcf,gln;
	void   gcf(), gser(), Error();

	if (x < 0.0 || a <= 0.0) Error("Invalid arguments in routine GAMMQ");
	if (x < (a+1.0)) 
	{
		gser(&gamser,a,x,&gln);
		return( 1.0-gamser );
	} else 
	{
		gcf(&gammcf,a,x,&gln);
		return( gammcf );
	}
}


/** Proc : fit() *************************************************************/
/*
	Given a set of points x[0..ndata-1], y[0..ndata-1] with standard 
	deviations sig[0..ndata-1], fit them to a straight line y = a + bx
	by minimizing chi square. Returned are a, b and their respective
	probable uncertainties siga and sigb, the chi-square chi2, and the
	goodness of fit probabiliy q. If mwt = 0 on input, then standard
	deviations are assumed to be unavailable : q is returned as 1.0
	and the normalization of chi2 is to unit standard deviation of
	all points.
*/


void fit(x,y,ndata,sig,mwt,a,b,siga,sigb,chi2,q)
double x[],y[],sig[],*a,*b,*siga,*sigb,*chi2,*q;
int    ndata,mwt;
{
	int i;
	double wt,t,sxoss,sx=0.0,sy=0.0,st2=0.0,ss,sigdat;
	double gammq();

	*b=0.0;
	if (mwt) {
		ss=0.0;
		for (i=0;i<ndata;i++) {
			wt=1.0/SQR(sig[i]);
			ss += wt;
			sx += x[i]*wt;
			sy += y[i]*wt;
		}
	} else {
		for (i=0;i<ndata;i++) {
			sx += x[i];
			sy += y[i];
		}
		ss=ndata;
	}
	sxoss=sx/ss;
	if (mwt) {
		for (i=0;i<ndata;i++) {
			t=(x[i]-sxoss)/sig[i];
			st2 += t*t;
			*b += t*y[i]/sig[i];
		}
	} else {
		for (i=0;i<ndata;i++) {
			t=x[i]-sxoss;
			st2 += t*t;
			*b += t*y[i];
		}
	}
	*b /= st2;
	*a=(sy-sx*(*b))/ss;
	*siga=sqrt((1.0+sx*sx/(ss*st2))/ss);
	*sigb=sqrt(1.0/st2);
	*chi2=0.0;
	if (mwt == 0) {
		for (i=0;i<ndata;i++)
			*chi2 += SQR(y[i]-(*a)-(*b)*x[i]);
		*q=1.0;
		sigdat=sqrt((*chi2)/(ndata-2));
		*siga *= sigdat;
		*sigb *= sigdat;
	} else {
		for (i=0;i<ndata;i++)
			*chi2 += SQR((y[i]-(*a)-(*b)*x[i])/sig[i]);
		*q=gammq(0.5*(ndata-2),0.5*(*chi2));
	}
}

/** Proc : fit1() *************************************************************/
/*
	Given a set of points x[0..ndata-1], y[0..ndata-1] with standard 
	deviations sig[0..ndata-1], fit them to a straight line y = bx
	by minimizing chi square. Returned are b and the probable uncertainties 
	sigb, the chi-square chi2, and the goodness of fit probabiliy q. 
	If mwt = 0 on input, then standard deviations are assumed to be 
	unavailable : q is returned as 1.0 and the normalization of chi2 is 
	to unit standard deviation of all points.
*/


void fit1(x,y,ndata,sig,mwt,b,sigb,chi2,q)
double x[],y[],sig[],*b,*sigb,*chi2,*q;
int    ndata,mwt;
{
	int i;
	double t,st2=0.0,sigdat;
	double gammq();

	*b=0.0;
	if (mwt) {
		for (i=0;i<ndata;i++) {
			t = x[i]/sig[i];
			st2 += t*t;
			*b += t*y[i]/sig[i];
		}
	} else {
		for (i=0;i<ndata;i++) {
			t = x[i];
			st2 += t*t;
			*b += t*y[i];
		}
	}
	*b /= st2;
	*sigb=sqrt(1.0/st2);
	*chi2=0.0;
	if (mwt == 0) {
		for (i=0;i<ndata;i++)
			*chi2 += SQR(y[i]-(*b)*x[i]);
		*q=1.0;
		sigdat=sqrt((*chi2)/(ndata-2));
		*sigb *= sigdat;
	} else {
		for (i=0;i<ndata;i++)
			*chi2 += SQR((y[i]-(*b)*x[i])/sig[i]);
		*q=gammq(0.5*(ndata-2),0.5*(*chi2));
	}
}

/* Proc : XX **************************************************************/
/*
		Fonction d'interruption de programme si presence d'une erreur.
*/
void Error(error_text)
	char error_text[];
{
	(void)fprintf(stderr, "\n%s \n", error_text);
	(void)fprintf(stderr, "\n...Retour au Systeme...\n");

	exit(1);
}


#undef SQR

#undef ITMAX
#undef EPS

