 /*
  * Khoros: $Id: output.c,v 1.1 1991/05/10 15:57:18 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: output.c,v 1.1 1991/05/10 15:57:18 khoros Exp $";
#endif

 /*
  * $Log: output.c,v $
 * Revision 1.1  1991/05/10  15:57:18  khoros
 * Initial revision
 *
  */ 


/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.

 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	    file name: output.c
   >>>>              
   >>>>   description: general output utilities for writing out
   >>>>			an image.
   >>>>               
   >>>>      routines: 
   >>>>			output_IFF_implicit
   >>>>			convert_2D_plot_to_IFF_implicit
   >>>>			convert_3D_plot_to_IFF
   >>>>			output_IFF_explicit
   >>>>			convert_2D_plot_to_IFF_explicit
   >>>>			convert_3D_plot_to_IFF_explicit
   >>>>			raster_screendump
   >>>>               
   >>>> modifications:
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "xprism.h"



/************************************************************
*
*  MODULE NAME: output_IFF_implicit
*
*      PURPOSE: Outputs a 2D or 3D plot in Image File Format w/
*		implicit location data
*
*        INPUT: filename - name of file to be created
*
*       OUTPUT: an image file in the global directory
*
*    CALLED BY: Drv_Output
*
*   WRITTEN BY: Danielle Argiro
*
*
*************************************************************/


int output_IFF_implicit(filename)

char  *filename;
{
      XPlot    *plot;
      struct xvimage  *image,
		      *convert_2d_plot_to_IFF_implicit(),
		      *convert_3d_plot_to_IFF_implicit();
      char *label = "Output Plot As Implicit Khoros File",
	   *prompt = "Enter number of plot to output";

      if ( mono_display )
	  plot = get_a_plot(label, prompt, APPEND_MARKER_TYPE);
      else 
          plot = get_a_plot(label, prompt, APPEND_COLOR);

      if (plot == NULL)
	  return(false);
      else
      {
           /*
	    *  convert the plot to an image
	    */
	   if (PLOT_TYPE_2D(gwin->plot_type))
	      image = convert_2d_plot_to_IFF_implicit(plot);
	   else 
	      image = convert_3d_plot_to_IFF_implicit(plot);

	   if (image == NULL)
	      return(false);

	   if(!writeimage(filename, image))
	   {
	    xvf_error_wait("Could not write image","output_IFF_implicit", NULL);
	      return(false);
	   }
           freeimage(image);
           return(true);
      }
}


/************************************************************
*
*  MODULE NAME: output_colormap
*
*      PURPOSE: Outputs the Y plot data as a colormap in a VIFF file
*
*        INPUT: filename - name of file to be created
*
*       OUTPUT: 
*
*   WRITTEN BY: Tom Sauer
*
*
*************************************************************/


int output_colormap(filename)

char  *filename;
{
      XPlot    *plot;
      struct xvimage  *image, *createimage();
      char *label = "Output Y Plot Data As A Colormap In A Khoros VIFF File",
	   *prompt = "Select Plot To Output";

	unsigned
	long    col_size,		/* size of a column */
       	 	row_size,		/* size of a row */
        	data_storage_type,	/* VFF_TYP_* define of image */
        	num_of_images,		/* # images pointed to by imagedata */
        	num_data_bands,		/* # of bands/pixel */
        	map_row_size,		/* # of columns in map array */
        	map_col_size,		/* # of rows in map array */
       	 	map_storage_type,	/* Storage type of cells in the maps */
       	 	map_scheme,	        /* mapping scheme */
        	location_type,		/* implied or explicit location data */
        	location_dim;		/* explicit locations dimension */
	char    *comment = "map generated by xprism2";
	char 	*cptr;
	int	i;

      if ( mono_display )
	  plot = get_a_plot(label, prompt, APPEND_MARKER_TYPE);
      else 
          plot = get_a_plot(label, prompt, APPEND_COLOR);

      if (plot == NULL)
	  return(false);
      else
      {
           /*
	    *  convert the plot to an image
	    */
	
	   row_size = 0;
	   col_size = 0;
	   data_storage_type = VFF_TYP_1_BYTE;
	   num_of_images = 1;
	   num_data_bands = 1;
	   map_row_size =  1;
	   map_col_size = plot->size;
	   map_storage_type = VFF_MAPTYP_1_BYTE;
	   map_scheme = VFF_MS_ONEPERBAND;
	   location_type = VFF_LOC_IMPLICIT;
	   location_dim = 0;
 	
	   image = createimage(col_size, row_size, data_storage_type,
		    num_of_images, num_data_bands, comment, map_row_size,
                    map_col_size, map_scheme, map_storage_type, location_type,
            	    location_dim);

	   if (image == NULL)
	       return(false);

	   image->subrow_size = 0;
	   image->color_space_model = VFF_CM_NONE;
  
           cptr = (char *) image->maps;
	   if (PLOT_TYPE_2D(gwin->plot_type))
	   {
              for (i=0; i < plot->size; i++)
	         *cptr++ = (char) plot->points[i].y;
	   }
	   else
	   {
               for (i=0; i < plot->size; i++)
	          *cptr++ = (char) plot->points[i].z;
	   }

	   if(!writeimage(filename, image))
	   {
	      xvf_error_wait("Could not write image","output_colormap", NULL);
	      return(false);
	   }
           freeimage(image);
           return(true);
      }
}


/************************************************************
*
*  MODULE NAME: convert_2D_plot_to_IFF_implicit
*
*      PURPOSE: Takes an Khoros image structure, and fills
*               it with values according to:
*               1) defaults, since xprism uses
*                  only a few of the provided fields in
*                  the Xvison Image File Format (see xviff.h)
*               2) imagedata according to the information in 
*                  the 2D plot structure passed in - plot information
*		   is stored implicitly.
*
*        INPUT: 1) plot  -- pointer to a (full) plot structure
*               2) image -- pointer to an (empty) xvimage
*
*       OUTPUT: image -- an xvimage structure, now filled with values
*
*    CALLED BY: Output
*
*   WRITTEN BY: Danielle Argiro and Mark Young
*
*
*************************************************************/

struct xvimage *convert_2d_plot_to_IFF_implicit(plot)

XPlot *plot;
{
	struct xvimage *image, *createimage();

	unsigned
	long    col_size,		/* size of a column */
       	 	row_size,		/* size of a row */
        	data_storage_type,	/* VFF_TYP_* define of image */
        	num_of_images,		/* # images pointed to by imagedata */
        	num_data_bands,		/* # of bands/pixel */
        	map_row_size,		/* # of columns in map array */
        	map_col_size,		/* # of rows in map array */
       	 	map_storage_type,	/* Storage type of cells in the maps */
       	 	map_scheme,	        /* mapping scheme */
        	location_type,		/* implied or explicit location data */
        	location_dim;		/* explicit locations dimension */
	char    *comment;		/* description of image */
	int	i;
        float *fptr;
	
	row_size = plot->row_size;
	col_size = plot->size/plot->row_size;
	data_storage_type = VFF_TYP_FLOAT;
	num_of_images = 1;
	num_data_bands = 1;
	map_row_size = map_col_size = 0;
	map_storage_type = VFF_MAPTYP_NONE;
	map_scheme = VFF_MS_NONE;
	location_type = VFF_LOC_IMPLICIT;
	location_dim = 0;
	comment = NULL;
	
	image = createimage(col_size, row_size, data_storage_type,
		    num_of_images, num_data_bands, comment, map_row_size,
                    map_col_size, map_scheme, map_storage_type, location_type,
            	    location_dim);

	image->subrow_size = 0;
	image->color_space_model = VFF_CM_NONE;

        fptr = (float *) image->imagedata;
        for (i=0; i < plot->size; i++)
	   *fptr++ = (float) plot->points[i].y;

        return(image);
}



/************************************************************
*
*  MODULE NAME: convert_3D_plot_to_IFF
*
*      PURPOSE: Takes an Khoros image structure, and fills
*               it with values according to:
*               1) defaults, since Khoros Plotting uses
*                  only a few of the provided fields in
*                  the Xvison Image File Format (see xviff.h)
*               2) imagedata according to the information in 
*                  the 3D plot structure passed in - x & y points
*		   are stored implicitly.
*
*        INPUT: 1) plot  -- pointer to a (full) plot structure
*               2) image -- pointer to an (empty) xvimage
*
*       OUTPUT: image -- an xvimage structure, now filled with values
*
*    CALLED BY: Output
*
*   WRITTEN BY: Danielle Argiro and Mark Young
*
*
*************************************************************/

struct xvimage *convert_3d_plot_to_IFF_implicit(plot)

XPlot *plot;
{
	struct xvimage *image, *createimage();

	unsigned
	long    col_size,		/* size of a column */
       	 	row_size,		/* size of a row */
        	data_storage_type,	/* VFF_TYP_* define of image */
        	num_of_images,		/* # images pointed to by imagedata */
        	num_data_bands,		/* # of bands/pixel */
        	map_row_size,		/* # of columns in map array */
        	map_col_size,		/* # of rows in map array */
       	 	map_storage_type,	/* Storage type of cells in the maps */
       	 	map_scheme,	        /* mapping scheme */
        	location_type,		/* implied or explicit location data */
        	location_dim;		/* explicit locations dimension */
	char    *comment;		/* description of image */
	int	i, j, l;
        float *fptr;
	

	row_size = plot->row_size;
	col_size = plot->size/plot->row_size;
	data_storage_type = VFF_TYP_FLOAT;
	num_of_images = 1;
	num_data_bands = 1;
	map_row_size = map_col_size = 0;
	map_storage_type = VFF_MAPTYP_NONE;
	location_type = VFF_LOC_IMPLICIT;
	map_scheme = VFF_MS_NONE;
	location_dim =  0;
	comment = NULL;
	
	image = createimage(col_size, row_size, data_storage_type,
		    num_of_images, num_data_bands, comment, map_row_size,
                    map_col_size, map_scheme, map_storage_type, location_type,
            	    location_dim);

	
	image->subrow_size = 0;
	image->color_space_model = VFF_CM_NONE;

        fptr = (float *)(image->imagedata);
        for (i = image->col_size-1; i >=0; i--)
        {
            l = i * image->row_size;
            for (j=0; j < image->row_size; j++)
            {
               *fptr++ = (float) plot->points[l].z; l++;
            }
        }

	return(image);
}



/************************************************************
*
*  MODULE NAME: output_IFF_explicit
*
*      PURPOSE: Outputs a 2D or 3D plot in Image File Format w/
*		explicit location data
*
*        INPUT: filename - name of file to be created
*
*       OUTPUT: an image file in the global directory
*
*    CALLED BY: Drv_Output
*
*
*   WRITTEN BY: Danielle Argiro 
*
*
*************************************************************/


int output_IFF_explicit(filename)

char  *filename;
{

      XPlot    *plot;
      struct xvimage  *image,
		      *convert_2d_plot_to_IFF_explicit(),
		      *convert_3d_plot_to_IFF_explicit();

      char *label = "Output Plot As Explicit Khoros File",
	   *prompt = "Enter number of plot to output";

      if ( mono_display )
	  plot = get_a_plot(label, prompt, APPEND_MARKER_TYPE);
      else 
          plot = get_a_plot(label, prompt, APPEND_COLOR);


      if (plot == NULL)
	 return(false);

      else
      {
           /*
	    *  convert the plot to an image
	    */
	   if (PLOT_TYPE_2D(gwin->plot_type))
	      image = convert_2d_plot_to_IFF_explicit(plot);
	   else 
	      image = convert_3d_plot_to_IFF_explicit(plot);

	   if (image == NULL)
	      return(false);
	      
	   if(!writeimage(filename, image))
	   {
	      xvf_error_wait("Could not write image","laser", NULL);
	      FLUSH();
	      return(false);
	   }
           freeimage(image);
      }
      return(true);
}


 
/************************************************************
*
*  MODULE NAME: convert_2D_plot_to_IFF_explicit
*
*      PURPOSE: Takes an Khoros image structure, and fills
*               it with values according to:
*               1) defaults, since xprism uses
*                  only a few of the provided fields in
*                  the Xvison Image File Format (see xviff.h)
*               2) imagedata according to the information in 
*                  the 2D plot structure passed in - plot information
*		   is stored explicitly.
*
*        INPUT: 1) plot  -- pointer to a (full) plot structure
*               2) image -- pointer to an (empty) xvimage
*
*       OUTPUT: image -- an xvimage structure, now filled with values
*
*    CALLED BY: Output
*
*   WRITTEN BY: Danielle Argiro and Mark Young
*
*
*************************************************************/

struct xvimage *convert_2d_plot_to_IFF_explicit(plot)

XPlot *plot;
{
	struct xvimage *image, *createimage();

	unsigned
	long    col_size,		/* size of a column */
       	 	row_size,		/* size of a row */
        	data_storage_type,	/* VFF_TYP_* define of image */
        	num_of_images,		/* # images pointed to by imagedata */
        	num_data_bands,		/* # of bands/pixel */
        	map_row_size,		/* # of columns in map array */
        	map_col_size,		/* # of rows in map array */
       	 	map_storage_type,	/* Storage type of cells in the maps */
       	 	map_scheme,	        /* mapping scheme */
        	location_type,		/* implied or explicit location data */
        	location_dim;		/* explicit locations dimension */
	int	i;
        float *fptr;
	
	row_size = 1;
	col_size = plot->size;
	data_storage_type = VFF_TYP_FLOAT;
	num_of_images = 1;
	num_data_bands = 1;
	map_row_size = map_col_size = 0;
	map_storage_type = VFF_MAPTYP_NONE;
	map_scheme = VFF_MS_NONE;
	location_type = VFF_LOC_EXPLICIT;
	location_dim = 1;
	
	image = createimage(col_size, row_size, data_storage_type,
		    num_of_images, num_data_bands, NULL, map_row_size,
                    map_col_size, map_scheme, map_storage_type, location_type,
            	    location_dim);

	image->color_space_model = VFF_CM_NONE;
	image->subrow_size = 0;
 
        fptr = (float *) image->imagedata;
        for (i=0; i < plot->size; i++)
        {
	   *fptr++ = (float) plot->points[i].y;
	   image->location[i] = (float) plot->points[i].x;
	}


	return(image);
}



/************************************************************
*
*  MODULE NAME: convert_3D_plot_to_IFF_explicit
*
*      PURPOSE: Takes an Khoros image structure, and fills
*               it with values according to:
*               1) defaults, since Khoros Plotting uses
*                  only a few of the provided fields in
*                  the Xvison Image File Format (see xviff.h)
*               2) imagedata according to the information in 
*                  the 3D plot structure passed in - x & y points
*		   are stored explicitly.
*
*        INPUT: 1) plot  -- pointer to a (full) plot structure
*               2) image -- pointer to an (empty) xvimage
*
*       OUTPUT: image -- an xvimage structure, now filled with values
*
*    CALLED BY: Output
*
*   WRITTEN BY: Danielle Argiro and Mark Young
*
*
*************************************************************/

struct xvimage *convert_3d_plot_to_IFF_explicit(plot)

XPlot *plot;
{
	struct xvimage *image, *createimage();

	int	i;
	unsigned
	long    col_size,		/* size of a column */
       	 	row_size,		/* size of a row */
        	data_storage_type,	/* VFF_TYP_* define of image */
        	num_of_images,		/* # images pointed to by imagedata */
        	num_data_bands,		/* # of bands/pixel */
        	map_row_size,		/* # of columns in map array */
        	map_col_size,		/* # of rows in map array */
       	 	map_storage_type,	/* Storage type of cells in the maps */
       	 	map_scheme,	        /* mapping scheme */
        	location_type,		/* implied or explicit location data */
        	location_dim;		/* explicit locations dimension */
        float *fptr;
	float *location1, *location2;
	

	row_size = plot->row_size;
	col_size = plot->size/plot->row_size;
	data_storage_type = VFF_TYP_FLOAT;
	num_of_images = 1;
	num_data_bands = 1;
	map_row_size = map_col_size = 0;
	map_storage_type = VFF_MAPTYP_NONE;
	map_scheme = VFF_MS_NONE;
	location_type = VFF_LOC_EXPLICIT;
	location_dim =  2;
	
	image = createimage(col_size, row_size, data_storage_type,
		    num_of_images, num_data_bands, NULL, map_row_size,
                    map_col_size, map_scheme, map_storage_type, location_type,
            	    location_dim);

	image->color_space_model = VFF_CM_NONE;
	image->subrow_size = 0;

	location1 = (float *) &image->location[0];
	location2 = (float *) &image->location[col_size * row_size];
        fptr = (float *) image->imagedata;

        for (i = 0; i < plot->size; i++)
        {
           *fptr++ = plot->points[i].z;
	   *location1++ = plot->points[i].x;
	   *location2++ = plot->points[i].y;
        }

	return(image);
}



/************************************************************
*
*  MODULE NAME: raster_screendump
*
*      PURPOSE: Dumps a graphics window to an xvimage file specified
*		by fid.
*               
*        INPUT: fid - the file descriptor to dump to
*				
*
*       OUTPUT: creates an image file of the window specified,  to be 
*		sent to vlaser before being erased.
*		returns TRUE on success, FALSE on failure
*
*    CALLED BY: Main
*
*   WRITTEN BY: Mark Young
*
*************************************************************/


struct xvimage *raster_screendump(bitmap)

int	bitmap;
{
	Window	workspace, root;
	struct  xvimage *image, *create_bitmap_image(), *create_pixmap_image();

	int         x, y;
	unsigned    int width, height, border_width, depth;

	XSynchronize(display, TRUE);

	/*
	 *  Check the size of the X window.  We need to get the width
	 *  height to see if it has changed.
	 */
	if ( !XGetGeometry(display, gwin->workspace, &root, &x, &y,
			   &width, &height,  &border_width, &depth))
	{
	   xvf_error_wait("Graphics Workspace does not exist!",
			  "raster_screendump", NULL);
	   return(NULL);
	}
	workspace = (Window) gwin->workspace;


	if (bitmap)
	{
	   image = create_bitmap_image(display, screen, workspace, 0,0,0,0,0);
	   if (image == NULL)
	   {
	      xvf_error_wait("Could not create image","raster_screendump",NULL);
	      return(NULL);
	   }
	}
	else
	{
	   image = create_pixmap_image(display, screen, workspace, 0,0,0,0,0);
	   if (image == NULL)
	   {
	      xvf_error_wait("Could not create image","raster_screendump",NULL);
	      return(NULL);
	   }
	}
	XSynchronize(display, FALSE);
	return(image);
}
