 /*
  * Khoros: $Id: unloadvector.c,v 1.1 1991/05/10 15:41:35 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: unloadvector.c,v 1.1 1991/05/10 15:41:35 khoros Exp $";
#endif

 /*
  * $Log: unloadvector.c,v $
 * Revision 1.1  1991/05/10  15:41:35  khoros
 * Initial revision
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.

 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	    file name: unloadvector.c                         <<<<
   >>>>                                                       <<<<
   >>>>   description: utilities for unloading                <<<<
   >>>>                 an image to a vector array            <<<<
   >>>>                                                       <<<<
   >>>>      routines: unload_float_vector(),                 <<<<
   >>>>                unload_complex_vector(), 	      <<<<
   >>>>                unload_float_vector(),                 <<<<
   >>>>                unload_complex_vector(),               <<<<
   >>>>                unload_int_vector(),                   <<<<
   >>>>                unload_byte_vector(),                  <<<<
   >>>>                unload_short_vector()                  <<<<
   >>>>                unload_double_vector(),                <<<<
   >>>>                unload_dcomplex_vector(),              <<<<
   >>>>                                                       <<<<
   >>>> modifications:	Mon Feb 25 1991	      		      <<<<
   >>>> 		added double and double complex       <<<<
   >>>>			capabilities - Donna Koechner	      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "vinclude.h"	


/************************************************************
*
*  Routine Name:  unload_vector()
*
*          Date:  Thu Apr  5 11:46:03 MDT 1990
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               char **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		type - data type (VFF_TYPX_XXXXX)
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/
char *unload_vector(input_vector, border, type, num_vects, dimension, nr, nc)
char **input_vector;
int border, type;
int num_vects, dimension, nr, nc;
{
    char  *image_data, *unload_char_vector();
    float *float_image_data, *unload_float_vector(), *unload_complex_vector();
    double *double_image_data, *unload_double_vector(), 
	   *unload_dcomplex_vector();
    int   *int_image_data, *unload_int_vector(); 
    short *unload_short_vector(), *short_image_data;
    char  *program = "load_vector";

    switch(type) 
    {
       case VFF_TYP_BIT:
          fprintf(stderr, "%s: Can not work on BIT data storage type\n",program);
          fprintf(stderr, "Can only convert type Byte, Short, Integer, ");
          fprintf(stderr, "Float\n and Complex\n");
          return(NULL);
          break;

       case VFF_TYP_1_BYTE:
          image_data = unload_char_vector((char **) input_vector, border, 
				  	   num_vects, dimension, nr, nc);
          if (image_data == NULL)
          {
             fprintf(stderr, "%s: unload_char_vector failed\n", program);
             return(NULL);
          }
          break;

       case VFF_TYP_2_BYTE:
          short_image_data = unload_short_vector((short **) input_vector, border,
					          num_vects, dimension, nr, nc);
          if (short_image_data == NULL)
          {
             fprintf(stderr, "%s: unload_short_vector failed\n", program);
             return(NULL);
          }
          image_data = (char *) short_image_data;
          break;

       case VFF_TYP_4_BYTE:
          int_image_data = unload_int_vector((int **) input_vector, border,
					         num_vects, dimension, nr, nc);
          if (int_image_data == NULL)
          {
             fprintf(stderr, "%s: unload_int_vector failed\n", program);
             return(NULL);
          }
          image_data = (char *) int_image_data;
          break;

       case VFF_TYP_FLOAT:
          float_image_data = unload_float_vector((float **)input_vector, border,
 						  num_vects, dimension, nr, nc);
          if (float_image_data == NULL)
          {
             fprintf(stderr, "%s: unload_float_vector failed\n", program);
             return(NULL);
          }
          image_data = (char *) float_image_data;
          break;

       case VFF_TYP_COMPLEX:
          float_image_data = unload_complex_vector((float **)input_vector,
				border, num_vects, dimension, nr, nc);
          if (float_image_data == NULL)
          {
             fprintf(stderr, "%s: unload_complex_vector failed\n", program);
             return(NULL);
          }

          image_data = (char *) float_image_data;
          break;

       case VFF_TYP_DOUBLE:
          double_image_data = unload_double_vector((double **)input_vector, 
			        border, num_vects, dimension, nr, nc);
          if (double_image_data == NULL)
          {
             fprintf(stderr, "%s: unload_double_vector failed\n", program);
             return(NULL);
          }
          image_data = (char *) double_image_data;
          break;

       case VFF_TYP_DCOMPLEX:
          double_image_data = unload_dcomplex_vector((double **)input_vector,
				border, num_vects, dimension, nr, nc);
          if (double_image_data == NULL)
          {
             fprintf(stderr, "%s: unload_dcomplex_vector failed\n", program);
             return(NULL);
          }

          image_data = (char *) double_image_data;
          break;

       default:
          fprintf(stderr, "%s: Invalid image Data Storage type\n", program);
          fprintf(stderr, "Can only convert type Byte, Short, Integer, ");
          fprintf(stderr, "Float\nand Complex\n");
          return(NULL);
          break;
      }

       return(image_data);
}

/************************************************************
*
*  Routine Name:  unload_char_vector()
*
*          Date:  Thu Apr  5 11:46:03 MDT 1990
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               char **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/

char *unload_char_vector(input_vectors, border, num_vects, dimension, nr, nc)

char **input_vectors;
int border, nr, nc;
int num_vects, dimension;
{

    char *imagedata, *cptr;
    int m, i, j, k;
    char *program = "unload_float_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (char *)malloc((unsigned int) nr * nc * dimension * sizeof (char));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (char));

          /* Assign vectors to the image data */

    cptr = (char *) imagedata;

          /* Assign ptr to input vectors minus the border */

    m = 0;
    for (i = border; i < nr - border; i++) 
    {
       for (j = border; j < nc - border; j++) 
       {
	  for (k = 0; k < dimension; k++) 
          {
	      cptr[(i * nc + j) + (k * nc * nr)] = input_vectors[m][k];
	  }
          m++;
       }
    }

       /* return the pointer to the image data */
    return(cptr);
}

/************************************************************
*
*  Routine Name:  unload_short_vector()
*
*          Date:  Thu Apr  5 11:46:03 MDT 1990
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               float **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/

short *unload_short_vector(input_vectors, border, num_vects, dimension, nr, nc)

short **input_vectors;
int border, nr, nc;
int num_vects, dimension;
{

    short *imagedata, *sptr;
    int m, i, j, k;
    char *program = "unload_float_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (short *)malloc((unsigned int) nr * nc * dimension * sizeof (short));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (short));

          /* Assign vectors to the image data */

    sptr = (short *) imagedata;

          /* Assign ptr to input vectors minus the border */

    m = 0;
    for (i = border; i < nr - border; i++) 
    {
       for (j = border; j < nc - border; j++) 
       {
	  for (k = 0; k < dimension; k++) 
          {
	      sptr[(i * nc + j) + (k * nc * nr)] = input_vectors[m][k];
	  }
          m++;
       }
    }

       /* return the pointer to the image data */
    return(sptr);
}


/************************************************************
*
*  Routine Name:  unload_int_vector()
*
*          Date:  Thu Apr  5 11:46:03 MDT 1990
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               int **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/

int *unload_int_vector(input_vectors, border, num_vects, dimension, nr, nc)

int **input_vectors;
int border, nr, nc;
int num_vects, dimension;
{

    int *imagedata, *iptr;
    int m, i, j, k;
    char *program = "unload_int_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (int *)malloc((unsigned int) nr * nc * dimension * sizeof (int));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (int));

          /* Assign vectors to the image data */

    iptr = (int *) imagedata;

          /* Assign ptr to input vectors minus the border */

    m = 0;
    for (i = border; i < nr - border; i++) 
    {
       for (j = border; j < nc - border; j++) 
       {
	  for (k = 0; k < dimension; k++) 
          {
	      iptr[(i * nc + j) + (k * nc * nr)] = input_vectors[m][k];
	  }
          m++;
       }
    }

       /* return the pointer to the image data */
    return(iptr);
}

/************************************************************
*
*  Routine Name:  unload_float_vector()
*
*          Date:  Thu Apr  5 11:46:03 MDT 1990
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               float **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/

float *unload_float_vector(input_vectors, border, num_vects, dimension, nr, nc)

float **input_vectors;
int border, nr, nc;
int num_vects, dimension;
{

    float *imagedata, *fptr;
    int m, i, j, k;
    char *program = "unload_float_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (float *)malloc((unsigned int) nr * nc * dimension * sizeof (float));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (float));

          /* Assign vectors to the image data */

    fptr = (float *) imagedata;

          /* Assign ptr to input vectors minus the border */

    m = 0;
    for (i = border; i < nr - border; i++) 
    {
       for (j = border; j < nc - border; j++) 
       {
	  for (k = 0; k < dimension; k++) 
          {
	      fptr[(i * nc + j) + (k * nc * nr)] = input_vectors[m][k];
	  }
          m++;
       }
    }

       /* return the pointer to the image data */
    return(fptr);
}


/************************************************************
*
*  Routine Name:  unload_complex_vector()
*
*          Date:  Thu Apr  5 11:46:03 MDT 1990
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               float **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/


float *unload_complex_vector(input_vectors, border, num_vects, dimension, nr, nc)

float **input_vectors;
int border, nr, nc;
int num_vects, dimension;

{

    float *imagedata, *fptr;
    int m, i, j, k, p, q;
    char *program = "unload_complex_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (float *)malloc((unsigned int) nr * nc * dimension * 2  * sizeof (float));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (float));

    fptr = (float *) imagedata;

          /* Assign ptr to input vectors minus the border */
          /* Assign vectors to the image data */

    m = 0;
    for (i = border; i < nr - border; i++)  
    {
        q = border*2;
        for (j = border; j < nc - border; j++ ) 
        {
            p = 0;
            for (k = 0; k < dimension; k++) 
            {
     	        fptr[(i * nc*2 + q)+(k * nc * nr*2)] = input_vectors[m][p];
                p++;
		fptr[(i * nc*2 + q+1)+(k * nc *nr*2)] = input_vectors[m][p];
                p++;
	    }
                 q +=2;
	         m++;
	 }
     }

       /* return the pointer to the image data */
    return(fptr);
}

/************************************************************
*
*  Routine Name:  unload_double_vector()
*
*          Date:  Mon Feb 25 15:15:57 MST 1991
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               double **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/

double *unload_double_vector(input_vectors, border, num_vects, dimension, nr, nc)

double **input_vectors;
int border, nr, nc;
int num_vects, dimension;
{

    double *imagedata, *dptr;
    int m, i, j, k;
    char *program = "unload_float_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (double *)malloc((unsigned int) nr * nc * dimension * sizeof (double));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (double));

          /* Assign vectors to the image data */

    dptr = (double *) imagedata;

          /* Assign ptr to input vectors minus the border */

    m = 0;
    for (i = border; i < nr - border; i++) 
    {
       for (j = border; j < nc - border; j++) 
       {
	  for (k = 0; k < dimension; k++) 
          {
	      dptr[(i * nc + j) + (k * nc * nr)] = input_vectors[m][k];
	  }
          m++;
       }
    }

       /* return the pointer to the image data */
    return(dptr);
}


/************************************************************
*
*  Routine Name:  unload_dcomplex_vector()
*
*          Date:  Mon Feb 25 15:18:41 MST 1991
*        
*       Purpose:  
*              Takes a 2-D array of vectorized data and rearanges the data so
*	       that the data is stored in an image format.
*
*	       Each vector represents a pixel and the dimension of the
*              vector represents the number of data bands for the image.
*              So the mth element of the vector represents t pixel in
*              the mth data band.
*              
*              The image is built by filling row by row from the
*	       linear vector data.
*
*              The border will be equal on all sides
*
*         Input: 
*               double **input_vectors - a pointer to the 2-D vectorized data
*		border - the border width to skip.
*		num_vects - total number of vectors
*		dimension - vector dimension
*		nr - number of rows in the image (col_size)
*		nc - number of columns in the image (row_size)
*
*        Output: returns a pointer to a 1-d array containing the image data
*
*     Called by: 
*
*    Written By:  Tom Sauer
*
*************************************************************/


double *unload_dcomplex_vector(input_vectors, border, num_vects, dimension, nr, nc)

double **input_vectors;
int border, nr, nc;
int num_vects, dimension;

{

    double *imagedata, *dptr;
    int m, i, j, k, p, q;
    char *program = "unload_dcomplex_vector";

       /* check to make sure there is enough vector data to fill
        * the image less the border
        */

    if ((nr-(border*2)) * (nc-(border*2)) != num_vects)
    {
       (void)fprintf(stderr,
                      "%s: the number of vectors is not equal to\n",program);
       (void)fprintf(stderr,"the image rows + border * image cols + border\n");
       return(NULL);
    }

            /* Allocate space for image data */

    imagedata = (double *)malloc((unsigned int) nr * nc * dimension * 2  * sizeof (double));
    if (imagedata == NULL) 
    {
        (void) fprintf (stderr, "%s: insufficient memory available\n",program);
	return (NULL);
    }

          /* zero the image data */
    bzero(imagedata, nr * nc * dimension * sizeof (double));

    dptr = (double *) imagedata;

          /* Assign ptr to input vectors minus the border */
          /* Assign vectors to the image data */

    m = 0;
    for (i = border; i < nr - border; i++)  
    {
        q = border*2;
        for (j = border; j < nc - border; j++ ) 
        {
            p = 0;
            for (k = 0; k < dimension; k++) 
            {
     	        dptr[(i * nc*2 + q)+(k * nc * nr*2)] = input_vectors[m][p];
                p++;
		dptr[(i * nc*2 + q+1)+(k * nc *nr*2)] = input_vectors[m][p];
                p++;
	    }
                 q +=2;
	         m++;
	 }
     }

       /* return the pointer to the image data */
    return(dptr);
}
