/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.

 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "vinclude.h"	


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	    file name: read_raw_mag.c                         <<<<
   >>>>                                                       <<<<
   >>>>   description: Utility that reads raw data files      <<<<
   >>>>			acquired using the magnets at CNID    <<<<
   >>>>                                                       <<<<
   >>>>      routines: read_raw_mag()<
   >>>>                                                       <<<<
   >>>> modifications:					      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


/**************************************************************
*
* MODULE NAME: 	read_raw_mag
*
*     PURPOSE: 	Read_raw_mag dynamically allocates space for the
*		data array associated with fid and returns the data
*		array to the calling routine.  The number of points
*		read is passed back through the pointer num_pts.
*		
*		Data from the magnets are stored in IEEE format, so
*		read_raw_mag checks the current machine type, and
*		if the machine does not use ieee format, the data is 
*		converted to ieee format before returning.
*
*		If the conv_to_float flag is set to 1, the data is
*		converted to float before returning.  This is used
*		primarily because VFF_TYP_COMPLEX requires float data.
*
*
*       INPUT:	fid		File descriptor pointing to the stream
*				in which the raw data is located
*
*		header_offset	Offset, in bytes, of data from the beginning
*				of raw file.
*
*		type		Data type of raw input file.
*
*		conv_to_float	Flag set to 1 if the VIFF file is to be 
*				converted to float before returning.
*
*		num_pts		Pointer to number of points found in raw file.
*
*      OUTPUT:  data		Returns the data points found in the raw
*				file.
*
* CALLED FROM:	main
*
* ROUTINES CALLED:	machtype
*
**************************************************************/

#define ALLOCSIZE 50000

/*ADD_SRC*/
char *read_raw_mag(fid, header_offset, num_pts, type, conv_to_float)
int	fid, header_offset, type, conv_to_float;
int 	*num_pts;

{
  int			num_bytes, num, size, i, unit_size;
  long 			machtype(), current_machine;
  unsigned char 	*bptr, *cptr, c;
  char			*data;
  short			*sptr;
  int			*iptr;
  float			*fptr, *temp;

 /* 
  * Make sure header offset value is valid.
  */
  if ( header_offset < 0 )
  {
    (void)fprintf(stderr,"Header offset must be >= 0\n");
    return(0);
  }

 /************************
  * Read in the raw image.
  ************************/

 /*
  * Allocate memory space for data.
  */
  size =  ALLOCSIZE;
  data = (char *) malloc ((unsigned int) size);

 /*
  * Read data and determine number of points.
  */
  num_bytes = 0;

  do
  {
    if (num_bytes == size)
    {
      size += ALLOCSIZE;
      if (!(data = (char *) realloc(data, (unsigned) size)))
      {
        (void) fprintf(stderr,"Not enough memory!  Tried to malloc %d bytes.", 
	size);
        return (NULL);
      }
    }

    if ((num = block_read(fid, (char *) (data + num_bytes),
	(size - num_bytes))) > 0)
    {
      num_bytes += num;
    }
  } while (num > 0);

  if (header_offset > 0)
  {
    num_bytes -= header_offset;
    data = &data[header_offset];
  }

 /*
  * Determine current machine type.
  */
  current_machine = machtype(NULL);

  switch (type)
  {
    case VFF_TYP_BIT:
	 unit_size = -1;
	 size = (int) (num_bytes * 8);
	 break;

    case VFF_TYP_1_BYTE:
	 unit_size = sizeof(char);
	 size = (int) (num_bytes / unit_size);
	 break;

    case VFF_TYP_2_BYTE:
	 sptr = (short *) data;
	 unit_size = sizeof(short);
	 size = (int) (num_bytes / unit_size);
 	
  	 /* If machine type does not use ieee order, convert to ieee */
  	 if (current_machine == VFF_DEP_DECORDER)
  	 {
           for (i=0; i<size; i++) { dectoieees(sptr); sptr++; }
  	 }
  	 else if (current_machine == VFF_DEP_NSORDER)
  	 {
      	   for (i=0; i<size; i++) { nstoieees(sptr); sptr++; }
  	 }

	 break;

    case VFF_TYP_4_BYTE:
	 iptr = (int *) data;
	 unit_size = sizeof(int);
	 size = (int) (num_bytes / unit_size);

  	 /* If machine type does not use ieee order, convert to ieee */
  	 if (current_machine == VFF_DEP_DECORDER)
  	 {
           for (i=0; i<size; i++) { dectoieeel(iptr); iptr++; }
  	 }
  	 else if (current_machine == VFF_DEP_NSORDER)
  	 {
      	   for (i=0; i<size; i++) { nstoieeel(iptr); iptr++; }
  	 }

	 break;

    case VFF_TYP_FLOAT:
    case VFF_TYP_COMPLEX:
	 fptr = (float *) data;
	 unit_size = sizeof(float);
	 size = (int) (num_bytes / unit_size);

  	 /* If machine type does not use ieee order, convert to ieee */
  	 if (current_machine == VFF_DEP_DECORDER)
  	 {
           for (i=0; i<size; i++) { dectoieeef(fptr); fptr++; }
  	 }
  	 else if (current_machine == VFF_DEP_NSORDER)
  	 {
      	   for (i=0; i<size; i++) { nstoieeef(fptr); fptr++; }
  	 }

	 break;
  }

  *num_pts = size;

 /*
  *  Round down to the nearest float.  Warn the user if the last point
  *  is not a complete float.
  */
  if (((num_bytes % unit_size) != 0) && (unit_size > 1))
  {
    (void) fprintf(stderr,"raw_sig.c: Warning!  Last point in the file\
 was an incomplete.\n(%d out of %d bytes found)\n", (num_bytes % unit_size),
 unit_size);
  }

  if (conv_to_float && (type != VFF_TYP_FLOAT))
  {
  
    if (!(temp = (float *) malloc((unsigned int) size * sizeof(float))))
    {
       return(NULL);
    }

    switch (type)
    {
      case VFF_TYP_BIT:
	   bptr = (unsigned char *) data;
           for (i = 0; i < size; i++)
           {
             c = (1 << (i % 8));
             temp[i] = (float) (c & (*bptr));

             if ((i % 8) == 7)
               bptr++;
           }
           break;

      case VFF_TYP_1_BYTE:
	   cptr = (unsigned char *) data;
           for (i = 0; i < size; i++)
             temp[i] = (float) cptr[i];
           break;

      case VFF_TYP_2_BYTE:
           sptr = (short *) data;
           for (i = 0; i < size; i++)
             temp[i] = (float) sptr[i];
           break;

      case VFF_TYP_4_BYTE:
           iptr = (int *) data;
           for (i = 0; i < size; i++)
             temp[i] = (float) iptr[i];
           break;

      case VFF_TYP_FLOAT:
      case VFF_TYP_COMPLEX:
           fptr = (float *) data;
           for (i = 0; i < size; i++)
             temp[i] = (float) fptr[i];
           break;
    }

 
    if(!(data = (char *) realloc(data, ((unsigned int) size * sizeof(float)) )))
    {
      (void) fprintf(stderr,"Not enough memory! \n");
      return (NULL);
    }
    data = (char *) temp;
    free(temp);

  }
  return(data);

}


/**************************************************************
*
* MODULE NAME: sig2viff
*
*     PURPOSE: 	sig2viff converts a raw complex data file generated
*		on the GE Signa system to a floating point complex file
*		(VIF format).
*		
*		The complex data storage format on the Signa is to 
*		specify all real-imaginary pairs corresponding to a
*		voxel (all band information for that voxel), before 
*		specifying data for the subsequent voxel.  (ie. the 
*		data is interleaved)
*
*		Sig2viff calls read_raw_mag to read in the raw data and
*		convert it to float, and calls load_viff_data to set up 
*		the data in the proper band format.  
*
*		If the single_voxel flag is set, all data points 
*		associated with that voxel are extracted and stored
*		as a single row by nbands-columns (instead of 1 row
*		by 1 col by nbands bands).  The data is stored in this 
*		format to simplify subsequent processing of the 1d signal.
*		The row and column positions of the voxel to be extracted
*		are given by row_offset and col_offset.  It is assumed
*		that the upper left hand pixel is at location [0][0] (not
*		[1][1]).  Row_offset and col_offset are used only if 
*		single_voxel is 1.
*
*
*       INPUT:	fid		File descriptor pointing to the
*				stream in which the raw data is located.
*
*		img1		Pointer to output (complex) VIFF file 
*				created by this routine.
*
*		nrows		number of rows
*	
*		ncols		number of columns
*
*		header_offset	Number of bytes of header information
*				in raw data file before actual data starts.
*
*		type		Data storage type of raw data.
*
*		single_voxel	Flag set to 1 if only a single pixel or
*				voxel is to be extracted from the file.
*
*		row_offset	If single_voxel is set, this is the row 
*				position of the pixel to be extracted.
*
*		col_offset	If single_voxel is set, this is the column
*				position of the pixel to be extracted.
*
*		numbands	The number of data bands associated with
*				each pixel.  In magnetic data, this refers
*				to the spectral or time domain data for
*				each voxel.
*
*      OUTPUT: does the work
*
* CALLED FROM: main
*
* ROUTINES CALLED:	read_raw_mag
*			load_viff_data
*
**************************************************************/

/*ADD_SRC*/

int	sig2viff(fid, img1, nrows, ncols, header_offset, type, 
		 single_voxel, row_offset, col_offset, numbands)


int     	fid, 			/* input file descriptor */
		nrows, 			/* number of rows */
		ncols, 			/* number of columns */
		header_offset, 		/* header offset in bytes */
		type,			/* input data storage type */
		single_voxel, 		/* flag set if only 1 voxel extracted */
		row_offset, 		/* row offset if single_voxel true */
		col_offset, 		/* column offset if single_voxel true */
		numbands; 		/* number of data bands */

struct xvimage 	**img1; 		/* input & complex output image */

{
  
  char		*data, 			/* raw data array */
		*read_raw_mag();	

  int		point_offset,		/* pixel offset if single_voxel set */
		conv_to_float,		/* flag set to 1 to convert to float */
		num_pts,
  		img_size;		/* calculated image size */

  struct xvimage	*img, *createimage();

 /*
  * Check parameter values to make sure they are resonable.
  */

  if (header_offset < 0 )
  {
    (void) fprintf(stderr,"Header offset must be >= 0\n");
    return(0);
  }

  if (numbands < 1 )
  {
    (void) fprintf(stderr,"Number of bands must be > 0\n");
    return(0);
  }

 /*
  * Calculate total image size (complex).
  */
  img_size = nrows * ncols * numbands * 2;

 /*
  * If only a single voxel is specified, calculate the position of that
  * voxel.
  */

  if (single_voxel)
  {
    /* multiply by 2 for complex, by 4 for float */
    point_offset = ( (row_offset * ncols) + col_offset ) * 2 * 4 * numbands;
  }

 /*
  * Read in raw data file.
  */
  conv_to_float = 1;

  if (!(data = read_raw_mag(fid, header_offset, &num_pts, type, conv_to_float)))
  {
    return(0);
  }

  if ( num_pts < img_size )
  {
    (void)fprintf(stderr,"\nOut of bounds! Number of requested rows and \
columns is greater than \nthe number of points found in the data file.\n\
Number of points is %d. (Expected number of points is %d\n\n",
num_pts, img_size);
    return(0);
  }

 /*
  * If only one voxel extracted, format as 1 row by numbands cols by 
  * 1 band, and return.
  */

  if (single_voxel)
  {
    data = &data[point_offset];
    nrows = 1;
    ncols = numbands;
    numbands = 1;
  }

 /*
  * Create the complex VIFF image.
  */
  img = createimage( (unsigned long) nrows,           	/* number of rows */
                     (unsigned long) ncols,             /* number of columns */
                     (unsigned long) VFF_TYP_COMPLEX,   /* data_storage_type */
                     (unsigned long) 1,                 /* number_of_images */
                     (unsigned long) numbands,          /* num_data_bands */
                     "", 			        /* comment */
                     (unsigned long) 0,                 /* map_row_size */
                     (unsigned long) 0,                 /* map_col_size */
                     (unsigned long) VFF_MS_NONE,       /* map_scheme */
                     (unsigned long) VFF_MAPTYP_NONE,   /* map_storage_type */
                     (unsigned long) VFF_LOC_IMPLICIT,  /* location_type */
                     (unsigned long) 0);        	/* location_dimension */

  if (img == NULL)
  {
    (void) fprintf(stderr,"sig2viff: Unable to allocate new image!\n");
    return(0);
  }
  *img1 = img;

 /*
  * Massage data points into specified form.
  */
  load_viff_data( data,				/* data */
		 (float *)img->imagedata, 	/* array to contain bands */
		 img->location, 		/* location */
		 nrows, 			/* number of rows */
		 ncols,				/* number of columns */
		 numbands, 			/* number of bands */
		 0,				/* location_dimension */
		 1,				/* interleaved = 1 */
		 0,				/* start */
		 1,				/* row_skip */
		 1,				/* col_skip */
		 1 );				/* complex data = 1 */

  free ((char *) data);
  return(1);
}
