/*ScianVisWindows.c
  Visualization window stuff
  Eric Pepke
  April 6, 1990
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianArrays.h"
#include "ScianLists.h"
#include "ScianWindows.h"
#include "ScianTextBoxes.h"
#include "ScianTitleBoxes.h"
#include "ScianObjWindows.h"
#include "ScianIcons.h"
#include "ScianColors.h"
#include "ScianControls.h"
#include "ScianButtons.h"
#include "ScianSpaces.h"
#include "ScianLights.h"
#include "ScianSliders.h"
#include "ScianIDs.h"
#include "ScianVisWindows.h"
#include "ScianDatasets.h"
#include "ScianPictures.h"
#include "ScianDialogs.h"
#include "ScianRecorders.h"
#include "ScianEvents.h"
#include "ScianScripts.h"
#include "ScianTimers.h"
#include "ScianErrors.h"
#include "ScianComplexControls.h"
#include "ScianMethods.h"
#include "ScianStyle.h"
#include "ScianVisObjects.h"
#include "ScianDraw.h"
#include "ScianDrawings.h"
#include "ScianObjFunctions.h"

ObjPtr visWindowClass;			/*Class for vis windows*/
ObjPtr visCorralClass;			/*Class for corral of a vis window*/
Bool showControlPanels = true;		/*True iff show control panels by default*/

static ObjPtr HideVisWindowPanel(window)
WinInfoPtr window;
/*Hides the panel of the controls on the vis window.  Returns true iff it did*/
{

    if (window)
    {
	ObjPtr panel, space, assocPanel;
	int panelWidth;
	int left, right, bottom, top;
	int nl, nr, nb, nt;

	GetWindowBounds(&left, &right, &bottom, &top);

	space = FindSpace(window);

        if (!space)
	{
	    return false;
	}

	panel = GetVar(space, OBJPANEL);
	if (panel)
	{
	    /*Get the panel's bounds*/
	    Get2DIntBounds(panel, &nl, &nr, &nb, &nt); 
	    panelWidth = nr - nl;

	    /*Change panel to be off the screen*/
	    nl = right + 1;
	    nr = nl + panelWidth;
	    Set2DIntBounds(panel, nl, nr, nb, nt);
	    ImInvalid(panel);
	}

	panel = GetVar(space, TOOLPANEL);
	if (panel)
	{
	    /*Get the panel's bounds*/
	    Get2DIntBounds(panel, &nl, &nr, &nb, &nt); 
	    panelWidth = nr - nl;

	    /*Change panel to be off the screen*/
	    nr = left - 1;
	    nl = nr - panelWidth;
	    Set2DIntBounds(panel, nl, nr, nb, nt);
	    ImInvalid(panel);
	}

	/*Now change the space*/
	Get2DIntBounds(space, &nl, &nr, &nb, &nt); 
	nr = right;
	nl = left;
	Set2DIntBounds(space, nl, nr, nb, nt);
	
	ImInvalid(space);

	/*And its panels*/
	assocPanel = GetVar(space, FRONTPANEL);
	if (assocPanel)
	{
	    int ol, or, ob, ot;

	    Get2DIntBounds(assocPanel, &ol, &or, &ob, &ot);
	    ReshapeObject(assocPanel, ol, or, ob, ot, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	assocPanel = GetVar(space, BACKPANEL);
	if (assocPanel)
	{
	    int ol, or, ob, ot;

	    Get2DIntBounds(assocPanel, &ol, &or, &ob, &ot);
	    ReshapeObject(assocPanel, ol, or, ob, ot, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	SetVar((ObjPtr) window, PANELHIDDEN, ObjTrue);

	DeleteMenus(window);

	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

void DoHidePanel()
{
    if (selWinInfo)
    {
	if (logging)
	{
	    Log("hide panel\n");
	}

	DeferMessage((ObjPtr) selWinInfo, HIDEPANEL);
    }
}

static ObjPtr ShowVisWindowPanel(window)
WinInfoPtr window;
/*Shows the panel of the controls on the vis window.  Returns true iff it did*/
{
    if (window)
    {
	ObjPtr panel, space, assocPanel;
	int lPanelWidth, rPanelWidth;
	int left, right, bottom, top;
	int nl, nr, nb, nt;

	GetWindowBounds(&left, &right, &bottom, &top);

	space = FindSpace(window);

        if (!space)
	{
	    return false;
	}

	panel = GetVar(space, OBJPANEL);
	if (panel)
	{
	    /*Get the panel's bounds*/
	    Get2DIntBounds(panel, &nl, &nr, &nb, &nt); 
	    rPanelWidth = nr - nl;

	    /*Change panel to be on the screen*/
	    nr = right;
	    nl = right - rPanelWidth;
	    Set2DIntBounds(panel, nl, nr, nb, nt);
	    ImInvalid(panel);
	}

	panel = GetVar(space, TOOLPANEL);
	if (panel)
	{
	    /*Get the panel's bounds*/
	    Get2DIntBounds(panel, &nl, &nr, &nb, &nt); 
	    lPanelWidth = nr - nl;

	    /*Change panel to be on the screen*/
	    nl = left;
	    nr = left + lPanelWidth;
	    Set2DIntBounds(panel, nl, nr, nb, nt);
	    ImInvalid(panel);
	}

	/*Now change the space*/
	Get2DIntBounds(space, &nl, &nr, &nb, &nt); 
	nr = right - rPanelWidth - 1;
	nl = left + lPanelWidth + 1;
	Set2DIntBounds(space, nl, nr, nb, nt);

	ImInvalid(space);

	/*And its panels*/
	assocPanel = GetVar(space, FRONTPANEL);
	if (assocPanel)
	{
	    int ol, or, ob, ot;

	    Get2DIntBounds(assocPanel, &ol, &or, &ob, &ot);
	    ReshapeObject(assocPanel, ol, or, ob, ot, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	assocPanel = GetVar(space, BACKPANEL);
	if (assocPanel)
	{
	    int ol, or, ob, ot;

	    Get2DIntBounds(assocPanel, &ol, &or, &ob, &ot);
	    ReshapeObject(assocPanel, ol, or, ob, ot, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	SetVar((ObjPtr) window, PANELHIDDEN, ObjFalse);

	DeleteMenus(window);

	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

void DoShowPanel()
{
    if (selWinInfo)
    {
	if (logging)
	{
	    Log("show panel\n");
	}

	DeferMessage((ObjPtr) selWinInfo, SHOWPANEL);
    }
}

int annotStagger = 0;
int nAnnot = 1;
int nRect = 1;
int nLine = 1;
int nTimeReadout = 1;

ObjPtr DeleteSpacePanelText(text)
ObjPtr text;
/*Delete text from a space panel*/
{
    return ObjTrue;
}

void AddAnnotation(name, bounds)
char *name;
real bounds[4];
/*Adds annotation with name and bounds to current window*/
{
    if (selWinInfo)
    {
	/*Search for a space*/
	ObjPtr space, frontPanel;

	space = FindSpace(selWinInfo);

	if (space)
	{
	    frontPanel = GetObjectVar("AddAnnotation", space, FRONTPANEL);
	    if (frontPanel)
	    {
		/*There's a front panel.  Stick an annotation there*/
		ObjPtr annotation;

		annotation = NewTextBox((int) bounds[0], (int) bounds[1], (int) bounds[2], (int) bounds[3],
					/*WITH_BG + */EDITABLE + ADJUSTABLE, name, "");
		SetMethod(annotation, CLONE, CloneDrawing);
		SetMethod(annotation, DUPLICATE, DuplicateDrawing);
		PrefixList(GetVar(frontPanel, CONTENTS), annotation);
		SetVar(annotation, PARENT, frontPanel);
		SetTextColor(annotation, NewInt(UIWHITE));
		ImInvalid(annotation);
		SetTextFont(annotation, ANNOTFONT);
		SetTextSize(annotation, ANNOTFONTSIZE);
		SetVar(annotation, STICKINESS, NewInt(FLOATINGLEFT + FLOATINGRIGHT + FLOATINGTOP + FLOATINGBOTTOM));
		SetMethod(annotation, DELETEICON, DeleteSpacePanelText);
		SetMethod(annotation, DELETE, DeleteObject); 
		SetMethod(annotation, PUSHTOBOTTOM, PushDrawingToBottom);
		SetMethod(annotation, BRINGTOTOP, BringDrawingToTop);
		SetMethod(annotation, MOVETOBACKPANEL, MoveDrawingToBackPanel);
		SetMethod(annotation, MOVETOFRONTPANEL, MoveDrawingToFrontPanel);

		DeselectAll();
		Select(annotation, true);
		MakeMeCurrent(annotation);

		if (logging)
		{
		    /*Log the action.*/
		    char cmd[256];
		    char *d;

		    d = &(cmd[0]);
		    sprintf(cmd, "annotation ");
		    while (*d) ++d;
		    MakeObjectName(d, annotation);
		    while (*d) ++d;
		    sprintf(d, " [%g %g %g %g]\n",
			bounds[0], bounds[1], bounds[2], bounds[3]);
		    Log(cmd);
		}
	    }
	}
    }
}

void AddTimeReadout(name, bounds)
char *name;
real bounds[4];
/*Adds annotation with name and bounds to current window*/
{
    if (selWinInfo)
    {
	/*Search for a space*/
	ObjPtr space, frontPanel;

	space = FindSpace(selWinInfo);

	if (space)
	{
	    frontPanel = GetObjectVar("AddTimeReadout", space, FRONTPANEL);
	    if (frontPanel)
	    {
		/*There's a front panel.  Stick a time readout there*/
		ObjPtr timeReadout;

		timeReadout = NewTimeReadout((int) bounds[0], (int) bounds[1], (int) bounds[2], (int) bounds[3],
					name, space);
		SetMethod(timeReadout, CLONE, CloneDrawing);
		SetMethod(timeReadout, DUPLICATE, DuplicateDrawing);
		PrefixList(GetVar(frontPanel, CONTENTS), timeReadout);
		SetVar(timeReadout, PARENT, frontPanel);
		SetTextColor(timeReadout, NewInt(UIWHITE));
		ImInvalid(timeReadout);
		SetTextFont(timeReadout, ANNOTFONT);
		SetTextSize(timeReadout, ANNOTFONTSIZE);
		SetVar(timeReadout, STICKINESS, NewInt(FLOATINGLEFT + FLOATINGRIGHT + FLOATINGTOP + FLOATINGBOTTOM));
		SetMethod(timeReadout, DELETEICON, DeleteSpacePanelText);
		SetMethod(timeReadout, DELETE, DeleteObject); 
		SetMethod(timeReadout, PUSHTOBOTTOM, PushDrawingToBottom);
		SetMethod(timeReadout, BRINGTOTOP, BringDrawingToTop);
		SetMethod(timeReadout, MOVETOBACKPANEL, MoveDrawingToBackPanel);
		SetMethod(timeReadout, MOVETOFRONTPANEL, MoveDrawingToFrontPanel);

		DeselectAll();
		Select(timeReadout, true);

		if (logging)
		{
		    /*Log the action.*/
		    char cmd[256];
		    char *d;

		    d = &(cmd[0]);
		    sprintf(cmd, "timereadout ");
		    while (*d) ++d;
		    MakeObjectName(d, timeReadout);
		    while (*d) ++d;
		    sprintf(d, " [%g %g %g %g]\n",
			bounds[0], bounds[1], bounds[2], bounds[3]);
		    Log(cmd);
		}
	    }
	}
    }
}

void AddRectangle(name, bounds)
char *name;
real bounds[4];
/*Adds rectangle with name and bounds to current window*/
{
    if (selWinInfo)
    {
	/*Search for a space*/
	ObjPtr space, frontPanel;

	space = FindSpace(selWinInfo);

	if (space)
	{
	    frontPanel = GetObjectVar("AddRectangle", space, FRONTPANEL);
	    if (frontPanel)
	    {
		/*There's a front panel.  Stick a rectangle there*/
		ObjPtr rectangle;

		rectangle = NewRectangle((int) bounds[0], (int) bounds[1], (int) bounds[2], (int) bounds[3],
					name);
		PrefixList(GetVar(frontPanel, CONTENTS), rectangle);
		SetVar(rectangle, PARENT, frontPanel);
		ImInvalid(rectangle);
 
		DeselectAll();
		Select(rectangle, true);

		if (logging)
		{
		    /*Log the action.*/
		    char cmd[256];
		    char *d;

		    d = &(cmd[0]);
		    sprintf(cmd, "rectangle ");
		    while (*d) ++d;
		    MakeObjectName(d, rectangle);
		    while (*d) ++d;
		    sprintf(d, " [%g %g %g %g]\n",
			bounds[0], bounds[1], bounds[2], bounds[3]);
		    Log(cmd);
		}
	    }
	}
    }
}

void AddLine(name, x1, y1, x2, y2)
char *name;
int x1, y1, x2, y2;
/*Adds line with name and endpoints (x1, y1)-(x2, y2) to current window*/
{
    if (selWinInfo)
    {
	/*Search for a space*/
	ObjPtr space, frontPanel;

	space = FindSpace(selWinInfo);

	if (space)
	{
	    frontPanel = GetObjectVar("AddLine", space, FRONTPANEL);
	    if (frontPanel)
	    {
		/*There's a front panel.  Stick a line there*/
		ObjPtr line;

		line = NewLine(x1, y1, x2, y2, name);
		PrefixList(GetVar(frontPanel, CONTENTS), line);
		SetVar(line, PARENT, frontPanel);
		ImInvalid(line);
 
		DeselectAll();
		Select(line, true);

		if (logging)
		{
		    /*Log the action.*/
		    char cmd[256];
		    char *d;

		    d = &(cmd[0]);
		    sprintf(cmd, "line ");
		    while (*d) ++d;
		    MakeObjectName(d, line);
		    while (*d) ++d;
		    sprintf(d, " %d %d %d %d\n",
			x1, y1, x2, y2);
		    Log(cmd);
		}
	    }
	}
    }
}

static void DoTogglePanel()
/*Toggles the panel of the controls on the top window.*/
{
    if (selWinInfo)
    {
	if (GetPredicate((ObjPtr) selWinInfo, PANELHIDDEN))
	{
	    DoShowPanel();
	}
	else
	{
	    DoHidePanel();
	}
    }
}

void DoShowControls()
/*Shows the controls in the selected objects in the window*/
{
    DoObjFunction(OF_SHOW_CONTROLS);
}

ObjPtr MakeLocalCopy(object)
ObjPtr object;
/*Makes a local copy of an object*/
{
    object = ObjectWhichRepresents(selWinInfo, object);

    if (object && IsIcon(object))
    {
	ObjPtr repObj, newObj;
	FuncTyp method;
	
	repObj = GetVar(object, REPOBJ);
	if (repObj)
	{
	    ObjPtr contentsList, corral = NULLOBJ;
	    contentsList = GetVar(object, PARENT);
	    if (contentsList || !IsList(contentsList))
	    {
		contentsList = GetVar(contentsList, CONTENTS);
	    }
	    if (!contentsList || !IsList(contentsList))
	    {
		ReportError("MakeLocalCopy","Internal error: cannot find contents list");
	    }

	    if (contentsList)
	    {
		corral = contentsList;
		while (corral && !IsCorral(corral))
		{
		    corral = GetVar(corral, PARENT);
		}
	    }

	    if (!corral)
	    {
		ReportError("MakeLocalCopy", "Cannot find local corral");
	    }

	    method = GetMethodSurely("MakeLocalCopy", repObj, CLONE);
	    if (!method)
	    {
		return ObjFalse;
	    }

	    method = GetMethod(object, DELETEICON);
	    if (method)
	    {
		if (IsTrue((*method)(object)))
		{
		    if (contentsList && IsList(contentsList))
		    {
			ImInvalid(corral);
			DeleteFromList(contentsList, object);
		    }
		}
	    }

	    method = GetMethod(repObj, CLONE);
	    if (method)
	    {
		ObjPtr name;
	        newObj = (*method)(repObj);
		object = Clone(object);
	        SetVar(object, REPOBJ, newObj);
		MakeVar(newObj, NAME);
		name = GetVar(newObj, NAME);
		SetVar(object, NAME, object);
		SetVar(object, SELECTED, ObjFalse);
	    }

	    method = GetMethod(corral, DROPINCONTENTS);
	    if (method)
	    {
		ObjPtr locArray;
		int x, y;
		real loc[2];
		locArray = GetVar(object, ICONLOC);
		if (locArray)
		{
		    Array2CArray(loc, locArray);
		    x = loc[0];
		    y = loc[1];
		}
		else
		{
		    x = y = 0;
		}
		(*method)(corral, object, x, y);
	    }
	    return ObjTrue;
	}
    }
    return ObjFalse;
}

ObjPtr CloneObject(object)
ObjPtr object;
/*Clones an individual object*/
{
    object = ObjectWhichRepresents(selWinInfo, object);
    if (object && IsIcon(object))
    {
	ObjPtr repObj, newVis;
	FuncTyp method;

	repObj = GetVar(object, REPOBJ);
	if (repObj && !GetPredicate(repObj, ONEONLY))
	{
	    ObjPtr corral = NULLOBJ;

	    corral = GetVar(object, PARENT);

	    if (corral)
	    {
		while (corral && !IsCorral(corral))
		{
		    corral = GetVar(corral, PARENT);
		}
	    }

	    if (!corral)
	    {
		ReportError("CloneObject", "Cannot find local corral");
		return ObjFalse;
	    }

	    method = GetMethodSurely("CloneObject", repObj, CLONE);
	    if (!method)
	    {
		return ObjFalse;
	    }

	    newVis = (*method)(repObj);
	    SetVar(newVis, SELECTED, ObjFalse);
	    if (IsController(newVis))
	    {
		AddControllerToSpace(newVis, FindSpace(selWinInfo), corral, NULLOBJ);
		ResolveController(newVis);
	    }
	    else
	    {
		IdleAllWindows();
		AddObjToSpace(newVis, FindSpace(selWinInfo), corral, NULLOBJ, NULLOBJ);
	    }
	    return ObjTrue;
	}
    }
    return ObjFalse;
}

int vwSerialNumber = 0;

void NewSerializedVisWindow()
/*Creates a new, serialized visualization window*/
{
    WinInfoPtr visWindow;

    /*Create a vis window*/
    sprintf(tempStr, "Visualization %d", ++vwSerialNumber); 
    visWindow = NewVisWindow(tempStr, WINDBUF + WINZBUF + WINRGB);
    SelWindow(visWindow);
}


int vawSerialNumber;

void DoShowFrontPanelControls()
/*Shows the controls for the front panel in the selected vis window*/
{
    ObjPtr space, panel;
    if (!selWinInfo)
    {
	return;
    }

    space = FindSpace(selWinInfo);
    if (!space)
    {
	return;
    }

    panel = GetVar(space, FRONTPANEL);
    if (!panel)
    {
	return;
    }

    if (logging)
    {
	Log("show front panel controls\n");
	InhibitLogging(true);
    }
    NewControlWindow(panel);
    if (logging)
    {
	InhibitLogging(false);
    }
}

void DoShowBackPanelControls()
/*Shows the controls for the back panel in the selected vis window*/
{
    ObjPtr space, panel;
    if (!selWinInfo)
    {
	return;
    }

    space = FindSpace(selWinInfo);
    if (!space)
    {
	return;
    }

    panel = GetVar(space, BACKPANEL);
    if (!panel)
    {
	return;
    }

    if (logging)
    {
	Log("show back panel controls\n");
	InhibitLogging(true);
    }

    NewControlWindow(panel);
    if (logging)
    {
	InhibitLogging(false);
    }
}

void DoShowSpaceControls()
/*Shows the controls for the space in the selected vis window*/
{
    ObjPtr space;
    if (!selWinInfo)
    {
	return;
    }

    space = FindSpace(selWinInfo);
    if (!space)
    {
	return;
    }

    if (logging)
    {
	Log("show space controls\n");
	InhibitLogging(true);
    }

    NewControlWindow(space);
    if (logging)
    {
	InhibitLogging(false);
    }
}

#define MINANNOTSIZE	40
#define MINRECTSIZE	0
#define MINLINESIZE	0

static ObjPtr DeleteClockReadout(readout)
ObjPtr readout;
/*Deletes a clock readout from a space*/
{
    return ObjTrue;
}

static ObjPtr MakeClockValue(clockDisplay)
ObjPtr clockDisplay;
/*Makes the VALUE of a clock display*/
{
    ObjPtr space, timeObj, format;
    real time;
    char s[400];

    space = GetObjectVar("MakeClockValue", clockDisplay, SPACE);
    if (!space)
    {
	return ObjFalse;
    }

    timeObj = GetVar(space, TIME);
    if (!timeObj)
    {
	time = 0.0;
    }
    else
    {
	time = GetReal(timeObj);
    }

    format = GetVar(clockDisplay, FORMAT);
    if (format)
    {
	PrintClock(s, GetString(format), time);
    }
    else
    {
        sprintf(s, "%#.2f", time);
    }

    SetVar(clockDisplay, VALUE, NewString(s));

    return ObjTrue;
}

ObjPtr ChangeClockDisplayFormat(textBox)
ObjPtr textBox;
/*Changes a clock display's FORMAT according to textBox*/
{
    ObjPtr repObj;
    repObj = GetObjectVar("ChangeClockFormat", textBox, REPOBJ);
    if (!repObj) return ObjFalse;
    SetVar(repObj, FORMAT, GetValue(textBox));
    ImInvalid(repObj);
    return ObjTrue;
}

static ObjPtr ShowClockDisplayControls(display, windowName)
ObjPtr display;
char *windowName;
/*Makes a new control window to control a clock display*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    real rgb[3], hsv[3];
    WinInfoPtr dialogExists;
    Bool hasBackground;

    dialogExists = DialogExists((WinInfoPtr) display, NewString("Controls"));
    controlWindow = GetDialog((WinInfoPtr) display, NewString("Controls"), windowName, 
	CDWINWIDTH, CDWINHEIGHT, CDWINWIDTH,
	CDWINHEIGHT, WINDBUF + WINRGB + WINFIXEDSIZE);
    
    if (!dialogExists)
    {
	long info;
	ObjPtr value;
	ObjPtr format;
	ObjPtr checkBox, icon, name, colorBar, titleBox, textBox, button;
	ObjPtr colorWheel, slider, radioGroup;
	int left, right, bottom, top;

	SetVar((ObjPtr) controlWindow, REPOBJ, display);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows controls for a clock display.  For information about any of the controls \
in the window, use Help In Context on the control.\n"));

	/*Add in a panel*/
	GetWindowBounds(&left, &right, &bottom, &top);
	panel = NewPanel(greyPanelClass, 0, right - left, 0, top - bottom);
	if (!panel)
	{
	    return NULLOBJ;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);

	contents = GetVar(panel, CONTENTS);

	/*Add in the group of controls for text color*/
	left = MINORBORDER;
	top = CDWINHEIGHT - MINORBORDER;
	right = left + 2 * MINORBORDER + 2 * MAJORBORDER + COLORWHEELWIDTH + SLIDERWIDTH;
	
	titleBox = NewTitleBox(left, right,
			top - TITLEBOXTOP - MINORBORDER - 2 * MINORBORDER - COLORWHEELWIDTH
				- CHECKBOXHEIGHT - TEXTBOXHEIGHT - TEXTBOXSEP,
			top, "Text");
	SetVar(titleBox, PARENT, panel);
	PrefixList(contents, titleBox);
	left += MINORBORDER;
	right -= MINORBORDER;
	top -= TITLEBOXTOP + MINORBORDER;

	/*Get the color for priming the controls*/
	var = GetVar(display, COLOR);
	if (IsArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	}
	else if (IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	}
	else
	{
	    ReportError("ShowClockDisplayControls", "Bad color");
	    rgb[0] = rgb[1] = rgb[2] = 1.0;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Text Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	SetValue(colorWheel, var);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, display);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeTextColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the text in the clock display.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Text Color Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, display);
	SetMethod(slider, CHANGEDVALUE, ChangeTextColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the text in the clock display.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	left -= MINORBORDER;	
	right += MINORBORDER;
	
	/*Make the background controls*/
	top = CDWINHEIGHT - MINORBORDER;
	right = CDWINWIDTH - MINORBORDER;
	left = right - (2 * MINORBORDER + 2 * MAJORBORDER + COLORWHEELWIDTH + SLIDERWIDTH);
	
	titleBox = NewTitleBox(left, right,
			top - TITLEBOXTOP - MINORBORDER - 2 * MINORBORDER - COLORWHEELWIDTH
				- CHECKBOXHEIGHT - TEXTBOXHEIGHT - TEXTBOXSEP,
			top, "Background");
	SetVar(titleBox, PARENT, panel);
	PrefixList(contents, titleBox);
	left += MINORBORDER;
	right -= MINORBORDER;
	top -= TITLEBOXTOP + MINORBORDER;

	/*Get the color for priming the controls*/
	var = GetVar(display, BACKGROUND);
	if (var && IsArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	    hasBackground = true;
	}
	else if (var && IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	    hasBackground = true;
	}
	else
	{
	    rgb[0] = rgb[1] = rgb[2] = 0.5;
	    hasBackground = false;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Background Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	SetValue(colorWheel, var);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, display);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeBackgroundColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the background of the clock display.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Background Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, display);
	SetMethod(slider, CHANGEDVALUE, ChangeBackgroundColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the background of the clock display.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	/*Make the check box*/
	top -= COLORWHEELWIDTH + TEXTBOXSEP + TEXTBOXHEIGHT + MINORBORDER;
	checkBox = NewCheckBox(left, right, 
		top - CHECKBOXHEIGHT, top,
		"No background", hasBackground ? false : true);
	SetVar(checkBox, PARENT, panel);
	SetVar(checkBox, REPOBJ, display);
	SetVar(checkBox, SLIDER, slider);
	SetVar(checkBox, COLORWHEEL, colorWheel);
	PrefixList(contents, checkBox);
	SetMethod(checkBox, CHANGEDVALUE, ChangeNoBackground);
	SetVar(checkBox, HELPSTRING, NewString("This checkbox controls whether \
a background is shown.  If it is selected, no background is shown, and the \
objects behind can be seen."));

	/*Link the checkbox to the slider and color wheel*/
	SetVar(colorWheel, CHECKBOX, checkBox);
	SetVar(slider, CHECKBOX, checkBox);

	top -= CHECKBOXHEIGHT + MINORBORDER + MAJORBORDER;
	left = MINORBORDER;
	right = CDWINWIDTH - MINORBORDER;

	/*Create the format text box*/
	textBox = NewTextBox(left, left + CDFORMATLABELWIDTH, 
				top - EDITBOXHEIGHT + (EDITBOXHEIGHT - TEXTBOXHEIGHT) / 2,
				top - (EDITBOXHEIGHT - TEXTBOXHEIGHT) / 2, 
				PLAIN, "Format:", "Format:");
	PrefixList(contents, textBox);
	SetVar(textBox, PARENT, panel);
	    
	/*Create the format editable text box*/
	left = left + CDFORMATLABELWIDTH;
	format = GetVar(display, FORMAT);
	textBox = NewTextBox(left, right, 
				top - EDITBOXHEIGHT,
				top, 
				EDITABLE + WITH_PIT + ONE_LINE, "Format",
				format ? GetString(format) : "%#.2t");
	PrefixList(contents, textBox);
	SetVar(textBox, PARENT, panel);
	SetVar(textBox, REPOBJ, display);
	SetVar(textBox, HELPSTRING, NewString(
		"This text box contains the format used to print the current time in \
the time readout.  It works very much like printf with the special conversion \
characters listed below.  Flags and field widths are exactly as in printf.  \
All times are printed as real numbers, so you may need to limit the number \
of decimal places.  \
There may be several conversions in one string, in which case, all will refer \
to the same time.  Hours and minutes are derives from the internal time specified in seconds.\n\
\n\
%h\t\tPrint the number of hours, from 1 to 12.\n\
%H\t\tPrint the number of hours, from 0 to 23.\n\
%m\t\tPrint the number of minutes, from 0 to 59.\n\
%M\t\tPrint the total number of minutes.\n\
%s\t\tPrint the number of seconds from 0 to 59.\n\
%S\t\tPrint the time in seconds.\n\
%t\t\tPrint the time in timesteps.  For this to work properly, \
each time step of data must have a time equal to the number of the time step.\n\
\n\
Examples for time = 43440 seconds:\n\
Format\t\t\t\tResult\t\tExplanation\n\
%.0t\t\t\t\t43440\t\tThe time is printed in seconds with no decimal places.\n\
%2.0h:%02.0m %a\t\t\t\t12:04 pm\t\tThe time is assumed to be 43440 seconds after \
midnight and is converted into 12-hour format.\n\
%.2M\t\t\t\t724\t\tThe time is printed in minutes with 2 decimal places."));
	SetMethod(textBox, CHANGEDVALUE, ChangeClockDisplayFormat);

    }
    return (ObjPtr) controlWindow;
}

#ifdef PROTO
static ObjPtr NewTimeReadout(int left, int right, int bottom, int top, char *name, ObjPtr space)
#else
static ObjPtr NewTimeReadout(left, right, bottom, top, name, space)
int left, right, bottom, top;
char *name;
ObjPtr space;
#endif
{
    ObjPtr clockThere;
    clockThere = NewTextBox(left, right, bottom, top, ADJUSTABLE, name, "");
    SetMethod(clockThere, DELETEICON, DeleteClockReadout);
    DeclareDependency(clockThere, VALUE, FORMAT);
    SetVar(clockThere, SPACE, space);
    DeclareIndirectDependency(clockThere, VALUE, SPACE, TIME);
    SetMethod(clockThere, NEWCTLWINDOW, ShowClockDisplayControls);
    SetMethod(clockThere, VALUE, MakeClockValue);
    SetVar(clockThere, FORMAT, NewString("%#.2t"));
    SetVar(clockThere, HELPSTRING, NewString("This time readout displays the time \
given by the clock in this space.  The display format is controlled by the Format \
text box in the readout's control panel.\n"));
    return clockThere;
}

#ifdef INTERACTIVE
static ObjPtr PressSpacePanel(object, x, y, flags)
ObjPtr object;
int x, y;
long flags;
/*Does a press in a field beginning at x and y.  Returns
  true iff the press really was in the field.*/
{
    int left, right, bottom, top;
    FuncTyp pressContents;		/*Routine to press the contents*/

    Get2DIntBounds(object, &left, &right, &bottom, &top);

    if (flags & F_OPTIONDOWN)
    {
	/*Ignore press if this is the front panel*/
	if (GetPredicate(object, FRONTPANEL))
	{
	    return ObjFalse;
	}
    }

    if (x >= left && x <= right && y >= bottom && y <= top)
    {
	/*Hey!  It really was a click in the field*/
	ObjPtr space;
	int tool;
	ObjPtr contents;
	ObjPtr retVal = ObjFalse;

	if (TOOL(flags) == T_ROTATE) flags |= F_SHIFTDOWN;

	SetScreenGrid(object);

	/*See if it's a dragBuffer click*/
	if (dragBuffer)
	{
	    /*Yes it is.  Move dragBuffer and exit*/
	    dropObject = dragBuffer;
	    dragBuffer = NULLOBJ;
	    return ObjTrue;
	}

    	/*Setup the new viewport*/
	StartPanel(left, right, bottom, top);

        x -= left;
        y -= bottom;
	
        contents = GetVar(object, CONTENTS);

	tool = ST_FINGER;
	space = GetVar(object, SPACE);
	if (space)
	{
	    ObjPtr var;
	    var = GetVar(space, EDITTOOL);
	    if (var)
	    {
		tool = GetInt(var); 
	    }
	}
	if (TOOL(flags) == T_HELP)
	{
	    tool = ST_FINGER;
	}

	switch (tool)
	{
	    case ST_FINGER:
		retVal = PressObject(contents, x, y, flags);

		if (TOOL(flags) == T_HELP && !IsTrue(retVal))
		{
		    ContextHelp(object);
		    StopPanel();
		    return ObjTrue;
		}
		else if (!IsTrue(retVal))
		{
		    if (TOOL(flags) == T_ROTATE)
		    {
			StopPanel();
			return ObjFalse;
		    }
		    else
		    {
			if (0 == (flags & F_SHIFTDOWN))
			{
			    DeselectAll();
			}
			StopPanel();
			return ObjFalse;
		    }
		}
		retVal = ObjTrue;
		break;
	    case ST_ANNOTATION:
	    case ST_RECTANGLE:
	    case ST_LINE:
	    case ST_TIME_READOUT:
		{
		    char name[30];
		    int l, r, b, t;
		    int minSize;
		    ObjPtr boundsArray;
		    ObjPtr screenObject;
		    int startX, startY, newX, newY, endX, endY;

		    if (!(flags & F_SHIFTDOWN))
		    {
			DeselectAll();
		    }

		    if (flags & F_SHIFTDOWN)
		    {
			x = GRIDX(x);
			y = GRIDY(y);
		    }

		    startX = x;
		    startY = y;

		    if (tool == ST_ANNOTATION || tool == ST_TIME_READOUT)
		    {
			minSize = MINANNOTSIZE;
		    }
		    else if (tool == ST_RECTANGLE)
		    {
			minSize = MINRECTSIZE;
		    }
		    else if (tool == ST_LINE)
		    {
			minSize = MINLINESIZE;
		    }

		    l = startX - HANDLESIZE / 2;
		    r = startX + minSize + HANDLESIZE / 2;
		    b = startY - minSize - HANDLESIZE / 2;
		    t = startY + HANDLESIZE / 2;

		    if (tool == ST_ANNOTATION)
		    {
			sprintf(name, "Annotation %d", nAnnot++);

		        screenObject = NewTextBox(l, r, b, t,
					/*WITH_BG + */EDITABLE + ADJUSTABLE, name, "");
			SetMethod(screenObject, CLONE, CloneDrawing);
			SetMethod(screenObject, DUPLICATE, DuplicateDrawing);
			SetMethod(screenObject, PUSHTOBOTTOM, PushDrawingToBottom);
			SetMethod(screenObject, BRINGTOTOP, BringDrawingToTop);
			SetMethod(screenObject, MOVETOBACKPANEL, MoveDrawingToBackPanel);
			SetMethod(screenObject, MOVETOFRONTPANEL, MoveDrawingToFrontPanel);
		    }
		    else if (tool == ST_TIME_READOUT)
		    {
			sprintf(name, "Time Readout %d", nTimeReadout++);

		        screenObject = NewTimeReadout(l, r, b, t, name, space);
			SetMethod(screenObject, CLONE, CloneDrawing);
			SetMethod(screenObject, DUPLICATE, DuplicateDrawing);
			SetMethod(screenObject, PUSHTOBOTTOM, PushDrawingToBottom);
			SetMethod(screenObject, BRINGTOTOP, BringDrawingToTop);
			SetMethod(screenObject, MOVETOBACKPANEL, MoveDrawingToBackPanel);
			SetMethod(screenObject, MOVETOFRONTPANEL, MoveDrawingToFrontPanel);
		    }
		    else if (tool == ST_RECTANGLE)
		    {
			sprintf(name, "Rectangle %d", nRect++);

			screenObject = NewRectangle(l, r, b, t, name);
		    }
		    else if (tool == ST_LINE)
		    {
			sprintf(name, "Line %d", nLine++);

			screenObject = NewLine(startX, startY, startX, startY, name);
		    }
		    SetVar(screenObject, STICKINESS, NewInt(FLOATINGLEFT + FLOATINGRIGHT + FLOATINGTOP + FLOATINGBOTTOM));
		    PrefixList(contents, screenObject);
		    SetVar(screenObject, PARENT, object);
		    if (tool == ST_ANNOTATION)
		    {
			SetTextColor(screenObject, NewInt(UIWHITE));
			ImInvalid(screenObject);
			SetTextFont(screenObject, ANNOTFONT);
			SetTextSize(screenObject, ANNOTFONTSIZE);
			SetMethod(screenObject, DELETEICON, DeleteSpacePanelText);
			SetMethod(screenObject, DELETE, DeleteObject);
		    }
		    else if (tool == ST_TIME_READOUT)
		    {
			SetTextColor(screenObject, NewInt(UIWHITE));
			ImInvalid(screenObject);
			SetTextFont(screenObject, ANNOTFONT);
			SetTextSize(screenObject, ANNOTFONTSIZE);
			SetMethod(screenObject, DELETEICON, DeleteClockReadout);
			SetMethod(screenObject, DELETE, DeleteObject);
		    }

		    if (tool == ST_ANNOTATION || tool == ST_TIME_READOUT)
		    {
			InhibitLogging(true);
			Select(screenObject, true);
			InhibitLogging(false);
		    }

		    /*Set up to draw in the overlay area*/
		    DrawMe(screenObject);

		    while (Mouse(&newX, &newY))
		    {

			if (tool == ST_ANNOTATION ||
			    tool == ST_TIME_READOUT ||
			    tool == ST_RECTANGLE ||
			    tool == ST_LINE)
			{
			    if (newX >= startX && newX - startX < minSize)
			    {
				newX = startX + minSize;
			    }
			    else if (newX < startX && startX - newX < minSize)
			    {
				newX = startX - minSize;
			    }

			    if (newY >= startY && newY - startY < minSize)
			    {
				newY = startY + minSize;
			    }
			    else if (newY < startY && startY - newY < minSize)
			    {
				newY = startY - minSize;
			    }
			}

			if (flags & F_SHIFTDOWN)
			{
			    newX = GRIDX(newX);
			    newY = GRIDY(newY);
			}
			if (newX != x || newY != y)
			{
			    x = newX;
			    y = newY;
			    Set2DIntBounds(screenObject,
					MIN(startX, x) - HANDLESIZE / 2,
					MAX(startX, x) + HANDLESIZE / 2,
					MIN(startY, y) - HANDLESIZE / 2, 
					MAX(startY, y) + HANDLESIZE / 2);
			    if (tool == ST_LINE)
			    {
				ObjPtr var;
				real loc[2];
				loc[0] = newX;
				loc[1] = newY;
				var = NewRealArray(1, 2L);
				CArray2Array(var, loc);
				SetVar(screenObject, ENDPOINT, var);
			    }
			    DrawMe(screenObject);
			}
		    }

		    Get2DIntBounds(screenObject, &l, &r, &b, &t);

		    if (logging)
		    {
			/*Log the action.*/
			char cmd[256];
			char *d, *s;

			s = name;
			d = &(cmd[0]);
			if (tool == ST_ANNOTATION)
			{
			    strcpy(cmd, "annotation ");
			}
			else if (tool == ST_RECTANGLE)
			{
			    strcpy(cmd, "rectangle ");
			}
			else if (tool == ST_LINE)
			{
			    strcpy(cmd, "line ");
			}
			else if (tool == ST_TIME_READOUT)
			{
			    strcpy(cmd, "timereadout ");
			}
			while (*d) ++d;
			MakeObjectName(d, screenObject);
			while (*d) ++d;
			if (tool == ST_LINE)
			{
			    sprintf(d, " %d %d %d %d\n",
				startX, startY, x, y);
			}
			else
			{
			    sprintf(d, " [%d %d %d %d]\n",
				l, r, b, t);
			}
			Log(cmd);

			if (flags & F_OPTIONDOWN)
			{
			    strcpy(d, "move ");
			    d = &(cmd[0]);
			    while (*d) ++d;
			    MakeObjectName(d, screenObject);
			    while (*d) ++d;
			    strcpy(d, " to back panel\n");
			    Log(cmd);
			}
		    }
		    SetValue(GetVar(space, TOOLGROUP), NewInt(ST_FINGER));

		    if (tool == ST_ANNOTATION)
		    {
			MakeMeCurrent(screenObject);
		    }
		    if (tool != ST_ANNOTATION && tool != ST_TIME_READOUT)
		    {
			Select(screenObject, true);
		    }
		}
		retVal = ObjTrue;
		break;
	}

	StopPanel();
	return retVal;
    }
    else
    {
	return ObjFalse;
    }
}
#endif

void NewVisAsWindow()
/*Brings up a new, empty visAs window*/
{
    WinInfoPtr newWindow;
    ObjPtr objList, contents;
    ObjPtr corral, panel, button;
    ThingListPtr runner;
    int l, r, b, t;
    int bw;

    /*Create a vis as window*/
    sprintf(tempStr, "Visualize As %d", ++vawSerialNumber); 
    newWindow = NewObjWindow(NULLOBJ, tempStr, WINDBUF + WINZBUF + WINRGB,
		VAWINWIDTH, VAWINHEIGHT, SCRWIDTH, SCRHEIGHT);
    SelWindow(newWindow);
    GetWindowBounds(&l, &r, &b, &t);

    SetVar((ObjPtr) newWindow, HELPSTRING,
	    NewString("This window shows all the possible ways to visualize a \
group of datasets."));

    /*Put in a panel*/
    panel = NewPanel(greyPanelClass, 0, r - l, 0, t - b);
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));

    contents = GetVar((ObjPtr) newWindow, CONTENTS);
    PrefixList(contents, panel);
    SetVar(panel, PARENT, (ObjPtr) newWindow);

    /*Put in buttons and an icon corral*/
    contents = GetListVar("NewVisAsWindow", panel, CONTENTS);
    if (!contents)
    {
	return;
    }

    /*Make an icon corral*/
    corral = NewIconCorral(NULLOBJ, l + MINORBORDER, r - MINORBORDER, b + 2 * MINORBORDER + BUTTONHEIGHT, t - MINORBORDER, OBJECTSFROMTOP + BARRIGHT + BARBOTTOM);
    SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(corral, NAME, NewString("Visualizations Corral"));
    SetVar((ObjPtr) newWindow, CORRAL, corral);
    SetVar(corral, HELPSTRING,
	NewString("This corral contains icons for all possible visualizations of a group of \
datasets.  You can visualize or show info on the visualizations by selecting \
some of them and pressing the buttons at the bottom of the window.  You can delete \
visualizations by selecting them and choosing Delete from the Object menu."));
    PrefixList(contents, corral);
    SetVar(corral, PARENT, panel);

    l += MINORBORDER;
    r -= MINORBORDER;
    b += MINORBORDER;
    t = b + BUTTONHEIGHT;
    bw = (r - l - MINORBORDER) / 2;

   /*Make a visualize button*/
    button = NewFunctionButton(newWindow,
		l, l + bw,
		b, b + BUTTONHEIGHT, OF_VISUALIZE); 
    if (button)
    {
	SetVar(button, PARENT, panel);
	SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
	PrefixList(contents, button);
    }

    /*Make a show info button*/
    button = NewFunctionButton(newWindow,
		r - bw, 
		r,
		b, b + BUTTONHEIGHT, OF_SHOW_CONTROLS); 
    if (button)
    {
	SetVar(button, PARENT, panel);
	SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
	PrefixList(contents, button);
    }
}

void VisObjectsAs()
/*Opens a new VisObjectsAs window using default for all of the
  selected objects.
*/
{
    DoObjFunction(OF_VISUALIZE_AS);
}

static ObjPtr DropInVisCorral(corral, object, x, y)
ObjPtr corral, object;
int x, y;
/*Drops an icon in a vis corral*/
{
    ObjPtr repObj;
    repObj = GetVar(object, REPOBJ);
    if (repObj)
    {
	real loc[2];
	ObjPtr locArray;
	loc[0] = x;
	loc[1] = y;
	locArray = NewRealArray(1, 2L);
	CArray2Array(locArray, loc);
	if (IsController(repObj))
	{
	    AddControllerToSpace(repObj, FindSpace(selWinInfo), GetVar((ObjPtr) selWinInfo, CORRAL), locArray);
	    ResolveController(repObj);
	}
	else
	{	
	    IdleAllWindows();
	    AddObjToSpace(object, FindSpace(selWinInfo), GetVar((ObjPtr) selWinInfo, CORRAL), locArray, NULLOBJ);
	}
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

void ForAllVisWindows(routine)
void (*routine)();
/*Performs (*routine)(window) on all vis windows*/
{
    WinInfoPtr curWindow;
    curWindow = allWindows;
    while (curWindow)
    {
	if (IsVisWindow((ObjPtr) curWindow))
	{
	    (*routine)(curWindow);
	}
	curWindow = curWindow -> next;
    }
}

void PushNonVisWindows()
/*Pushes all the non vis windows*/
{
#ifdef GRAPHICS
#ifdef WINDOWS4D
    int maxDepth;
    WinInfoPtr curWindow;
    maxDepth = -1;

    curWindow = allWindows;
    while (curWindow)
    {
	if (IsVisWindow((ObjPtr) curWindow))
	{
	    if ((curWindow) -> id)
	    {
		maxDepth = MAX(maxDepth, windepth((curWindow) -> id));
	    }
	}
	curWindow = curWindow -> next;
    }

    curWindow = allWindows;
    while (curWindow)
    {
	if (!IsVisWindow((ObjPtr) curWindow))
	{
	    if ((curWindow) -> id && windepth((curWindow) -> id) < maxDepth)
	    {
		SelWindow(curWindow);
		--maxDepth;
		winpush();
	    }
	}
	curWindow = curWindow -> next;
    }
#endif
#endif
}

void Tile(width, height)
int width, height;
/*Tiles all the vis windows in the lower left corner as big as width, height*/
{
#ifdef GRAPHICS
#ifdef WINDOWS4D
    int nWide, nHigh;		/*# of windows wide and high*/
    int i, j;
    int nVisWindows;
    WinInfoPtr curWindow;
    long oldWindow;

    if (logging)
    {
	char cmd[256];
	sprintf(cmd, "tile %d %d\n", width, height);
	Log(cmd);
    }

    /*Count the number of vis windows*/
    nVisWindows = 0;
    curWindow = allWindows;
    while (curWindow)
    {
	if (IsVisWindow((ObjPtr) curWindow))
	{
	    ++nVisWindows;
	}
	curWindow = curWindow -> next;
    }
    if (!nVisWindows) return;

    /*Estimate the number of windows high*/
    nHigh = sqrt((double) nVisWindows) + 0.49;
    nWide = nVisWindows / nHigh;
    if (nWide * nHigh < nVisWindows) ++nHigh;

    /*Go through and tile the windows*/
    i = 0;
    j = nHigh - 1;
    curWindow = allWindows;
    while (curWindow)
    {
	if (IsVisWindow((ObjPtr) curWindow))
	{
	    long l, r, b, t;
	    
	    if (i)
	    {
		l = i * width / nWide + 1;
	    }
	    else
	    {
		l = 0;
	    }
	    r = (i + 1) * width / nWide;
	    if (j)
	    {
		b = j * height / nHigh + 1;
	    }
	    else
	    {
		b = 0;
	    }
	    t = (j + 1) * height / nHigh;
	    if (++i >= nWide)
	    {
		i = 0;
		--j;
	    }

	    /*Move the window*/
	    SelWindow(curWindow);
	    winposition(l, r, b, t);
	}
	curWindow = curWindow -> next;
    }
#endif
#endif
}

void DoTileFullScreen()
/*Tiles all the visualization windows within the full screen*/
{
    Tile(SCRWIDTH, SCRHEIGHT);
}

void DoTileVideoScreen()
/*Tiles all the visualization windows within the full screen*/
{
    Tile(recScrWidth, recScrHeight);
}

static void RGBTask()
/*Task to change the current window to RGB*/
{
    if (selWinInfo)
    {
	selWinInfo -> flags |= WINRGB;

	SelWindow(selWinInfo);
	SetMode(selWinInfo);
	ImInvalid((ObjPtr) selWinInfo);
    }
}

static void CMapTask()
/*Task to change the current window to CMap mode*/
{
    if (selWinInfo)
    {
	selWinInfo -> flags &= ~WINRGB;

	SelWindow(selWinInfo);
	SetMode(selWinInfo);
	ImInvalid((ObjPtr) selWinInfo);
    }
}

static ObjPtr ChangeVisColorMode(radio)
ObjPtr radio;
/*Changes the color mode of the current visualization window according to
  the value of radio*/
{
    int rgbp;

    rgbp = GetInt(GetValue(radio));

    if (rgbp)
    {
	DoTask(RGBTask);
    }
    else
    {
	DoTask(CMapTask);
    }
    return ObjTrue;
}

ObjPtr ChangeShowGrid(checkBox)
ObjPtr checkBox;
/*Changes show grid according to a check box*/
{
    ObjPtr repObj;
    ObjPtr value;
    real hsv[3], rgb[3];

    repObj = GetObjectVar("ChangeShowGrid", checkBox, REPOBJ);
    if (!repObj)
    {
	return ObjFalse;
    }

    value = GetValue(checkBox);
    if (!value)
    {
	return ObjFalse;
    }

    SetVar(repObj, SHOWGRID, value);
    ImInvalid(repObj);

    return ObjTrue;
}

static ObjPtr ShowFrontSpacePanelControls(spacePanel, windowName)
ObjPtr spacePanel;
char *windowName;
/*Makes a new control window to control a space panel*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    real rgb[3], hsv[3];
    WinInfoPtr dialogExists;
    Bool hasBackground;

    dialogExists = DialogExists((WinInfoPtr) spacePanel, NewString("Controls"));
    controlWindow = GetDialog((WinInfoPtr) spacePanel, NewString("Controls"), windowName, 
	SPWINWIDTH, SPWINHEIGHT, SPWINWIDTH,
	SPWINHEIGHT, WINDBUF + WINRGB + WINFIXEDSIZE);
    
    if (!dialogExists)
    {
	long info;
	ObjPtr value;
	
	ObjPtr checkBox, icon, name, colorBar, titleBox, textBox, button;
	ObjPtr colorWheel, slider, radioGroup;
	int left, right, bottom, top;

	SetVar((ObjPtr) controlWindow, REPOBJ, spacePanel);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows controls for a space panel.  For information about any of the controls \
in the window, use Help In Context on the control.\n"));

	/*Add in a panel*/
	GetWindowBounds(&left, &right, &bottom, &top);
	panel = NewPanel(greyPanelClass, 0, right - left, 0, top - bottom);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);

	contents = GetVar(panel, CONTENTS);

	/*Add in the group of controls for text color*/
	left = MAJORBORDER;
	top = SPWINHEIGHT - MAJORBORDER;
	right = SPWINWIDTH - MAJORBORDER;
	bottom = SPWINHEIGHT - (MAJORBORDER + (COLORWHEELWIDTH + TEXTBOXSEP + TEXTBOXHEIGHT + CHECKBOXHEIGHT + TITLEBOXTOP + 3 * MINORBORDER))
;

	titleBox = NewTitleBox(left, right, bottom, top, "Background");
	PrefixList(contents, titleBox);
	SetVar(titleBox, PARENT, panel);

	left += MINORBORDER;
	right -= MINORBORDER;
	top -= MINORBORDER + TITLEBOXTOP;
	
	/*Get the color for priming the controls*/
	var = GetVar(spacePanel, BACKGROUND);
	if (var && IsRealArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	    hasBackground = true;
	}
	else if (var && IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	    hasBackground = true;
	}
	else
	{
	    rgb[0] = rgb[1] = rgb[2] = 0.0;
	    hasBackground = false;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Background Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	InhibitLogging(true);
	SetValue(colorWheel, var);
	InhibitLogging(false);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, spacePanel);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeBackgroundColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the background of the space panel.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Background Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, spacePanel);
	SetMethod(slider, CHANGEDVALUE, ChangeBackgroundColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the background of the space panel.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	/*Make the check box*/
	top -= COLORWHEELWIDTH + TEXTBOXSEP + TEXTBOXHEIGHT + MINORBORDER;
	checkBox = NewCheckBox(left, right, 
		top - CHECKBOXHEIGHT, top,
		"No background", hasBackground ? false : true);
	SetVar(checkBox, PARENT, panel);
	SetVar(checkBox, REPOBJ, spacePanel);
	SetVar(checkBox, SLIDER, slider);
	SetVar(checkBox, COLORWHEEL, colorWheel);
	PrefixList(contents, checkBox);
	SetMethod(checkBox, CHANGEDVALUE, ChangeNoBackground);
	SetVar(checkBox, HELPSTRING, NewString("This checkbox controls whether \
a background is shown.  If it is selected, no background is shown, and the \
objects behind can be seen."));

	/*Link slider and color wheel to check box*/
	SetVar(slider, CHECKBOX, checkBox);
	SetVar(colorWheel, CHECKBOX, checkBox);

	/*Make check box for show grid*/
	left = MAJORBORDER;
	top = MAJORBORDER + CHECKBOXHEIGHT;
	right = SPWINWIDTH - MAJORBORDER;
	bottom = MAJORBORDER;

	checkBox = NewCheckBox(left, right, bottom, top, "Show grid",
			GetPredicate(spacePanel, SHOWGRID));
	PrefixList(contents, checkBox);
	SetVar(checkBox, PARENT, panel);
	SetVar(checkBox, REPOBJ, spacePanel);
	SetMethod(checkBox, CHANGEDVALUE, ChangeShowGrid);
    }
    return (ObjPtr) controlWindow;
}

static ObjPtr ShowBackSpacePanelControls(spacePanel, windowName)
ObjPtr spacePanel;
char *windowName;
/*Makes a new control window to control a space panel*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    real rgb[3], hsv[3];
    WinInfoPtr dialogExists;
    Bool hasBackground;

    dialogExists = DialogExists((WinInfoPtr) spacePanel, NewString("Controls"));
    controlWindow = GetDialog((WinInfoPtr) spacePanel, NewString("Controls"), windowName, 
	SPWINWIDTH, SPWINHEIGHT, SPWINWIDTH,
	SPWINHEIGHT, WINDBUF + WINRGB + WINFIXEDSIZE);
    
    if (!dialogExists)
    {
	long info;
	ObjPtr value;
	
	ObjPtr checkBox, icon, name, colorBar, titleBox, textBox, button;
	ObjPtr colorWheel, slider, radioGroup;
	int left, right, bottom, top;

	SetVar((ObjPtr) controlWindow, REPOBJ, spacePanel);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows controls for a space panel.  For information about any of the controls \
in the window, use Help In Context on the control.\n"));

	/*Add in a panel*/
	GetWindowBounds(&left, &right, &bottom, &top);
	panel = NewPanel(greyPanelClass, 0, right - left, 0, top - bottom);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);

	contents = GetVar(panel, CONTENTS);

	/*Add in the group of controls for text color*/
	left = MAJORBORDER;
	top = SPWINHEIGHT - MAJORBORDER;
	right = SPWINWIDTH - MAJORBORDER;
	bottom = SPWINHEIGHT - (MAJORBORDER + (COLORWHEELWIDTH + TEXTBOXSEP + TEXTBOXHEIGHT + CHECKBOXHEIGHT + TITLEBOXTOP + 3 * MINORBORDER))
;

	titleBox = NewTitleBox(left, right, bottom, top, "Background");
	PrefixList(contents, titleBox);
	SetVar(titleBox, PARENT, panel);

	left += MINORBORDER;
	right -= MINORBORDER;
	top -= MINORBORDER + TITLEBOXTOP;
	
	/*Get the color for priming the controls*/
	var = GetVar(spacePanel, BACKGROUND);
	if (var && IsRealArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	    hasBackground = true;
	}
	else if (var && IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	    hasBackground = true;
	}
	else
	{
	    rgb[0] = rgb[1] = rgb[2] = 0.0;
	    hasBackground = false;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Background Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	InhibitLogging(true);
	SetValue(colorWheel, var);
	InhibitLogging(false);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, spacePanel);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeBackgroundColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the background of the space panel.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Background Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, spacePanel);
	SetMethod(slider, CHANGEDVALUE, ChangeBackgroundColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the background of the space panel.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	/*Make check box for show grid*/
	left = MAJORBORDER;
	top = MAJORBORDER + CHECKBOXHEIGHT;
	right = SPWINWIDTH - MAJORBORDER;
	bottom = MAJORBORDER;

	checkBox = NewCheckBox(left, right, bottom, top, "Show grid",
			GetPredicate(spacePanel, SHOWGRID));
	PrefixList(contents, checkBox);
	SetVar(checkBox, PARENT, panel);
	SetVar(checkBox, REPOBJ, spacePanel);
	SetMethod(checkBox, CHANGEDVALUE, ChangeShowGrid);
    }
    return (ObjPtr) controlWindow;
}

static ObjPtr ChangeSpaceTools(toolGroup)
ObjPtr toolGroup;
/*Changes a space's tools based on toolGroup*/
{
    ObjPtr value;
    ObjPtr space;

    value = GetValue(toolGroup);

    space = GetObjectVar("ChangeSpaceTools", toolGroup, SPACE);
    if (!space)
    {
	return ObjFalse;
    }
    SetVar(space, EDITTOOL, value);
    return ObjTrue;
}

WinInfoPtr NewVisWindow(title, flags)
char *title;
long flags;
/*Creates a new visualization window with title title and flags flags*/
{
    WinInfoPtr visWindow;
    ObjPtr space, panel, corral, contents, panelContents;
    ObjPtr button, radioGroup;
    int left, right, bottom, top;
    int l, r, b, t;
    ObjPtr lights;
    ThingListPtr runner;

    visWindow = NewObjWindow(visWindowClass, title, flags, VWINWIDTH, VWINHEIGHT, SCRWIDTH, SCRHEIGHT);

    SetMinSize(visWindow, 100, 100);

    SetVar((ObjPtr) visWindow, HIDEPANEL, ObjTrue);
    DefineMenuItem((ObjPtr) visWindow, WINDOWSMENU, "Show Front Panel Controls", DoShowFrontPanelControls);
    DefineMenuItem((ObjPtr) visWindow, WINDOWSMENU, "Show Back Panel Controls", DoShowBackPanelControls);
    DefineMenuItem((ObjPtr) visWindow, WINDOWSMENU, "Show Space Controls", DoShowSpaceControls);
    DefineMenuItem((ObjPtr) visWindow, WINSIZEMENU, "Tile Full Screen", DoTileFullScreen);
    DefineMenuItem((ObjPtr) visWindow, WINSIZEMENU, "Tile Video Screen", DoTileVideoScreen);
    DefineMenuItem((ObjPtr) visWindow, WINSIZEMENU, "PHSCologram Screen", DoPhscoScreen);
    DefineMenuItem((ObjPtr) visWindow, TEXTMENU, "", 0);

    GetWindowBounds(&left, &right, &bottom, &top);

    contents = GetVar((ObjPtr) visWindow, CONTENTS);

    /*Add in the panel behind the space*/
    panel = NewPanel(spaceBackPanelClass, left + VWINTOOLPWIDTH + 1, right - VWINPANELWIDTH - 1, bottom, top);
    if (!panel)
    {
	return (WinInfoPtr) 0;
    }
    SetVar(panel, NAME, ConcatStrings(NewString(title), NewString(" Back Panel")));
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("This panel holds 2-dimensional objects \
and is always drawn behind the space."));
    SetVar(panel, OWNERWINDOW, (ObjPtr) visWindow);
#ifdef INTERACTIVE
    SetMethod(panel, PRESS, PressSpacePanel);
#endif
    PrefixList(contents, panel);
    
    /*Add in a space*/
    space = NewSpace(left + VWINTOOLPWIDTH + 1, right - VWINPANELWIDTH - 1, bottom, top);
    if (!space)
    {
	return (WinInfoPtr) 0;
    }
    SetVar(space, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(space, NAME, ConcatStrings(NewString(title), NewString(" Space")));
    SetVar(space, OWNERWINDOW, (ObjPtr) visWindow);
    PrefixList(contents, space);
    SetVar(space, PARENT, (ObjPtr) visWindow);

    /*Notify the space that the panel is behind the space*/
    SetVar(panel, SPACE, space);
    SetVar(space, BACKPANEL, panel);
    SetVar(panel, PARENT, space);
    SetMethod(panel, NEWCTLWINDOW, ShowBackSpacePanelControls);
    SetMethod(panel, SHOWCONTROLS, NewControlWindow);

    /*Add in the panel to cover the space*/
    panel = NewPanel(spacePanelClass, left + VWINTOOLPWIDTH + 1, right - VWINPANELWIDTH - 1, bottom, top);
    if (!panel)
    {
	return (WinInfoPtr) 0;
    }
    SetVar(panel, NAME, ConcatStrings(NewString(title), NewString(" Front Panel")));
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("This panel holds 2-dimensional objects \
and is always drawn in front of the space."));
    SetVar(panel, OWNERWINDOW, (ObjPtr) visWindow);
    SetVar(panel, SPACE, space);
#ifdef INTERACTIVE
    SetMethod(panel, PRESS, PressSpacePanel);
#endif
    PrefixList(contents, panel);
    SetVar(panel, PARENT, space);

    /*Notify the space that the panel covers the space*/
    SetVar(space, FRONTPANEL, panel);
    SetVar(panel, FRONTPANEL, ObjTrue);
    SetMethod(panel, NEWCTLWINDOW, ShowFrontSpacePanelControls);
    SetMethod(panel, SHOWCONTROLS, NewControlWindow);

    /*Add in a tools panel*/
    panel = NewPanel(greyPanelClass, left, left + VWINTOOLPWIDTH, bottom, top);
    if (!panel)
    {
	return (WinInfoPtr) 0;
    }
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYBOTTOM + STICKYTOP));
    PrefixList(contents, panel);
    SetVar(panel, PARENT, (ObjPtr) visWindow);
    SetVar(space, TOOLPANEL, panel);
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("You can show or hide this panel by choosing the Toggle Panel \
item from the Windows menu."));
    panelContents = GetVar(panel, CONTENTS);

    /*Put tools in the space tools group*/
    radioGroup = NewRadioButtonGroup("Space Tools");
    SetVar(space, TOOLGROUP, radioGroup);
    SetVar(radioGroup, HELPSTRING,
	   NewString("This is a group of tools that allow you to modify the \
space using the mouse.  To use one of these tools, click on the icon representing \
the tool before clicking in the space.  To find out what each tool does, use \
help in context on each individual button."));
    SetVar(radioGroup, PARENT, panel);
    PrefixList(panelContents, radioGroup);
    SetVar(radioGroup, HALTHELP, ObjTrue);
    SetVar(radioGroup, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));

    /*Motion/rotation tool*/
    l = left + VWTOOLBORDER;
    t = top - VWTOOLBORDER;
    button = NewIconButton(l, l + SMALLICONBUTTONSIZE,
			t - SMALLICONBUTTONSIZE, t,
			ICONMOVEROTATE, UIYELLOW, "Move/Rotate Space", BS_PLAIN);
    SetVar(button, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
    AddRadioButton(radioGroup, button);
    SetVar(button, HELPSTRING, NewString("This button selects the space rotation and \
motion tool.  When this tool is selected, you can rotate and move the entire space \
using the mouse.\n\
\n\
Click and drag within the space using the left \
mouse button to move all the objects within the space.  Hold down the Shift \
key while dragging to contrain motion to an orthogonal axis of the space.  \
If Space Motion Guides is selected in the Preferences window, you will see the \
objects as embedded within a gray lattice showing the orthogonal axes.\n\
\n\
Click and drag within the space ucing the center mouse button to rotate \
the entire space around the focus point of the observer at the center of the \
space.  The space can be rotated around any axis.  Hold down the Shift key \
to constrain to rotation around an orthogonal axis of the space.  Double-click \
to snap the space to the nearest straight-on orientation.  \
If Space Rotation Guides is selected in the Preferences window, you will see the \
virtual trackball that you are rotation as a wire frame sphere.  If Rotation Inertia \
is selected in the Preferences window, the space will continue to rotate if you \
give it a spin and let go while moving the mouse.\n\
\n\
Both motion and rotation will affect other spaces controlled by the same observer."));

    t = t - SMALLICONBUTTONSIZE - VWTOOLBORDER;
    button = NewIconButton(l, l + SMALLICONBUTTONSIZE,
			t - SMALLICONBUTTONSIZE, t,
			ICONFINGER, UIYELLOW, "Panel Selection", BS_PLAIN);
    SetVar(button, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
    AddRadioButton(radioGroup, button);
    SetVar(button, HELPSTRING, NewString("This button selects the space panel selection tool.\
Use this tool to select and move objects on the front or back panels of the space."));

    t = t - SMALLICONBUTTONSIZE - VWTOOLBORDER;
    button = NewIconButton(l, l + SMALLICONBUTTONSIZE,
			t - SMALLICONBUTTONSIZE, t,
			ICONANNOTATION, UIYELLOW, "Draw Annotation", BS_PLAIN);
    SetVar(button, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
    AddRadioButton(radioGroup, button);
    SetVar(button, HELPSTRING, NewString("This button selects the annotation drawing tool.  \
Use this button to drag out annotation boxes on the panels of the space.  \
Normally this tool draws on the front panel.  Hold down the Alt or Ctrl key while drawing to draw \
on the back panel.  When the box has been drawn, enter text by typing.  The appearance of the boxes can be modified by selecting them and choosing Show Controls \
from the Objects menu."));

    t = t - SMALLICONBUTTONSIZE - VWTOOLBORDER;
    button = NewIconButton(l, l + SMALLICONBUTTONSIZE,
			t - SMALLICONBUTTONSIZE, t,
			ICONRECTANGLE, UIYELLOW, "Draw Rectangle", BS_PLAIN);
    SetVar(button, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
    AddRadioButton(radioGroup, button);
    SetVar(button, HELPSTRING, NewString("This button selects the rectangle drawing tool.  \
Use this button to drag out rectangular boxes on the panels of the space.  \
Normally this tool draws on the front panel.  Hold down the Alt or Ctrl key while drawing to draw \
on the back panel.  The appearance of the boxes can be modified by selecting them and choosing Show Controls \
from the Objects menu."));

    t = t - SMALLICONBUTTONSIZE - VWTOOLBORDER;
    button = NewIconButton(l, l + SMALLICONBUTTONSIZE,
			t - SMALLICONBUTTONSIZE, t,
			ICONLINE, UIYELLOW, "Draw Line", BS_PLAIN);
    SetVar(button, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
    AddRadioButton(radioGroup, button);
    SetVar(button, HELPSTRING, NewString("This button selects the line drawing tool.  \
Use this button to drag out lines on the panels of the space.  \
Normally this tool draws on the front panel.  Hold down the Alt or Ctrl key while drawing to draw \
on the back panel.  The appearance of the lines can be modified by selecting them and choosing Show Controls \
from the Objects menu.  You can also add arrowheads from within this control panel."));

    t = t - SMALLICONBUTTONSIZE - VWTOOLBORDER;
    button = NewIconButton(l, l + SMALLICONBUTTONSIZE,
			t - SMALLICONBUTTONSIZE, t,
			ICONCLOCK, UIYELLOW, "Draw Time Readout", BS_PLAIN);
    SetVar(button, STICKINESS, NewInt(STICKYLEFT + STICKYTOP));
    AddRadioButton(radioGroup, button);
    SetVar(button, HELPSTRING, NewString("This button selects the time readout drawing tool.  \
Use this button to drag out time readouts on the panels of the space.  The time readout is \
a text box which always shows the current time in the space.  To change the \
format in which the time is displayed, show the control panel by \
selecting the readout and choosing Show Controls from the Objects menu.  \
Normally this tool draws on the front panel.  Hold down the Alt or Ctrl key while drawing to draw \
on the back panel."));

    /*Give the radio group a default value*/
    SetValue(radioGroup, NewInt(ST_MOVEROTATE));
    SetVar(space, EDITTOOL, NewInt(ST_MOVEROTATE));
    SetVar(radioGroup, SPACE, space);
    SetMethod(radioGroup, CHANGEDVALUE, ChangeSpaceTools);

    /*Add in an object control panel*/
    panel = NewPanel(greyPanelClass, right - VWINPANELWIDTH, right, bottom, top);
    if (!panel)
    {
	return (WinInfoPtr) 0;
    }
    SetVar(panel, STICKINESS, NewInt(STICKYRIGHT + STICKYBOTTOM + STICKYTOP));
    PrefixList(contents, panel);
    SetVar(panel, PARENT, (ObjPtr) visWindow);
    SetVar(space, OBJPANEL, panel);
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("You can show or hide this panel by choosing the Toggle Panel \
item from the Windows menu."));

    contents = GetVar(panel, CONTENTS);
    if (!contents)
    {
	return (WinInfoPtr) 0;
    }

    /*Add in an icon corral*/
    corral = NewIconCorral(visCorralClass,
		MINORBORDER, VWINPANELWIDTH - MINORBORDER, 
		bottom + MINORBORDER + MAJORBORDER + CHECKBOXHEIGHT,
		top - MINORBORDER,
		BARRIGHT + BARBOTTOM);

    if (!corral)
    {
	return (WinInfoPtr) 0;
    }
    SetVar((ObjPtr) visWindow, CORRAL, corral);

    SetVar(corral, NAME, NewString("Space Contents"));
    SetVar(corral, HELPSTRING,
	NewString("This corral shows the objects and space controllers present \
in this visualization window.  You can drag new datasets from the datasets window \
into this corral to visualize them along with the other objects.  You can also \
drag in observer, renderer, clock, and lights icons.  There can only be one \
observer, renderer, and clock in a space, so dragging in these will replace the \
existing controller.  This will probably change soon.\n"));
    SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    PrefixList(contents, corral);
    SetVar(corral, PARENT, panel);
    SetVar(corral, SPACE, space);
    SetVar(space, CORRAL, corral);

    {
	ObjPtr radio, button;
	/*Make some radio buttons for RGB versus color map mode*/
	radio = NewRadioButtonGroup("Color Mode");
	SetVar(radio, HELPSTRING,
		NewString("These radio buttons control the color mode of the \
visualization window.  Full color mode is more useful for most visualization \
techniques.  Color map mode can be used on computers that do not have enough \
bit planes to do full color well.  It also provides a different way of doing \
interpolation on smooth color shaded visualizations, which is sometimes useful."));
	PrefixList(contents, radio);
	SetVar(radio, PARENT, panel);

	button = NewRadioButton(
		MINORBORDER, VWINPANELWIDTH / 2,
		MINORBORDER, MINORBORDER + CHECKBOXHEIGHT,
		"Color Map");
	SetVar(button, HELPSTRING,
	    NewString("This button puts the window into color map color mode.  \
This prevents lighting and image filtering from working, but it can be useful \
on simple color shaded visualizations.  Color map mode only works on hardware with \
color maps."));
	AddRadioButton(radio, button);
#ifdef GRAPHICS
	if (!hasCmap)
	{
	    ActivateButton(button, false);
	}
#endif

	button = NewRadioButton(
		VWINPANELWIDTH / 2, VWINPANELWIDTH - MINORBORDER,
		MINORBORDER, MINORBORDER + CHECKBOXHEIGHT,
		"Full Color");
	SetVar(button, HELPSTRING,
	    NewString("This button puts the window into full color mode.  \
In full color mode, all the lighting and filtering is available.  Full color \
mode only works if the hardware has enough bitplanes to support it."));
	AddRadioButton(radio, button);
#ifdef GRAPHICS
	if (!hasRGB)
	{
	    ActivateButton(button, false);
	}
#endif

	InhibitLogging(true);
	SetValue(radio, NewInt((flags & WINRGB) && hasRGB ? 1 : 0));
	InhibitLogging(false);
	SetVar((ObjPtr) visWindow, CMODERADIO, radio);
	SetMethod(radio, CHANGEDVALUE, ChangeVisColorMode);
    }
    /*Add an observer*/
    AddControllerToSpace(NewObserver(), space, corral, NULLOBJ);

    /*Add a renderer*/
    AddControllerToSpace(NewRenderer(), space, corral, NULLOBJ);

    /*Add some lights*/
#ifdef GRAPHICS
    if (hasRGB)
#endif
    {
	lights = NewLights();
	runner = LISTOF(lights);
	while (runner)
	{
	    AddControllerToSpace(runner -> thing, space, corral, NULLOBJ);
	    runner = runner -> next;
	}
    }

    /*Add a clock*/
    AddControllerToSpace(NewClock(), space, corral, NULLOBJ);

    return visWindow;
}

ObjPtr FindSpace(win)
WinInfoPtr win;
/*Finds the space in win*/
{
    ThingListPtr contents;

    /*Search for a space in win*/
    contents = LISTOF(GetVar((ObjPtr) win, CONTENTS));
    while (contents)
    {
	if (IsSpace(contents -> thing))
	{
	    return contents -> thing;
	}
	contents = contents -> next;
    }
    return NULLOBJ;
}

ObjPtr FindObserver(win)
WinInfoPtr win;
/*Finds the observer in win*/
{
    ObjPtr space, retVal;
    retVal = GetVar((ObjPtr) win, REPOBJ);
    if (retVal && (IsObserver(retVal)))
    {
	return retVal;
    }
    space = FindSpace(win);
    if (space)
    {
	retVal = GetObjectVar("FindObserver", space, OBSERVER);
    }
    else
    {
	ReportError("FindObserver", "No space in window");
    }
    return retVal;
}

#ifdef PROTO
void RotateWindowTo(WinInfoPtr window, Matrix r)
#else
void RotateWindowTo(window, r)
WinInfoPtr window;
Matrix r;
#endif
/*Rotates the space in window to rotation matrix given by r*/
{
    ObjPtr xformMatrix;
    ObjPtr observer;

    observer = FindObserver(window);
    if (!observer)
    {
	ReportError("RotateWindowTo", "No observer for window");
	return;
    }
	xformMatrix = NewMatrix();
	MATCOPY(MATRIXOF(xformMatrix), r);
	SetVar(observer, XFORM, xformMatrix);

	ResolveController(observer);
	if (logging)
	{
	    char cmd[256];
	    MatrixToText(tempStr, xformMatrix);
	    sprintf(cmd, "rotate to %s\n", tempStr);
	    Log(cmd);
	}
}

#ifdef PROTO
void RotateOrthoWindow(WinInfoPtr window, char axis, float amount)
#else
void RotateOrthoWindow(window, axis, amount)
WinInfoPtr window;
char axis;
float amount;
#endif
/*Rotates the space in window about orthogonal axis axis by amount, and 
  stops continuous rotation.*/
{
    ObjPtr observer;
    ObjPtr xformMatrix;
    Matrix newXform;
    real av[3];

    observer = FindObserver(window);
    if (!observer) return;

    /*Get the rotation matrix*/
    xformMatrix = (ObjPtr) GetVar(observer, XFORM);
    if (!xformMatrix || !IsMatrix(xformMatrix))
    {
	xformMatrix = NewMatrix();
	SetVar(observer, XFORM, xformMatrix);
    }
    MATCOPY(newXform, MATRIXOF(xformMatrix));

    switch(axis)
    {
	case 'x':
	case 'X':
	    av[0] = newXform[0][0];
	    av[1] = newXform[0][1];
	    av[2] = newXform[0][2];
	    break;
	case 'y':
	case 'Y':
	    av[0] = newXform[1][0];
	    av[1] = newXform[1][1];
	    av[2] = newXform[1][2];
	    break;
	case 'z':
	case 'Z':
	    av[0] = newXform[2][0];
	    av[1] = newXform[2][1];
	    av[2] = newXform[2][2];
	    break;
	default:
	    ReportError("RotateOrthoWindow", "Bad axis");
	    return;
    }

    RotateObserver(observer, av, amount * M_PI / 180.0, false);
}

#ifdef PROTO
void ShearOrthoWindow(WinInfoPtr window, float amount)
#else
void ShearOrthoWindow(window, amount)
WinInfoPtr window;
float amount;
#endif
/*Shears the space in window by amount, and 
  stops continuous rotation.*/
{
    ObjPtr space;

#ifdef GRAPHICS
    space = FindSpace(window);
    if (space)
    {
	/*Get the rotation matrix*/
	Matrix shearMatrix;
	ObjPtr matObj;

	MATCOPY(shearMatrix, Identity);
	shearMatrix[2][1] = amount;
	matObj = NewMatrix();
	MATCOPY(MATRIXOF(matObj), shearMatrix);
	SetVar(space, SHEAR, matObj);
	ImInvalid(space);
    }
#endif
}

static ObjPtr ChangeWindowToRGB(window, button)
WinInfoPtr window;
int button;
/*If button is 1, changes the color mode of window to RGB*/
{
    if (button == 1)
    {
	ObjPtr radio = NULLOBJ;
	SelWindow(window);
	if (selWinInfo)
	{
	    radio = GetVar((ObjPtr) selWinInfo, CMODERADIO);
	}
	if (radio)
	{
	    SetValue(radio, NewInt(1));
	}
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

ObjPtr SetVisWindowRGB (window)
ObjPtr window;
/*Deferred message to choose whether to make window RGB or not*/
{
#ifdef GRAPHICS
    char tempBuf[400];
    WinInfoPtr errWindow;
    if (!hasRGB) return;

    sprintf(tempBuf, "You will see no effects from this action in window %s \
until you change it to full color mode.  Would you like to change it now?",
	((WinInfoPtr) window) -> winTitle);

    errWindow = AlertUser(UICAUTIONALERT, (WinInfoPtr) window, tempBuf,
	ChangeWindowToRGB, 2, "Cancel", "Change");
    SetVar((ObjPtr) errWindow, HELPSTRING,
	NewString("Visualization windows have two color modes: full color and color map.  \
Full color specifies full 24-bit RGB color.  Color map specifies mapped colors using a color \
table.  Some features, such as lighting, only work on windows set to full color mode.  \
If you click on the Change button, the affected window will automatically be set \
to this mode."));
#endif
}

void InitVisWindows()
/*Initializes the vis windows*/
{
    visWindowClass = NewObject(objWindowClass, 0);
    AddToReferenceList(visWindowClass);
    SetVar(visWindowClass, CLASSID, NewInt(CLASS_VISWINDOW));
    SetVar(visWindowClass, HELPSTRING,
	NewString("This is a visualizaton window.  On the right is a panel which \
shows the visualization objects shown in the window, the controllers which \
affect the appearance of the object, and some buttons which affect the window \
as a whole or the selected icons.  If this panel is not shown, it can be brought \
back with the Toggle Panel item in the Window menu.  On the left is a \
3-dimensional space in which \
the visualization object are shown.  There is one clear panel in front of the and \
one clear panel behind the space.  These panels are used to hold 2-dimensional \
objects such as clock readouts and text boxes, which can be created and modified \
with items in the Text menu.")); 
    SetMethod(visWindowClass, SETRGBMESSAGE, SetVisWindowRGB);
    SetMethod(visWindowClass, HIDEPANEL, HideVisWindowPanel);
    SetMethod(visWindowClass, SHOWPANEL, ShowVisWindowPanel);

    /*Class for the corral in a vis window*/
    visCorralClass = NewObject(corralClass, 0);
    AddToReferenceList(visCorralClass);
    SetMethod(visCorralClass, DROPINCONTENTS, DropInVisCorral);
}

void KillVisWindows()
/*Kills the vis windows*/
{
    DeleteThing(visCorralClass);
    DeleteThing(visWindowClass);
}
