/*	ScianTextBoxes.c
	Routines for text boxes

	J. Lyons
	9/28/90

	3/26/91	added simple editing
	3/28/91 added routines to set font, size, line spacing
	3/29/91 added provision for different style text boxes
	4/19/91 added one-line editing 
	5/13/91 EMP moved an #ifdef FONTS4D around a fontHandle
	5/29/91 EMP removed lib headers
	7/17/91 added multiline editing
	7/22/91 added incredibly fancy frame for adjustable text boxes
	8/12/91 modified to use ScianFontSystem routines
	8/16/91 modified to use Set/Get2DIntBounds
	8/27/91 added ACTIVATED attribute
	8/28/91 added CUT, COPY, PASTE methods
	9/5/91	added TextHeight routine
	9/6/91	added help
	10/30/91 added shift-clicking to press method
	11/1/91 added double-clicking to press method
	11/4/91 added control window for textboxes (with EMP)
	1/8/92	added indentation control
	1/20/92	fixed linespace problem for large sizes
	1/22/92	made indentation behavior more like tabs, sort of
*/

#include "Scian.h"
#include "ScianStyle.h"
#include "ScianTypes.h"
#include "ScianFontSystem.h"
#include "ScianWindows.h"
#include "ScianArrays.h"
#include "ScianDraw.h"
#include "ScianColors.h"
#include "ScianIDs.h"
#include "ScianErrors.h"
#include "ScianEvents.h"
#include "ScianScripts.h"
#include "ScianMethods.h"
#include "ScianControls.h"
#include "ScianHelp.h"
#include "ScianDialogs.h" /* should really be ScianParts.h */
#include "ScianTitleBoxes.h"
#include "ScianTextBoxes.h"
#include "ScianSliders.h"
#include "ScianButtons.h"

/* #define NEWWAY */

#define MAXLINE		255
#define MAXNLINES	20
#define MAXTEXTSIZE	96

/*** put in ScianStyle.h ***/
#define PLAIN		0
#define TEXTBOXCOLOR	UIPINK
#ifndef INDENTAMT
#define INDENTAMT	5	/* number of 'chars' to indent */
#endif


/* parameters for incredibly fancy frame for moving and resizing text boxes */
#define HANDLESIZE		12
#define INSIDEFRAMEWEIGHT	4
#define OUTSIDEFRAMEWEIGHT	2
#define INSET ((HANDLESIZE - (2*OUTSIDEFRAMEWEIGHT + INSIDEFRAMEWEIGHT))/2)
#define	OUTSIDEFRAMECOLOR	UIWHITE
#define	INSIDEFRAMECOLOR	UIBLACK

/* static function prototypes */
#ifdef PROTO

static ObjPtr MakeHelpString(ObjPtr, ObjPtr);
static ObjPtr SetTBVal(ObjPtr, ObjPtr);
static void NewText(ObjPtr);
static ObjPtr DrawTextBox(ObjPtr);
static ObjPtr PressTextBox(ObjPtr, int, int, long);
static ObjPtr EditTextBox(ObjPtr, short, long);
static ObjPtr ShowTextBoxControls(ObjPtr, WinInfoPtr, char *);
static ObjPtr SelectTextBox(ObjPtr, Bool);
static ObjPtr TextBoxNotCurrent(ObjPtr);
static ObjPtr CutText(ObjPtr);
static ObjPtr CopyText(ObjPtr);
static ObjPtr PasteText(ObjPtr, ObjPtr);
static char *NextLine(char *, int, char *);

#else

static ObjPtr MakeHelpString();
static ObjPtr SetTBVal();
static void NewText();
static ObjPtr DrawTextBox();
static ObjPtr PressTextBox();
static ObjPtr EditTextBox();
static ObjPtr ShowTextBoxControls();
static ObjPtr SelectTextBox();
static ObjPtr TextBoxNotCurrent();
static ObjPtr CutText();
static ObjPtr CopyText();
static ObjPtr PasteText();
static char *NextLine();

#endif


/*** remove when moved to ScianDraw.c ***/
void DrawPitEdge(int, int, int,int);

/*********************************************************************** GLOBALS */
ObjPtr textBoxClass;
static char *textBuf; /* pointer to buffer for drawing and editing text */
static int textBufSize; /* current size of the buffer */

#define TEXTBUFFERSIZE	1000
#define TEXTBUFFERINCR	100

static int indent;	/* number of pixels to indent text from left bound */

void InitTextBoxes()
{
	textBoxClass = NewObject(controlClass, 0);
	AddToReferenceList(textBoxClass);
	SetVar(textBoxClass, NAME, NewString("Text Box"));
	SetVar(textBoxClass, TYPESTRING, NewString("text box"));
	SetMethod(textBoxClass, DRAW, DrawTextBox);
	SetMethod(textBoxClass, SETVAL, SetTBVal);
	SetMethod(textBoxClass, PRESS, PressTextBox);
	SetMethod(textBoxClass, KEYDOWN, EditTextBox);
	SetMethod(textBoxClass, CUT, CutText);
	SetMethod(textBoxClass, COPY, CopyText);
	SetMethod(textBoxClass, PASTE, PasteText);
	SetMethod(textBoxClass, SELECTALL, SelectAll);
	SetMethod(textBoxClass, MAKE1HELPSTRING, MakeHelpString);
	SetMethod(textBoxClass, NEWCTLWINDOW, ShowTextBoxControls);
	SetMethod(textBoxClass, SELECT, SelectTextBox);
	SetMethod(textBoxClass, YOURENOTCURRENT, TextBoxNotCurrent);

	/* get a buffer for drawing and editing text */
	textBuf = (char *) malloc(textBufSize = TEXTBUFFERSIZE);
	if (!textBuf)
	{
		OMErr();
		textBufSize = 0;
	}
}

void KillTextBoxes()
{
	DeleteThing(textBoxClass);
	if (textBuf) free(textBuf);
}

#ifdef PROTO
ObjPtr NewTextBox(int left, int right, int bottom , int top, 
					int style, char *name, char *text)
#else
ObjPtr NewTextBox(left, right, bottom, top, style, name, text)
int left, right, bottom, top, style;
char *name, *text;
#endif
{
	ObjPtr newBox;

	if (left > right)
	{
		register int n;
		n = left; left = right; right = n;
	}
	if (bottom > top)
	{
		register int n;
		n = bottom; bottom = top; top = n;
	}
	newBox = NewObject(textBoxClass, 0);
	Set2DIntBounds(newBox, left, right, bottom, top);
	SetVar(newBox, ACTIVATED, NewInt(true));
	SetVar(newBox, STYLE, NewInt(style));
	SetVar(newBox, NAME, NewString(name));
	SetVar(newBox, VALUE, NewString(text));
	SetVar(newBox, CHANGED, ObjTrue); /* flag to line-break text before drawing first time */
	if (style & EDITABLE)
	{
		ObjPtr theLen = NewInt(strlen(text));
		SetVar(newBox, BGNSEL, theLen);
		SetVar(newBox, ENDSEL, theLen);
	}
	return newBox;
}

#ifdef PROTO
static ObjPtr MakeHelpString(ObjPtr textBox, ObjPtr theClass)
#else
static ObjPtr MakeHelpString(textBox, theClass)
ObjPtr textBox, theClass;
#endif
{
	int style = GetInt(GetVar(textBox, STYLE));

	*textBuf = '\0';

	if (style & EDITABLE) strcpy(textBuf,"This is an editable text box. \
Click the left mouse button anywhere in the text to place the insertion point \
for adding new text. Drag through text to select it for editing. ");

	if (style & ADJUSTABLE) strcat(textBuf, "To change the size or location \
of this text box, first click anywhere in the text to select it. A frame will \
appear around the text with eight small handles. Drag any of the handles to change \
the size of the text box. Drag the frame itself to reposition the text.");

	SetVar(theClass, HELPSTRING, *textBuf ? NewString(textBuf) : NULLOBJ);
	return ObjTrue;
}

#ifdef PROTO
void ActivateTextBox(ObjPtr textBox, Bool act)
#else
void ActivateTextBox(textBox, act)
ObjPtr textBox;
Bool act;
#endif
{
	if (act) SetVar(textBox, ACTIVATED, ObjTrue);
	else
	{
		SetVar(textBox, ACTIVATED, ObjFalse);
		MakeMeCurrent(NULLOBJ);
	}
	ImInvalid(textBox);
}

#ifdef PROTO
static ObjPtr SetTBVal(ObjPtr textBox, ObjPtr theText)
#else
static ObjPtr SetTBVal(textBox, theText)
ObjPtr textBox, theText;
#endif
{
	Bool deferChange;

	if (IsString(theText)) SetTextBox(textBox, GetString(theText));
	else if (IsInt(theText))
	{
		sprintf(tempStr, "%d", GetInt(theText));
		SetTextBox(textBox, tempStr);
	}
	else if (IsReal(theText))
	{
		sprintf(tempStr, "%g", GetReal(theText));
		SetTextBox(textBox, tempStr);
	}
	else return ObjFalse;

	return ObjTrue;
}

/*********************************************************************** SET TEXT ROUTINES */
#ifdef PROTO
ObjPtr SetTextBox(ObjPtr textBox, char *text)
#else
ObjPtr SetTextBox(textBox, text)
ObjPtr textBox;
char *text;
#endif
{
	SetVar(textBox, VALUE, NewString(text));
	if (GetInt(GetVar(textBox, STYLE)) & EDITABLE)
	{
		ObjPtr theLen = NewInt(strlen(text));
		SetVar(textBox, BGNSEL, theLen);
		SetVar(textBox, ENDSEL, theLen);
	}
	SetVar(textBox, CHANGED, ObjTrue); 
	ImInvalid(textBox);
	ChangedValue(textBox);
	if (logging) LogControl(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextFont(ObjPtr textBox, char *fontName)
#else
ObjPtr SetTextFont(textBox, fontName)
ObjPtr textBox;
char *fontName;
#endif
{
	SetVar(textBox, TEXTFONT, NewString(fontName));
	SetVar(textBox, CHANGED, ObjTrue); 
	ImInvalid(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextSize(ObjPtr textBox, int textSize)
#else
ObjPtr SetTextSize(textBox, textSize)
ObjPtr textBox;
int textSize;
#endif
{
	if (textSize > 0 && textSize <= MAXTEXTSIZE)
		SetVar(textBox, TEXTSIZE, NewInt(textSize));
	SetVar(textBox, CHANGED, ObjTrue); 
	ImInvalid(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextLineSpace(ObjPtr textBox, int lineSpace)
#else
ObjPtr SetTextSize(textBox, lineSpace)
ObjPtr textBox;
int lineSpace;
#endif
{
	if (lineSpace > 0)
		SetVar(textBox, LINESPACE, NewInt(lineSpace));
	ImInvalid(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextAlign(ObjPtr textBox, int value)
#else
ObjPtr SetTextAlign(textBox, value)
ObjPtr textBox;
int value;
#endif
{
	SetVar(textBox, ALIGNMENT, NewInt(value));
	ImInvalid(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextColor(ObjPtr textBox, ObjPtr color)
#else
ObjPtr SetTextColor(textBox, color)
ObjPtr textBox, color;
#endif
{
	SetVar(textBox, COLOR, color);
	ImInvalid(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextBGColor(ObjPtr textBox, ObjPtr color)
#else
ObjPtr SetTextBGColor(textBox, color)
ObjPtr textBox, color;
#endif
{
	SetVar(textBox, BACKGROUND, color);
	ImInvalid(textBox);
	return NULLOBJ;
}

#ifdef PROTO
ObjPtr SetTextBoxStyle(ObjPtr textBox, int style)
#else
ObjPtr SetTextBoxStyle(textBox, style)
ObjPtr textBox;
int style;
#endif
{
	SetVar(textBox, STYLE, NewInt(style));
	SetVar(textBox, CHANGED, ObjTrue); 
	ImInvalid(textBox);
	return NULLOBJ;
}

/*********************************************************** NEW TEXT */
#ifdef PROTO
static void NewText(ObjPtr textBox)
#else
static void NewText(textBox)
ObjPtr textBox;
#endif
{
	/* puts "soft" eols (\r is used) into the text as needed */

	int left, right, bottom, top;
	int textLeft, textRight;
	ObjPtr theObj;
	char *text, *nxtLine, *bgnLine, *s, *t, *fontName, lineBuf[MAXLINE];
	int len, width, size, style;

	Get2DIntBounds(textBox, &left, &right, &bottom, &top);

	text = GetString(GetVar(textBox, VALUE));

	theObj = GetVar(textBox, STYLE);
	if (theObj) style = GetInt(theObj);
	else style = PLAIN;

	if (style & ONE_LINE)
	{
		SetVar(textBox, VALUE, NewString(text));
		return;
	}

	theObj = GetVar(textBox, TEXTFONT);
	if (theObj) fontName = GetString(theObj);
	else fontName = DEFFONT;

	theObj = GetVar(textBox, TEXTSIZE);
	if (theObj) size = GetInt(theObj);
	else size = DEFTEXTSIZE;

	if (style & WITH_PIT)
	{
		textLeft = left + TEXTMARGIN;
		textRight = right - TEXTMARGIN;
	}
	else if (style & (EDITABLE | ADJUSTABLE) || GetVar(textBox, BACKGROUND))
	{
		textLeft = left + HANDLESIZE + 2;
		textRight = right - HANDLESIZE - 2;
	}
	else /* default is no inset */
	{
		textLeft = left;
		textRight = right;
	}

	/*** CHANGE FOR INDENTATION ***/
	SetupFont(fontName, size);
	width = textRight - textLeft;
	nxtLine = text; /* set up for loop */
	t = textBuf;
	do {
		if (t - textBuf + MAXLINE > textBufSize) /* increase buffer size */
		{
			int tIndex = t - textBuf; /* remember position */
			textBuf = (char *) realloc(textBuf, textBufSize + TEXTBUFFERINCR);
			if (!textBuf) /* failed */
			{
				OMErr();
				textBufSize = 0;
				return;
			}
			t = tIndex + textBuf; /* reset pointer in new buffer */
		}
		bgnLine = nxtLine;
		nxtLine = NextLine(bgnLine, width, s=lineBuf); /* Note: NextLine ignores existing soft CRs */
		while (*t++ = *s++) ; /* copy line into textBuf */
		--t; /* back up to terminator */
		if (*nxtLine == '\n') ++nxtLine, *t++ = '\n';	/* paragraph */
		else *t++ = '\r'; /* soft eol */
	} while (*nxtLine);
	*t = '\0'; /* stuff terminator */
	SetVar(textBox, VALUE, NewString(textBuf));
	SetVar(textBox, CHANGED, ObjFalse);
	/* calling routine will handle draw, changed value */
}

/*********************************************************** TEXT HEIGHT */
#ifdef PROTO
int TextHeight(ObjPtr textBox)
#else
int TextHeight(textBox)
ObjPtr textBox;
#endif
{
#ifndef NEWWAY
	int left, right, bottom, top;
	int textLeft, textRight, textTop;
	ObjPtr theObj;
	char *text, *nxtLine, *bgnLine, lineBuf[MAXLINE], *fontName;
	int y, width, size, spWid, lineSpace, style;

	Get2DIntBounds(textBox, &left, &right, &bottom, &top);

	theObj = GetVar(textBox, VALUE);
	if (theObj) text = GetString(theObj);
	else text = "\0";
		
	theObj = GetVar(textBox, STYLE);
	if (theObj) style = GetInt(theObj);
	else style = PLAIN;

	theObj = GetVar(textBox, TEXTFONT);
	if (theObj) fontName = GetString(theObj);
	else fontName = DEFFONT;

	theObj = GetVar(textBox, TEXTSIZE);
	if (theObj) size = GetInt(theObj);
	else size = DEFTEXTSIZE;

	theObj = GetVar(textBox, LINESPACE);
	if (theObj) lineSpace = GetInt(theObj);
	else lineSpace = 4 + size/3;

	if (style & WITH_PIT)
	{
		textLeft = left + TEXTMARGIN;
		textRight = right - TEXTMARGIN;
		textTop = top - TEXTMARGIN;
	}
	else if (style & (EDITABLE | ADJUSTABLE) || GetVar(textBox, BACKGROUND))
	{
		textLeft = left + HANDLESIZE + 2;
		textRight = right - HANDLESIZE - 2;
		textTop = top - HANDLESIZE - 2;
	}
	else /* default is no inset */
	{
		textLeft = left;
		textRight = right;
		textTop = top;
	}

	SetupFont(fontName, size);
	spWid = ChrWidth(' ');
	y = textTop - size;
	width = textRight - textLeft;
	bgnLine = nxtLine = text; /* set up for loop */
	do {
		bgnLine = nxtLine;
		nxtLine = NextLine(bgnLine, width, lineBuf);
		if (*nxtLine == '\t')
		{
			++nxtLine;
			indent += spWid*INDENTAMT;
			continue;
		}
		y -= size + lineSpace;
		if (*nxtLine == '\n')	/* paragraph */
		{
			++nxtLine;
			indent = 0;
		}
	} while (*nxtLine);
	return textTop - y + size;
#else
	ObjPtr theObj;
	int n, size, lineSpace;
	char *t;
		
	if (GetPredicate(textBox, CHANGED)) NewText(textBox);

	t = GetString(GetVar(textBox, VALUE));

	theObj = GetVar(textBox, TEXTSIZE);
	if (theObj) size = GetInt(theObj);
	else size = DEFTEXTSIZE;

	theObj = GetVar(textBox, LINESPACE);
	if (theObj) lineSpace = GetInt(theObj);
	else lineSpace = 4 + size/3;

	n = 0;
	while (*t)
	{
		if (*t == '\r' || *t == '\n') ++n;
		++t;
	}
	return n*(size + lineSpace);
#endif
}

/******************************************************************************* DRAW TEXT BOX */
#ifdef PROTO
static ObjPtr DrawTextBox(ObjPtr textBox)
#else
static ObjPtr DrawTextBox(textBox)
ObjPtr textBox;
#endif
{
#ifdef GRAPHICS
	int left, right, bottom, top;
	Bool active;
	int textLeft, textRight, textBottom, textTop;
	int x,y;
	ObjPtr theObj, color, bgColor;
	char *text, *nxtLine, *bgnLine, lineBuf[MAXLINE], *fontName;
	int lineStart, selStart, selEnd; /* distances for line and selection */
	int bgnSel, endSel; /* index to text selection */
	int align, length, width, center, size, spWid, lineSpace, style; 

	Get2DIntBounds(textBox, &left, &right, &bottom, &top);

	active = GetPredicate(textBox, ACTIVATED);
	
	theObj = GetStringVar("DrawTextBox", textBox, VALUE);
	if (theObj) text = GetString(theObj);
	else text = "\0";
		
	if ((length = strlen(text)) > textBufSize) /* grow buffer */
	{
		textBuf = (char *) realloc(textBuf, length + TEXTBUFFERINCR);
		if (!textBuf) /* failed */
		{
			OMErr();
			textBufSize = 0;
			return NULLOBJ;
		}
	}

	theObj = GetIntVar("DrawTextBox", textBox, STYLE);
	if (theObj) style = GetInt(theObj);
	else style = PLAIN;

	theObj = GetVar(textBox, ALIGNMENT);
	if (theObj) align = GetInt(theObj);
	else align = DEFALIGN;

	theObj = GetVar(textBox, TEXTFONT);
	if (theObj) fontName = GetString(theObj);
	else fontName = DEFFONT;

	theObj = GetVar(textBox, TEXTSIZE);
	if (theObj) size = GetInt(theObj);
	else size = DEFTEXTSIZE;

	theObj = GetVar(textBox, LINESPACE);
	if (theObj) lineSpace = GetInt(theObj);
	else lineSpace = 4 + size/3;

	color = GetVar(textBox, COLOR);

	bgColor = GetVar(textBox, BACKGROUND);

	if (active && AmICurrent(textBox) && style & EDITABLE)
	{
		theObj = GetVar(textBox, BGNSEL);
		if (theObj) bgnSel = GetInt(theObj);
		else bgnSel = strlen(text);

		theObj = GetVar(textBox, ENDSEL);
		if (theObj) endSel = GetInt(theObj);
		else endSel = bgnSel;
	}

	if (style & WITH_PIT)
	{
		DrawPitEdge(left, right, bottom, top);
		SetUIColor(active ? TEXTBOXCOLOR : UIBACKGROUND);
		FillRect(left + EDGE, right - EDGE,
				bottom + EDGE, top - EDGE);
		textLeft = left + TEXTMARGIN;
		textRight = right - TEXTMARGIN;
		textBottom = bottom + TEXTMARGIN;
		textTop = top - TEXTMARGIN;
	}
	else if (bgColor)
	{
		SetObjectColor(bgColor);
		FillRect(left, right, bottom, top);
		textLeft = left + HANDLESIZE + 2;
		textRight = right - HANDLESIZE - 2;
		textBottom = bottom + HANDLESIZE + 2;
		textTop = top - HANDLESIZE - 2;
	}
	else if (style & (EDITABLE | ADJUSTABLE))
	{
		textLeft = left + HANDLESIZE + 2;
		textRight = right - HANDLESIZE - 2;
		textBottom = bottom + HANDLESIZE + 2;
		textTop = top - HANDLESIZE - 2;
	}
	else /* default is no inset */
	{
		textLeft = left;
		textRight = right;
		textBottom = bottom;
		textTop = top;
	}

	/* wake up, time to draw */
	SetupFont(fontName, size);
	y = textTop - size;
	width = textRight - textLeft;
	indent = 0; spWid = ChrWidth(' ');
	center = textLeft + width/2;
	nxtLine = text; /* set up for loop */
	do {
		bgnLine = nxtLine;
		nxtLine = NextLine(bgnLine, width, lineBuf);
		length = StrWidth(lineBuf);
		{
			switch (align)
			{
			case CENTERALIGN:
           			lineStart = center - length/2;
				break;

			case RIGHTALIGN:
				lineStart = textRight - length;
				break;

			default:	/* assume align left */
				lineStart = textLeft + indent;
				break;
			}
			if (active && AmICurrent(textBox) && style & EDITABLE)
			{
				/* draw text cursor if in this line */
				if (bgnSel == endSel
					&& text + bgnSel <= nxtLine && text + bgnSel >= bgnLine)
				{
					/* find position of insertion point */
					int i = bgnSel - (bgnLine - text);

					strncpy(textBuf, bgnLine, i);
					textBuf[i] = '\0';
					selStart = StrWidth(textBuf);

					/* draw text cursor before bgnSel */
					FillUIRect(lineStart + selStart, lineStart + selStart + 1,
						y - lineSpace + 2, y + size + 2, TEXTCURSORCOLOR);
				}
				else if (text + endSel > bgnLine && text + bgnSel < nxtLine)
				{
					/* some of the selection is on this line */
					if (text + bgnSel < bgnLine) selStart = lineStart;
					else /* compute dist to start of selection */
					{
						int i = bgnSel - (bgnLine - text);

						strncpy(textBuf, bgnLine, i);
						textBuf[i] = '\0';
						selStart = lineStart + StrWidth(textBuf);
					}
					if (text + endSel >= nxtLine) selEnd = lineStart + length;
					else /* compute dist to end of selection */
					{
						int i = endSel - (bgnLine - text);

						strncpy(textBuf, bgnLine, i);
						textBuf[i] = '\0';
						selEnd = lineStart + StrWidth(textBuf);
					}
					/* now draw the selection rectangle for this line */
					FillUIRect(selStart, selEnd, y - lineSpace + 2, y + size + 2,
								TEXTSELECTCOLOR);
				}
			}
			/* now draw the text */
			if (color) SetObjectColor(color);
			else SetUIColor(UITEXT);
			DrawString(lineStart, y, lineBuf);
		}
		if (*nxtLine == '\t')
		{
			++nxtLine;
			indent += spWid*INDENTAMT;
			continue;
		}
		if ((y -= size + lineSpace) < textBottom) break; /* whoa! past bottom of box */
		if (*nxtLine == '\n')	/* paragraph */
		{
			++nxtLine;
			indent = 0;
		}
	} while (*nxtLine);

	if (AmICurrent(textBox) && style & ADJUSTABLE)
	{
		/* Draw incredibly fancy frame for moving and resizing text box */

		int horCent = (left + right)/2;
		int vertCent = (bottom + top)/2;

		DrawFrameRect(left+INSET, right-INSET,
				bottom+INSET, top-INSET,
				OUTSIDEFRAMECOLOR, OUTSIDEFRAMEWEIGHT);
		DrawFrameRect(left+INSET+OUTSIDEFRAMEWEIGHT,
				right-INSET-OUTSIDEFRAMEWEIGHT,
				bottom+INSET+OUTSIDEFRAMEWEIGHT,
				top-INSET-OUTSIDEFRAMEWEIGHT,
				INSIDEFRAMECOLOR, INSIDEFRAMEWEIGHT);
		DrawFrameRect(left+INSET+OUTSIDEFRAMEWEIGHT+INSIDEFRAMEWEIGHT,
				right-INSET-OUTSIDEFRAMEWEIGHT-INSIDEFRAMEWEIGHT,
				bottom+INSET+OUTSIDEFRAMEWEIGHT+INSIDEFRAMEWEIGHT,
				top-INSET-OUTSIDEFRAMEWEIGHT-INSIDEFRAMEWEIGHT,
				OUTSIDEFRAMECOLOR, OUTSIDEFRAMEWEIGHT);

		/* Now draw the handles */
		/* center of sides */
		FillUIRect(left, left+HANDLESIZE,
				vertCent-HANDLESIZE/2, vertCent+HANDLESIZE/2, OUTSIDEFRAMECOLOR);
		FillUIRect(left+OUTSIDEFRAMEWEIGHT,
				left+HANDLESIZE-OUTSIDEFRAMEWEIGHT,
				vertCent-HANDLESIZE/2+OUTSIDEFRAMEWEIGHT,
				vertCent+HANDLESIZE/2-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);

		FillUIRect(right-HANDLESIZE, right,
				vertCent-HANDLESIZE/2, vertCent+HANDLESIZE/2, OUTSIDEFRAMECOLOR);
		FillUIRect(right-HANDLESIZE+OUTSIDEFRAMEWEIGHT,
				right-OUTSIDEFRAMEWEIGHT,
				vertCent-HANDLESIZE/2+OUTSIDEFRAMEWEIGHT,
				vertCent+HANDLESIZE/2-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);

		/* top edge */
		FillUIRect(horCent-HANDLESIZE/2, horCent+HANDLESIZE/2,
				top-HANDLESIZE, top, OUTSIDEFRAMECOLOR);
		FillUIRect(horCent-HANDLESIZE/2+OUTSIDEFRAMEWEIGHT,
				horCent+HANDLESIZE/2-OUTSIDEFRAMEWEIGHT,
				top-HANDLESIZE+OUTSIDEFRAMEWEIGHT,
				top-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);

		FillUIRect(left, left+HANDLESIZE,
				top-HANDLESIZE, top, OUTSIDEFRAMECOLOR);
		FillUIRect(left+OUTSIDEFRAMEWEIGHT,
				left+HANDLESIZE-OUTSIDEFRAMEWEIGHT,
				top-HANDLESIZE+OUTSIDEFRAMEWEIGHT,
				top-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);
		
		FillUIRect(right-HANDLESIZE, right,
				top-HANDLESIZE, top, OUTSIDEFRAMECOLOR);
		FillUIRect(right-HANDLESIZE+OUTSIDEFRAMEWEIGHT,
				right-OUTSIDEFRAMEWEIGHT,
				top-HANDLESIZE+OUTSIDEFRAMEWEIGHT,
				top-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);

		/* bottom edge */
		FillUIRect(horCent-HANDLESIZE/2, horCent+HANDLESIZE/2,
				bottom, bottom+HANDLESIZE, OUTSIDEFRAMECOLOR);
		FillUIRect(horCent-HANDLESIZE/2+OUTSIDEFRAMEWEIGHT,
				horCent+HANDLESIZE/2-OUTSIDEFRAMEWEIGHT,
				bottom+OUTSIDEFRAMEWEIGHT,
				bottom+HANDLESIZE-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);

		FillUIRect(left, left+HANDLESIZE,
				bottom, bottom+HANDLESIZE, OUTSIDEFRAMECOLOR);
		FillUIRect(left+OUTSIDEFRAMEWEIGHT,
				left+HANDLESIZE-OUTSIDEFRAMEWEIGHT,
				bottom+OUTSIDEFRAMEWEIGHT,
				bottom+HANDLESIZE-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);
		
		FillUIRect(right-HANDLESIZE, right,
				bottom, bottom+HANDLESIZE, OUTSIDEFRAMECOLOR);
		FillUIRect(right-HANDLESIZE+OUTSIDEFRAMEWEIGHT,
				right-OUTSIDEFRAMEWEIGHT,
				bottom+OUTSIDEFRAMEWEIGHT,
				bottom+HANDLESIZE-OUTSIDEFRAMEWEIGHT, INSIDEFRAMECOLOR);
	}
	else if (!active) /* cover with gauze */
		FillUIGauzeRect(left, right, bottom, top, UIBACKGROUND);
#endif
	return NULLOBJ;
}

#ifdef PROTO
static ObjPtr PressTextBox(ObjPtr textBox, int mouseX, int mouseY, long flags)
#else
static ObjPtr PressTextBox(textBox, mouseX, mouseY, flags)
ObjPtr textBox;
int mouseX, mouseY;
long flags;
#endif
{
#ifdef INTERACTIVE
	ObjPtr theObj;
	char *text, *t, *fontName, lineBuf[MAXLINE];
	int i, j, k, l, n, width, length, spWid;
	int style, align, size, nLines, lineSpace;
	float x;
	int y, mX, mY, offset[MAXNLINES * MAXLINE], lnx[3*(MAXNLINES + 1)];
	int left, right, bottom, top, hCent, vCent;

	Get2DIntBounds(textBox, &left, &right, &bottom, &top);

	/* return if mouse outside text box */
	if (mouseX < left || mouseX > right || mouseY < bottom 
			|| mouseY > top) return ObjFalse;

	if (TOOL(flags) == T_HELP) /* help mode? */
	{
		ContextHelp(textBox);
		return ObjTrue;
	}

	/* return if not active */
	if (!GetPredicate(textBox, ACTIVATED)) return ObjTrue;

	theObj = GetIntVar("PressTextBox", textBox, STYLE);
	if (!theObj) style = PLAIN;
	else style = GetInt(theObj);

	/* return if not editable or adjustable */
	if (!(style & (EDITABLE | ADJUSTABLE))) return ObjTrue;
	Select(textBox, TRUE); /* make text box current for editing or adjusting */
	DrawMe(textBox);
	UpdateDrawing();

	hCent = (left + right)/2;
	vCent = (bottom + top)/2;

	if (style & ADJUSTABLE)
	{
		Bool ml, mr, mb, mt;
		ml = mr = mb = mt = FALSE;

		if (mouseX < left + HANDLESIZE) /* on left side */
		{
			if (mouseY > top - HANDLESIZE) /* top-left handle */
				mt = ml = TRUE;
			else if (mouseY < bottom + HANDLESIZE) /* bottom-left handle */
				mb = ml = TRUE;
			else if (mouseY > vCent - HANDLESIZE/2 && mouseY < vCent + HANDLESIZE/2)
				ml = TRUE; /* bottom middle handle */
			else ml = mr = mb = mt = TRUE; /* in frame */
		}
		else if (mouseX > right - HANDLESIZE) /* on right side */
		{
			if (mouseY > top - HANDLESIZE) /* top-right handle */
				mt = mr = TRUE;
			else if (mouseY < bottom + HANDLESIZE) /* bottom-right handle */
				mb = mr = TRUE;
			else if (mouseY > vCent - HANDLESIZE/2 && mouseY < vCent + HANDLESIZE/2)
				mr = TRUE;
			else ml = mr = mb = mt = TRUE; /* in frame */
		}
		else if (mouseY < bottom + HANDLESIZE) /* on bottom */
		{
			/* already handled (heh heh) corners */
			if (mouseX > hCent - HANDLESIZE/2 && mouseX < hCent + HANDLESIZE/2)
				mb = TRUE; /* bottom middle handle */
			else ml = mr = mb = mt = TRUE; /* in frame */
		}
		else if (mouseY > top - HANDLESIZE) /* on top */
		{
			/* already handled (heh heh) corners */
			if (mouseX > hCent - HANDLESIZE/2 && mouseX < hCent + HANDLESIZE/2)
				mt = TRUE; /* middle top handle */
			else ml = mr = mb = mt = TRUE; /* in frame */
		}
		else /* not on incredibly fancy frame */
		{
			if (!(style & EDITABLE)) return ObjTrue;
		}
		if (mr || ml || mb || mt) /* drag the incredibly fancy frame around */
		{
			/* I am greatly indebted to my friend and colleague, Eric Pepke,
			   for the following code. Any errors or obfuscations are his. */
			int initX = mouseX, initY = mouseY;
			int newLeft, newRight, newBottom, newTop;
			int oldNewLeft, oldNewRight, oldNewBottom, oldNewTop;

			newLeft = oldNewLeft = left;
			newRight = oldNewRight = right;
			newBottom = oldNewBottom = bottom;
			newTop = oldNewTop = top;

			while (Mouse(&mX, &mY))
			{
				if (ml) newLeft = left + mX - initX;
				if (mr) newRight = right + mX - initX;
				if (mb) newBottom = bottom + mY - initY;
				if (mt) newTop = top + mY - initY;

				if (flags & F_OPTIONDOWN)
				{
					/*Grid drag*/
					if (ml && mr && mb && mt)
					{
					    /*Special case--whole object gridded
					      Only grid top left*/
					    int width, height;
					    width = newRight - newLeft;
					    height = newTop - newBottom;
					    newLeft = GRIDX(newLeft);
					    newRight = newLeft + width;
					    newTop = top - (GRIDY(top - newTop));
					    newBottom = newTop - height;
					}
					else
					{
					    /*Normal case*/
					    if (ml) newLeft = GRIDX(newLeft);
					    if (mr) newRight = right - GRIDX(right - newRight);
					    if (mb) newBottom = GRIDY(newBottom);
					    if (mt) newTop = top - GRIDY(top - newTop);
					}
				}
				if (ml && newLeft + 3 * HANDLESIZE > newRight)
					newLeft = newRight - 3 * HANDLESIZE;
				if (mr && newLeft + 3 * HANDLESIZE > newRight)
					newRight = newLeft + 3 * HANDLESIZE;
				if (mb && newBottom + 3 * HANDLESIZE > newTop)
					newBottom = newTop - 3 * HANDLESIZE;
				if (mt && newBottom + 3 * HANDLESIZE > newTop)
					newTop = newBottom + 3 * HANDLESIZE;
				if ((newLeft != oldNewLeft ||
					 newRight != oldNewRight ||
					 newBottom != oldNewBottom ||
					 newTop != oldNewTop) &&
					 newLeft < newRight &&
					 newBottom < newTop)
				{
					Set2DIntBounds(textBox, newLeft, newRight, newBottom, newTop);
					oldNewLeft = newLeft;
					oldNewRight = newRight;
					oldNewBottom = newBottom;
					oldNewTop = newTop;
					DrawMe(textBox);
				}
			}
			if (logging)
			{
				char cmd[256];
				MakeObjectName(tempStr, textBox);
				sprintf(cmd, "set bounds %s [%d %d %d %d]\n",
					tempStr, newLeft, newRight,
					newBottom, newTop);
				Log(cmd);
			}
			return ObjTrue;
		}
	}

	/* click is in text of editable textbox; track mouse to select text */

	theObj = GetVar(textBox, VALUE);
	if (theObj) text = GetString(theObj);
	else text = "\0";

	theObj = GetVar(textBox, ALIGNMENT);
	if (theObj) align = GetInt(theObj);
	else align = DEFALIGN;

	theObj = GetVar(textBox, TEXTFONT);
	if (theObj) fontName = GetString(theObj);
	else fontName = DEFFONT;

	theObj = GetVar(textBox, TEXTSIZE);
	if (theObj) size = GetInt(theObj);
	else size = DEFTEXTSIZE;

	theObj = GetVar(textBox, LINESPACE);
	if (theObj) lineSpace = GetInt(theObj);
	else lineSpace = 4 + size/3;
	
	/* adjust bounds for background or pit */
	if (style & WITH_PIT)
	{
		left = left + TEXTMARGIN;
		right = right - TEXTMARGIN;
		bottom = bottom + TEXTMARGIN;
		top = top - TEXTMARGIN;
	}
	else if (style & ADJUSTABLE || GetVar(textBox, BACKGROUND))
	{
		left = left + (HANDLESIZE+2);
		right = right - (HANDLESIZE+2);
		bottom = bottom + (HANDLESIZE+2);
		top = top - (HANDLESIZE+2);
	}
	width = right - left;

	SetupFont(fontName, size);
	spWid = ChrWidth(' ');
/*
*	Scheme for selecting text:
*	Build two arrays to find character positions of mouseDown and mouseUp.
*	lnx[l] is the index of the first character of line l in text.
*	offset[i] is the total horiz offset to the middle of character i in text.
*/
	t = text; l = 0; nLines = 0; lnx[0] = 0; indent = 0; /* setup for loop */
	do {
		k = lnx[l];
		t = NextLine(t, width, lineBuf);
		lnx[++l] = t - text; /* index to first char of next segment */
		length = StrWidth(lineBuf);
			
		/* figure out where line begins */
		switch (align)
		{
		case CENTERALIGN:
           		x = (left + right)/2 - length/2;
			break;

		case RIGHTALIGN:
			x = right - length;
			break;

		default:	/* assume align left */
			x = left + indent;
			break;
		}

		/* build next segment of char offsets */
		for (i=k; i<lnx[l]; ++i)
		{
			register int w;
			if (text[i] == '\t')
			{
				indent += spWid*INDENTAMT;
				offset[i] = indent - spWid;
				x = indent;
			}
			else
			{
				w = ChrWidth(text[i]);
				offset[i] = x + w/2;
				x += w;
			}
		}
		if (*t == '\t') /* indent */
		{
			++t;
			--l; /* don't count as line */
			continue;
		}
		if (*t == '\n') /* paragraph */
		{
			++t;
			indent = 0;
		}
		nLines = l;
		if (nLines > MAXNLINES) break;
	} while (*t);
	
	if (flags & F_SHIFTDOWN)
	{
		i = GetInt(GetVar(textBox, BGNSEL));
	}
	else
	{
		/* find line and character position of mouse down */
		l = (top - mouseY)/(size + lineSpace);
		if (l < 0) i = 0;
		else if (l >= nLines)
		{
			i = lnx[nLines];
		}
		else
		{
			i=lnx[l];
			while (offset[i] < mouseX && i<lnx[l+1]) ++i;
		}
	}

	/* now track mouse and find line and char pos of end of selection */
	mX = mouseX; mY = mouseY; /* setup for at least one loop */
	do {
		l = (top - mY)/(size + lineSpace);
		if (l < 0) j = 0;
		else if (l >= nLines)
		{
			j = lnx[nLines];
		}
		else 
		{
			j=lnx[l];
			while (offset[j] < mX && j<lnx[l+1]) ++j;
		}
		if (i > j) /* make sure bgnSel < endSel */
		{
			SetVar(textBox, BGNSEL, NewInt(j));
			SetVar(textBox, ENDSEL, NewInt(i));
		}
		else
		{
			SetVar(textBox, BGNSEL, NewInt(i));
			SetVar(textBox, ENDSEL, NewInt(j));
		}
		DrawMe(textBox);
		UpdateDrawing(); /* force redraw */
	} while (Mouse(&mX, &mY));

	if (i == j && flags & F_DOUBLECLICK)
	{
		/* select the word at insPt */
		while (i > 0)
		{
			if (IsAlphaNum(text[i-1])) --i;
			else break;
		}
		while (text[j] != '\0')
		{
			if (IsAlphaNum(text[j])) ++j;
			else break;
		}
		SetVar(textBox, BGNSEL, NewInt(i));
		SetVar(textBox, ENDSEL, NewInt(j));
	}
#endif
	return ObjTrue;
}

int IsAlphaNum(c)
char c;
{
	return c >= 'a' && c <= 'z' || c >= 'A' && c <= 'Z' || c >= '0' && c <='9';
}

#ifdef PROTO
static ObjPtr EditTextBox(ObjPtr textBox, short key, long flags)
#else
static ObjPtr EditTextBox(textBox, key, flags)
ObjPtr textBox;
short key;
long flags;
#endif
{
#ifdef INTERACTIVE
	ObjPtr theObj, theText, theStyle, theInsPt, theEndSel;
	char *text;
	int i, j, len, style, insPt, endSel;

	if (!AmICurrent(textBox)) return ObjFalse; /* keyboard input not for me */

	theStyle = GetIntVar("EditTextBox", textBox, STYLE);
	if (!theStyle) return ObjTrue;
	style = GetInt(theStyle);
	if (!(style & EDITABLE)) return ObjTrue;

	theText = GetStringVar("EditTextBox", textBox, VALUE);
	if (theText) text = GetString(theText);
	else text = "\0";
	
	if ((len = strlen(text)) > textBufSize) /* grow buffer */
	{
		textBuf = (char *) realloc(textBuf, len + TEXTBUFFERINCR);
		if (!textBuf) /* failed */
		{
			OMErr();
			textBufSize = 0;
			return NULLOBJ;
		}
	}

	theInsPt = GetVar(textBox, BGNSEL);
	if (!theInsPt) insPt = len;
	else insPt = GetInt(theInsPt);

	theEndSel = GetVar(textBox, ENDSEL);
	if (!theEndSel) endSel = len;
	else endSel = GetInt(theEndSel);

	/* copy up to the insertion point */
	for(i=0; i<insPt; ++i) textBuf[i] = text[i];

	/* process every key press */
	if (key >= ' ' && key <= '~' || key == '\t')
	{
		/* add key at insertion point */
		textBuf[insPt] = key;
		for(i = endSel, j = ++insPt; i < len; ++i, ++j)
			textBuf[j] = text[i];
		textBuf[j] = '\0';
		SetVar(textBox, VALUE, NewString(textBuf));
		SetVar(textBox, BGNSEL, NewInt(insPt));
		SetVar(textBox, ENDSEL, NewInt(insPt));
		SetVar(textBox, CHANGED, ObjTrue);
		DrawMe(textBox);
	}
	else if (key == '\b')
	{
		if (insPt == endSel && insPt > 0) /* delete character before insPt */
		{
			for(i = insPt, j = --insPt; i<len; ++i, ++j)
				textBuf[j] = text[i];
			textBuf[j] = '\0';
		}
		else /* just delete selection */
		{
			for(i = endSel, j = insPt; i < len; ++i, ++j)
				textBuf[j] = text[i];
			textBuf[j] = '\0';
		}
		SetVar(textBox, VALUE, NewString(textBuf));
		SetVar(textBox, BGNSEL, NewInt(insPt));
		SetVar(textBox, ENDSEL, NewInt(insPt));
		SetVar(textBox, CHANGED, ObjTrue);
		DrawMe(textBox);
	}
	else if (key == '\r')
	{
		if (style & ONE_LINE) /* check for ENTERMETHOD */
		{
			FuncTyp EnterMethod = GetMethod(textBox, ENTERMETHOD);
			if (EnterMethod) EnterMethod(textBox);
			else
			{
				ChangedValue(textBox);
				if (logging) LogControl(textBox);
			}
		}
		else /* put newline into string at insertion point */
		{
			textBuf[insPt] = '\n';
			for(i=insPt; i<len; ++i) textBuf[i+1] = text[i + endSel - insPt];
			textBuf[len+1] = '\0';
			SetVar(textBox, VALUE, NewString(textBuf));
			SetVar(textBox, BGNSEL, NewInt(insPt + 1));
			SetVar(textBox, ENDSEL, NewInt(insPt + 1));
			SetVar(textBox, CHANGED, ObjTrue);
			DrawMe(textBox);
		}
	}
	else if (key == KLEFTARROW)
	{
		/* just move the insertion point (and endSel) back */
		if (insPt > 0) --insPt;
		SetVar(textBox, BGNSEL, theObj = NewInt(insPt));
		SetVar(textBox, ENDSEL, theObj);
		DrawMe(textBox);
	}
	else if (key == KRIGHTARROW)
	{
		/* just move endSel (and insPt) forward */
		if (endSel < len) ++endSel;
		SetVar(textBox, BGNSEL, theObj = NewInt(endSel));
		SetVar(textBox, ENDSEL, theObj);
		DrawMe(textBox);
	}
	else if (key == KUPARROW || key == KDOWNARROW)
	{
		/* Oh, so you want to move the cursor up or down, eh? Well,
		   we can do that, it's no bother.  Really.  So what if we have 
		   to read all of the text and create a few arrays?  We don't
 		   mind. It's no trouble at all, really.  Have a seat. */

		int size, align, l, offset[MAXNLINES * MAXLINE], lnx[MAXNLINES + 1];
		char *t, *fontName, lineBuf[MAXLINE];
		int left, right, bottom, top, width, length;
		float x;

		theObj = GetVar(textBox, ALIGNMENT);
		if (theObj) align = GetInt(theObj);
		else align = DEFALIGN;

		theObj = GetVar(textBox, TEXTFONT);
		if (theObj) fontName = GetString(theObj);
		else fontName = DEFFONT;

		theObj = GetVar(textBox, TEXTSIZE);
		if (theObj) size = GetInt(theObj);
		else size = DEFTEXTSIZE;

		Get2DIntBounds(textBox, &left, &right, &bottom, &top);

		/* adjust bounds */
		if (style & WITH_PIT)
		{
			left += TEXTMARGIN;
			right -= TEXTMARGIN;
			bottom += TEXTMARGIN;
			top -= TEXTMARGIN;
		}
		else if (style & ADJUSTABLE || GetVar(textBox, BACKGROUND))
		{
			left += HANDLESIZE + 2;
			right -= HANDLESIZE + 2;
			bottom += HANDLESIZE + 2;
			top -= HANDLESIZE + 2;
		}
		width = right - left;
		SetupFont(fontName, size);

		t = text; l = 0; lnx[0] = 0; /* setup for loop */
		do {
			t = NextLine(t, width, lineBuf);
			lnx[++l] = t - text; /* index to first char of next line */
			length = StrWidth(lineBuf);
			
			/* figure out where line begins */
			switch (align)
			{
			case CENTERALIGN:
           			x = (left + right)/2 - length/2;
				break;

			case RIGHTALIGN:
				x = right - length;
				break;

			default:	/* assume align left */
				x = left;
				break;
			}

			/* build next line of char offsets */
			for (i=lnx[l-1]; i<lnx[l]; ++i)
			{
				register int w = ChrWidth(text[i]);
				offset[i] = x + w/2;
				x += w;
			}

			if (*t == '\n') ++t; /* paragraph */
			if (l > MAXNLINES) break;
		} while (*t);

		/* find the line containing insPt */
		for (i=1; i<=l; ++i) if (lnx[i] >= insPt) break;

		/* line is i-1 */

		if (key == KUPARROW)
		{
			int k;

			if (i-1 == 0) return ObjTrue; /* already on first line */

			for (k=lnx[i-2]; k<lnx[i-1]; ++k)
				if (offset[k] > offset[insPt]) break;
			SetVar(textBox, BGNSEL, theObj = NewInt(k));
			SetVar(textBox, ENDSEL, theObj);
			DrawMe(textBox);
		}
		else if (key == KDOWNARROW)
		{
			int k;

			if (i == l) return ObjTrue; /* already on last line */

			for (k=lnx[i]; k<lnx[i+1]; ++k)
				if (offset[k] > offset[insPt]) break;
			SetVar(textBox, BGNSEL, theObj = NewInt(k));
			SetVar(textBox, ENDSEL, theObj);
			DrawMe(textBox);
		}
	}
	else if (key == '\0')
	{
		if (!(style & DEFERCHANGE)) 
		{
			ChangedValue(textBox);
			if (logging) LogControl(textBox);
		}
	}
#endif
	return ObjTrue;
}

/************************************************************ CUT, COPY, PASTE*/
#ifdef PROTO
static ObjPtr CutText(ObjPtr textBox)
#else
static ObjPtr CutText(textBox)
ObjPtr textBox;
#endif
{
	ObjPtr theText, theInsPt, theEndSel;
	int i, j, len, insPt, endSel;
	char *text;
	Bool deferChange;

	theText = GetStringVar("CutText", textBox, VALUE);
	if (theText) text = GetString(theText);
	else text = "\0";
	
	len = strlen(text);

	theInsPt = GetVar(textBox, BGNSEL);
	if (!theInsPt) insPt = len;
	else insPt = GetInt(theInsPt);

	theEndSel = GetVar(textBox, ENDSEL);
	if (!theEndSel) endSel = len;
	else endSel = GetInt(theEndSel);

	deferChange = GetInt(GetVar(textBox, STYLE)) & DEFERCHANGE;

	/* copy the selection to textBuf */
	for (i=insPt, j=0; i<endSel; ++i, ++j) textBuf[j] = text[i];
	textBuf[j] = '\0';

	/* delete selection */
	for (j=insPt, i=endSel; text[i]!='\0'; ++i, ++j) text[j] = text[i];
	text[j] = '\0';

	SetVar(textBox, VALUE, NewString(text));
	SetVar(textBox, ENDSEL, NewInt(insPt));
	SetVar(textBox, CHANGED, ObjTrue);

	if (!deferChange) 
	{
		ChangedValue(textBox);
		if (logging) LogControl(textBox);
	}
	ImInvalid(textBox);
	return NewString(textBuf);
}

#ifdef PROTO
static ObjPtr CopyText(ObjPtr textBox)
#else
static ObjPtr CopyText(textBox)
ObjPtr textBox;
#endif
{
	ObjPtr theText, theInsPt, theEndSel;
	int i, j, len, insPt, endSel;
	char *text;

	theText = GetStringVar("CopyText", textBox, VALUE);
	if (theText) text = GetString(theText);
	else text = "\0";
	
	len = strlen(text);

	theInsPt = GetVar(textBox, BGNSEL);
	if (!theInsPt) insPt = len;
	else insPt = GetInt(theInsPt);

	theEndSel = GetVar(textBox, ENDSEL);
	if (!theEndSel) endSel = len;
	else endSel = GetInt(theEndSel);

	/* copy the selection to textBuf */
	for (i=insPt, j=0; i<endSel; ++i, ++j) textBuf[j] = text[i];

	/* terminate and return */
	textBuf[j] = '\0';
	return NewString(textBuf);
}

#ifdef PROTO
static ObjPtr PasteText(ObjPtr textBox, ObjPtr newText)
#else
static ObjPtr PasteText(textBox, newText)
ObjPtr textBox, newText;
#endif
{
	ObjPtr theText, theInsPt, theEndSel;
	int i, j, len, style, insPt, endSel;
	char *text, *p;
	Bool deferChange;

	if (!IsString(newText)) /* check paste object */
	{
		ReportError("PasteText", "Paste object not a string");
		return NULLOBJ;
	}

	p = GetString(newText);

	theText = GetStringVar("PasteText", textBox, VALUE);
	if (theText) text = GetString(theText);
	else text = "\0";
	
	len = strlen(text);

	theInsPt = GetVar(textBox, BGNSEL);
	if (!theInsPt) insPt = len;
	else insPt = GetInt(theInsPt);

	theEndSel = GetVar(textBox, ENDSEL);
	if (!theEndSel) endSel = len;
	else endSel = GetInt(theEndSel);

	deferChange = GetInt(GetVar(textBox, STYLE)) & DEFERCHANGE;

	if ((len += strlen(p)) > textBufSize) /* grow buffer */
	{
		textBuf = (char *) realloc(textBuf, len + TEXTBUFFERINCR);
		if (!textBuf) /* failed */
		{
			OMErr();
			textBufSize = 0;
			return NULLOBJ;
		}
	}

	/* copy up to the insertion point */
	for (i=0; i<insPt; ++i) textBuf[i] = text[i];

	/* copy the paste text */
	for (j=0; p[j]!='\0'; ++i, ++j) textBuf[i] = p[j];

	insPt = i; /* remember new insertion point */

	/* copy text after endSel */
	for (j=endSel; text[j]!='\0'; ++i, ++j) textBuf[i] = text[j];

	textBuf[i] = '\0';
	SetVar(textBox, VALUE, NewString(textBuf));
	SetVar(textBox, BGNSEL, theInsPt = NewInt(insPt));
	SetVar(textBox, ENDSEL, theInsPt);
	SetVar(textBox, CHANGED, ObjTrue);

	if (!deferChange) 
	{
		ChangedValue(textBox);
		if (logging) LogControl(textBox);
	}
	ImInvalid(textBox);
	return NULLOBJ;
}

/********************************************************** SELECT ALL */
/* globals used for SelectAll deferred task */
static WinInfoPtr gTBWin = NULL;
static ObjPtr gTextBox;

void DoSelectAll(void)
{
	if (gTBWin)
	{
		SelectWindow(gTBWin->id);
		Select(gTextBox, TRUE);
		gTBWin = NULL;
	}
	else ReportError("DoSelectAll", "No owner window!");

}

#ifdef PROTO
ObjPtr SelectAll(ObjPtr textBox)
#else
ObjPtr SelectAll(textBox)
ObjPtr textBox;
#endif
{
	/*** err check */
	SetVar(textBox, BGNSEL, NewInt(0));
	SetVar(textBox, ENDSEL, NewInt(strlen(GetString(GetVar(textBox,VALUE)))));
	gTBWin = (WinInfoPtr) GetVar(GetVar(textBox, PARENT), PARENT);
	gTextBox = textBox;
	DoTask(DoSelectAll);
	return ObjTrue;
}

/* adapted from SelectPaletteDisplay by Eric Pepke */
#ifdef PROTO
static ObjPtr SelectTextBox(ObjPtr object, Bool selectp)
#else
static ObjPtr SelectTextBox(object, selectp)
ObjPtr object;
Bool selectp;
#endif
/*Selects an icon*/
{
    Bool alreadySelected;

    alreadySelected = GetPredicate(object, SELECTED);
    if (selectp == alreadySelected)
    {
	return ObjTrue;
    }

    if (selectp)
    {
	MakeMeCurrent(object);
    }

    if (logging)
    {
	char logLine[256];
	MakeObjectName(tempStr, object);
	if (tempStr[0])
	{
	    sprintf(logLine, "%s %s\n", selectp ? "select" : "deselect", tempStr);
	    Log(logLine);
	}
    }
    SetVar(object, SELECTED, NewInt(selectp));
    ImInvalid(object);
/*    ChangedValue(object);  EMP */
    return ObjTrue;
}


static ObjPtr TextBoxNotCurrent(object)
ObjPtr object;
/*Makes a palette display not current*/
{
    Select(object, false);
}

/* adapted from ShowPaletteDisplayControls by Eric Pepke */
static ObjPtr ShowTextBoxControls(display, ownerWindow, windowName)
ObjPtr display;
WinInfoPtr ownerWindow;
char *windowName;
/*Makes a new control window to control a palette display*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    real rgb[3], hsv[3];
    WinInfoPtr dialogExists;
    Bool hasBackground;

    dialogExists = DialogExists((WinInfoPtr) 0, display);
    controlWindow = GetDialog((WinInfoPtr) 0, display, windowName, 
	TBPALWINWIDTH, TBPALWINHEIGHT, TBPALWINWIDTH,
	TBPALWINHEIGHT, WINDBUF + WINRGB + WINFIXEDSIZE);
    
    if (!dialogExists)
    {
	long info;
	ObjPtr value;
	
	ObjPtr checkBox, icon, name, colorBar, titleBox, textBox, button;
	ObjPtr colorWheel, slider, radioGroup;
	int left, right, bottom, top;

	SetVar((ObjPtr) controlWindow, REPOBJ, display);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows controls for a color palette legend.  For information about any of the controls \
in the window, use Help In Context on the control.\n"));

	/*Add in a panel*/
	panel = NewPanel(greyPanelClass, 0, TBPALWINWIDTH, 0, TBPALWINHEIGHT);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);

	contents = GetVar(panel, CONTENTS);

	/*Add in the group of controls for text color*/
	left = MAJORBORDER;
	top = TBPALWINHEIGHT - MAJORBORDER;
	right = left + 4 * MAJORBORDER + COLORWHEELWIDTH + SLIDERWIDTH;
	
	titleBox = NewTitleBox(left, right,
			top - TITLEBOXTOP - MAJORBORDER - 2 * MINORBORDER - COLORWHEELWIDTH
				- CHECKBOXHEIGHT - TEXTBOXHEIGHT - TEXTBOXSEP,
			top, "Text Color");
	SetVar(titleBox, PARENT, panel);
	PrefixList(contents, titleBox);
	left += MAJORBORDER;
	right -= MAJORBORDER;
	top -= TITLEBOXTOP + MAJORBORDER;

	/*Get the color for priming the controls*/
	var = GetVar(display, COLOR);
	if (IsArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	}
	else if (IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	}
	else
	{
	    ReportError("ShowPaletteDisplayControls", "Bad color");
	    rgb[0] = rgb[1] = rgb[2] = 1.0;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Text Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	SetValue(colorWheel, var);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, display);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeTextColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the text and lines in the palette legend.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Text Color Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, display);
	SetMethod(slider, CHANGEDVALUE, ChangeTextColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the text and lines in the palette legend.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	left -= MINORBORDER;	
	right += MINORBORDER;
	
	/*Make the background controls*/
	top = TBPALWINHEIGHT - MAJORBORDER;
	right = TBPALWINWIDTH - MAJORBORDER;
	left = right - (4 * MAJORBORDER + COLORWHEELWIDTH + SLIDERWIDTH);
	
	titleBox = NewTitleBox(left, right,
			top - TITLEBOXTOP - MAJORBORDER - 2 * MINORBORDER - COLORWHEELWIDTH
				- CHECKBOXHEIGHT - TEXTBOXHEIGHT - TEXTBOXSEP,
			top, "Background");
	SetVar(titleBox, PARENT, panel);
	PrefixList(contents, titleBox);
	left += MAJORBORDER;
	right -= MAJORBORDER;
	top -= TITLEBOXTOP + MAJORBORDER;

	/*Get the color for priming the controls*/
	var = GetVar(display, BACKGROUND);
	if (var && IsArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	    hasBackground = true;
	}
	else if (var && IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	    hasBackground = true;
	}
	else
	{
	    rgb[0] = rgb[1] = rgb[2] = 0.5;
	    hasBackground = false;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Background Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	SetValue(colorWheel, var);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, display);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeBackgroundColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the background of the text box.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Background Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, display);
	SetMethod(slider, CHANGEDVALUE, ChangeBackgroundColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the background of the text box.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	left -= MINORBORDER;	
	right += MINORBORDER;
	/*Make the check box*/
	top -= COLORWHEELWIDTH + TEXTBOXSEP + TEXTBOXHEIGHT + MINORBORDER;
	checkBox = NewCheckBox(left, right, 
		top - CHECKBOXHEIGHT, top,
		"No background", hasBackground ? false : true);
	SetVar(checkBox, PARENT, panel);
	SetVar(checkBox, REPOBJ, display);
	SetVar(checkBox, SLIDER, slider);
	SetVar(checkBox, COLORWHEEL, colorWheel);
	PrefixList(contents, checkBox);
	SetMethod(checkBox, CHANGEDVALUE, ChangeNoBackground);
	SetVar(checkBox, HELPSTRING, NewString("This checkbox controls whether \
a background is shown.  If it is selected, no background is shown, and the \
objects behind can be seen."));

	/*Link the checkbox to the slider and color wheel*/
	SetVar(colorWheel, CHECKBOX, checkBox);
	SetVar(slider, CHECKBOX, checkBox);

	top -= CHECKBOXHEIGHT + MINORBORDER + MINORBORDER;
	left = MAJORBORDER;
	right = TBPALWINWIDTH - MAJORBORDER;
    }
    return ObjTrue;
}

/************************************************************* LINE COUNT */
#ifdef PROTO
int LineCount(char *text, int width)
#else
int LineCount(text, width)
char *text;
int width;
#endif
{
	char lineBuf[MAXLINE];
	int lineCount = 0;

	do {
		text = NextLine(text, width, lineBuf);
		if (*text == '\n') ++text;
		if (*lineBuf == '\0') break; /* no text or width too small */
		++lineCount;
	} while (*text);
	return lineCount;
}

/*************************************************************** NEXTLINE ******/

/*	This function assembles words into line buffer from text buffer. It
	returns a pointer to the next word in text or to terminator if done. 
	A newline indicates end of paragraph and terminates current line. The 
	return pointer points to the newline. All spaces are copied to the 
	new line except extra spaces that won't fit at the end of a line.  
	If a single word is too long for the line, it will be broken
	after the last letter that fits. If the line buffer is not big enough 
	to hold the number of characters that will fit in the width given, the 
	line will be terminated after the last word that fits.  Other than 
	newlines and soft eol (\r), no control characters should be in text.

	1/21/92: indentation scheme added. Now the tab character can be included
	in the text. There are fixed tab stops at multiples of INDENTAMT * width
	of a space. The amount of indentation after a tab is a function only of the
	number of tabs, not the size of the text between tabs; to avoid overlapping
	text, it may be necessary to use multiple tabs. After the last tab, wrapped 
	lines will continue to be indented to that tab stop until a newline is 
	encountered. Indentation only applies to left-aligned text.  
*/

static char *NextLine(text, width, line)
char *text; 	/* pointer into source text buffer */
int width;	 	/* width of destination space in pixels */
char *line; 	/* line output buffer MAXLINE chars long */
{
	char *curWord;
	int inSpace;
	int totWid, n, m, spWid;

	spWid = ChrWidth(' ');
	curWord = text;	/* point to beginning of current word in input text */
	totWid = n = m = 0;	/* reset offsets in output line */
	inSpace = TRUE;	/* reading space flag */
	/* move words until:
		-fill up line buffer
		-width of characters in line buffer exceed given width
		-run out of input text
		-encounter newline
	*/
	while (n < MAXLINE - 1)
	{
		if (*text == '\r') /* "soft" EOL -- ignore */
		{
			++text;
		}
		else if (*text == '\t')
		{
			/* terminate output here */
			if (inSpace) line[m] = '\0';
			else line[n] = '\0';
			return text;
		}
		else if (*text == ' ')
		{
			if (!inSpace)
			{
				/* the previous word fit; update fallback pointer */
				m = n;	/* where line will end if next word doesn't fit */
				inSpace = TRUE;
			}
			curWord = text++;
			line[n++] = ' ';
			totWid += spWid;
		}
		else if (*text == '\n')
		{
			/* terminate line right here */
			if (inSpace) line[m] = '\0';
			else line[n] = '\0';
			return text;
		}
		else if (*text == '\0')
		{
			/* terminate line right here */
			if (inSpace) line[m] = '\0';
			else line[n] = '\0';
			return text;
		}
		else
		{
			/* non-space; move it */
			totWid += ChrWidth(*text);
			if (totWid > width - indent) break;
			if (inSpace)	/* beginning new word */
			{
				curWord = text;
				inSpace = FALSE;
			}
			line[n++] = *text++;
		}
	}
	if (m == 0 && !inSpace)	/* single word too long for line, break it */
	{
		line[n] = '\0';
		return text;
	}
	else
	{
		line[m] = '\0';
		if (inSpace)	/* skip to next word */
		{
			while (*curWord==' ') ++curWord;
		}
		return curWord;
	}
}
