/*ScianDraw.c
  Drawing routines for various types of objects
  Eric Pepke
  March 28, 1990
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianArrays.h"
#include "ScianLists.h"
#include "ScianDraw.h"
#include "ScianIDs.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianColors.h"
#include "ScianPictures.h"

#ifdef GRAPHICS
Bool drawMouse = false;				/*True iff draw on next mouse*/
extern int curMouse;				/*Current mouse button*/

static int translations[200][2];		/*Translations recorded so far*/
static int curTranslation = 0;
static Screencoord vpl, vpr, vpb, vpt;		/*Current viewport bounds*/
static int screenmasks[200][4];			/*Screen masks recorded so far*/
static int originX, originY;
static int curScreenmask = 0;
static Bool inSpace[200];					/*True iff we're in a space*/
static int curInSpace = 0;
static Matrix savedMatrices[20];
static int curMatrix = 0;

Matrix Identity = {{1.0, 0.0, 0.0, 0.0},
		   {0.0, 1.0, 0.0, 0.0},
		   {0.0, 0.0, 1.0, 0.0},
		   {0.0, 0.0, 0.0, 1.0}};

#ifdef PROTO
void RegisterInSpace(Bool whether)
#else
void RegisterInSpace(whether)
Bool whether;
#endif
/*Registers inSpace as whether*/
{
    inSpace[++curInSpace] = whether;
}

void EndRegister()
/*Ends a registration of in space or not*/
{
    --curInSpace;
}

void MyPushMatrix()
{
    getmatrix(savedMatrices[curMatrix++]);
}

void MyPopMatrix()
{
    loadmatrix(savedMatrices[--curMatrix]);
}

void CurOffset(x, y)
int *x, *y;
/*Returns the current offset of the internal mouse coords from global*/
{
    if (inSpace[curInSpace])
    {
	*x = 0;
	*y = 0;
    }
    else
    {
	*x = originX + translations[curTranslation][0];
	*y = originY + translations[curTranslation][1];
    }
}

void UpdateDrawing(void)
/*Doesn't do a damn thing*/
{
}

void UD(void)
/*Updated drawing according to belated DrawMe's*/
{
    if (drawMouse)
    {
	InitClipRect();
	IdleAllWindows();
	ResetClipRect();
	drawMouse = false;
    }
}

Bool Mouse(x, y)
int *x, *y;
/*Returns in x and y the current mouse position in local coordinates*/
{
    long ox, oy;

    UD();

    getorigin(&ox, &oy);

    if (inSpace[curInSpace])
    {
	*x = getvaluator(MOUSEX) - ox;
	*y = getvaluator(MOUSEY) - oy;
    }
    else
    {
	*x = getvaluator(MOUSEX) - ox - originX - translations[curTranslation][0];
	*y = getvaluator(MOUSEY) - oy - originY - translations[curTranslation][1];
    }
    return getbutton(curMouse);
}

void SetOrigin(x, y)
int x, y;
/*Sets the origin to x, y WITHIN current translation*/
{
    ++curTranslation;
    translations[curTranslation][0] = translations[curTranslation - 1][0] + x;
    translations[curTranslation][1] = translations[curTranslation - 1][1] + y;
    translate((Coord) x, (Coord) y, 0.0);
}

void GetOrigin(x, y)
int *x, *y;
/*Gets the current origin*/
{
    *x = originX;
    *y = originY;
}

void RestoreOrigin()
/*Restores the origin to what it was*/
{
    translate((Coord)  (translations[curTranslation - 1][0] -
			translations[curTranslation][0]),
	      (Coord)  (translations[curTranslation - 1][1] -
			translations[curTranslation][1]), 0.0);
    --curTranslation;
}

void InitClipRect()
/*Initializes the clipping rects*/
{
    MyPushMatrix();

    /*Make lower left 0, 0*/
    getviewport(&vpl, &vpr, &vpb, &vpt);

    /*First translation is the offset within the current viewport*/
    ++curTranslation;
    translations[curTranslation][0] = 0;
    translations[curTranslation][1] = 0;

    /*First screenmask*/
    ++curScreenmask;
    screenmasks[curScreenmask][0] = vpl;
    screenmasks[curScreenmask][1] = vpr;
    screenmasks[curScreenmask][2] = vpb;
    screenmasks[curScreenmask][3] = vpt;
    originX = vpl;
    originY = vpb;
}

void ResetClipRect()
/*Resets the clip rect to what it was before*/
{
    int sl, sr, sb, st;

    sl = screenmasks[curScreenmask][0];
    sr = screenmasks[curScreenmask][1];
    sb = screenmasks[curScreenmask][2];
    st = screenmasks[curScreenmask][3];

    getviewport(&vpl, &vpr, &vpb, &vpt);
    
    originX = vpl;
    originY = vpb;

    scrmask(sl, sr, sb, st);
    --curScreenmask;
    --curTranslation;
    MyPopMatrix();
}

#ifdef PROTO
void SetClipRect(int left, int right, int bottom,
	int top)
#else
void SetClipRect(left, right, bottom, top)
int left, right, bottom, top;
#endif
/*Sets the current clipping rectangle to left, right, bottom, top*/
{
    int sl, sr, sb, st;

    left += translations[curTranslation][0];
    right += translations[curTranslation][0];
    bottom += translations[curTranslation][1];
    top += translations[curTranslation][1];

    left += originX;
    right += originX;
    bottom += originY;
    top += originY;

    sl = screenmasks[curScreenmask][0];
    sr = screenmasks[curScreenmask][1];
    sb = screenmasks[curScreenmask][2];
    st = screenmasks[curScreenmask][3];

    if (left > sl) sl = left;
    if (right < sr) sr = right;
    if (bottom > sb) sb = bottom;
    if (top < st) st = top;

    ++curScreenmask;
    scrmask(sl, sr, sb, st);

    screenmasks[curScreenmask][0] = sl;
    screenmasks[curScreenmask][1] = sr;
    screenmasks[curScreenmask][2] = sb;
    screenmasks[curScreenmask][3] = st;
}

void RestoreClipRect()
/*Restores the clipping rectangle to what it was before*/
{
    int sl, sr, sb, st;

    --curScreenmask;
    sl = screenmasks[curScreenmask][0];
    sr = screenmasks[curScreenmask][1];
    sb = screenmasks[curScreenmask][2];
    st = screenmasks[curScreenmask][3];

    scrmask(sl, sr, sb, st);
}

void DrawObj2(origObj, obj)
ObjPtr origObj, obj;
/*Draws origObj according to method in obj*/
{
    FuncTyp DrawFunction;			/*Method*/

    if (obj)
    {
	DrawObj2(origObj, ClassOf(obj));
	DrawFunction = Get1Method(obj, DRAW);
	if (DrawFunction)
	{
	    (*DrawFunction)(origObj);
	}
    }
}

void DrawObject(obj)
ObjPtr obj;
/*Draws obj.*/
{
    ObjPtr picState;
    picState = GetVar(obj, PICSTATE);
    if (picState)
    {
	SetVar(obj, PICSTATE, 
		   NewInt(GetInt(picState) & FG ? FGBG : FGBB));
    }

    if (GetPredicate(obj, INHIBITDRAWING))
    {
	return;
    }

    if (GetPredicate(obj, HIDDEN))
    {
	return;
    }

    if (GetPredicate(obj, MULTIDRAW))
    {
	DrawObj2(obj, obj);
    }
    else
    {
	FuncTyp DrawFunction;			/*Method*/

	DrawFunction = GetMethod(obj, DRAW);
	if (DrawFunction)
	{
	    (*DrawFunction)(obj);
	}
    }
}

void ClipToMe(object)
ObjPtr object;
/*Sets the clipping rectangle to object*/
{
    int l, r, b, t;

    Get2DIntBounds(object, &l, &r, &b, &t);

    SetClipRect(l, r, b, t);
}
#endif

#ifdef INTERACTIVE
ObjPtr PressObject(obj, x, y, flags)
ObjPtr obj;
int x, y;
long flags;
/*Presses in obj at x, y*/
{
    FuncTyp PressFunction;
    PressFunction = GetMethod(obj, PRESS);
    if (PressFunction)
    {
	return (*PressFunction)(obj, x, y, flags);
    }
    else
    {
	return ObjFalse;
    }
}

ObjPtr DropObjects(obj, dropObj, x, y)
ObjPtr obj, dropObj;
int x, y;
/*Drops dropObj in obj at x, y*/
{
    FuncTyp method;
    method = GetMethod(obj, DROPOBJECTS);
    if (method)
    {
	return (*method)(obj, dropObj, x, y);
    }
    else
    {
	return ObjFalse;
    }
}

#ifdef PROTO
ObjPtr KeyDownObject(ObjPtr obj, short key, int flags)
#else
ObjPtr KeyDownObject(obj, key, flags)
ObjPtr obj;
short key;
int flags;
#endif
/*Does a keydown in obj*/
{
    FuncTyp method;
    method = GetMethod(obj, KEYDOWN);
    if (method)
    {
	return (*method)(obj, key, flags);
    }
    else
    {
	return ObjFalse;
    }
}
#endif

#ifdef GRAPHICS
void FrameUIRect(left, right, bottom, top, uiColor)
int left, right, bottom, top, uiColor;
/*Frames a user interface rectangle*/
{
    SetUIColor(uiColor);
    recti(left, bottom, right, top);
}

void FrameRect(left, right, bottom, top)
int left, right, bottom, top;
/*Frames a rectangle*/
{
    recti(left, bottom, right, top);
}

void FillUIRect(left, right, bottom, top, uiColor)
int left, right, bottom, top, uiColor;
/*Fills a user interface rectangle*/
{
    SetUIColor(uiColor);
    rectfi(left, bottom, right, top);
}

void FillRect(left, right, bottom, top)
int left, right, bottom, top;
/*Fills a rectangle*/
{
    rectfi(left, bottom, right, top);
}

void FillUIGauzeRect(left, right, bottom, top, uiColor)
int left, right, bottom, top, uiColor;
/*Fills a user interface rectangle as if it were gauze*/
{
    SetUIColor(uiColor);
    setpattern(GREYPAT);
    rectfi(left, bottom, right, top);
    setpattern(SOLIDPAT);
}

void DrawUILine(x1, y1, x2, y2, uiColor)
int x1, y1, x2, y2, uiColor;
{
    SetUIColor(uiColor);
    move2i(x1, y1);
    draw2i(x2, y2);
}

void DrawLine(x1, y1, x2, y2)
int x1, y1, x2, y2;
{
    move2i(x1, y1);
    draw2i(x2, y2);
}

void DrawVCursor(x, yt, yb)
int x, yt, yb;
/*Draws a vertical cursor*/
{
    SetUIColor(UIORANGE);
    move2i(x, yt);
    draw2i(x, yb);
    SetUIColor(UIBLACK);
    move2i(x + 1, yt);
    draw2i(x + 1, yb);
}

#ifdef FONTS4D
extern fmfonthandle currentFontHandle;
#endif

#ifdef PROTO
void DrawSpaceString(real x, real y, real z, char *s)
#else
void DrawSpaceString(x, y, z, s)
real x, y, z;
char *s;
#endif
/*Draws a string beginning at x, y, z in a space*/
{

    cmov((Coord) x, (Coord) y, (Coord) z);
#ifdef FONTS4D
	if (currentFontHandle)
	{
		fmprstr(s);
	}
	else
	{
		charstr(s);
	}
#else
	charstr(s);
#endif
}

#ifdef PROTO
void DrawSpaceLine(real x1, real y1, real z1, real x2, real y2, real z2, int uiColor)
#else
void DrawSpaceLine(x1, y1, z1, x2, y2, z2, uiColor)
real x1, y1, z1, x2, y2, z2;
int uiColor;
#endif
/*Draws a line in a space*/
{
    SetUIColor(uiColor);
    move((Coord) x1, (Coord) y1, (Coord) z1);
    draw((Coord) x2, (Coord) y2, (Coord) z2);
}

#ifdef PROTO
void DrawWFSphere(real x, real y, real z, real radius, int uiColor)
#else
void DrawWFSphere(x, y, z, radius, uiColor)
real x, y, z, radius;
int uiColor;
#endif
/*Draws a wire frame sphere*/
{
    SetUIColor(uiColor);
    pushmatrix();
    circ(x, y, radius);
    translate(0.0, 0.0, radius * SQ22);
    circ(x, y, radius * SQ22);
    translate(0.0, 0.0, -radius * 2.0 * SQ22);
    circ(x, y, radius * SQ22);
    translate(0.0, 0.0, radius * SQ22);
    rotate(900, 'y');
    circ(x, y, radius);
    rotate(450, 'x');
    circ(x, y, radius);
    rotate(450, 'x');
    circ(x, y, radius);
    rotate(450, 'x');
    circ(x, y, radius);
    popmatrix();
}
#endif
