/*---------------------------------------------------------------------------*/
/*                                                                           */
/* File:     TEST_IM.C                                                       */
/*                                                                           */
/* Purpose:  The purpose of this program is to test the image routines.      */
/*                                                                           */
/* Contains: main                                                            */
/*           Create                                                          */
/*           Open                                                            */
/*           Close                                                           */
/*           Read                                                            */
/*           Write                                                           */
/*           GetPix                                                          */
/*           PutPix                                                          */
/*           Header                                                          */
/*           Dim                                                             */
/*           Bounds                                                          */
/*           GetDesc                                                         */
/*           GetTitle                                                        */
/*           PutTitle                                                        */
/*           InfoIds                                                         */
/*           GetInfo                                                         */
/*           PutInfo                                                         */
/*           CopyInfo                                                        */
/*           Error                                                           */
/*                                                                           */
/* Author:   John Gauch                                                      */
/*                                                                           */
/* Date:     March 22, 1987                                                  */
/*                                                                           */
/*---------------------------------------------------------------------------*/

/* Include files */
#include <stdio.h>
#include <math.h>
#include "image.h"

/* Global variables */
IMAGE *image1;
GREYTYPE *pixel1;

char *malloc();

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This program is used to test the IMAGE routines.  It uses a     */
/*	     crude menu based interface.                                     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
main()
   {
   /* Local Variables */
   int Test = -1;

   /* Loop until user quits */
   while (Test != 0)
      {
      /* Print menu */
      printf("\nIMAGE Test Program\n");
      printf("-----------------\n\n");
      printf(" 0)  Quit\n");
      printf(" 1)  Create\n");
      printf(" 2)  Open\n");
      printf(" 3)  Close\n");
      printf(" 4)  Read\n");
      printf(" 5)  Write\n");
      printf(" 6)  GetPix\n");
      printf(" 7)  PutPix\n");
      printf(" 8)  Header\n");
      printf(" 9)  Dim\n");
      printf("10)  Bounds\n");
      printf("11)  GetDesc\n");
      printf("12)  GetTitle\n");
      printf("13)  PutTitle\n");
      printf("14)  InfoIds\n");
      printf("15)  GetInfo\n");
      printf("16)  PutInfo\n");
      printf("17)  CopyInfo\n");
      printf("18)  Error\n");

      /* Read user selection */
      printf("\nEnter test number:");
      scanf("%d", &Test);
      printf("\n");

      /* Handle the test cases */
      switch(Test){
         case  0:  break;
         case  1:  Create(); break;
         case  2:  Open(); break;
         case  3:  Close(); break;
         case  4:  Read(); break;
         case  5:  Write(); break;
         case  6:  GetPix(); break;
         case  7:  PutPix(); break;
         case  8:  Header(); break;
         case  9:  Dim(); break;
         case 10:  Bounds(); break;
         case 11:  GetDesc(); break;
         case 12:  GetTitle(); break;
         case 13:  PutTitle(); break;
         case 14:  InfoIds(); break;
         case 15:  GetInfo(); break;
         case 16:  PutInfo(); break;
         case 17:  CopyInfo(); break;
         case 18:  Error(); break;
         default:  printf("Invalid test number\n"); break;
         }
      }
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine creates an image.                                  */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Create()
   {
   /* Local variables */
   char name1[80];
   int protection;
   char pixformatstr[20];
   int pixformat= -1;
   int dimc;
   int dimv[nDIMV];
   int i;

   /* Get image file names */
   printf("Enter image name:");
   scanf("%s", name1);

   /* Get protection mode */
   printf("Enter protection:");
   scanf("%o", &protection);

   /* Get open mode */
   printf("Enter pixel format (GREY or REAL):");
   scanf("%s", pixformatstr);
   if (pixformatstr[0] == 'G') pixformat = GREY;
   if (pixformatstr[0] == 'R') pixformat = REAL;

   /* Get image dimensions */
   printf("Enter dimc:");
   scanf("%d", &dimc);
   printf("Enter dimv:");
   for (i=0; i<dimc; i++)
      scanf("%d", &dimv[i]);

   /* Create image files */
   if ((image1 = imcreat(name1,protection, pixformat, dimc, dimv)) == NULL){
      fprintf(stderr,"Can not create image\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine opens an image.                                    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Open()
   {
   /* Local variables */
   char name1[80];
   char modestr[20];
   int mode = -1;

   /* Get image file names */
   printf("Enter image name:");
   scanf("%s", name1);

   /* Get open mode */
   printf("Enter open mode (UPDATE or READ):");
   scanf("%s", modestr);
   if (modestr[0] == 'U') mode = UPDATE;
   if (modestr[0] == 'R') mode = READ;

   /* Open image files */
   if ((image1 = imopen(name1,mode)) == NULL){
      fprintf(stderr,"Can not open image\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine closes an image.                                   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Close()
   {
   /* Close image */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine gets the image header.                             */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Header()
   {
   int pixformat;
   int pixsize;
   int pixcnt;
   int dimc;
   int dimv[nDIMV];
   int maxmin[nMAXMIN];
   int i;

   /* Get header info */
   if (imheader(image1, &pixformat, &pixsize, &pixcnt, &dimc, dimv, maxmin) 
   == INVALID){
      fprintf(stderr,"Can not get image header\n"); 
      return(1);}

   /* Print header info */
   printf("pixformat = %d\n", pixformat);
   printf("pixsize = %d\n", pixsize);
   printf("pixcnt = %d\n", pixcnt);
   printf("dimc = %d\n", dimc);
   for (i=0; i<dimc; i++)
      printf("dimv[%d] = %d\n", i, dimv[i]);
   printf("maxmim = %d %d \n", maxmin[0], maxmin[1]);

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine gets the image dimension count and pixel format.   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Dim()
   {
   int pixformat;
   int dimc;

   /* Get header info */
   if (imdim(image1, &pixformat, &dimc) == INVALID){
      fprintf(stderr,"Can not get image dimension\n"); 
      return(1);}

   /* Print header info */
   printf("pixformat = %d\n", pixformat);
   printf("dimc = %d\n", dimc);

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine gets the image dimension vector.                   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Bounds()
   {
   int dimv[nDIMV];
   int i;

   /* Get header info */
   if (imbounds(image1, dimv) == INVALID){
      fprintf(stderr,"Can not get image dimension\n"); 
      return(1);}

   /* Print header info */
   for (i=0; i<nDIMV; i++)
      printf("dimv[%d] =%d\n", i, dimv[i]);

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine gets the image maxmin or histogram.                */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetDesc()
   {
   char typestr[20];
   int type = -1;
   int buffer[nHISTOGRAM];
   int i;

   /* Get description type */
   printf("Enter description type (MINMAX or HISTO):");
   scanf("%s", typestr);
   if (typestr[0] == 'M') type = MINMAX;
   if (typestr[0] == 'H') type = HISTO;

   /* Get header info */
   if (imgetdesc(image1, type, buffer) == INVALID){
      fprintf(stderr,"Can not get image description\n"); 
      return(1);}

   /* Print maxmin */
   if (type == MINMAX)
      printf("maxmim = %d %d \n", buffer[0], buffer[1]);

   /* Print histogram */
   else if (type == HISTO)
      {
      for (i=0; i<nHISTOGRAM; i++)
         printf("%d", buffer[i]);
      printf("\n");
      }

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads image pixels.                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Read()
   {
   int low;
   int high;
   int pixcnt;

   /* Get endpoints */
   printf("Enter endpoints for pixels:");
   scanf("%d %d", &low, &high);

   /* Allocate space for pixels */
   pixcnt = high - low + 1;
   pixel1 = (GREYTYPE *)malloc((unsigned)pixcnt*sizeof(GREYTYPE));
   if (pixel1 == NULL) {
      fprintf(stderr,"Can not allocate pixels\n"); 
      return(1);}

   /* Get image title */
   if (imread(image1, low, high, pixel1) == INVALID){
      fprintf(stderr,"Can not get pixels\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes image pixels.                               */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Write()
   {
   int low;
   int high;

   /* Get endpoints */
   printf("Enter endpoints for pixels:");
   scanf("%d %d", &low, &high);

   /* Get image title */
   if (imwrite(image1, low, high, pixel1) == INVALID){
      fprintf(stderr,"Can not get pixels\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads image pixels.                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetPix()
   {
   int endpts[nDIMV][2];
   int coarseness[nDIMV];
   int length;
   int pixcnt;
   int i;

   /* Get endpoints and coarseness */
   pixcnt = 1;
   for (i=0; i<image1->Dimc; i++)
      {
      printf("Enter endpoints for dimension %d:", i);
      scanf("%d %d", &endpts[i][0], &endpts[i][1]);
      coarseness[i] = 1;

      /* Determine number of pixels */
      length = (endpts[i][1] - endpts[i][0] + 1);
      if (length <= 0) {
         fprintf(stderr,"Bad endoints\n"); 
         return(1);}
      pixcnt = pixcnt * length;
      }

   /* Allocate space for pixels */
   pixel1 = (GREYTYPE *)malloc((unsigned)pixcnt*sizeof(GREYTYPE));
   if (pixel1 == NULL) {
      fprintf(stderr,"Can not allocate pixels\n"); 
      return(1);}

   /* Get image title */
   if (imgetpix(image1, endpts, coarseness, pixel1) == INVALID){
      fprintf(stderr,"Can not get pixels\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes image pixels.                               */
/*                                                                           */
/*---------------------------------------------------------------------------*/
PutPix()
   {
   int endpts[nDIMV][2];
   int coarseness[nDIMV];
   int length;
   int i;

   /* Get endpoints and coarseness */
   for (i=0; i<image1->Dimc; i++)
      {
      printf("Enter endpoints for dimension %d:", i);
      scanf("%d %d", &endpts[i][0], &endpts[i][1]);
      coarseness[i] = 1;

      /* Determine number of pixels */
      length = (endpts[i][1] - endpts[i][0] + 1);
      if (length <= 0) {
         fprintf(stderr,"Bad endoints\n"); 
         return(1);}
      }

   /* Get image title */
   if (imputpix(image1, endpts, coarseness, pixel1) == INVALID){
      fprintf(stderr,"Can not put pixels\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine prints the image title.                            */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetTitle()
   {
   char title[nTITLE];

   /* Get image title */
   if (imgettitle(image1, title) == INVALID){
      fprintf(stderr,"Can not get image title string\n"); 
      return(1);}

   /* Print image title */
   printf("title = %s\n", title);

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine assigns the image title.                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/
PutTitle()
   {
   char title[nTITLE];

   /* Get image title */
   printf("Enter image title:");
   scanf("%s", title);
   
   /* Assign image title */
   if (imputtitle(image1, title) == INVALID){
      fprintf(stderr,"Can not put image title string\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine prints all information field names.                */
/*                                                                           */
/*---------------------------------------------------------------------------*/
InfoIds()
   {
   char **name;
   int i;

   /* Get information field names */
   if ((name = (char **)iminfoids(image1)) == NULL){
      fprintf(stderr,"Can not get information field\n"); 
      return(1);}

   /* Print field names */
   i = 0;
   while ((*name) != '\0')
      {
      printf("name[%d] = %s\n", i, *name);
      i++;
      name++;
      }
      
   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine prints the information field.                      */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetInfo()
   {
   char name[20];
   char *data;

   /* Get field name */
   printf("Enter information field name:");
   scanf("%s", name);
   
   /* Get field data */
   if ((data = imgetinfo(image1, name)) == NULL){
      fprintf(stderr,"Can not get image information string\n"); 
      return(1);}

   /* Print field data */
   printf("data = %s\n", data);

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine assigns the information field.                     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
PutInfo()
   {
   char name[20];
   char data[200];

   /* Get field name */
   printf("Enter information field name:");
   scanf("%s", name);
   
   /* Get field data */
   printf("Enter information field data:");
   scanf("%s", data);
   
   /* Assign information field */
   if (imputinfo(image1, name, data) == INVALID){
      fprintf(stderr,"Can not put image information string\n"); 
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine copys all info fields from one image to another.   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
CopyInfo()
   {
   /* Local variables */
   IMAGE *image1;
   char name1[80];
   IMAGE *image2;
   char name2[80];

   /* Get image file names */
   printf("Enter source image name:");
   scanf("%s", name1);
   printf("Enter destination image name:");
   scanf("%s", name2);

   /* Open image files */
   if ((image1 = imopen(name1,READ)) == NULL){
      fprintf(stderr,"Can not open source image\n"); 
      return(1);}
   if ((image2 = imopen(name2,UPDATE)) == NULL){
      fprintf(stderr,"Can not open destination image\n"); 
      return(1);}

   /* Copy info fields to specified image */
   if (imcopyinfo(image1,image2) == INVALID) {
      fprintf(stderr, "Copy error: %s\n", _imerrbuf);
      return(1);}

   /* Close images */
   if (imclose(image1) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}
   if (imclose(image2) == INVALID){
      fprintf(stderr,"Can not close image\n");
      return(1);}

   return(0);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine prints the image error buffer.                     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
Error()
   {
   /* Get error string */
   if (imerror(image1) == NULL){
      fprintf(stderr,"Can not get image error string\n"); 
      return(1);}

   /* Print error string */
   printf("error = %s\n", imerror(image1));

   return(0);
   }

