/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *     Copyright (c) 1990,1991,1992,1993 Bellcore           *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *     dfs@bellcore.com      dcook@stat.rutgers.edu         *
 *      (201) 829-4263                                      *
 *                                                          *
 ************************************************************/

/*
 * The code in this file was written by Nancy Hubbell during the
 * summer of 1990.
*/

#include <stdio.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"

/* Functions used in this file */
float NiceValue();
double ceil(), floor();
void find_tick_label();
void plane_to_screen();
/* */

XSegment tick_segs[100];
static lcoords axes[3];

/* --------- Dynamic allocation section ----------- */
void
alloc_axis_arrays(xg)
/*
 * Dynamically allocate arrays.
*/
  xgobidata *xg;
{
  Cardinal nc = (Cardinal) xg->ncols;

  xg->nicelim = (lims *) XtMalloc(nc * sizeof(lims));
  xg->tickdelta = (float *) XtMalloc(nc * sizeof(float));
  xg->deci = (int *) XtMalloc(nc * sizeof(int));
}

void
free_axis_arrays(xg)
/*
 * Dynamically free arrays.
*/
  xgobidata *xg;
{
  XtFree((XtPointer) xg->nicelim);
  XtFree((XtPointer) xg->tickdelta);
  XtFree((XtPointer) xg->deci);
}

/* --------- End of dynamic allocation section ----------- */

static void
make_axes()
/*
 * Calculate plane-coordinates of axes
*/
{
  float precis = PRECISION1;

/*   .(axes[0].x, axes[0].y
 *   |
 *   |
 *   |
 *   |
 *   |
 *   |
 *   .----------------. (axes[2].x, axes[2].y)
 * (axes[1].x, axes[1].y)
*/

  axes[2].x = (long)(precis + (0.1 * precis));
  axes[0].y = (long)(precis + (0.1 * precis));

}

void
convert_axes(tix, xg)
/*
 * convert plane-axes to screen-axes
*/
  tickinfo *tix;
  xgobidata *xg;
{
  long ltmp;
  icoords tmin;

  ltmp = axes[2].x + xg->shift_wrld.x * (1 - xg->scale_cntr);
  xg->screen_axes[2].x = (int) ((ltmp * xg->is.x) >> EXP1);
  xg->screen_axes[2].x += (xg->mid.x + xg->shift_scrn.x * xg->scale_cntr);

  /*
   * Note:  this is a subtraction though it's an addition for the points.
  */
  ltmp = axes[0].y - xg->shift_wrld.y * (1-xg->scale_cntr);
  xg->screen_axes[0].y = (int) ((ltmp * xg->is.y) >> EXP1);
  xg->screen_axes[0].y += (xg->mid.y + xg->shift_scrn.y * xg->scale_cntr);

/*
 * Find the screen value of the minimum x and y coordinates,
 * using the fact that the minimum value in world coordinates for
 * every column is -PRECISION1.
*/

  if (xg->is_xyplotting)
  {
    ltmp = (long) -PRECISION1 + xg->shift_wrld.x * (1-xg->scale_cntr);
    tmin.x = (int) ((ltmp * xg->is.x) >> EXP1);
    tmin.x += (xg->mid.x + xg->shift_scrn.x * xg->scale_cntr);

    if (tmin.x - tix->screen[0].x < 10)
      xg->screen_axes[0].x = xg->screen_axes[1].x = tix->screen[0].x - 10;
    else
      xg->screen_axes[0].x = xg->screen_axes[1].x = tix->screen[0].x;
  }
  else if (xg->is_dotplotting)
  {
    int i, min;
    /*
     * Find the minimum screen value of the data and subtract 10.
    */
    min = xg->screen[0].x ;
    for (i=0; i<xg->nrows_in_plot; i++)
      if (xg->screen[i].x < min)
        min = xg->screen[i].x;
    xg->screen_axes[0].x = xg->screen_axes[1].x = min - 10 ;
  }

  ltmp = (long) -PRECISION1 - xg->shift_wrld.y * (1-xg->scale_cntr);
  tmin.y = (int) ((ltmp * xg->is.y) >> EXP1);
  tmin.y += (xg->mid.y + xg->shift_scrn.y * xg->scale_cntr);

  if (tix->screen[0].y - tmin.y < 10)
    xg->screen_axes[2].y = xg->screen_axes[1].y = tix->screen[0].y + 10;
  else
    xg->screen_axes[2].y = xg->screen_axes[1].y = tix->screen[0].y;
}

int
set_deci(tickdelt)
  float tickdelt;
/*
 * Figure out how many decimal places each tickdelta contains
*/
{
  int j;
  int is_set;
  float deci;
  double ddelt = (double) tickdelt;

  j=0;
  is_set = 0;
  while (!is_set)
  {
    if (j == 10 ||
      (fabs(ddelt) > .1 &&
        ( fabs(ddelt - floor(ddelt)) < .0001 ||
          fabs(ddelt - ceil(ddelt)) < .0001 )))
    {
      deci = j;
      is_set = 1;
    }
    else
    {
      ddelt = 10.0 * ddelt;
      j++;
    }
  }
  return(deci);
}

void
number_length(xy_tick, length)
  float xy_tick;
  int *length;
/* find the length of the number */
{
  int j;
  int is_num;
  float tmpnum;

  j = 0;
  tmpnum = xy_tick;
  is_num = 0;
  while (!is_num)
  {
    if ((int)tmpnum == 0)
    {
      *length = j;
      is_num = 1;
    }
    else
    {
      tmpnum = tmpnum/10;
      j++;
    }
  }
}

int
plot_too_big(xg)
/*
 * Find out if the labels are too close to the left or the bottom
 * of the plotting window.
*/
  xgobidata *xg;
{
  int shrink = 0;
/*
 * Make sure the code here matches add_x_axis() and add_y_axis() in
 * plot_once.c.
*/
  if ((xg->screen_axes[1].x - xg->maxwidth - 7) < 5)
  {
    shrink = 1;
  }

  if (shrink == 0)
  {
    if ((xg->screen_axes[1].y + 2*FONTHEIGHT(appdata.plotFont) + 14) >
        (xg->max.y - 5) )
    {
       shrink = 1;
    }
  }
  return(shrink);
}


void
extend_axes(xindx, yindx, tix, xg)
/*
 * If the last tick is larger than the end of an axis, then extend the axis.
*/
  int xindx, yindx;
  tickinfo *tix;
  xgobidata *xg;
{
  int k;

  if (xg->is_xyplotting)
  {
    k = tix->screen[tix->nticks[xindx]-1].x;
    if (xg->screen_axes[2].x < k)
      xg->screen_axes[2].x = k;
  }

  k = tix->screen[tix->nticks[yindx]-1].y;
  if (xg->screen_axes[0].y > k)
    xg->screen_axes[0].y = k;
}

int
find_minindex(limits, xg)
/*
 * Find the index of the column that has the smallest lim[].min.
*/
  lims *limits;
  xgobidata *xg;
{
  int j, minindex;
  float ftmp;

  minindex = 0;
  ftmp = limits[0].min;
  for (j=0; j<xg->ncols_used; j++)
  {
    if (limits[j].min < ftmp)
    {
      ftmp = limits[j].min;
      minindex = j;
    }
  }
  return(minindex);
}

int
find_maxwidth(limits, xg)
/*
 * Find the longest tick label for all of the variables, and
 * use maxwidth to determine whether the plot needs to be
 * repositioned.
*/
  lims *limits;
  xgobidata *xg;
{
  int j;
  float tmp, ftmp;
  char str[50];
  int width, length, maxwidth;

  maxwidth = 0;
  for (j=0; j<xg->ncols_used; j++)
  {
    tmp = xg->nicelim[j].min;
    while (tmp + xg->tickdelta[j] <=
         limits[j].max + xg->tickdelta[j]/3)
    {
      tmp = tmp + xg->tickdelta[j];
    }

    if ((fabs((double) tmp) >= 10000. ||
       xg->deci[j] > 4) &&
       tmp != 0.0)
    {

      /* Use exponential notation */
      (void) number_length(tmp, &length);
      ftmp = tmp/ (float) pow((double)10, (double)(length-1));
      if (ftmp == (float)(int)ftmp)
        (void) sprintf (str, "%.0e", tmp);
      else
        (void) sprintf (str, "%.2e", tmp);
    }
    else
    {
      switch(xg->deci[j])
      {
        case 0:
          (void) sprintf(str, "%3.0f", tmp);
          break;
        case 1:
          (void) sprintf(str, "%3.1f", tmp);
          break;
        case 2:
          (void) sprintf(str, "%3.2f", tmp);
          break;
        case 3:
          (void) sprintf(str, "%3.3f", tmp);
          break;
        case 4:
          (void) sprintf(str, "%3.4f", tmp);
          break;
      }
    }
    width = XTextWidth(appdata.plotFont, str, strlen(str));
    if (width > maxwidth)
      maxwidth = width;
  }
  return(maxwidth);
}


float
NiceValue(x)
  float x;
{
  float fvalue;

  if (x <= 0)
    fvalue = 0.0;
  else
  {
    double dx = (double) x;
    double dilx, lx, vv1, vv2, vv3, vv4;

    lx = log(dx) / log(10.0);
    dilx = floor(lx);

    vv1 = pow(10.0, dilx);
    vv2 = vv1 * 2.0;
    vv3 = vv1 * 5.0;
    vv4 = vv1 * 10.0;

    if ((fabs(dx - vv1) < fabs(dx - vv2))
     && (fabs(dx - vv1) < fabs(dx - vv3))
     && (fabs(dx - vv1) < fabs(dx - vv4)))
      fvalue = (float) vv1;

    else if ((fabs(dx - vv2) < fabs(dx - vv3))
        && (fabs(dx - vv2) < fabs(dx - vv4)))
      fvalue = (float) vv2;

    else if (fabs(dx - vv3) < fabs(dx - vv4))
      fvalue = (float) vv3;

    else
      fvalue = (float) vv4;
  }

  return(fvalue);
}

void
SetNiceRange(j, xg)
/*
 * ticks: number of ticks used by each column
 * tickdelta: increment between subsequent ticks for each column
*/
  int j;
  xgobidata *xg;
{
  float nicearg;

  xg->ticks.nticks[j] = 4;

  nicearg = (xg->nicelim[j].max - xg->nicelim[j].min) /
            (float) (xg->ticks.nticks[j] - 1) ;
  xg->tickdelta[j] = NiceValue( nicearg );
  xg->nicelim[j].min = (float) floor( (double) (
    xg->nicelim[j].min / xg->tickdelta[j])) * xg->tickdelta[j];
  xg->nicelim[j].max = (float) ceil( (double) (
    xg->nicelim[j].max / xg->tickdelta[j])) * xg->tickdelta[j];

  /* add .01 for rounding */
  xg->ticks.nticks[j] = 1 +
    (.01 + (xg->nicelim[j].max - xg->nicelim[j].min) / xg->tickdelta[j]);
  /*
   * Initialize ticks0.nticks[]
  */
  xg->ticks0.nticks[j] = xg->ticks.nticks[j];
}

void
generate_ticks(xindx, yindx, limits, tix, xg)
/*
 *  Generate a vector of x ticks and y ticks.
*/
  tickinfo *tix;
  lims *limits;
  int xindx, yindx;
  xgobidata *xg;
{
  int j;

  if (xg->is_xyplotting)  /* skip if dotplotting */
  {
    tix->xticks[0] = xg->nicelim[xindx].min;
    j = 1;
    while (tix->xticks[j-1] + xg->tickdelta[xindx] <=
       limits[xindx].max + xg->tickdelta[xindx]/2)
    {
      tix->xticks[j] = tix->xticks[j-1] + xg->tickdelta[xindx];
      if (j++ == NTICKS-1)
      {
        fprintf(stderr, "warning: (generate_ticks) too many x ticks\n");
        break;
      }
    }
    tix->nticks[xindx] = j;
  }

  tix->yticks[0] = xg->nicelim[yindx].min;
  j = 1;
  while (tix->yticks[j-1] + xg->tickdelta[yindx] <=
       limits[yindx].max + xg->tickdelta[yindx]/2)
  {
    tix->yticks[j] = tix->yticks[j-1] + xg->tickdelta[yindx];
    if (j++ == NTICKS-1)
    {
      fprintf(stderr, "warning: (generate_ticks) too many y ticks\n");
      break;
    }
  }
  tix->nticks[yindx] = j;
}

void
scale_ticks(xindx, yindx, limits, tix, xg)
/*
 * calculate plane-coordinates of ticks
*/
  int xindx, yindx;
  lims *limits;
  tickinfo *tix;
  xgobidata *xg;
{
  int j;
  float tmpf;
  float precis;

  precis = PRECISION1;

  if (xg->is_xyplotting)   /* skip if dotplotting */
  {
    for (j=0; j<tix->nticks[xindx]; j++)
    {
      tmpf = -1.0 + 2.0*(tix->xticks[j] - limits[xindx].min)
        /(limits[xindx].max - limits[xindx].min);
      tix->plane[j].x = (long) (precis * tmpf);
    }
  }

  for (j=0; j<tix->nticks[yindx]; j++)
  {
    tmpf = -1.0 + 2.0*(tix->yticks[j] - limits[yindx].min)
      /(limits[yindx].max - limits[yindx].min);
    tix->plane[j].y = (long) (precis * tmpf);
  }
}

void
convert_ticks(xindx, yindx, tix, xg)
/*
 * Convert plane-ticks to screen-ticks
*/
  int xindx, yindx;
  tickinfo *tix;
  xgobidata *xg;
{
  int j;
  long ltmp;

  if (xg->is_xyplotting)   /* skip if dotplotting */
  {
    for (j=0; j<tix->nticks[xindx]; j++)
    {
      ltmp = tix->plane[j].x + xg->shift_wrld.x * (1-xg->scale_cntr);
      tix->screen[j].x = (int) ((ltmp * xg->is.x) >> EXP1);
      tix->screen[j].x += (xg->mid.x + xg->shift_scrn.x * xg->scale_cntr);
    }
  }

  for (j=0; j<tix->nticks[yindx]; j++)
  {
    /*
     *  Note:  this is a subtraction though it's an addition for the points.
    */
    ltmp = tix->plane[j].y - xg->shift_wrld.y * (1-xg->scale_cntr);
    tix->screen[j].y = (int) ((ltmp * xg->is.y) >> EXP1);
    tix->screen[j].y += (xg->mid.y + xg->shift_scrn.y * xg->scale_cntr);
  }
}

void
build_ticks(xindx, yindx, tix, xg)
/*
 * build x and y tick segments
*/
  int xindx, yindx;
  tickinfo *tix;
  xgobidata *xg;
{
  int j, k;

  if (xg->is_xyplotting)   /* skip if dotplotting */
  {
    /* build x tick segments */
    for (j=0; j<tix->nticks[xindx]; j++)
    {
      tick_segs[j].x1 = tix->screen[j].x;
      tick_segs[j].x2 = tix->screen[j].x;
      tick_segs[j].y1 = xg->screen_axes[1].y;
      tick_segs[j].y2 = xg->screen_axes[1].y + 5;
    }
  }

  /* build y tick segments */
  for (j=0; j<tix->nticks[yindx]; j++)
  {
    if (xg->is_dotplotting)
      k = j;
    else if (xg->is_xyplotting)
      k = j + tix->nticks[xindx];
    tick_segs[k].x1 = xg->screen_axes[1].x;
    tick_segs[k].x2 = xg->screen_axes[1].x - 5;
    tick_segs[k].y1 = tix->screen[j].y;
    tick_segs[k].y2 = tix->screen[j].y;
  }
}

void
init_axes(xg, firsttime)
/*
 * When reading in data, use the limits determined using min-max
 * standardization method to define the axes for xy plotting.
*/
  xgobidata *xg;
  int firsttime;
{
  int j;

  if (firsttime)
    make_axes();

  for (j=0; j<xg->ncols_used; j++)
  {
    /*
     * Find the nicelim values, which are chosen to make reasonable
     * placement of the tick marks.
    */
    xg->nicelim[j].min = xg->lim0[j].min;
    xg->nicelim[j].max = xg->lim0[j].max;
    SetNiceRange(j, xg);
    xg->deci[j] = set_deci(xg->tickdelta[j]);
  }
  xg->minindex = find_minindex(xg->lim0, xg);
  generate_ticks(xg->minindex, xg->minindex, xg->lim0, &xg->ticks0, xg);
  scale_ticks(xg->minindex, xg->minindex, xg->lim0, &xg->ticks0, xg);

  /* This part is redone when shift and scale change */
  convert_ticks(xg->minindex, xg->minindex, &xg->ticks0, xg);
  convert_axes(&xg->ticks0, xg);

  xg->maxwidth = find_maxwidth(xg->lim0, xg);
  while (plot_too_big(xg))
  {
    xg->scale0.x = .97 * xg->scale0.x;
    xg->scale.x = xg->scale0.x;
    xg->scale0.y = .97 * xg->scale0.y;
    xg->scale.y = xg->scale0.y;

    plane_to_screen(xg);
    convert_ticks(xg->minindex, xg->minindex, &xg->ticks0, xg);
    convert_axes(&xg->ticks0, xg);
  }
}

void
init_tickdelta(xg)
  xgobidata *xg;
{
/*
 * Initialite the limits and tickdelta for the current variables.
*/
  int j;

  for (j=0; j<xg->ncols_used; j++)
  {
    xg->nicelim[j].min = xg->lim[j].min;
    xg->nicelim[j].max = xg->lim[j].max;
    SetNiceRange(j, xg);
    xg->deci[j] = set_deci(xg->tickdelta[j]);
  }
}

void
init_ticks(vars, xg)
  icoords *vars;
  xgobidata *xg;
{
/*
 * Set the ticks for the current standardization method.
 *
 * Construct current ticks.
 * This is redone when choosing new variables or when
 * re-entering xyplotting.
*/
  /* use minindex and ticks0 to set the axes */
  convert_ticks(xg->minindex, xg->minindex, &xg->ticks0, xg);
  convert_axes(&xg->ticks0, xg);
  /* now generate and scale current ticks */
  generate_ticks(vars->x, vars->y, xg->lim, &xg->ticks, xg);
  scale_ticks(vars->x, vars->y, xg->lim, &xg->ticks, xg);
  convert_ticks(vars->x, vars->y, &xg->ticks, xg);
  extend_axes(vars->x, vars->y, &xg->ticks, xg);
  build_ticks(vars->x, vars->y, &xg->ticks, xg);
}

/* For drawing ticks and axes; used in plot_once() */

int
check_x_axis(xg, xvar, tix)
/*
 * Check whether the first ticks are to the left of the perpendicular axis.
 * If so, they will simply not be drawn.
*/
  xgobidata *xg;
  int xvar;
  tickinfo *tix;
{
  int i;
  int xt = 0;

  for (i=0; i<tix->nticks[xvar]; i++)
    if (xg->screen_axes[1].x > tix->screen[i].x)
      xt++;
  return(xt);
}

int
check_y_axis(xg, yvar, tix)
/*
 * Check whether the first ticks are to the left of the perpendicular axis.
 * If so, they will simply not be drawn.
*/
  xgobidata *xg;
  int yvar;
  tickinfo *tix;
{
  int i;
  int yt = 0;

  for (i=0; i<tix->nticks[yvar]; i++)
    if (xg->screen_axes[1].y < tix->screen[i].y)
      yt = yt + 1;
  return(yt);
}

void
add_x_axis(xg, vars, xstart, tix)
  xgobidata *xg;
  icoords *vars;
  int xstart;
  tickinfo *tix;
{
  int j, width, src, dest;
  char str[50];
  XPoint vpnts[2];
/*
 * Draw x axis
*/
  for (j=0; j<2; j++)
  {
    vpnts[j].x = xg->screen_axes[j+1].x;
    vpnts[j].y = xg->screen_axes[j+1].y;
  }
  XDrawLines(display, xg->pixmap0, copy_GC, vpnts, 2, CoordModeOrigin);

  /*
   * Draw ticks
  */
  XDrawSegments(display, xg->pixmap0, copy_GC,
    tick_segs + xstart,
    tix->nticks[vars->x] - xstart);
  /*
   * Simply skip the first tick if it's to the left of the Y axis
  */
  for (j=xstart; j<tix->nticks[vars->x]; j++)
  {
    find_tick_label(xg, vars->x, j, tix->xticks, str);
    /*
     * To get better placement, strip blanks.
    */
    for (src=0, dest=0; src<(strlen(str)+1); src++)
      if (str[src] != ' ')
        str[dest++] = str[src];

    width = XTextWidth(appdata.plotFont, str, strlen(str));
    XDrawImageString(display, xg->pixmap0, copy_GC,
      tix->screen[j].x - width/2,
      xg->screen_axes[1].y + FONTHEIGHT(appdata.plotFont)+7,
      str, strlen(str));
  }
  /*
   * Draw X Axis label
  */
  strcpy(str, xg->collab_tform[vars->x]);
  XDrawImageString(display, xg->pixmap0, copy_GC,
    xg->screen_axes[1].x +
     (xg->screen_axes[2].x - xg->screen_axes[1].x)/2 - strlen(str),
    xg->screen_axes[1].y + 2*FONTHEIGHT(appdata.plotFont) + 14,
    str,
    strlen(str));

  return;
}

void
add_y_axis(xg, vars, ystart, tix)
  xgobidata *xg;
  icoords *vars;
  int ystart;
  tickinfo *tix;
{
  int j, nxticks;
  char str[50];
  XPoint vpnts[2];
/*
 * Draw y axis
*/
  for (j=0; j<2; j++)
  {
    vpnts[j].x = xg->screen_axes[j].x;
    vpnts[j].y = xg->screen_axes[j].y;
  }
  XDrawLines(display, xg->pixmap0, copy_GC, vpnts, 2, CoordModeOrigin);

  /*
   * Draw ticks
  */
  if (xg->is_dotplotting)
    nxticks = 0;
  else if (xg->is_xyplotting)
    nxticks = tix->nticks[vars->x];
  XDrawSegments(display, xg->pixmap0, copy_GC,
    tick_segs + nxticks + ystart,
    tix->nticks[vars->y] - ystart);

  /*
   * Draw y tick labels.
  */
  for (j=ystart; j<tix->nticks[vars->y]; j++)
  {
    find_tick_label(xg, vars->y, j, tix->yticks, str);

    XDrawImageString(display, xg->pixmap0, copy_GC,
      xg->screen_axes[1].x -
        XTextWidth(appdata.plotFont, str, strlen(str)) - 7,
      tix->screen[j].y + 5,
      str, strlen(str));
  }
  /*
   * Draw Y Axis label
  */
  strcpy(str, xg->collab_tform[vars->y]);
  XDrawImageString(display, xg->pixmap0, copy_GC,
    xg->screen_axes[0].x - strlen(str)/2 - 6,
    xg->screen_axes[0].y - 10,
    str,
    strlen(str));

  return;
}
