/* nurbssrf.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: nurbssrf.c,v 1.6 1995/11/03 14:14:40 brianp Exp $

$Log: nurbssrf.c,v $
 * Revision 1.6  1995/11/03  14:14:40  brianp
 * Bogdan's November 3, 1995 updates
 *
 * Revision 1.5  1995/09/20  18:25:57  brianp
 * removed Bogdan's old email address
 *
 * Revision 1.4  1995/08/04  13:09:59  brianp
 * include gluP.h to define NULL, just in case
 *
 * Revision 1.3  1995/08/02  15:12:29  brianp
 * use MEMCPY macro instead of memcpy
 *
 * Revision 1.2  1995/07/28  21:37:30  brianp
 * updates from Bogdan on July 28
 *
 * Revision 1.1  1995/07/28  14:45:01  brianp
 * Initial revision
 *
 */


/*
 * NURBS implementation written by Bogdan Sikorski (bogdan@cira.it)
 * See README-nurbs for more info.
 */


#include <stdlib.h>
#include <string.h> /* prototype of memcpy() */
#include <math.h>
#include "nurbs.h"

static int
get_surface_dim(GLenum type)
{
	switch(type)
	{
		case GL_MAP2_VERTEX_3:			return 3;
		case GL_MAP2_VERTEX_4:			return 4;
		case GL_MAP2_INDEX:				return 1;
		case GL_MAP2_COLOR_4:			return 4;
		case GL_MAP2_NORMAL:			return 3;
		case GL_MAP2_TEXTURE_COORD_1:	return 1;
		case GL_MAP2_TEXTURE_COORD_2:	return 2;
		case GL_MAP2_TEXTURE_COORD_3:	return 3;
		case GL_MAP2_TEXTURE_COORD_4:	return 4;
	}
}

static GLenum
test_nurbs_surface(GLUnurbsObj *nobj, surface_attribs *attrib)
{
	GLenum err;
	GLint tmp_int;

	if(attrib->sorder < 0 || attrib->torder < 0)
	{
		call_user_error(nobj,GLU_INVALID_VALUE);
		return GLU_ERROR;
	}
	glGetIntegerv(GL_MAX_EVAL_ORDER,&tmp_int);
	if(attrib->sorder > tmp_int || attrib->sorder < 2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR1);
		return GLU_ERROR;
	}
	if(attrib->torder > tmp_int || attrib->torder < 2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR1);
		return GLU_ERROR;
	}
	if(attrib->sknot_count < attrib->sorder +2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR2);
		return GLU_ERROR;
	}
	if(attrib->tknot_count < attrib->torder +2)
	{
		call_user_error(nobj,GLU_NURBS_ERROR2);
		return GLU_ERROR;
	}
	if(attrib->s_stride < 0 || attrib->t_stride < 0)
	{
		call_user_error(nobj,GLU_NURBS_ERROR34);
		return GLU_ERROR;
	}
	if(attrib->sknot==NULL || attrib->tknot==NULL || attrib->ctrlarray==NULL)
	{
		call_user_error(nobj,GLU_NURBS_ERROR36);
		return GLU_ERROR;
	}
	if((err=test_knot(attrib->tknot_count,attrib->tknot,attrib->torder))
		!=GLU_NO_ERROR)
	{
		call_user_error(nobj,err);
		return GLU_ERROR;
	}
	if((err=test_knot(attrib->sknot_count,attrib->sknot,attrib->sorder))
		!=GLU_NO_ERROR)
	{
		call_user_error(nobj,err);
		return GLU_ERROR;
	}
	return GLU_NO_ERROR;
}

static GLenum
test_nurbs_surfaces(GLUnurbsObj *nobj)
{
	/* test the geometric data */
	if(test_nurbs_surface(nobj,&(nobj->surface.geom))!=GLU_NO_ERROR)
		return GLU_ERROR;
	/* now test the attributive data */
	/* color */
	if(nobj->surface.color.type!=GLU_INVALID_ENUM)
		if(test_nurbs_surface(nobj,&(nobj->surface.color))!=GLU_NO_ERROR)
			return GLU_ERROR;
	/* normal */
	if(nobj->surface.normal.type!=GLU_INVALID_ENUM)
		if(test_nurbs_surface(nobj,&(nobj->surface.normal))!=GLU_NO_ERROR)
			return GLU_ERROR;
	/* texture */
	if(nobj->surface.texture.type!=GLU_INVALID_ENUM)
		if(test_nurbs_surface(nobj,&(nobj->surface.texture))!=GLU_NO_ERROR)
			return GLU_ERROR;
	return GLU_NO_ERROR;
}

static GLenum
convert_surf(knot_str_type *s_knot, knot_str_type *t_knot,
	surface_attribs *attrib, GLfloat **new_ctrl,
	GLint *s_n_ctrl, GLint *t_n_ctrl)
{
	GLfloat **tmp_ctrl;
	GLfloat *ctrl_offset;
	GLint tmp_n_control;
	GLint i,j,t_cnt,s_cnt;
	GLint tmp_stride;
	GLint dim;
	GLenum err;

	t_cnt=attrib->tknot_count-attrib->torder;
	s_cnt=attrib->sknot_count-attrib->sorder;
	if((tmp_ctrl=(GLfloat **)malloc(sizeof(GLfloat *)*t_cnt))==NULL)
		return GLU_OUT_OF_MEMORY;
	if((err=explode_knot(s_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		if(s_knot->unified_knot)
		{
			free(s_knot->unified_knot);
			s_knot->unified_knot=NULL;
		}
		return err;
	}
	if(s_knot->unified_knot)
	{
		free(s_knot->unified_knot);
		s_knot->unified_knot=NULL;
	}
	if((err=calc_alphas(s_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		free(s_knot->new_knot);
		return err;
	}
	free(s_knot->new_knot);
	ctrl_offset=attrib->ctrlarray;
	dim=attrib->dim;
	for(i=0;i<t_cnt;i++)
	{
		if((err=calc_new_ctrl_pts(ctrl_offset,attrib->s_stride,s_knot,
			dim,&(tmp_ctrl[i]),&tmp_n_control))!=GLU_NO_ERROR)
		{
			for(--i;i<=0;i--)
				free(tmp_ctrl[i]);
			free(tmp_ctrl);
			free(s_knot->alpha);
			return err;
		}
		ctrl_offset+=attrib->t_stride;
	}
	free(s_knot->alpha);
	tmp_stride=dim*tmp_n_control;
	if((*new_ctrl=(GLfloat *)malloc(sizeof(GLfloat)*tmp_stride*t_cnt))
		==NULL)
	{
		for(i=0;i<t_cnt;i++)
			free(tmp_ctrl[i]);
		free(tmp_ctrl);
		return GLU_OUT_OF_MEMORY;
	}
	for(i=0;i<tmp_n_control;i++)
		for(j=0;j<t_cnt;j++)
			MEMCPY(*new_ctrl+j*dim+i*dim*t_cnt,tmp_ctrl[j]+dim*i,
				sizeof(GLfloat)*dim);
	for(i=0;i<t_cnt;i++)
		free(tmp_ctrl[i]);
	free(tmp_ctrl);
	*s_n_ctrl=tmp_n_control;
	
	if((tmp_ctrl=(GLfloat **)malloc(sizeof(GLfloat *)*(*s_n_ctrl)))==NULL)
	{
		return GLU_OUT_OF_MEMORY;
	}
	if((err=explode_knot(t_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		if(t_knot->unified_knot)
		{
			free(t_knot->unified_knot);
			t_knot->unified_knot=NULL;
		}
		return err;
	}
	if(t_knot->unified_knot)
	{
		free(t_knot->unified_knot);
		t_knot->unified_knot=NULL;
	}
	if((err=calc_alphas(t_knot))!=GLU_NO_ERROR)
	{
		free(tmp_ctrl);
		free(t_knot->new_knot);
		return err;
	}
	free(t_knot->new_knot);
	ctrl_offset=*new_ctrl;
	for(i=0;i<(*s_n_ctrl);i++)
	{
		if((err=calc_new_ctrl_pts(ctrl_offset,dim,t_knot,
			dim,&(tmp_ctrl[i]),&tmp_n_control))!=GLU_NO_ERROR)
		{
			for(--i;i<=0;i--)
				free(tmp_ctrl[i]);
			free(tmp_ctrl);
			free(t_knot->alpha);
			return err;
		}
		ctrl_offset+=dim*t_cnt;
	}
	free(t_knot->alpha);
	free(*new_ctrl);
	tmp_stride=dim*tmp_n_control;
	if((*new_ctrl=(GLfloat *)malloc(sizeof(GLfloat)*tmp_stride*(*s_n_ctrl)))
		==NULL)
	{
		for(i=0;i<(*s_n_ctrl);i++)
			free(tmp_ctrl[i]);
		free(tmp_ctrl);
		return GLU_OUT_OF_MEMORY;
	}
	for(i=0;i<(*s_n_ctrl);i++)
	{
		MEMCPY(*new_ctrl+i*tmp_stride,tmp_ctrl[i],sizeof(GLfloat)*tmp_stride);
		free(tmp_ctrl[i]);
	}
	free(tmp_ctrl);
	*t_n_ctrl=tmp_n_control;
	return GLU_NO_ERROR;
}

/* prepare the knot information structures */
static GLenum
fill_knot_structures(GLUnurbsObj *nobj,
	knot_str_type *geom_s_knot, knot_str_type *geom_t_knot,
	knot_str_type *color_s_knot, knot_str_type *color_t_knot,
	knot_str_type *normal_s_knot,  knot_str_type *normal_t_knot,
	knot_str_type *texture_s_knot, knot_str_type *texture_t_knot)
{
	GLint order;
	GLfloat *knot;
	GLint nknots;
	GLint t_min,t_max;

	geom_s_knot->unified_knot=NULL;
	knot=geom_s_knot->knot=nobj->surface.geom.sknot;
	nknots=geom_s_knot->nknots=nobj->surface.geom.sknot_count;
	order=geom_s_knot->order=nobj->surface.geom.sorder;
	geom_s_knot->delta_nknots=0;
	t_min=geom_s_knot->t_min=order-1;
	t_max=geom_s_knot->t_max=nknots-order;
	if(fabs(knot[t_min]-knot[t_max])<EPSILON)
	{
		call_user_error(nobj,GLU_NURBS_ERROR3);
		return GLU_ERROR;
	}
	if(fabs(knot[0]-knot[t_min])<EPSILON)
	{
		/* knot open at beggining */
		geom_s_knot->open_at_begin=GL_TRUE;
	}
	else
		geom_s_knot->open_at_begin=GL_FALSE;
	if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
	{
		/* knot open at end */
		geom_s_knot->open_at_end=GL_TRUE;
	}
	else
		geom_s_knot->open_at_end=GL_FALSE;
	geom_t_knot->unified_knot=NULL;
	knot=geom_t_knot->knot=nobj->surface.geom.tknot;
	nknots=geom_t_knot->nknots=nobj->surface.geom.tknot_count;
	order=geom_t_knot->order=nobj->surface.geom.torder;
	geom_t_knot->delta_nknots=0;
	t_min=geom_t_knot->t_min=order-1;
	t_max=geom_t_knot->t_max=nknots-order;
	if(fabs(knot[t_min]-knot[t_max])<EPSILON)
	{
		call_user_error(nobj,GLU_NURBS_ERROR3);
		return GLU_ERROR;
	}
	if(fabs(knot[0]-knot[t_min])<EPSILON)
	{
		/* knot open at beggining */
		geom_t_knot->open_at_begin=GL_TRUE;
	}
	else
		geom_t_knot->open_at_begin=GL_FALSE;
	if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
	{
		/* knot open at end */
		geom_t_knot->open_at_end=GL_TRUE;
	}
	else
		geom_t_knot->open_at_end=GL_FALSE;

	if(nobj->surface.color.type!=GLU_INVALID_ENUM)
	{
		color_s_knot->unified_knot=(GLfloat *)1;
		knot=color_s_knot->knot=nobj->surface.color.sknot;
		nknots=color_s_knot->nknots=nobj->surface.color.sknot_count;
		order=color_s_knot->order=nobj->surface.color.sorder;
		color_s_knot->delta_nknots=0;
		t_min=color_s_knot->t_min=order-1;
		t_max=color_s_knot->t_max=nknots-order;
		if(fabs(knot[t_min]-knot[t_max])<EPSILON)
		{
			call_user_error(nobj,GLU_NURBS_ERROR3);
			return GLU_ERROR;
		}
		if(fabs(knot[0]-knot[t_min])<EPSILON)
		{
			/* knot open at beggining */
			color_s_knot->open_at_begin=GL_TRUE;
		}
		else
			color_s_knot->open_at_begin=GL_FALSE;
		if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
		{
			/* knot open at end */
			color_s_knot->open_at_end=GL_TRUE;
		}
		else
			color_s_knot->open_at_end=GL_FALSE;
		color_t_knot->unified_knot=(GLfloat *)1;
		knot=color_t_knot->knot=nobj->surface.color.tknot;
		nknots=color_t_knot->nknots=nobj->surface.color.tknot_count;
		order=color_t_knot->order=nobj->surface.color.torder;
		color_t_knot->delta_nknots=0;
		t_min=color_t_knot->t_min=order-1;
		t_max=color_t_knot->t_max=nknots-order;
		if(fabs(knot[t_min]-knot[t_max])<EPSILON)
		{
			call_user_error(nobj,GLU_NURBS_ERROR3);
			return GLU_ERROR;
		}
		if(fabs(knot[0]-knot[t_min])<EPSILON)
		{
			/* knot open at beggining */
			color_t_knot->open_at_begin=GL_TRUE;
		}
		else
			color_t_knot->open_at_begin=GL_FALSE;
		if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
		{
			/* knot open at end */
			color_t_knot->open_at_end=GL_TRUE;
		}
		else
			color_t_knot->open_at_end=GL_FALSE;
	}
	else
	{
		color_s_knot->unified_knot=NULL;
		color_t_knot->unified_knot=NULL;
	}

	if(nobj->surface.normal.type!=GLU_INVALID_ENUM)
	{
		normal_s_knot->unified_knot=(GLfloat *)1;
		knot=normal_s_knot->knot=nobj->surface.normal.sknot;
		nknots=normal_s_knot->nknots=nobj->surface.normal.sknot_count;
		order=normal_s_knot->order=nobj->surface.normal.sorder;
		normal_s_knot->delta_nknots=0;
		t_min=normal_s_knot->t_min=order-1;
		t_max=normal_s_knot->t_max=nknots-order;
		if(fabs(knot[t_min]-knot[t_max])<EPSILON)
		{
			call_user_error(nobj,GLU_NURBS_ERROR3);
			return GLU_ERROR;
		}
		if(fabs(knot[0]-knot[t_min])<EPSILON)
		{
			/* knot open at beggining */
			normal_s_knot->open_at_begin=GL_TRUE;
		}
		else
			normal_s_knot->open_at_begin=GL_FALSE;
		if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
		{
			/* knot open at end */
			normal_s_knot->open_at_end=GL_TRUE;
		}
		else
			normal_s_knot->open_at_end=GL_FALSE;
		normal_t_knot->unified_knot=(GLfloat *)1;
		knot=normal_t_knot->knot=nobj->surface.normal.tknot;
		nknots=normal_t_knot->nknots=nobj->surface.normal.tknot_count;
		order=normal_t_knot->order=nobj->surface.normal.torder;
		normal_t_knot->delta_nknots=0;
		t_min=normal_t_knot->t_min=order-1;
		t_max=normal_t_knot->t_max=nknots-order;
		if(fabs(knot[t_min]-knot[t_max])<EPSILON)
		{
			call_user_error(nobj,GLU_NURBS_ERROR3);
			return GLU_ERROR;
		}
		if(fabs(knot[0]-knot[t_min])<EPSILON)
		{
			/* knot open at beggining */
			normal_t_knot->open_at_begin=GL_TRUE;
		}
		else
			normal_t_knot->open_at_begin=GL_FALSE;
		if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
		{
			/* knot open at end */
			normal_t_knot->open_at_end=GL_TRUE;
		}
		else
			normal_t_knot->open_at_end=GL_FALSE;
	}
	else
	{
		normal_s_knot->unified_knot=NULL;
		normal_t_knot->unified_knot=NULL;
	}

	if(nobj->surface.texture.type!=GLU_INVALID_ENUM)
	{
		texture_s_knot->unified_knot=(GLfloat *)1;
		knot=texture_s_knot->knot=nobj->surface.texture.sknot;
		nknots=texture_s_knot->nknots=nobj->surface.texture.sknot_count;
		order=texture_s_knot->order=nobj->surface.texture.sorder;
		texture_s_knot->delta_nknots=0;
		t_min=texture_s_knot->t_min=order-1;
		t_max=texture_s_knot->t_max=nknots-order;
		if(fabs(knot[t_min]-knot[t_max])<EPSILON)
		{
			call_user_error(nobj,GLU_NURBS_ERROR3);
			return GLU_ERROR;
		}
		if(fabs(knot[0]-knot[t_min])<EPSILON)
		{
			/* knot open at beggining */
			texture_s_knot->open_at_begin=GL_TRUE;
		}
		else
			texture_s_knot->open_at_begin=GL_FALSE;
		if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
		{
			/* knot open at end */
			texture_s_knot->open_at_end=GL_TRUE;
		}
		else
			texture_s_knot->open_at_end=GL_FALSE;
		texture_t_knot->unified_knot=(GLfloat *)1;
		knot=texture_t_knot->knot=nobj->surface.texture.tknot;
		nknots=texture_t_knot->nknots=nobj->surface.texture.tknot_count;
		order=texture_t_knot->order=nobj->surface.texture.torder;
		texture_t_knot->delta_nknots=0;
		t_min=texture_t_knot->t_min=order-1;
		t_max=texture_t_knot->t_max=nknots-order;
		if(fabs(knot[t_min]-knot[t_max])<EPSILON)
		{
			call_user_error(nobj,GLU_NURBS_ERROR3);
			return GLU_ERROR;
		}
		if(fabs(knot[0]-knot[t_min])<EPSILON)
		{
			/* knot open at beggining */
			texture_t_knot->open_at_begin=GL_TRUE;
		}
		else
			texture_t_knot->open_at_begin=GL_FALSE;
		if(fabs(knot[t_max]-knot[nknots-1])<EPSILON)
		{
			/* knot open at end */
			texture_t_knot->open_at_end=GL_TRUE;
		}
		else
			texture_t_knot->open_at_end=GL_FALSE;
	}
	else
	{
		texture_s_knot->unified_knot=NULL;
		texture_t_knot->unified_knot=NULL;
	}
	return GLU_NO_ERROR;
}

static void
free_unified_knots(knot_str_type *geom_knot, knot_str_type *color_knot,
	knot_str_type *normal_knot, knot_str_type *texture_knot)
{
	if(geom_knot->unified_knot)
		free(geom_knot->unified_knot);
	if(color_knot->unified_knot)
		free(color_knot->unified_knot);
	if(normal_knot->unified_knot)
		free(normal_knot->unified_knot);
	if(texture_knot->unified_knot)
		free(texture_knot->unified_knot);
}

/* qsort function */
static int
knot_sort(const void *a, const void *b)
{
	GLfloat x,y;

	x=*((GLfloat *)a);
	y=*((GLfloat *)b);
	if(fabs(x-y) < EPSILON)
		return 0;
	if(x > y)
		return 1;
	return -1;
}

/* insert into dest knot all values within the valid range from src knot */
/* that do not appear in dest */
/* First we have to transfor all knot values of src in such a way that */
/* the valid knot range boundaries allign with the ones of dest */
static void
collect_unified_knot(knot_str_type *dest, knot_str_type *src,
	GLfloat *mult_factor, GLfloat *add_factor)
{
	GLfloat *src_knot,*dest_knot;
	GLint src_t_min,src_t_max,dest_t_min,dest_t_max;
	GLint src_nknots,dest_nknots;
	GLint i,j,k,new_cnt;

	src_knot=src->unified_knot;
	dest_knot=dest->unified_knot;
	src_t_min=src->t_min;
	src_t_max=src->t_max;
	dest_t_min=dest->t_min;
	dest_t_max=dest->t_max;
	src_nknots=src->unified_nknots;
	dest_nknots=dest->unified_nknots;
	/* convert src knot to have the same range values for its valid range */
	*mult_factor=(dest_knot[dest_t_max]-dest_knot[dest_t_min]) /
		(src_knot[src_t_max]-src_knot[src_t_min]);
	for(i=0;i<src_nknots;i++)
		src_knot[i] *= *mult_factor;
	*add_factor=dest_knot[dest_t_min]-src_knot[src_t_min];
	if(fabs(*add_factor)>EPSILON)
		for(i=0;i<src_nknots;i++)
			src_knot[i] += *add_factor;

	for(i=src_t_min+1 , j=dest_t_min+1 , k=dest_nknots , new_cnt=dest_nknots;
		i<src_t_max;
		i++)
	{
		if(fabs(dest_knot[j]-src_knot[i]) < EPSILON)
			continue; /* knot from src already appears in dest */
		else
		if(dest_knot[j] > src_knot[i])
		{
			/* knot from src is not in dest - add this knot to dest */
			dest_knot[k++]=src_knot[i];
			++new_cnt;
			++(dest->t_max); /* the valid range widens */
			++(dest->delta_nknots); /* increment the extra knot value counter */
		}
		else
		{
			/* the knot value at dest is smaller than the knot from src */
			/* scan the dest, and stop when we find a greater or equal value */
			/* after positioning, the former code will decide if to insert */
			/* or not the current src knot value */
			while(src_knot[i] > dest_knot[j])
				j++;
			--i;
		}
	}
	dest->unified_nknots=new_cnt;
	qsort((void *)dest_knot,(size_t)new_cnt,(size_t)sizeof(GLfloat),
		&knot_sort);
}

/* similar as above - insert into the dest knot new values from src knot */
/* since the dest knot had its values "scaled" to the same valid range */
/* as the src previously, we have revert this process */
static void
fill_unified_knot(knot_str_type *dest, knot_str_type *src,
	GLfloat mult_factor, GLfloat add_factor)
{
	GLfloat *src_knot,*dest_knot;
	GLint src_t_min,src_t_max,dest_t_min;
	GLint src_nknots,dest_nknots;
	GLint i,j,k,new_cnt;
	GLfloat delta;

	src_knot=src->unified_knot;
	dest_knot=dest->unified_knot;
	src_t_min=src->t_min;
	src_t_max=src->t_max;
	dest_t_min=dest->t_min;
	src_nknots=src->unified_nknots;
	dest_nknots=dest->unified_nknots;
	for(i=src_t_min+1 , j=dest_t_min+1 , k=dest_nknots , new_cnt=dest_nknots;
		i<src_t_max;
		i++)
	{
		if(fabs(dest_knot[j]-src_knot[i]) < EPSILON)
			continue;
		else
		if(dest_knot[j] > src_knot[i])
		{
			/* insert */
			dest_knot[k++]=src_knot[i];
			++new_cnt;
			++(dest->t_max);
			++(dest->delta_nknots);
		}
		else
		{
			while(src_knot[i] > dest_knot[j])
				j++;
			--i;
		}
	}
	dest->unified_nknots=new_cnt;
	qsort((void *)dest_knot,(size_t)new_cnt,(size_t)sizeof(GLfloat),
		&knot_sort);
	/* return back to the original values of knot */
	for(i=0;i<new_cnt;i++)
	{
		dest_knot[i] -= add_factor;
		dest_knot[i] /= mult_factor;
	}
}

/* modify all knot valid ranges in such a way that all have the same number */
/* of knots in between, but preserve scaling */
/* do this by knot insertion */
static GLenum
unify_knots(GLUnurbsObj *nobj,knot_str_type *geom_knot,
	knot_str_type *color_knot, knot_str_type *normal_knot,
	knot_str_type *texture_knot)
{
	GLint max_nknots;
	GLfloat color_mult, color_add;
	GLfloat normal_mult, normal_add;
	GLfloat texture_mult, texture_add;
	GLint i;

	/* find the maximum modified knot length */
	max_nknots=geom_knot->nknots;
	if(color_knot->unified_knot)
		max_nknots+=color_knot->nknots;
	if(normal_knot->unified_knot)
		max_nknots+=normal_knot->nknots;
	if(texture_knot->unified_knot)
		max_nknots+=texture_knot->nknots;
	/* any attirb data ? */
	if(max_nknots!=geom_knot->nknots)
	{
		/* allocate space for the unified knots */
		if((geom_knot->unified_knot=
				(GLfloat *)malloc(sizeof(GLfloat)*max_nknots))==NULL)
		{
			call_user_error(nobj,GLU_OUT_OF_MEMORY);
			return GLU_ERROR;
		}
		/* copy the original knot to the unified one */
		geom_knot->unified_nknots=geom_knot->nknots;
		for(i=0;i<geom_knot->nknots;i++)
			(geom_knot->unified_knot)[i]=(geom_knot->knot)[i];
		if(color_knot->unified_knot)
		{
			if((color_knot->unified_knot=
					(GLfloat *)malloc(sizeof(GLfloat)*max_nknots))==NULL)
			{
				free(geom_knot->unified_knot);
				call_user_error(nobj,GLU_OUT_OF_MEMORY);
				return GLU_ERROR;
			}
			/* copy the original knot to the unified one */
			color_knot->unified_nknots=color_knot->nknots;
			for(i=0;i<color_knot->nknots;i++)
				(color_knot->unified_knot)[i]=(color_knot->knot)[i];
		}
		if(normal_knot->unified_knot)
		{
			if((normal_knot->unified_knot=
					(GLfloat *)malloc(sizeof(GLfloat)*max_nknots))==NULL)
			{
				free(geom_knot->unified_knot);
				free(color_knot->unified_knot);
				call_user_error(nobj,GLU_OUT_OF_MEMORY);
				return GLU_ERROR;
			}
			/* copy the original knot to the unified one */
			normal_knot->unified_nknots=normal_knot->nknots;
			for(i=0;i<normal_knot->nknots;i++)
				(normal_knot->unified_knot)[i]=(normal_knot->knot)[i];
		}
		if(texture_knot->unified_knot)
		{
			if((texture_knot->unified_knot=
					(GLfloat *)malloc(sizeof(GLfloat)*max_nknots))==NULL)
			{
				free(geom_knot->unified_knot);
				free(color_knot->unified_knot);
				free(normal_knot->unified_knot);
				call_user_error(nobj,GLU_OUT_OF_MEMORY);
				return GLU_ERROR;
			}
			/* copy the original knot to the unified one */
			texture_knot->unified_nknots=texture_knot->nknots;
			for(i=0;i<texture_knot->nknots;i++)
				(texture_knot->unified_knot)[i]=(texture_knot->knot)[i];
		}
		/* fill in the geometry knot with all additional knot values */
		/* appearing in attirbutive knots */
		if(color_knot->unified_knot)
			collect_unified_knot(geom_knot,color_knot,&color_mult,&color_add);
		if(normal_knot->unified_knot)
			collect_unified_knot(geom_knot,normal_knot,&normal_mult,&normal_add);
		if(texture_knot->unified_knot)
			collect_unified_knot(geom_knot,texture_knot,&texture_mult,&texture_add);
		/* since we have now built the "unified" geometry knot */
		/* add same knot values to all attributive knots */
		if(color_knot->unified_knot)
			fill_unified_knot(color_knot,geom_knot,color_mult,color_add);
		if(normal_knot->unified_knot)
			fill_unified_knot(normal_knot,geom_knot,normal_mult,normal_add);
		if(texture_knot->unified_knot)
			fill_unified_knot(texture_knot,geom_knot,texture_mult,texture_add);
	}
	return GLU_NO_ERROR;
}

void
free_new_ctrl(new_ctrl_type *p)
{
	free(p->geom_ctrl);
	if(p->geom_offsets)
		free(p->geom_offsets);
	if(p->color_ctrl)
	{
		free(p->color_ctrl);
		if(p->color_offsets)
			free(p->color_offsets);
	}
	if(p->normal_ctrl)
	{
		free(p->normal_ctrl);
		if(p->normal_offsets)
			free(p->normal_offsets);
	}
	if(p->texture_ctrl)
	{
		free(p->texture_ctrl);
		if(p->texture_offsets)
			free(p->texture_offsets);
	}
}

/* convert surfaces - geometry and possible attribute ones into equivalent */
/* sequence of adjacent Bezier patches */
static GLenum
convert_surfs(GLUnurbsObj *nobj, new_ctrl_type *new_ctrl)
{
	knot_str_type	geom_s_knot,color_s_knot,normal_s_knot,texture_s_knot;
	knot_str_type	geom_t_knot,color_t_knot,normal_t_knot,texture_t_knot;
	GLenum			err;

	if((err=fill_knot_structures(nobj,&geom_s_knot,&geom_t_knot,
		&color_s_knot,&color_t_knot,&normal_s_knot,&normal_t_knot,
		&texture_s_knot,&texture_t_knot)) !=GLU_NO_ERROR)
	{
		return err;
	}
	/* unify knots - all knots should have the same number of working */
	/* ranges */
	if((err=unify_knots(nobj,&geom_s_knot,&color_s_knot,&normal_s_knot,
		&texture_s_knot)) !=GLU_NO_ERROR)
	{
		call_user_error(nobj,err);
		return err;
	}
	if((err=unify_knots(nobj,&geom_t_knot,&color_t_knot,&normal_t_knot,
		&texture_t_knot)) !=GLU_NO_ERROR)
	{
		free_unified_knots(&geom_s_knot,&color_s_knot,&normal_s_knot,&texture_s_knot);
		call_user_error(nobj,err);
		return err;
	}

	/* convert the geometry surface */
	nobj->surface.geom.dim=get_surface_dim(nobj->surface.geom.type);
	if((err=convert_surf(&geom_s_knot,&geom_t_knot,&(nobj->surface.geom),
		&(new_ctrl->geom_ctrl),&(new_ctrl->geom_s_pt_cnt),
		&(new_ctrl->geom_t_pt_cnt)))!=GLU_NO_ERROR)
	{
		free_unified_knots(&geom_s_knot,&color_s_knot,&normal_s_knot,
			&texture_s_knot);
		free_unified_knots(&geom_t_knot,&color_t_knot,&normal_t_knot,
			&texture_t_knot);
		call_user_error(nobj,err);
		return err;
	}
	/* if additional attributive surfaces are given convert them as well */
	if(color_s_knot.unified_knot)
	{
		nobj->surface.color.dim=get_surface_dim(nobj->surface.color.type);
		if((err=convert_surf(&color_s_knot,&color_t_knot,&(nobj->surface.color),
			&(new_ctrl->color_ctrl),&(new_ctrl->color_s_pt_cnt),
			&(new_ctrl->color_t_pt_cnt)))!=GLU_NO_ERROR)
		{
			free_unified_knots(&color_s_knot,&color_s_knot,&normal_s_knot,
				&texture_s_knot);
			free_unified_knots(&color_t_knot,&color_t_knot,&normal_t_knot,
				&texture_t_knot);
			free_new_ctrl(new_ctrl);
			call_user_error(nobj,err);
			return err;
		}
	}
	if(normal_s_knot.unified_knot)
	{
		nobj->surface.normal.dim=get_surface_dim(nobj->surface.normal.type);
		if((err=convert_surf(&normal_s_knot,&normal_t_knot,&(nobj->surface.normal),
			&(new_ctrl->normal_ctrl),&(new_ctrl->normal_s_pt_cnt),
			&(new_ctrl->normal_t_pt_cnt)))!=GLU_NO_ERROR)
		{
			free_unified_knots(&normal_s_knot,&normal_s_knot,&normal_s_knot,
				&texture_s_knot);
			free_unified_knots(&normal_t_knot,&normal_t_knot,&normal_t_knot,
				&texture_t_knot);
			free_new_ctrl(new_ctrl);
			call_user_error(nobj,err);
			return err;
		}
	}
	if(texture_s_knot.unified_knot)
	{
		nobj->surface.texture.dim=get_surface_dim(nobj->surface.texture.type);
		if((err=convert_surf(&texture_s_knot,&texture_t_knot,&(nobj->surface.texture),
			&(new_ctrl->texture_ctrl),&(new_ctrl->texture_s_pt_cnt),
			&(new_ctrl->texture_t_pt_cnt)))!=GLU_NO_ERROR)
		{
			free_unified_knots(&texture_s_knot,&texture_s_knot,&texture_s_knot,
				&texture_s_knot);
			free_unified_knots(&texture_t_knot,&texture_t_knot,&texture_t_knot,
				&texture_t_knot);
			free_new_ctrl(new_ctrl);
			call_user_error(nobj,err);
			return err;
		}
	}
	return GLU_NO_ERROR;
}

/* draw NURBS surface in OUTLINE POLYGON mode */
static void
draw_polygon_mode( GLenum display_mode, GLUnurbsObj *nobj,
	new_ctrl_type *new_ctrl, GLint *sfactors, GLint *tfactors )
{
	GLsizei		offset;
	GLint		t_bezier_cnt,s_bezier_cnt;
	GLboolean	do_color,do_normal,do_texture;
	GLint		i,j;

	t_bezier_cnt=new_ctrl->t_bezier_cnt;
	s_bezier_cnt=new_ctrl->s_bezier_cnt;
	glEnable(nobj->surface.geom.type);
	if(new_ctrl->color_ctrl)
	{
		glEnable(nobj->surface.color.type);
		do_color=GL_TRUE;
	}
	else
		do_color=GL_FALSE;
	if(new_ctrl->normal_ctrl)
	{
		glEnable(nobj->surface.normal.type);
		do_normal=GL_TRUE;
	}
	else
		do_normal=GL_FALSE;
	if(new_ctrl->texture_ctrl)
	{
		glEnable(nobj->surface.texture.type);
		do_texture=GL_TRUE;
	}
	else
		do_texture=GL_FALSE;
	for(j=0; j<s_bezier_cnt; j++)
	{
		for(i=0; i<t_bezier_cnt; i++)
		{
			offset=j*t_bezier_cnt + i;
			if(fine_culling_test_3D(nobj,*(new_ctrl->geom_offsets + offset),
					nobj->surface.geom.sorder,nobj->surface.geom.torder,
					new_ctrl->geom_s_stride,new_ctrl->geom_t_stride,
					nobj->surface.geom.dim))
				continue;
			glMap2f(nobj->surface.geom.type,0.0,1.0,new_ctrl->geom_s_stride,
				nobj->surface.geom.sorder,0.0,1.0,new_ctrl->geom_t_stride,
				nobj->surface.geom.torder,*(new_ctrl->geom_offsets + offset));
			if(do_color)
			{
				glMap2f(nobj->surface.color.type,0.0,1.0,
					new_ctrl->color_s_stride,nobj->surface.color.sorder,
					0.0,1.0,new_ctrl->color_t_stride,nobj->surface.color.torder,
					*(new_ctrl->color_offsets + offset));
			}
			if(do_normal)
			{
				glMap2f(nobj->surface.normal.type,0.0,1.0,
					new_ctrl->normal_s_stride,nobj->surface.normal.sorder,
					0.0,1.0,new_ctrl->normal_t_stride,
					nobj->surface.normal.torder,
					*(new_ctrl->normal_offsets+offset));
			}
			if(do_texture)
			{
				glMap2f(nobj->surface.texture.type,0.0,1.0,
					new_ctrl->texture_s_stride,nobj->surface.texture.sorder,
					0.0,1.0,new_ctrl->texture_t_stride,
					nobj->surface.texture.torder,
					*(new_ctrl->texture_offsets+offset));
			}
			glMapGrid2f(sfactors[j],0.0,1.0,tfactors[i],0.0,1.0);
			glEvalMesh2(display_mode,0,sfactors[j],0,tfactors[i]);
		}
	}
}

/* draw NURBS surface in OUTLINE POLYGON mode */
static void
draw_patch_mode( GLenum display_mode, GLUnurbsObj *nobj,
	new_ctrl_type *new_ctrl, GLint *sfactors, GLint *tfactors )
{
	GLsizei		offset;
	GLint		t_bezier_cnt,s_bezier_cnt;
	GLboolean	do_color,do_normal,do_texture;
	GLint		i,j;

	t_bezier_cnt=new_ctrl->t_bezier_cnt;
	s_bezier_cnt=new_ctrl->s_bezier_cnt;
	glEnable(nobj->surface.geom.type);
	if(new_ctrl->color_ctrl)
	{
		glEnable(nobj->surface.color.type);
		do_color=GL_TRUE;
	}
	else
		do_color=GL_FALSE;
	if(new_ctrl->normal_ctrl)
	{
		glEnable(nobj->surface.normal.type);
		do_normal=GL_TRUE;
	}
	else
		do_normal=GL_FALSE;
	if(new_ctrl->texture_ctrl)
	{
		glEnable(nobj->surface.texture.type);
		do_texture=GL_TRUE;
	}
	else
		do_texture=GL_FALSE;
	for(j=0; j<s_bezier_cnt; j++)
	{
		for(i=0; i<t_bezier_cnt; i++)
		{
			offset=j*t_bezier_cnt + i;
			if(fine_culling_test_3D(nobj,*(new_ctrl->geom_offsets + offset),
					nobj->surface.geom.sorder,nobj->surface.geom.torder,
					new_ctrl->geom_s_stride,new_ctrl->geom_t_stride,
					nobj->surface.geom.dim))
				continue;
			glMap2f(nobj->surface.geom.type,0.0,1.0,new_ctrl->geom_s_stride,
				nobj->surface.geom.sorder,0.0,1.0,new_ctrl->geom_t_stride,
				nobj->surface.geom.torder,*(new_ctrl->geom_offsets + offset));
			if(do_color)
			{
				glMap2f(nobj->surface.color.type,0.0,1.0,
					new_ctrl->color_s_stride,nobj->surface.color.sorder,
					0.0,1.0,new_ctrl->color_t_stride,nobj->surface.color.torder,
					*(new_ctrl->color_offsets + offset));
			}
			if(do_normal)
			{
				glMap2f(nobj->surface.normal.type,0.0,1.0,
					new_ctrl->normal_s_stride,nobj->surface.normal.sorder,
					0.0,1.0,new_ctrl->normal_t_stride,
					nobj->surface.normal.torder,
					*(new_ctrl->normal_offsets+offset));
			}
			if(do_texture)
			{
				glMap2f(nobj->surface.texture.type,0.0,1.0,
					new_ctrl->texture_s_stride,nobj->surface.texture.sorder,
					0.0,1.0,new_ctrl->texture_t_stride,
					nobj->surface.texture.torder,
					*(new_ctrl->texture_offsets+offset));
			}
			glMapGrid2f(sfactors[j],0.0,1.0,tfactors[i],0.0,1.0);
			glEvalMesh2(display_mode,0,sfactors[j],0,tfactors[i]);
		}
	}
}

void
init_new_ctrl(new_ctrl_type *p)
{
	p->geom_ctrl=p->color_ctrl=p->normal_ctrl=p->texture_ctrl=NULL;
	p->geom_offsets=p->color_offsets=p->normal_offsets=p->texture_offsets=NULL;
	p->s_bezier_cnt=p->t_bezier_cnt=0;
}

GLenum
augment_new_ctrl(GLUnurbsObj *nobj, new_ctrl_type *p)
{
	GLsizei offset_size;
	GLint	i,j;

	p->s_bezier_cnt=(p->geom_s_pt_cnt)/(nobj->surface.geom.sorder);
	p->t_bezier_cnt=(p->geom_t_pt_cnt)/(nobj->surface.geom.torder);
	offset_size=(p->s_bezier_cnt)*(p->t_bezier_cnt);
	p->geom_t_stride=nobj->surface.geom.dim;
	p->geom_s_stride=(p->geom_t_pt_cnt)*(nobj->surface.geom.dim);
	p->color_t_stride=nobj->surface.color.dim;
	p->color_s_stride=(p->color_t_pt_cnt)*(nobj->surface.color.dim);
	p->normal_t_stride=nobj->surface.normal.dim;
	p->normal_s_stride=(p->normal_t_pt_cnt)*(nobj->surface.normal.dim);
	p->texture_t_stride=nobj->surface.texture.dim;
	p->texture_s_stride=(p->texture_t_pt_cnt)*(nobj->surface.texture.dim);
	if((p->geom_offsets=(GLfloat **)malloc(sizeof(GLfloat *)*offset_size))==NULL)
	{
		call_user_error(nobj,GLU_OUT_OF_MEMORY);
		return GLU_ERROR;
	}
	if(p->color_ctrl)
		if((p->color_offsets=(GLfloat **)malloc(sizeof(GLfloat *)*offset_size))==NULL)
		{
			free_new_ctrl(p);
			call_user_error(nobj,GLU_OUT_OF_MEMORY);
			return GLU_ERROR;
		}
	if(p->normal_ctrl)
		if((p->normal_offsets=(GLfloat **)malloc(sizeof(GLfloat *)*offset_size))==NULL)
		{
			free_new_ctrl(p);
			call_user_error(nobj,GLU_OUT_OF_MEMORY);
			return GLU_ERROR;
		}
	if(p->texture_ctrl)
		if((p->texture_offsets=(GLfloat **)malloc(sizeof(GLfloat *)*offset_size))==NULL)
		{
			free_new_ctrl(p);
			call_user_error(nobj,GLU_OUT_OF_MEMORY);
			return GLU_ERROR;
		}
	for(i=0;i<p->s_bezier_cnt;i++)
		for(j=0;j<p->t_bezier_cnt;j++)
			*(p->geom_offsets + i*(p->t_bezier_cnt) + j) =
				p->geom_ctrl + i*(nobj->surface.geom.sorder)*
				(nobj->surface.geom.dim)*(p->geom_t_pt_cnt) +
				j*(nobj->surface.geom.dim)*(nobj->surface.geom.torder);
	if(p->color_ctrl)
		for(i=0;i<p->s_bezier_cnt;i++)
			for(j=0;j<p->t_bezier_cnt;j++)
				*(p->color_offsets + i*(p->t_bezier_cnt) + j) =
					p->color_ctrl + i*(nobj->surface.color.sorder)*
					(nobj->surface.color.dim)*(p->color_t_pt_cnt) +
					j*(nobj->surface.color.dim)*(nobj->surface.color.torder);
	if(p->normal_ctrl)
		for(i=0;i<p->s_bezier_cnt;i++)
			for(j=0;j<p->t_bezier_cnt;j++)
				*(p->normal_offsets + i*(p->t_bezier_cnt) + j) =
					p->normal_ctrl + i*(nobj->surface.normal.sorder)*
					(nobj->surface.normal.dim)*(p->normal_t_pt_cnt) +
					j*(nobj->surface.normal.dim)*(nobj->surface.normal.torder);
	if(p->texture_ctrl)
		for(i=0;i<p->s_bezier_cnt;i++)
			for(j=0;j<p->t_bezier_cnt;j++)
				*(p->texture_offsets + i*(p->t_bezier_cnt) + j) =
					p->texture_ctrl + i*(nobj->surface.texture.sorder)*
					(nobj->surface.texture.dim)*(p->texture_t_pt_cnt) +
					j*(nobj->surface.texture.dim)*(nobj->surface.texture.torder);
	return GLU_NO_ERROR;
}

/* main NURBS surface procedure */
void
do_nurbs_surface( GLUnurbsObj *nobj )
{
	GLint			*sfactors,*tfactors;
	new_ctrl_type	new_ctrl;

	/* test user supplied data */
	if(test_nurbs_surfaces(nobj)!=GLU_NO_ERROR)
		return;

	init_new_ctrl(&new_ctrl);

	if(convert_surfs(nobj,&new_ctrl)!=GLU_NO_ERROR)
		return;
	if(augment_new_ctrl(nobj,&new_ctrl)!=GLU_NO_ERROR)
		return;
	if(glu_do_sampling_3D(nobj,new_ctrl.geom_ctrl,new_ctrl.geom_s_pt_cnt,
		new_ctrl.geom_t_pt_cnt,&sfactors,&tfactors)!=GLU_NO_ERROR)
	{
		free_new_ctrl(&new_ctrl);
		return;
	}
	switch(nobj->display_mode)
	{
		case GLU_FILL:
/*			if(polygon_trimming(nobj,&new_ctrl,sfactors,tfactors)==GLU_NO_ERROR)*/
				draw_polygon_mode(GL_FILL,nobj,&new_ctrl,sfactors,tfactors);
			break;
		case GLU_OUTLINE_POLYGON:
			/* TODO - missing trimming handeling */
/* just for now - no OUTLINE_PATCH mode 
			draw_patch_mode(GL_LINE,nobj,&new_ctrl,sfactors,tfactors);
			break; */
		case GLU_OUTLINE_PATCH:
/*			if(polygon_trimming(nobj,&new_ctrl,sfactors,tfactors)==GLU_NO_ERROR)*/
				draw_polygon_mode(GL_LINE,nobj,&new_ctrl,sfactors,tfactors);
			break;
	}
	free(sfactors);
	free(tfactors);
	free_new_ctrl(&new_ctrl);
}

