/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvasphere.c
 >>>>
 >>>>      Program Name: vasphere
 >>>>
 >>>> Date Last Updated: Tue Apr 23 02:07:18 1991 
 >>>>
 >>>>          Routines: lvasphere - the library call for vasphere
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvasphere - library call for vasphere
*
* Purpose:
*    
*    Generate an aspheric surface of revolution
*    
*    
* Input:
*    
*    image          input image structure
*    
*    rows           the desired number of rows in output image
*    
*    cols           the desired number of columns in output image
*    
*    type           the data storage type of the resulting image
*    
*    cx             the x pixel coordinate of the origin of the physi-
*                   cal coordinate system
*    
*    cy             the y pixel coordinate of the origin of the physi-
*                   cal coordinate system
*    
*    x              the X offset of the center of symmetry  (meters)
*    
*    y              the Y offset of the center of symmetry (meters)
*    
*    s              the size if a pixel (meters)
*    
*    curv           curvature parameter
*    
*    a2             A2 coefficient
*    
*    a4             A4 coefficient
*    
*    a6             A6 coefficient
*    
*    a8             A8 coefficient
*    
*    a10            A10 coefficient
*    
*    cutoff         cutoff radius (meters)
*    
*    piston         piston offset (meters)
*    
*    
* Output:
*    
*    image          holds the resulting output image.
*    
*    
*
* Written By: Scott Wilson
*    
*    
****************************************************************/


/* -library_def */
int
lvasphere(image,rows,cols,type,cx,cy,x,y,s,curv,a2,a4,a6,a8,a10,cutoff,piston)
struct xvimage **image;
int rows,cols,type,cx,cy;
float x,y,s,cutoff,piston;
float curv,a2,a4,a6,a8,a10;
/* -library_def_end */

/* -library_code */
{
  int i,j;
  unsigned char *c;
  float *f,xhat,yhat;
  double *d,z,eval_asphere();
  struct xvimage *img, *createimage();

  img   = createimage((unsigned long) rows,  /* number of rows */
                     (unsigned long) cols,   /* number of columns */
                     (unsigned long) type,   /* data storage type */
                     (unsigned long) 1,      /* num_of_images */
                     (unsigned long) 1,      /* num_data_bands */
                     "created by vasphere",    /* comment */
                     (unsigned long) 0,      /* map_row_size */
                     (unsigned long) 0,      /* map_col_size */
                     (unsigned long)
                     VFF_MS_NONE,            /* map_scheme */
                     (unsigned long)
                     VFF_MAPTYP_NONE,        /* map_storage_type */
                     (unsigned long)
                     VFF_LOC_IMPLICIT,       /* location_type */
                     (unsigned long) 0);     /* location_dim */
  if (img == NULL)
    {
      fprintf(stderr,"lvasphere: Unable to allocate new image!\n");
      *image = NULL;
      return(0);
    }
  img->pixsizx = s;
  img->pixsizy = s;

  *image = img;

  switch(type)
    {
      case VFF_TYP_1_BYTE:
        c = (unsigned char *)(img->imagedata);
        for (i=0; i<rows; i++)
          {
            for (j=0; j<cols; j++)
              {
                xhat = (j-cx)*s-x;
                yhat = (i-cy)*s-y;
                z = xhat*xhat+yhat*yhat;
                z = sqrt(z);
                if (cutoff == 0.0)
                  c[PIXEL(j,i,rows,cols)] = 
                    eval_asphere(curv,a2,a4,a6,a8,a10,piston,z);
                else {
                  if (z <= cutoff)
                    c[PIXEL(j,i,rows,cols)] = 
                      eval_asphere(curv,a2,a4,a6,a8,a10,piston,z);
                  else
                    c[PIXEL(j,i,rows,cols)]= 0.0;
                }
              }
          }
        break;
      case VFF_TYP_FLOAT:
        f = (float *)(img->imagedata);
        for (i=0; i<rows; i++)
          {
            for (j=0; j<cols; j++)
              {
                xhat = (j-cx)*s-x;
                yhat = (i-cy)*s-y;
                z = xhat*xhat+yhat*yhat;
                z = sqrt(z);
                if (cutoff == 0.0)
                  f[PIXEL(j,i,rows,cols)] = 
                    eval_asphere(curv,a2,a4,a6,a8,a10,piston,z);
                else {
                  if (z <= cutoff)
                    f[PIXEL(j,i,rows,cols)] = 
                      eval_asphere(curv,a2,a4,a6,a8,a10,piston,z);
                  else
                    f[PIXEL(j,i,rows,cols)]= 0.0;
                }
              }
          }
        break;
      case VFF_TYP_DOUBLE:
        d = (double *)(img->imagedata);
        for (i=0; i<rows; i++)
          {
            for (j=0; j<cols; j++)
              {
                xhat = (j-cx)*s-x;
                yhat = (i-cy)*s-y;
                z = xhat*xhat+yhat*yhat;
                z = sqrt(z);
                if (cutoff == 0.0)
                  d[PIXEL(j,i,rows,cols)] = 
                    eval_asphere(curv,a2,a4,a6,a8,a10,piston,z);
                else {
                  if (z <= cutoff)
                    d[PIXEL(j,i,rows,cols)] = 
                      eval_asphere(curv,a2,a4,a6,a8,a10,piston,z);
                  else
                    d[PIXEL(j,i,rows,cols)]= 0.0;
                }
              }
          }
        break;
      default:
        fprintf(stderr,"lvasphere: Unsupported data storage type requested\n");
        fprintf(stderr,"         Supported types: BYTE, FLOAT, and DOUBLE\n");
        return(0);
    }

  return(1);
}

double
eval_asphere(curv,a2,a4,a6,a8,a10,piston,r)
float curv,a2,a4,a6,a8,a10,piston;
double r;
  {
    double x,r2,r4,r6,r8,r10;

    r2 = r*r;
    r4 = r2*r2;
    r6 = r2*r4;
    r8 = r4*r4;
    r10 = r4*r6;

    x = piston +
        (curv*r2/(1+sqrt(1-curv*curv*r2)))+a2*r2+a4*r4+a6*r6+a8*r8+a10*r10;
    return(x);
  }
/* -library_code_end */
