/* $Id: postscript.c,v 2.3 89/09/20 17:01:44 mbp Exp $
 *
 * postscript.c: procedures for writing PostScript files
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

#include <stdio.h>
#include <math.h>
#include "gr.h"
#include "internal.h"

#define RADTODEG(a) ((a)*180/M_PI);

#define HEADER 1
#define TRAILER 2

static int current_x=0;
static int current_y=0;

static FILE *postscript_fp=NULL;
static FILE *find_ps_file();
char *getenv();

/*-----------------------------------------------------------------------
 * Function:	GR_begin_postscript_file
 * Description:	initialize PostScript output to a file
 * Args  IN:	fp: pointer to already opened file
 */
GR_begin_postscript_file(fp, logo, label, time)
     FILE *fp;
     char *logo, *label, *time;
{
  FILE *header_fp;
  int c;

  /* Make sure we're not already plotting */
  if (postscript_fp != NULL) return(1);

  /* Save the file pointer */
  postscript_fp = fp;

  /* Write initial lines of PostScript to file */
  if (fprintf(postscript_fp, "%%!\n") == EOF) return(1);
  if (fprintf(postscript_fp, "/CanvasWidth %1d def\n", GR_canvas_width)
      == EOF) return(1);
  if (fprintf(postscript_fp, "/CanvasHeight %1d def\n", GR_canvas_height)
      == EOF) return(1);

  if ( (label != NULL) && (strlen(label)>0) ) {
    if (fprintf(postscript_fp, "/LabelPresent 1 def\n") == EOF) return(1);
    if (fprintf(postscript_fp, "/Label (%s) def\n",label) == EOF) return(1);
  }
  else
    if (fprintf(postscript_fp, "/LabelPresent 0 def\n") == EOF) return(1);

  if ( (time != NULL) && (strlen(time)>0) ) {
    if (fprintf(postscript_fp, "/TimePresent 1 def\n") == EOF) return(1);
    if (fprintf(postscript_fp, "/Time (%s) def\n",time) == EOF) return(1);
  }
  else
    if (fprintf(postscript_fp, "/TimePresent 0 def\n") == EOF) return(1);

  if ( (logo != NULL) && (strlen(logo)>0) ) {
    if (fprintf(postscript_fp, "/LogoPresent 1 def\n") == EOF) return(1);
    if (fprintf(postscript_fp, "/Logo (%s) def\n",logo) == EOF) return(1);
  }
  else
    if (fprintf(postscript_fp, "/LogoPresent 0 def\n") == EOF) return(1);

  /* Copy PostScript header file into new file */
  header_fp = find_ps_file( HEADER );
  if (header_fp == NULL) {
    GR_error("I can't find the PostScript header file !");
    return(1);
  }
  else {
    while ((c=getc(header_fp))!=EOF) putc(c,postscript_fp);
    fclose(header_fp);
  }
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	GR_end_postscript_file
 * Description:	terminate PostScript output to a file
 * Args:	(none)
 * Notes:	Should only be called after a call to
 *		GR_begin_postscript_file.   This call does NOT close
 *		the file.
 */
int
GR_end_postscript_file()
{
  FILE *trailer_fp;
  int c;

  /* Make sure we're plotting */
  if (postscript_fp == NULL) return(1);
  
  /* Copy PostScript trailer file into the new file */
  trailer_fp = find_ps_file( TRAILER );
  if (trailer_fp == NULL) {
    GR_error("I can't find the PostScript trailer file !");
    return(1);
  }
  else {
    while ((c=getc(trailer_fp))!=EOF) putc(c,postscript_fp);
    fclose(trailer_fp);
  }

  /* Reset the global file pointer so we know we're not plotting */
  postscript_fp = NULL;

  return(0);
}

int
GR_postscript_move(x,y)
int x,y;
{
  fprintf( postscript_fp, "%1d %1d m\n", x, y );
}

int
GR_postscript_draw(x,y)
int x,y;
{
  fprintf( postscript_fp, "%1d %1d d\n", x, y );
}

int
GR_postscript_point()
{
  fprintf( postscript_fp, "p\n" );
}

int
GR_postscript_arc( xc,yc, x1,y1, x2,y2 )
     int xc,yc,x1,y1,x2,y2;
{
  int xdiff, ydiff, r;
  double ang1,ang2;
  
  /* Compute radius in device coords: */
  xdiff = x1 - xc;
  ydiff = y1 - yc;
  r = (int) (sqrt( (double)(xdiff*xdiff + ydiff*ydiff) ) + .5);

  /* Check to see if the two points on the arc are equal: */
  if ( (x1==x2) && (y1==y2) ) {

    /* If they are, draw a cicle through that point */
    fprintf( postscript_fp, "%1d %1d %1d c\n", xc, yc, r );

  }
  else {
    /* Otherwise, draw the arc. The roles of x and y are switched in
     * calculating the angles here, because in the PostScript file x
     * is vertical and y is horizontal (the picture is drawn
     * sideways).  The PostScript file automatically switches the x &
     * y coordinates of points, however, which is why we DON'T
     * interchange xc and yc.  This is terribly confusing, but it's
     * the first combination I came up with which works! */
    ang1 = RADTODEG( atan2( (double)xdiff, (double)ydiff ) );
    ang2 = RADTODEG( atan2( (double)(x2 - xc), (double)(y2 - yc) ) );
    fprintf( postscript_fp, "%1d %1d %1d %f %f a\n", xc, yc, r, ang1, ang2 );
  }
}

/*-----------------------------------------------------------------------
 * Function:     find_ps_file
 * Description:  return FILE ptr to .ps header or trailer file
 * Arguments:    type: tells which file to look for:
 *                 HEADER means look for header file
 *                 TRAILER means look for trailer file
 * Returns:      ptr to file opened for reading
 * Notes:        This procedure looks for this files "gr_header.ps" nad
 *               "gr_trailer.ps" in the following places, in this order:
 *               1. In the directory given by the environment variable
 *                  GR_LIB, if this variable is set
 *               2. In the "hardwired" location, given by the strings
 *                  HEADER_FILE, and TRAILER_FILE.
 */
static FILE *
find_ps_file(c)
int c;
{
  char fname[256], fbase[20], *gr_lib;
  int found;
  FILE *fp;

  sprintf(fbase, "%s", c==HEADER ? "gr_header.ps" : "gr_trailer.ps");

  /* First look in GR_LIB */
  gr_lib = getenv("GR_LIB");
  if (gr_lib != NULL) {
    sprintf(fname,"%s%s%s",
	    gr_lib, (gr_lib[strlen(gr_lib)-1]=='/') ? "" : "/",
	    fbase);
    fp = fopen(fname,"r");
    if (fp != NULL) return(fp);
  }
 
  /* Then in hardwired location */
  sprintf(fname,"%s", c==HEADER ? HEADER_FILE : TRAILER_FILE );
  fp = fopen(fname,"r");
  if (fp != NULL) return(fp);

  /* If we got this far, it doesn't exist */
  return(NULL);
}
