/*ScianWindows.c
  Eric Pepke
  March 8, 1990
  Window handling routines in Scian
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianWindows.h"
#include "ScianVisWindows.h"
#include "ScianLists.h"
#include "ScianEvents.h"
#include "ScianColors.h"
#include "ScianErrors.h"
#include "ScianIDs.h"
#include "ScianScripts.h"
#include "ScianTextBoxes.h"
#include "ScianFileSystem.h"
#include "ScianPreferences.h"
#include "ScianDatasets.h"
#include "ScianHelp.h"
#include "ScianDialogs.h"
#include "ScianArrays.h"
#include "ScianFiles.h"

int updateBatch = 0;			/*Current batch of updates*/
WinInfoPtr allWindows = 0;		/*Window info on all windows*/
extern long inputWindow;		/*Window receiving input focus*/
short firstTime = 1;			/*First time a new window is gotten*/
WinInfoPtr curIntWindow = 0;		/*The current window being interacted*/
int curDisplayMode;			/*Current display mode*/
Bool rgbp;				/*True iff in rgb mode*/
extern Bool phsco;			/*Phscologram mode*/
int selWindow = 0;			/*Currently selected window*/
WinInfoPtr selWinInfo;			/*Info for currently selected window*/
ObjPtr windowClass = 0;			/*Window class*/
short curCursor = 0;			/*Current cursor*/
extern WinInfoPtr logWindow;		/*Logging window*/
int recScrWidth = VSCRWIDTH;		/*Video screen size*/
int recScrHeight = VSCRHEIGHT;
ObjPtr clipboard;			/*Clipboard*/

#ifdef WINNOCLOSE
#define MAXCACHEDWINDOWS 100
long cachedWindows[MAXCACHEDWINDOWS];
int nCachedWindows = 0;
#endif

int movingDrawingQuality = DQ_FULL;	/*Drawing quality while moving*/
int drawingQuality = DQ_FULL;   	/*The current drawing quality*/

long fontSubMenu;
long sizeSubMenu;
long alignSubMenu;

#define STAGX		100		/*Minimum x for window staggering*/
#define STAGY		800		/*Maximum y for window staggering*/
#define STAGSTEP	40		/*Step for staggering*/
#define STAGMAX		200		/*Maximum staggering*/

int stagger = 0;			/*Current staggering amount*/

int textSizes[] =
    {
	9,
	10,
	12,
	14,
	18,
	24,
	36,
	48,
	72
    };

/*Names of all the main level menus*/
char *menuName[NMENUS] =
    {
	"File",
	"Edit",
	"Object",
	"Text",
	"Arrange",
	"Windows"
    };

void GetWindowBounds(l, r, b, t)
int *l, *r, *b, *t;
/*Gets the viewport of the current window into l, r, b, t*/
{
    Screencoord left, right, bottom, top;
    getviewport(&left, &right, &bottom, &top);
    *l = left;
    *r = right;
    *b = bottom;
    *t = top;
}

void GetWindowOrigin(x, y)
int *x, *y;
/*Gets the origin of the current window*/
{
    long lx, ly;
    getorigin(&lx, &ly);
    *x = lx;
    *y = ly;
}

void PopWindow(winInfo)
WinInfoPtr winInfo;
/*Pops winInfo*/
{
    if (winInfo -> id)
    {
	SelectWindow(winInfo -> id);
	winpop();
    }
    else
    {
	prefposition(winInfo -> nl, winInfo -> nr, winInfo -> nb, winInfo -> nt);
#ifdef WINNOCLOSE
	if (nCachedWindows)
	{
	    --nCachedWindows;
	    winInfo -> id = cachedWindows[nCachedWindows];
	}
	else
#endif
	winInfo -> id = winopen(winInfo -> winTitle);
	winset(winInfo -> id);
	setcursor(curCursor, 1, 0);
	minsize(winInfo -> minWidth, winInfo -> minHeight);
	maxsize(winInfo -> maxWidth, winInfo -> maxHeight);
	/*Set to the current window and set the graphics state*/
	winconstraints();
	SelectWindow(winInfo -> id);
	wintitle(winInfo -> winTitle);
	SetMode(winInfo);
	SetVar((ObjPtr) winInfo, PICSTATE, NewInt(FBBB));
    }
}

void ResetMovingQuality()
/*Resets the moving quality to the highest.  Called by ScianEvents 
  heuristically*/
{
    movingDrawingQuality = DQ_FULL;
}


void DoShowFrame()
/*Shows the window frame*/
{
#ifndef WINNOCLOSE
    Screencoord l, r, b, t;
    long ox, oy;
    long sx, sy;
    long oldWindow;
    if (!(selWinInfo -> flags & WINNOFRAME))
    {
	return;
    }
    if (logging)
    {
	Log("show frame\n");
    }
    /*Create a new window in the same place*/
    getviewport(&l, &r, &b, &t);
    getorigin(&ox, &oy);
    getsize(&sx, &sy);
    prefposition(ox, ox + sx - 1, oy, oy + sy - 1);
    oldWindow = selWindow;
    selWindow = winopen(selWinInfo -> winTitle);
    selWinInfo -> id = selWindow;
    wintitle(selWinInfo -> winTitle);
    winclose(oldWindow);
    winset(selWindow);
    setcursor(curCursor, 1, 0);
    minsize(selWinInfo -> minWidth, selWinInfo -> minHeight);
    maxsize(selWinInfo -> maxWidth, selWinInfo -> maxHeight);
    /*Set to the current window and set the graphics state*/
    SetMode(selWinInfo);
    winconstraints();
    if (oldWindow == inputWindow)
    {
	inputWindow = selWinInfo -> id;
    }

    selWinInfo -> flags &= ~WINNOFRAME;
    SetVar((ObjPtr) selWinInfo, PICSTATE, NewInt(FBBB));
    DeleteMenus(selWinInfo);
#endif
}

void DoHideFrame()
/*Hides the window frame*/
{
#ifndef WINNOCLOSE
    Screencoord l, r, b, t;
    long ox, oy;
    long sx, sy;
    long oldWindow;
    if (selWinInfo -> flags & WINNOFRAME)
    {
	return;
    }
    if (logging)
    {
	Log("hide frame\n");
    }
    /*Create a new window in the same place*/
    getviewport(&l, &r, &b, &t);
    getorigin(&ox, &oy);
    getsize(&sx, &sy);
    prefposition(ox, ox + sx - 1, oy, oy + sy - 1);
    oldWindow = selWindow;
    noborder();
    selWindow = winopen(selWinInfo -> winTitle);
    selWinInfo -> id = selWindow;
    wintitle(selWinInfo -> winTitle);
    winclose(oldWindow);
    winset(selWindow);
    setcursor(curCursor, 1, 0);
    minsize(selWinInfo -> minWidth, selWinInfo -> minHeight);
    maxsize(selWinInfo -> maxWidth, selWinInfo -> maxHeight);
    /*Set to the current window and set the graphics state*/
    SetMode(selWinInfo);
    winconstraints();
    if (oldWindow == inputWindow)
    {
	inputWindow = selWinInfo -> id;
    }

    selWinInfo -> flags |= WINNOFRAME;
    SetVar((ObjPtr) selWinInfo, PICSTATE, NewInt(FBBB));
    DeleteMenus(selWinInfo);
#endif
}

void ScrSaveFailed(void)
/*Alerts the user that a screen save failed*/
{
    WinInfoPtr errWindow;
    errWindow = AlertUser(UIRED, (WinInfoPtr) 0, "The attempt to save the screen failed.", 0, 0, " "); 
SetVar((ObjPtr) errWindow, HELPSTRING,
	NewString("The attempt to save the screen failed.  This is probably \
because the current directory is a directory to which you don't have write \
access.  It may also be that the disk is full."));
}

void SaveScreen(window, flags)
WinInfoPtr window;
int flags;
/*Saves a screen*/
{
    long l, r, b, t;
    int k;

    if (flags & F_OPTIONDOWN)
    {
	l = b = 0;
	r = SCRWIDTH;
	t = SCRHEIGHT;
    }
    else
    {
	long sx, sy;
	getorigin(&l, &b);
	getsize(&sx, &sy);
	r = l + sx - 1;
	t = b + sy - 1;

	if (!(flags & F_SHIFTDOWN || ((WinInfoPtr) window) -> flags & WINNOFRAME))
	{
	    l -= WINBL;
	    r += WINBR;
	    b -= WINBB;
	    t += WINBT;
	}
    }
    for (k = 0; ; ++k)
    {
	FILE *test;
	sprintf(tempStr, "screen%d", k);
	if (test = fopen(tempStr, "r"))
	{
	    fclose(test);
	}
	else
	{

	    sprintf(tempStr, "scrsave screen%d %d %d %d %d", k, l, r, b, t);
	    if (system(tempStr))
	    {
		DoTask(ScrSaveFailed);
	    }
	    else
	    {
		printf("Screen saved in screen%d\n", k);
	    }
	    return;
	}
    }
}

void DeleteMenus(winInfo)
WinInfoPtr winInfo;
/*Deletes the menus in winInfo*/
{
    int k;
    if (winInfo -> mainMenu)
    {
	freepup(winInfo -> mainMenu);
	winInfo -> mainMenu = 0;
    }
    for (k = 0; k < NMENUS; ++k)
    {
	if (winInfo -> subMenus[k])
	{
	    freepup(winInfo -> subMenus[k]);
	    winInfo -> subMenus[k] = 0;
	}
    }
}

void SetFont(object, f)
ObjPtr object;
char *f;
/*Sets object to have font f*/
{
    if (object)
    {
	if (logging)
	{
	    char cmd[256];
	    char *s;

	    sprintf(cmd, "set font ");
	    s = &(cmd[0]);
	    while (*s) ++s;
	    MakeObjectName(s, object);
	    while (*s) ++s;
	    *s++ = ' ';
	    strcpy(s, f);
	    while (*s) ++s;
	    *s++ = '\n';
	    *s = 0;
	    Log(cmd);
	}
#if 0
	if (InClass(object, textBoxClass))
#endif
	{
	    SetTextFont(object, f);
	}
    }
}

void MenuSetFont(n)
int n;
/*Sets font n*/
{
    if (selWinInfo)
    {
	ObjPtr current;
	current = GetVar((ObjPtr) selWinInfo, CURRENT);
	if (current)
	{
	    SetFont(current, fonts[n - 1]);
	}
    }
}

void SetSize(object, size)
ObjPtr object;
int size;
/*Sets object to have font size size*/
{
    if (object)
    {
	if (logging)
	{
	    char cmd[256];
	    char *s;

	    sprintf(cmd, "set size ");
	    s = &(cmd[0]);
	    while (*s) ++s;
	    MakeObjectName(s, object);
	    while (*s) ++s;
	    *s++ = ' ';
	    sprintf(s, "%d", size);
	    while (*s) ++s;
	    *s++ = '\n';
	    *s = 0;
	    Log(cmd);
	}
#if 0
	if (InClass(object, textBoxClass))
#endif
	{
	    SetTextSize(object, size);
	}
    }
}

void MenuSetSize(n)
int n;
/*Sets font size n*/
{
    if (selWinInfo)
    {
	ObjPtr current;
	current = GetVar((ObjPtr) selWinInfo, CURRENT);
	if (current)
	{
	    SetSize(current, textSizes[n - 1]);
	}
    }
}

void SetAlignment(object, alignment)
ObjPtr object;
int alignment;
/*Sets object to have alignment*/
{
    if (object)
    {
	if (logging)
	{
	    char cmd[256];
	    char *s;

	    sprintf(cmd, "set alignment ");
	    s = &(cmd[0]);
	    while (*s) ++s;
	    MakeObjectName(s, object);
	    while (*s) ++s;
	    *s++ = ' ';
	    sprintf(s, "%d", alignment);
	    while (*s) ++s;
	    *s++ = '\n';
	    *s = 0;
	    Log(cmd);
	}
#if 0
	if (InClass(object, textBoxClass))
#endif
	{
	    SetTextAlign(object, alignment);
	}
    }
}

void MenuSetAlignment(n)
int n;
/*Sets alignment of text to n, n is ALIGNLEFT, ALIGNRIGHT, or ALIGNCENTER*/
{
    if (selWinInfo)
    {
	ObjPtr current;
	current = GetVar((ObjPtr) selWinInfo, CURRENT);
	if (current)
	{
	    SetAlignment(current, n);
	}
    }
}

void PopDatasetsWindow(void)
{
    Log("show datasets\n");
    InhibitLogging(true);
    PopWindow(DatasetsWindow());
    InhibitLogging(false);
}

void PopFileReadersWindow(void)
{
    Log("show filereaders\n");
    InhibitLogging(true);
    PopWindow(FileReadersWindow());
    InhibitLogging(false);
}

void CutText(void)
/*Cuts text*/
{
    if (selWinInfo)
    {
	ObjPtr current;
	current = GetVar((ObjPtr) selWinInfo, CURRENT);
	if (current)
	{
	    FuncTyp method;
	    method = GetMethod(current, CUT);
	    if (method)
	    {
		ObjPtr value;
		value = (*method)(current);
		if (value)
		{
		    ObjPtr *elements;
		    elements = ELEMENTS(clipboard);
		    elements[0] = value;
		}
	    }
	}
    }
}

void CopyText(void)
/*Cuts text*/
{
    if (selWinInfo)
    {
	ObjPtr current;
	current = GetVar((ObjPtr) selWinInfo, CURRENT);
	if (current)
	{
	    FuncTyp method;
	    method = GetMethod(current, COPY);
	    if (method)
	    {
		ObjPtr value;
		value = (*method)(current);
		if (value)
		{
		    ObjPtr *elements;
		    elements = ELEMENTS(clipboard);
		    elements[0] = value;
		}
	    }
	}
    }
}

void PasteText(void)
/*Cuts text*/
{
    if (selWinInfo)
    {
	ObjPtr current;
	current = GetVar((ObjPtr) selWinInfo, CURRENT);
	if (current)
	{
	    FuncTyp method;
	    method = GetMethod(current, PASTE);
	    if (method)
	    {
		ObjPtr value;
		ObjPtr *elements;
		elements = ELEMENTS(clipboard);
		value = elements[0];
		(*method)(current, value);
	    }
	}
    }
}

void DoUndoReshape()
/*Undoes a reshape of the window*/
{
    if (selWinInfo)
    {
	winposition(selWinInfo -> ol, selWinInfo -> or, selWinInfo -> ob, selWinInfo -> ot);
    }
}

void BuildMenu(winInfo)
WinInfoPtr winInfo;
/*Builds a menu in winInfo*/
{
    long menus[NMENUS];		/*All the main level menus*/
    long mainMenu;		/*Main menu*/
    int k;
    Bool textMenu = false;	/*Text menu has been added*/

    /*Make the initial menus null*/
    DeleteMenus(winInfo);
    for (k = 0; k < NMENUS; ++k)
    {
	menus[k] = 0;
    }

    /*Set up the prefix parts of the menus*/

    /*Arrange menu*/
    if ((winInfo -> flags & WINFIXEDSIZE) == 0)
    {
	/*It can move; give it moving items*/
	menus[ARRANGEMENU] = newpup();
	addtopup(menus[ARRANGEMENU], "Full Screen%f", DoMaxScreen);
	addtopup(menus[ARRANGEMENU], "Video Screen%f", DoVideoScreen);
	addtopup(menus[ARRANGEMENU], "Double Video Screen%f", Do2VideoScreen);
	addtopup(menus[ARRANGEMENU], "PHSCologram Screen%f", DoPhscoScreen);
	addtopup(menus[ARRANGEMENU], "Previous%f", DoUndoReshape);
	addtopup(menus[ARRANGEMENU], "--------");
    }

    /*Windows menu*/
    menus[WINDOWSMENU] = newpup();
    addtopup(menus[WINDOWSMENU], "Help%f", DoShowHelp);
    addtopup(menus[WINDOWSMENU], "Preferences%f", DoShowPreferences);
    addtopup(menus[WINDOWSMENU], "Datasets%f", PopDatasetsWindow);
    addtopup(menus[WINDOWSMENU], "File Readers%f", PopFileReadersWindow);

    /*File menu*/
    menus[FILEMENU] = newpup();
    addtopup(menus[FILEMENU], "New File Window%f", DoNewFileWindow);

    /*Add the window frame show or hide*/
    addtopup(menus[WINDOWSMENU], "--------");

    /*Go through the defined menu items*/
    for (k = 0; k < winInfo -> nMenuItems; ++k)
    {
	int whichMenu;
	whichMenu = winInfo -> menuItems[k] . menu;
	if (!menus[whichMenu])
	{
	    menus[whichMenu] = newpup();
	}
	if ((whichMenu == TEXTMENU) && !textMenu)
	{
	    textMenu = true;
	    addtopup(menus[TEXTMENU], "Cut Text%f", CutText);
	    addtopup(menus[TEXTMENU], "Copy Text%f", CopyText);
	    addtopup(menus[TEXTMENU], "Paste Text%f", PasteText);
	    if (fontSubMenu)
	    {
		addtopup(menus[TEXTMENU], "Text Font%m", fontSubMenu);
	    }
	    if (sizeSubMenu)
	    {
		addtopup(menus[TEXTMENU], "Text Size%m", sizeSubMenu);
	    }
	    if (alignSubMenu)
	    {
		addtopup(menus[TEXTMENU], "Text Alignment%m", alignSubMenu);
	    }
	    addtopup(menus[TEXTMENU], "");
	}
	sprintf(tempStr, "%s%%f", winInfo -> menuItems[k] . name);
	addtopup(menus[whichMenu], tempStr, winInfo -> menuItems[k] . action);
    }

    /*Set up the postfix parts of the menus*/
    if (GetPredicate((ObjPtr) winInfo, HIDEPANEL))
    {
	if (GetPredicate((ObjPtr) winInfo, PANELHIDDEN))
	{
	    addtopup(menus[WINDOWSMENU], "Show Control Panel%f", DoShowPanel);
	}
	else
	{
	    addtopup(menus[WINDOWSMENU], "Hide Control Panel%f", DoHidePanel);
	}
    }
    if (winInfo -> flags & WINNOFRAME)
    {
	addtopup(menus[WINDOWSMENU], "Show Window Frame%f", DoShowFrame);
    }
    else
    {
	addtopup(menus[WINDOWSMENU], "Hide Window Frame%f", DoHideFrame);
    }
    addtopup(menus[WINDOWSMENU], "Close%f", DoClose);

    /*Now create the main menu*/
    sprintf(tempStr, "%s%%t", winInfo -> winTitle);
    mainMenu = defpup(tempStr);
    addtopup(mainMenu, "Help%f", DoShowHelp);

    /*Add in the little menus*/
    for (k = 0; k < NMENUS; ++k)
    {
	if (menus[k])
	{
	    sprintf(tempStr, "%s%%m", menuName[k]);
	    addtopup(mainMenu, tempStr, menus[k]);
	}
	winInfo -> subMenus[k] = menus[k];
    }

    /*Add the quit*/
    addtopup(mainMenu, "Quit SciAn%f", DoQuit);
    winInfo -> mainMenu = mainMenu;
}

void DefineMenuItem(window, menu, name, action)
ObjPtr window;
int menu;
char *name;
void (*action)();
/*Defines a menu item in window under menu with name and action*/
{
    int nMenuItems;

    /*Add it to the window's list of menu items*/
    nMenuItems = ((WinInfoPtr) window) -> nMenuItems;
    ((WinInfoPtr) window) -> menuItems[nMenuItems] . menu = menu;
    strcpy(((WinInfoPtr) window) -> menuItems[nMenuItems] . name, name);
    ((WinInfoPtr) window) -> menuItems[nMenuItems] . action = action;
    ((WinInfoPtr) window) -> nMenuItems = nMenuItems + 1;
}

void MySetCursor(cursorNumber)
int cursorNumber;
/*Sets the cursor to cursorNumber*/
{
#ifdef CURSORS4D
    if (cursorNumber != curCursor)
    {
	Screencoord l, r, b, t; 
	WinInfoPtr curWindow;
	long oldWindow;
	curCursor = cursorNumber;

	oldWindow = winget();

	/*Save everything*/
	if (oldWindow)
	{
	    getscrmask(&l, &r, &b, &t);
	    pushattributes();
	    pushviewport();
	}

	curWindow = allWindows;
	while (curWindow)
	{
	    if (curWindow -> id)
	    {
		winset(curWindow -> id);
		if (curCursor < 0)
		{
		    cursoff();
		}
		else
		{
		    curson();
		    setcursor(curCursor, (Colorindex) 1, (Colorindex) 0);
		}
	    }
	    curWindow = curWindow -> next;
	}
	if (oldWindow)
	{
	    winset(oldWindow);

	    popviewport();
	    popattributes();
	    scrmask(l, r, b, t);
	}
    }
#endif
}

WindowID NewOpenedWindow(title, minWidth, minHeight, maxWidth, maxHeight, flags)
char *title;
int minWidth, minHeight, maxWidth, maxHeight;
long flags;
/*Creates a new opened window with title, and size determined by minWidth,
  minHeight, maxWidth, and maxHeight and returns a window ID which will
  differ from machine to machine.*/
{
    WindowID retVal;
    long ox, oy;
    int stagX, stagY;

    if (flags & WINNOFRAME)
    {
	noborder();
    }

    if (flags & WINCENTERED)
    {
	ox = SCRWIDTH / 2 - minWidth / 2;
	oy = SCRHEIGHT / 2 - minHeight / 2;
	prefposition(ox, ox + minWidth,
		     oy, oy + minHeight);
    }
    else
#ifndef WINNOCLOSE
    if (curScript || runningRemote || GetPrefTruth(PREF_NEWWINPLACE))
#endif
    {
	prefposition(stagX = STAGX + stagger, STAGX + stagger + minWidth,
		     stagY = STAGY - stagger - minHeight, STAGY - stagger);
	stagger += STAGSTEP;
	if (stagger > STAGMAX) stagger = 0;
    }
#ifndef WINNOCLOSE
    else
    {
	minsize(minWidth, minHeight);
	maxsize(maxWidth, maxHeight);
    }
#endif

#ifdef WINNOCLOSE
	if (nCachedWindows)
	{
	    --nCachedWindows;
	    retVal = cachedWindows[nCachedWindows];
	    winset(retVal);
	    if (flags & WINCENTERED)
	    {
		ox = SCRWIDTH / 2 - minWidth / 2;
		oy = SCRHEIGHT / 2 - minHeight / 2;
		minsize(minWidth, minHeight);
		maxsize(maxWidth, maxHeight);
		winconstraints();
		winposition(ox, ox + minWidth,
		     oy, oy + minHeight);
	    }
	    else
	    {
		minsize(minWidth, minHeight);
		maxsize(maxWidth, maxHeight);
		winconstraints();
		winposition(stagX, stagX + minWidth,
		     stagY, stagY + minHeight);
	    }
	}
	else
#endif
    retVal = winopen(title);
    InitStuff();

	minsize(minWidth, minHeight);
	maxsize(maxWidth, maxHeight);
	winconstraints();
    wintitle(title);
/*
    winset(retVal);
    setcursor(curCursor, 1, 0);*/

    shademodel(FLAT);

    return retVal;
}

void SetMinSize(winInfo, mw, mh)
WinInfoPtr winInfo;
int mw, mh;
/*Sets the minimum size of window winInfo to mw, mh*/
{
    SelectWindow(winInfo -> id);
    minsize(mw, mh);
    winconstraints();
}

void SetMode(curWinInfo)
WinInfoPtr curWinInfo;
/*Sets the mode of the current window according to the flags*/
{
	if (curWinInfo -> flags & WINRGB)
	{
	    if (hasRGB)
	    {
		RGBmode();
	    }
	    else
	    {
		cmode();
	    }
	}
	else
	{
	    if (hasCmap)
	    {
		cmode();
	    }
	    else
	    {
		RGBmode();
	    }
	}
	if (hasDouble && (curWinInfo -> flags & WINDBUF))
	{
	    doublebuffer();
	}
	else
	{
	    singlebuffer();
	}
	gconfig();
	SelectWindow(curWinInfo -> id);
}

void DoClose()
/*Closes the current window*/
{
    if (selWindow)
    {
	DisposeWindow(selWindow);
    }
}


void DoMaxScreen()
/*Makes the current window go to the max screen*/
{
    if (logging)
    {
	Log("fullscreen\n");
    }
    winposition(0, XMAXSCREEN, 0, YMAXSCREEN);
}


void DoVideoScreen()
/*Makes the current window go to the video screen*/
{
    if (logging)
    {
	Log("videoscreen\n");
    }
    winposition(0, recScrWidth, 0, recScrHeight);
}

void Do2VideoScreen()
/*Makes the current window go to twice the video screen*/
{
    if (logging)
    {
	char cmd[256];
	sprintf(cmd, "locate %d %d %d %d\n", 0, 2 * recScrWidth, 0, 2 * recScrHeight);
	Log(cmd);
    }
    winposition(0, 2 * recScrWidth, 0, 2 * recScrHeight);
}

void LocateWindow(l, r, b, t)
int l, r, b, t;
/*Locates current window at l, r, b, t*/
{
    winposition(l, r, b, t);
}

void DoPhscoScreen()
/*Makes the current window go to the phscologram screen*/
{
    winposition(0, 1100, 0, 615);
    phsco = true;
}

static void NullAndVoid()
/*Default do-nothing void routine*/
{
}

static Bool TrueBool()
/*Boolean routine that returns true*/
{
}

static void SetModeTo(flags)
long flags;
/*Sets the current mode to what flags says.*/
{
}

WinInfoPtr NewWinInfo(superClass, windowID, flags, title, 
		minWidth, minHeight, maxWidth, maxHeight)
ObjPtr superClass;
long windowID;
long flags;
char *title;
int minWidth, minHeight, maxWidth, maxHeight;
/*Creates a new window information record associated with windowID 
  and returns it, or NIL if it couldn't 
  allocate one.  Sets everything to the default.  Sets the superclass to
  superClass so that it can inherit*/
{
    WinInfoPtr newInfo;

    if (firstTime)
    {
	/*Finish the initialization that we couldn't do until the 1st window*/
	singlebuffer();
#ifdef GL4D
	cmode();
#endif
	zbuffer(FALSE);
	gconfig();
	firstTime = 0;
    }

    if (superClass == 0)
    {
	superClass = windowClass;
    }

    if (logging)
    {
	logWindow = 0;
    }

    newInfo = (WinInfoPtr) NewObject(superClass, sizeof(WinInfo) - sizeof(Thing));
    if (newInfo)
    {
	int k;
	WinInfoPtr *runner;
	long ox, oy, sx, sy;

	/*Set up the new info to point to this window*/
	newInfo -> id = windowID;

	/*Link the new info into the list*/
	runner = &(allWindows);
	while (*runner)
	{
	    runner = &((*runner) -> next);
	}
	*runner = newInfo;
	newInfo -> next = 0;

	/*Initialize values to defaults*/
        newInfo -> thing . flags = WINDOW;
	AddToReferenceList((ObjPtr) newInfo);
        
	newInfo -> flags = flags;
	SetVar((ObjPtr) newInfo, PICSTATE, NewInt(FBBB));
	newInfo -> mainMenu = 0;
	for (k = 0; k < NMENUS; ++k)
	{
	    newInfo -> subMenus[k] = 0;
	}
	newInfo -> nMenuItems = 0;

	newInfo -> minWidth = minWidth;
	newInfo -> maxWidth = maxWidth;
	newInfo -> minHeight = minHeight;
	newInfo -> maxHeight = maxHeight;

	/*Set to the current window and set the graphics state*/
	SetMode(newInfo);
	
	/*Save the title*/
	strcpy(newInfo -> winTitle, title);
	SetVar((ObjPtr) newInfo, NAME, NewString(title));

	/*Select the new window*/
	SelectWindow(windowID);

	/*Stuff ol, or, ob, ot*/
	getorigin(&ox, &oy);
	getsize(&sx, &sy);
	newInfo -> nl = ox;
	newInfo -> nr = ox + sx - 1;
	newInfo -> nb = oy;
	newInfo -> nt = oy + sy - 1;
	newInfo -> ol = ox;
	newInfo -> or = ox + sx - 1;
	newInfo -> ob = oy;
	newInfo -> ot = oy + sy - 1;

	/*Return the new info*/
	return newInfo;
    }
    else
    {
	/*Failed to allocate info.  Return NIL*/
	OMErr();
	return (WinInfoPtr) 0;
    }
}

void SetWindowTitle(win, title)
ObjPtr win;
char *title;
/*Sets window title to title*/
{
    strcpy(((WinInfoPtr) win) -> winTitle, title);
    SetVar((ObjPtr) win, NAME, NewString(title));
    if (!(win -> flags & WINNOFRAME))
    {
	wintitle(title);
    }
}

WinInfoPtr GetWinInfo(winID)
long winID;
/*Gets the winInfo record associated with winID or returns NIL*/
{
    register WinInfoPtr runner;
    if (winID == 0) return 0;

    runner = allWindows;
    while (runner)
    {
	if (runner -> id == winID)
	{
	    return runner;
	}
	runner = runner -> next;
    }
    return (WinInfoPtr) 0;
}

int strcmp2(s1, s2)
char s1[], s2[];
/*Compares s1 and s2 without regard to case*/
{
    int k;
    for (k = 0; s1[k]; ++k)
    {
	if (toupper(s1[k]) < toupper(s2[k])) return -k - 1;
	if (toupper(s1[k]) > toupper(s2[k])) return k + 1;
    }
    return s2[k] ? -k - 1 : 0;
}

WinInfoPtr GetWinFromTitle(title)
char *title;
/*Gets the winInfo record associated with title.  If there is none or there
  is more than one, returns nil*/
{
    register WinInfoPtr runner;
    runner = allWindows;
    while (runner)
    {
	if (!strcmp2(runner -> winTitle, title))
	{
	    return runner;
	}
	runner = runner -> next;
    }
    return (WinInfoPtr) 0;
}

Bool DisposeWindow(winID)
long winID;
/*Disposes of the given window and closes the window.*/
{
    WinInfoPtr *runner;
    runner = &allWindows;
    while (*runner)
    {
	if ((*runner) -> id == winID)
	{
	    WinInfoPtr next;
	    FuncTyp closeRoutine;
	    
	    /*Found it.  Close it.*/
	    if (logging)
	    {
		Log("close\n");
	    }

	    /*First check for hide routine*/
	    closeRoutine = GetMethod((ObjPtr) *runner, HIDE);
	    if (closeRoutine && IsTrue((*closeRoutine)(*runner)))
	    {
#ifdef WINNOCLOSE
		cachedWindows[nCachedWindows] = winID;
		++nCachedWindows;
		SelectWindow(winID);
		minsize(100, 100);
		maxsize(1280, 1024);
		winconstraints();
		winposition(-200, -100, -200, -100);
#else
		winclose(winID);
#endif
		if (winID == inputWindow)
		{
		    inputWindow = 0;
		}
		(*runner) -> id = 0;
		SelectWindow(0);

		/*Just advance*/
		runner = &((*runner) -> next);
	    }
	    else
	    {
		closeRoutine = GetMethod((ObjPtr) *runner, CLOSE);
		if (closeRoutine && !IsTrue((*closeRoutine)(*runner)))
		{
		    return false;
		}
		SelectWindow(winID);

#ifdef WINNOCLOSE
		cachedWindows[nCachedWindows] = winID;
		++nCachedWindows;
		minsize(100, 100);
		maxsize(1280, 1024);
		winconstraints();
		winposition(-200, -100, -200, -100);
#else
		winclose(winID);
#endif
		if (winID == inputWindow)
		{
		    inputWindow = 0;
		}
		SelectWindow(0);
		next = (*runner) -> next;
		(*runner) -> id = 0;
		DeleteThing((ObjPtr) *runner);
		*runner = next;
	    }
	}
	else
	{
	    /*Not found.  Just advance.*/
	    runner = &((*runner) -> next);
	}
    }
    if (!allWindows)
    {
	DoQuit();
    }
    return true;
}

static ObjPtr WindowDispose(window)
ObjPtr window;
/*Calls DisposeWindow on window*/
{
    if (((WinInfoPtr) window) -> id)
    {
	return DisposeWindow(((WinInfoPtr) window) -> id) ? ObjTrue : ObjFalse;
    }
    else
    {
	return ObjFalse;
    }
}

Bool DisposeAllWindows()
/*Disposes of all the windows.  Returns true iff successful.*/
{
    while (allWindows)
    {
	WinInfoPtr next;
	FuncTyp closeRoutine;

	closeRoutine = GetMethod((ObjPtr) allWindows, CLOSE);
	if (closeRoutine && !IsTrue((*closeRoutine)(allWindows)))
	{
	    return false;
	}
#ifdef WINNOCLOSE
	cachedWindows[nCachedWindows] = allWindows -> id;
	++nCachedWindows;
	SelectWindow(allWindows -> id);
	minsize(100, 100);
	maxsize(1280, 1024);
	winconstraints();
	winposition(-200, -100, -200, -100);
#else
	if (allWindows -> id)
	{
	    winclose(allWindows -> id);
	}
#endif
	next = allWindows -> next;
	DeleteThing((ObjPtr) allWindows);
	allWindows = next;
    }
    inputWindow = 0;
    return true;
}

void DrawWindowInfo(theWindow)
WinInfoPtr theWindow;
/*Draws the window based on WinInfoPtr theWindow into the current set window.
  This is the fundamental window drawing routine.*/
{
    Screencoord l, r, b, t;
    FuncTyp drawRoutine;

    SetModeTo(theWindow -> flags);

    getviewport(&l, &r, &b, &t);

    drawRoutine = GetMethodSurely("DrawWindowInfo", (ObjPtr) theWindow, DRAW);

    if (drawRoutine)
    {
	(*drawRoutine)(theWindow);
    }
}

int ReshapeWindow(window, left, right, bottom, top)
ObjPtr window;
int left, right, bottom, top;
/*Reshapes a window to a new viewport.  Old viewport is left, right, bottom,
  top.  Old origin is in ox and oy.  Does not redraw.*/
{
    if (selWinInfo)
    {
	FuncTyp reshapeMethod;
	reshapeMethod = GetMethod(window, RESHAPE);
	if (reshapeMethod)
	{
	    (*reshapeMethod)(window, (int) left, (int) right, (int) bottom, (int) top);
	}
    }
    return true;
}

static ObjPtr MakeWindowDrawn(curWindow)
WinInfoPtr curWindow;
/*Method to make a window drawn.  Returns true iff drawing done*/
{
    ObjPtr retVal = ObjFalse;
    int picState;
    ObjPtr picVar;

    picVar = GetVar((ObjPtr) curWindow, PICSTATE);
    if (picVar)
    {
	picState = GetInt(picVar);
    }
    else
    {
	picState = FBBB;
    }    

    if (curWindow -> flags & WINDBUF)
    {
	/*Double buffer*/
	if (!(picState & BG))
	{
	    /*Back buffer is bad, have to redraw it*/
	    SetVar((ObjPtr) curWindow, PICSTATE, 
		   NewInt(picState & FG ? FGBG : FGBB));
	    SelectWindow(curWindow -> id);
	    DrawWindowInfo(curWindow);
	    retVal = ObjTrue;
	}
    }
    else
    {
	/*Single buffer*/
	if (!(picState & FG))
	{
	    SetVar((ObjPtr) curWindow, PICSTATE, NewInt(FG));
	    SelectWindow(curWindow -> id);
	    DrawWindowInfo(curWindow);
	    retVal = ObjTrue;
	}
    }
    return retVal;
}

Bool IdleAllWindows()
/*Idles all the windows, redrawing double-buffered ones.  Returns true
  iff any redrawing was done*/
{
    WinInfoPtr curWindow;
    long oldWindow;
    Bool retVal = false;
    struct tms buffer;
    long beginDraw;
    int dummy;
    int prefDrawMoving;

    oldWindow = winget();
    ++updateBatch;

    prefDrawMoving = GetPrefInteger(PREF_DRAWMOVING);

    if (prefDrawMoving == DM_SKELETON)
    {
	movingDrawingQuality = DQ_SKELETON;
    }

    if (interactiveMoving)
    {
	beginDraw = times(&buffer);
	drawingQuality = movingDrawingQuality;
    }    
    else
    {
	drawingQuality = DQ_FULL;
    }

    curWindow = allWindows;
    for (; curWindow; curWindow = curWindow -> next)
    {
	FuncTyp method;

	if (curWindow -> id == 0) continue;

	if (AltDown() && (curWindow -> id != inputWindow))
	{
	    continue;
	}

	if (!(interactiveMoving || GetPredicate((ObjPtr) curWindow, DRAWNFULL)))
	{
	    ImInvalid((ObjPtr) curWindow);
	}

	method = GetMethod((ObjPtr) curWindow, MAKEDRAWN);
	if (method)
	{
#if MACHINE == RS6000 && 0
	    /*Stupid kludge for stupid IBM stupid window problem*/
	    long startTime;
	    struct tms *buffer;
	    startTime = times(&buffer);
#endif
	    if (IsTrue((*method)(curWindow)))
	    {
		retVal = true;
		SetVar((ObjPtr) curWindow, DRAWNFULL,
			interactiveMoving ? ObjFalse : ObjTrue);
#ifdef GL4D
		if (curWindow -> flags & WINDBUF)
		{
		    swapbuffers();
		}
#endif
#if MACHINE == RS6000 && 0
		/*Wait until 8 clock ticks have passed for stupid IBM*/
		while (times(&buffer) < startTime + 8);
#endif
	    }
	}
    }
#ifndef GL4D
    swapbuffers();
#endif
    if (oldWindow && (oldWindow != selWindow))
    {
	SelectWindow(oldWindow);
    }

    if (interactiveMoving)
    {
	if (prefDrawMoving == DM_AUTO)
	{
		if (times(&buffer) > beginDraw + MAXMOVETIME)
		{
		    /*Back off on drawing quality*/
		    if (movingDrawingQuality > 0)
		    {
			--movingDrawingQuality;
		    }
		}
	}
    }

    return retVal;
}

void InitWindows()
/*Initializes the window system*/
{
    long dim;
    windowClass = NewObject(NULLOBJ, 0);
    SetMethod(windowClass, MAKEDRAWN, MakeWindowDrawn);
    SetMethod(windowClass, DISPOSE, WindowDispose);
    SetVar(windowClass, TYPESTRING, NewString("window"));
    AddToReferenceList(windowClass);

#ifdef FONTS4D
    if (nFonts)
    {
	int k;
	/*Make the fonts submenu*/
	fontSubMenu = newpup();
	for (k = 0; k < nFonts; ++k)
	{
	    strcpy(tempStr, fonts[k]);
	    strcat(tempStr, "%f%n");
	    addtopup(fontSubMenu, tempStr, MenuSetFont);
	}

	sizeSubMenu = newpup();
	for (k = 0; k < (sizeof(textSizes) / sizeof(int)); ++k)
	{
	    sprintf(tempStr, "%d%%f%%n", textSizes[k]);
	    addtopup(sizeSubMenu, tempStr, MenuSetSize);
	}
    }
#endif
    alignSubMenu = newpup();
    sprintf(tempStr, "Align Left%%f%%x%d", LEFTALIGN);
    addtopup(alignSubMenu, tempStr, MenuSetAlignment);
    sprintf(tempStr, "Align Center%%f%%x%d", CENTERALIGN);
    addtopup(alignSubMenu, tempStr, MenuSetAlignment);
    sprintf(tempStr, "Align Right%%f%%x%d", RIGHTALIGN);
    addtopup(alignSubMenu, tempStr, MenuSetAlignment);

    /*Create the clipboard*/
    dim = 1;
    clipboard = NewArray(AT_OBJECT, 1, &dim);
    AddToReferenceList(clipboard);
}

void KillWindows()
/*Kills the window system*/
{
    DeleteThing(clipboard);
    DeleteThing(windowClass);
}

