/*ScianVisWindows.c
  Visualization window stuff
  Eric Pepke
  April 6, 1990
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianArrays.h"
#include "ScianLists.h"
#include "ScianWindows.h"
#include "ScianTextBoxes.h"
#include "ScianTitleBoxes.h"
#include "ScianObjWindows.h"
#include "ScianIcons.h"
#include "ScianColors.h"
#include "ScianControls.h"
#include "ScianButtons.h"
#include "ScianSpaces.h"
#include "ScianLights.h"
#include "ScianSliders.h"
#include "ScianIDs.h"
#include "ScianVisWindows.h"
#include "ScianDatasets.h"
#include "ScianPictures.h"
#include "ScianDialogs.h"
#include "ScianRecorders.h"
#include "ScianEvents.h"
#include "ScianScripts.h"
#include "ScianTimers.h"
#include "ScianErrors.h"
#include "ScianComplexControls.h"
#include "ScianMethods.h"
#include "ScianStyle.h"
#include "ScianVisObjects.h"

ObjPtr visWindowClass;			/*Class for vis windows*/
ObjPtr visCorralClass;			/*Class for corral of a vis window*/

ObjPtr NewControlWindow(visObj, name)
ObjPtr visObj;
char *name;
/*Opens a new control window for visualization object visObj using slave
  window ownerWindow*/
{
    FuncTyp newCtlWindow;
    ObjPtr retVal;

    newCtlWindow = GetMethod(visObj, NEWCTLWINDOW);
    if (newCtlWindow)
    {
	PauseDrawing(true);
	retVal = (*newCtlWindow)(visObj, NULLOBJ, name);
	PauseDrawing(false);
    }
    else
    {
	retVal = NULLOBJ;
    }
    return retVal;
}

ObjPtr VisIcon(object)
ObjPtr object;
/*Visualizes an icon*/
{
    DoTask(VisObjects);
    return ObjTrue;
}

Bool DoHidePanel()
/*Hides the panel of the controls on the top window.  Returns true iff it did*/
{
    if (selWinInfo)
    {
	ObjPtr panel, space, assocPanel;
	int panelWidth;
	Screencoord left, right, bottom, top;
	int nl, nr, nb, nt;

	getviewport(&left, &right, &bottom, &top);

	panel = FindPanel(selWinInfo);
	space = FindSpace(selWinInfo);

        if (!panel || !space)
	{
	    return false;
	}

	/*Get the panel's bounds*/
	Get2DIntBounds(panel, &nl, &nr, &nb, &nt); 
	panelWidth = nr - nl;

	/*Change panel to be off the screen*/
	nl = right + 1;
	nr = nl + panelWidth;
	Set2DIntBounds(panel, nl, nr, nb, nt);

	/*Now change the space*/
	Get2DIntBounds(space, &nl, &nr, &nb, &nt); 
	nr = right;
	Set2DIntBounds(space, nl, nr, nb, nt);
	
	ImInvalid(panel);
	ImInvalid(space);

	/*And its panels*/
	assocPanel = GetVar(space, FRONTPANEL);
	if (assocPanel)
	{
	    Set2DIntBounds(assocPanel, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	assocPanel = GetVar(space, BACKPANEL);
	if (assocPanel)
	{
	    Set2DIntBounds(assocPanel, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	SetVar((ObjPtr) selWinInfo, PANELHIDDEN, ObjTrue);

	if (logging)
	{
	    Log("hide panel\n");
	}

	DeleteMenus(selWinInfo);

	return true;
    }
    else
    {
	return false;
    }
}

Bool DoShowPanel()
/*Shows the panel of the controls on the top window.  Returns true iff it did*/
{
    if (selWinInfo)
    {
	ObjPtr panel, space, assocPanel;
	int panelWidth;
	Screencoord left, right, bottom, top;
	int nl, nr, nb, nt;

	getviewport(&left, &right, &bottom, &top);

	panel = FindPanel(selWinInfo);
	space = FindSpace(selWinInfo);

        if (!panel || !space)
	{
	    return false;
	}

	/*Get the panel's bounds*/
	Get2DIntBounds(panel, &nl, &nr, &nb, &nt); 
	panelWidth = nr - nl;

	/*Change panel to be on the screen*/
	nr = right;
	nl = right - panelWidth;
	Set2DIntBounds(panel, nl, nr, nb, nt);

	/*Now change the space*/
	Get2DIntBounds(space, &nl, &nr, &nb, &nt); 
	nr = right - panelWidth;
	Set2DIntBounds(space, nl, nr, nb, nt);

	ImInvalid(panel);
	ImInvalid(space);

	/*And its panels*/
	assocPanel = GetVar(space, FRONTPANEL);
	if (assocPanel)
	{
	    Set2DIntBounds(assocPanel, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	assocPanel = GetVar(space, BACKPANEL);
	if (assocPanel)
	{
	    Set2DIntBounds(assocPanel, nl, nr, nb, nt);
	    ImInvalid(assocPanel);
	}

	SetVar((ObjPtr) selWinInfo, PANELHIDDEN, ObjFalse);

	if (logging)
	{
	    Log("show panel\n");
	}

	DeleteMenus(selWinInfo);

	return true;
    }
    else
    {
	return false;
    }
}

int annotStagger = 0;
int nAnnot = 1;

ObjPtr DeleteSpacePanelText(text)
ObjPtr text;
/*Delete text from a space panel*/
{
    return ObjTrue;
}

void AddAnnotation(name, bounds)
char *name;
real bounds[4];
/*Adds annotation with name and bounds to current window*/
{
    if (selWindow)
    {
	/*Search for a space*/
	ObjPtr space, frontPanel;

	space = FindSpace(selWinInfo);

	if (space)
	{
	    frontPanel = GetObjectVar("AddAnnotation", space, FRONTPANEL);
	    if (frontPanel)
	    {
		/*There's a front panel.  Stick an annotation there*/
		ObjPtr annotation;

		annotation = NewTextBox(bounds[0], bounds[1], bounds[2], bounds[3],
					/*WITH_BG + */EDITABLE + ADJUSTABLE, name, "");
		PrefixList(GetVar(frontPanel, CONTENTS), annotation);
		SetVar(annotation, PARENT, frontPanel);
		SetTextColor(annotation, NewInt(UIGRAY75));
		ImInvalid(annotation);
		SetTextFont(annotation, ANNOTFONT);
		SetTextSize(annotation, ANNOTFONTSIZE);
		SetVar(annotation, STICKINESS, NewInt(FLOATINGLEFT + FLOATINGRIGHT + FLOATINGTOP + FLOATINGBOTTOM));
		SetMethod(annotation, DELETEICON, DeleteSpacePanelText);
 
		if (!runningScript)
		{
		    MakeMeCurrent(annotation);
		}

		if (logging)
		{
		    /*Log the action.  Destructive to name*/
		    char cmd[256];
		    char *d;

		    d = &(cmd[0]);
		    sprintf(cmd, "annotation ");
		    while (*d) ++d;
		    if (isdigit(*name))
		    {
			*d++ = '\\';
		    }
		    while (*name)
		    {
			if (!isalpha(*name) && !isdigit(*name))
			{
			    *d++ = '\\';
			}
			*d++ = *name++;
		    }
		    sprintf(d, " [%g %g %g %g]\n",
			bounds[0], bounds[1], bounds[2], bounds[3]);
		    Log(cmd);
		}
	    }
	}
    }
}

void DoAddAnnotation()
/*Add an annotation*/
{
    if (selWindow)
    {
	/*Search for a space*/
	ObjPtr space, frontPanel;

	space = FindSpace(selWinInfo);

	if (space)
	{
	    frontPanel = GetVar(space, FRONTPANEL);
	    if (frontPanel)
	    {
		char name[30];
		real bounds[4];
		real bb[4];
		ObjPtr boundsArray;

		boundsArray = GetFixedArrayVar("DoAddAnnotation", frontPanel, BOUNDS, 1, 4L);
		if (!boundsArray) return;
		Array2CArray(bounds, boundsArray);

		sprintf(name, "Annotation %d", nAnnot++);
		bb[0] = bounds[0] + ANNOTMINX + annotStagger;
		bb[1] = bounds[0] + ANNOTMINX + annotStagger + ANNOTWIDTH;
		bb[2] = bounds[3] - ANNOTMAXY - annotStagger - ANNOTHEIGHT;
		bb[3] = bounds[3] - ANNOTMAXY - annotStagger;
		AddAnnotation(name, bb);
		annotStagger += ANNOTSTAGGER;
		if (annotStagger > ANNOTMAXSTAGGER)
		{
		    annotStagger = 0;
		}
	    }
	}
    }
}

static void DoTogglePanel()
/*Toggles the panel of the controls on the top window.*/
{
    if (selWinInfo)
    {
	if (GetPredicate((ObjPtr) selWinInfo, PANELHIDDEN))
	{
	    DoShowPanel();
	}
	else
	{
	    DoHidePanel();
	}
    }
}


void DoShowControls()
/*Shows the controls in the selected objects in the window*/
{
    ObjPtr var;
    ThingListPtr list;
    char windowName[256];
    ObjPtr name, repObj;
    ObjPtr current, parent, contents;

    if (!selWindow)
    {
	return;
    }

    if (logging)
    {
	Log("show controls\n");
	InhibitLogging(true);
    }

    var = FindPanelContents(selWinInfo);
    if (!var)
    {
	return;
    }
    list = LISTOF(var);
    while (list)
    {
	ObjPtr corral;

	corral = list -> thing;
	if (corral && InClass(corral, corralClass) && !GetPredicate(corral, ONEICON))
	{
	    ThingListPtr contents;
	    contents = LISTOF(GetVar(corral, CONTENTS));
	    while (contents)
	    {
		if (IsObject(contents -> thing) &&
		    InClass(contents -> thing, iconClass))
		{
		    if (GetPredicate(contents -> thing, SELECTED))
		    {
			name = GetVar(contents -> thing, NAME);
			if (name)
			{
			    strncpy(windowName, GetString(name), 255);
			    windowName[255] = 0;
			}
			else
			{
			    strcpy(windowName, "");
			}
			repObj = GetVar(contents -> thing, REPOBJ);
			if (!repObj)
			{
			    if (logging)
			    {
				InhibitLogging(false);
			    }
			    return;
			}
			name = GetVar(contents -> thing, FORMAT);
			if (name)
			{
			    strcat(windowName, " ");
			    strcat(windowName, GetString(name));
    			}
			NewControlWindow(repObj, windowName);
		    }
		}
		contents = contents -> next;
	    }
	}
	list = list -> next;
    }

    /*Search for something else to delete*/
    current = GetVar((ObjPtr) selWinInfo, CURRENT);
    if (current)
    {
	parent = GetVar(current, PARENT);
	if (parent && InClass(parent, panelClass))
	{
	    name = GetVar(current, NAME);
	    if (name)
	    {
		strncpy(windowName, GetString(name), 255);
		windowName[255] = 0;
	    }
	    NewControlWindow(current, windowName);
	}
    }
    if (logging)
    {
	InhibitLogging(false);
    }
}

void DoTurnOn()
/*Turns on the selected objects in the window*/
{
    ObjPtr corral;
    Bool didit = false;

    if (logging)
    {
	InhibitLogging(true);
    }
    if (selWindow)
    {
	corral = FindMainCorral(selWinInfo);
	if (corral)
	{
	    ThingListPtr contents;
	    contents = LISTOF(GetVar(corral, CONTENTS));
	    while (contents)
	    {
		if (IsObject(contents -> thing) &&
		    InClass(contents -> thing, iconClass))
		{
		    if (GetPredicate(contents -> thing, SELECTED))
		    {
			ObjPtr repObj;
			repObj = GetObjectVar("DoTurnOn",
				contents -> thing, REPOBJ);
			if (repObj)
			{
			    FuncTyp method;
			    method = GetMethod(repObj, HIDE);
			    if (method)
			    {
				if (IsTrue((*method)(repObj, ObjFalse)))
				{
				    didit = true;
				    SetVar(contents -> thing, ICONGREYED, ObjFalse);
				    ImInvalid(contents -> thing);
				}
			    }
			}
		    }
		}
		contents = contents -> next;
	    }
	}
	else
	{
	    Error("DoTurnOn", NOCORRAL, (ObjPtr) selWinInfo);
	}
    }
    if (logging)
    {
	InhibitLogging(false);
	if (didit)
	{
	    Log("turn on\n");
	}
    }
}

void DoTurnOff()
/*Hides the visualization of vis icons*/
{
    ObjPtr corral;
    Bool didit = false;

    if (logging)
    {
	InhibitLogging(true);
    }
    if (selWindow)
    {
	corral = FindMainCorral(selWinInfo);
	if (corral)
	{
	    ThingListPtr contents;
	    contents = LISTOF(GetVar(corral, CONTENTS));
	    while (contents)
	    {
		if (IsObject(contents -> thing) &&
		    InClass(contents -> thing, iconClass))
		{
		    if (GetPredicate(contents -> thing, SELECTED))
		    {
			ObjPtr repObj;
			repObj = GetObjectVar("DoTurnOff",
				contents -> thing, REPOBJ);

			if (repObj)
			{
			    FuncTyp method;
			    method = GetMethod(repObj, HIDE);
			    if (method)
			    {
				if (IsTrue((*method)(repObj, ObjTrue)))
				{
				    didit = true;
				    SetVar(contents -> thing, ICONGREYED, ObjTrue);
				    ImInvalid(contents -> thing);
				}
			    }
			}
		    }
		}
		contents = contents -> next;
	    }
	}
	else
	{
	    Error("DoTurnOff", NOCORRAL, (ObjPtr) selWinInfo);
	}
    }
    if (logging)
    {
	InhibitLogging(false);
	if (didit)
	{
	    Log("turn off\n");
	}
    }
}

#if 0
void DoSetObserver()
/*Sets the current observer to the selected ones*/
{
    ObjPtr corral;
    Bool didit = false;

    if (logging)
    {
	InhibitLogging(true);
    }
    if (selWindow)
    {
	corral = FindMainCorral(selWinInfo);
	if (corral)
	{
	    ThingListPtr contents;
	    contents = LISTOF(GetVar(corral, CONTENTS));
	    while (contents)
	    {
		if (IsObject(contents -> thing) &&
		    InClass(contents -> thing, iconClass))
		{
		    if (GetPredicate(contents -> thing, SELECTED))
		    {
			ObjPtr repObj;
			repObj = GetObjectVar("DoSetObserver",
				contents -> thing, REPOBJ);

			if (repObj && InClass(repObj, observerClass))
			{
			    ObjPtr space /*, observers*/;
			    space = FindSpace(selWinInfo);
			    if (space)
			    {
				observers = GetListVar("DoSetObserver", space, OBSERVERS);
				DeleteFromList(observers, repObj);
				PrefixList(observers, repObj);
				ImInvalid(corral);
				ImInvalid(space);
			    } 
			}
		    }
		}
		contents = contents -> next;
	    }
	}
	else
	{
	    Error("DoSetObserver", NOCORRAL, (ObjPtr) selWinInfo);
	}
    }
    if (logging)
    {
	InhibitLogging(false);
	if (didit)
	{
	    Log("set observer\n");
	}
    }
}
#endif

void DoMakeLocalCopy()
/*Makes local copies of the icons in the window*/
{
    ObjPtr corral;

    if (selWindow)
    {
	corral = FindMainCorral(selWinInfo);
	if (corral)
	{
	    ThingListPtr contents;
	    ObjPtr contentsList;
	    contentsList = GetVar(corral, CONTENTS);
	    contents = LISTOF(contentsList);
	    while (contents)
	    {
		ThingListPtr next;
		next = contents -> next;
		if (contents -> thing && IsObject(contents -> thing) &&
		    InClass(contents -> thing, iconClass))
		{
		    if (GetPredicate(contents -> thing, SELECTED))
		    {
			ObjPtr repObj, newObj, icon;
			FuncTyp method;
	
			icon = contents -> thing;

			repObj = GetVar(contents -> thing, REPOBJ);
			if (repObj)
			{
			    method = GetMethodSurely("DoMakeLocalCopy", repObj, CLONE);
			    if (!method)
			    {
				return;
			    }

			    method = GetMethod(icon, DELETEICON);
			    if (method)
			    {
				if (IsTrue((*method)(icon)))
				{
				    DeleteFromList(contentsList, icon);
				}
			    }

			    method = GetMethod(repObj, CLONE);
			    if (method)
			    {
				ObjPtr name;
			        newObj = (*method)(repObj);
				icon = Clone(icon);
			        SetVar(icon, REPOBJ, newObj);
			        MakeVar(newObj, NAME);
				name = GetVar(newObj, NAME);
				SetVar(icon, NAME, name);
			    }

			    method = GetMethod(corral, DROPINCONTENTS);
			    if (method)
			    {
				ObjPtr locArray;
				int x, y;
				real loc[2];
				locArray = GetVar(icon, ICONLOC);
				if (locArray)
				{
				    Array2CArray(loc, locArray);
				    x = loc[0];
				    y = loc[1];
				}
				else
				{
				    x = y = 0;
				}
				(*method)(corral, icon, x, y);
				RecalcScroll(corral);
			    }
			}
		    }
		}
		contents = next;
	    }
	    if (logging)
	    {
		Log("localcopy\n");
	    }
	}
	else
	{
	    Error("DoMakeLocalCopy", NOCORRAL, (ObjPtr) selWinInfo);
	}
    }
}

void DoCloneIcon()
/*Clones the icons in the window*/
{
    ObjPtr corral;

    if (selWindow)
    {
	corral = FindMainCorral(selWinInfo);
	if (corral)
	{
	    ThingListPtr contents;
	    contents = LISTOF(GetVar(corral, CONTENTS));
	    while (contents)
	    {
		if (IsObject(contents -> thing) &&
		    InClass(contents -> thing, iconClass))
		{
		    if (GetPredicate(contents -> thing, SELECTED))
		    {
			ObjPtr repObj, newVis;
			FuncTyp method;

			repObj = GetVar(contents -> thing, REPOBJ);
			if (repObj && !GetVar(repObj, CONTROLLERCLASS))
			{
			    method = GetMethodSurely("DoCloneIcon", repObj, CLONE);
			    if (!method)
			    {
				return;
			    }

			    newVis = (*method)(repObj);
			    if (InClass(newVis, controllerClass))
			    {
				AddControllerToSpace(newVis, FindSpace(selWinInfo), corral, NULLOBJ);
				ResolveController(newVis);
			    }
			    else
		 	    {
				AddObjToSpace(newVis, FindSpace(selWinInfo), corral, NULLOBJ, NULLOBJ);
				IdleAllWindows();
			    }
			}
		    }
		}
		contents = contents -> next;
	    }
	    if (logging)
	    {
		Log("duplicate\n");
	    }
	}
	else
	{
	    Error("CloneIcons", NOCORRAL, (ObjPtr) selWinInfo);
	}
    }
}

int vwSerialNumber = 0;

void VisObjects()
/*Opens a new visualization window using default visualizations for all of the
  selected objects in the icon corral in the foremost window.  The frontmost
  window had better be an object window
*/
{
    WinInfoPtr visWindow;
    WinInfoPtr sourceWindow;

    Log("visualize\n");

    /*Save the current window for later*/
    sourceWindow = selWinInfo;

    /*Create a vis window*/
    sprintf(tempStr, "Visualization %d", ++vwSerialNumber); 
    visWindow = NewVisWindow(tempStr, WINDBUF + WINZBUF + (hasRGB ? WINRGB : 0));
    SetMinSize(visWindow, 100, 100);

    if (sourceWindow)
    {
	/*Great, there's a window full of icons and stuff*/

	ObjPtr objList;
	ObjPtr corral;
	ThingListPtr runner;

	corral = FindMainCorral(sourceWindow);
	if (!corral) return;
	objList = GetListVar("VisObjects", corral, CONTENTS);
	if (objList)
	{
	    ThingListPtr theIcons;
	    /*There's a list of icons.  Add them to the vis window*/
		    
	    theIcons = LISTOF(objList);
	    while (theIcons)
	    {
		if (IsObject(theIcons -> thing) &&
		    InClass(theIcons -> thing, iconClass))
		{
		    /*Boy, it's an icon.  If selected, add it*/
		    if (GetPredicate(theIcons -> thing, SELECTED))
		    {
			IdleAllWindows();
			AddObjToSpace(theIcons -> thing, FindSpace(visWindow), FindMainCorral(visWindow), NULLOBJ, NULLOBJ);
		    }
		}
		theIcons = theIcons -> next;
	    }
	}
	else
	{
	    return;
	}
    }
}

int vawSerialNumber;

void DoShowFrontPanelControls()
/*Shows the controls for the front panel in the selected vis window*/
{
    char windowName[255];
    ObjPtr space, panel;
    if (!selWinInfo)
    {
	return;
    }

    space = FindSpace(selWinInfo);
    if (!space)
    {
	return;
    }

    panel = GetVar(space, FRONTPANEL);
    if (!panel)
    {
	return;
    }

    strcpy(windowName, selWinInfo -> winTitle);
    strcat(windowName, " Front Panel");
    NewControlWindow(panel, windowName);
}

void DoShowBackPanelControls()
/*Shows the controls for the back panel in the selected vis window*/
{
    char windowName[255];
    ObjPtr space, panel;
    if (!selWinInfo)
    {
	return;
    }

    space = FindSpace(selWinInfo);
    if (!space)
    {
	return;
    }

    panel = GetVar(space, BACKPANEL);
    if (!panel)
    {
	return;
    }

    strcpy(windowName, selWinInfo -> winTitle);
    strcat(windowName, " Back Panel");
    NewControlWindow(panel, windowName);
}

void VisObjectsAs()
/*Opens a new VisObjectsAs window using default for all of the
  selected objects in the icon corral in the foremost window.  The frontmost
  window had better be an object window
*/
{
    WinInfoPtr newWindow;
    WinInfoPtr sourceWindow;
    ObjPtr objList, contents;
    ObjPtr corral, sourceCorral, panel, button;
    ThingListPtr runner;
    Screencoord l, r, b, t;
    int bw;

    sourceWindow = selWinInfo;
    if (!sourceWindow)
    {
	return;
    }

    Log("visobjectsas\n");

    /*Create a vis as window*/
    sprintf(tempStr, "Visualize As %d", ++vawSerialNumber); 
    newWindow = NewObjWindow(NULLOBJ, tempStr, WINDBUF + WINZBUF + (hasRGB ? WINRGB : 0),
		VAWINWIDTH, VAWINHEIGHT, SCRWIDTH, SCRHEIGHT);
    getviewport(&l, &r, &b, &t);

    /*Add menu entries*/
    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Turn On", DoTurnOn);
    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Turn Off", DoTurnOff);
    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Visualize", VisObjects);
    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Show Controls", DoShowControls);

    SetVar((ObjPtr) newWindow, HELPSTRING,
	    NewString("This window shows all the possible ways to visualize a \
group of datasets."));

    /*Put in a panel*/
    panel = NewPanel(greyPanelClass, l, r, b, t);
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));

    contents = GetVar((ObjPtr) newWindow, CONTENTS);
    PrefixList(contents, panel);
    SetVar(panel, PARENT, (ObjPtr) newWindow);

    /*Put in buttons and an icon corral*/
    contents = GetListVar("NewVisAsWindow", panel, CONTENTS);
    if (!contents)
    {
	return;
    }

    /*Make an icon corral*/
    corral = NewIconCorral(NULLOBJ, l + MINORBORDER, r - MINORBORDER, b + 2 * MINORBORDER + BUTTONHEIGHT, t - MINORBORDER, BARRIGHT + BARBOTTOM);
    SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(corral, NAME, NewString("Datasets Corral"));
    SetVar(corral, HELPSTRING,
	NewString("This corral contains icons for all possible visualizations of a group of \
datasets.  You can visualize or show info on the visualizations by selecting \
some of them and pressing the buttons at the bottom of the window.  You can delete \
visualizations by selecting them and choosing Delete from the Object menu."));
    PrefixList(contents, corral);
    SetVar(corral, PARENT, panel);

    l += MINORBORDER;
    r -= MINORBORDER;
    b += MINORBORDER;
    t = b + BUTTONHEIGHT;
    bw = (r - l - MINORBORDER) / 2;

    /*Make a visualize button*/
    button = NewButton(l, l + bw,
		b, b + BUTTONHEIGHT, "Visualize");
    SetVar(button, PARENT, panel);
    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + STICKYLEFT + FLOATINGRIGHT));
    PrefixList(contents, button);
    SetMethod(button, CHANGEDVALUE, VisualizeButton);
    SetVar(corral, VISBUTTON, button);
    ActivateButton(button, false);
    SetVar(button, HELPSTRING, 
	NewString("Pressing this button will open the selected visualization objects \
together in a new window."));

    /*Make a show info button*/
    button = NewButton(r - bw, 
		r,
		b, b + BUTTONHEIGHT, "Show Controls");
    SetVar(button, PARENT, panel);
    SetVar(button, STICKINESS, NewInt(STICKYBOTTOM + FLOATINGLEFT + STICKYRIGHT));
    PrefixList(contents, button);
    SetMethod(button, CHANGEDVALUE, ShowControlsButton);
    SetVar(corral, SHOWCNTBUTTON, button);
    ActivateButton(button, false);
    SetVar(button, HELPSTRING, 
	NewString("Pressing this button will bring up a window that shows \
controls for the \
selected visualization objects."));

    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Select All", DoSelectAllIcons);
    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Deselect All", DoDeselectAllIcons);
    DefineMenuItem((ObjPtr) newWindow, OBJECTMENU, "Delete", DoDelete);

    sourceCorral = FindMainCorral(sourceWindow);
    if (!sourceCorral) return;

    objList = GetListVar("VisObjectsAs", sourceCorral, CONTENTS);
    if (objList)
    {
	ThingListPtr theIcons;
	/*There's a list of icons.  Add their visualization to the corral*/
		    
	theIcons = LISTOF(objList);
	while (theIcons)
	{
	    if (IsObject(theIcons -> thing) &&
		InClass(theIcons -> thing, iconClass))
	    {
		/*Boy, it's an icon.  If selected, add it*/
		if (GetPredicate(theIcons -> thing, SELECTED))
		{
		    ObjPtr repObj;
		    repObj = GetVar(theIcons -> thing, REPOBJ);
		    if (InClass(repObj, visClass))
		    {
			repObj = GetVar(repObj, MAINDATASET);
		    }
		    if (InClass(repObj, datasetClass))
		    {
			ObjPtr visList;
			visList = GetAllVis(repObj, false);
			if (visList)
			{
			    ThingListPtr runner;
			    runner = LISTOF(visList);
			    while (runner)
			    {
				ObjPtr vis, icon;
				vis = runner -> thing;
				if (vis)
				{
				    icon = NewVisIcon(vis);
				    if (icon)
				    {
					ObjPtr format;
					format = GetVar(icon, FORMAT);
					if (format)
					{
					    SetVar(icon, FORMAT,
						ConcatStrings(format, NewString(" Template")));
					}
					SetVar(vis, TEMPLATEP, ObjTrue);
					SetVar(icon, CORRAL, corral);
					SetMethod(icon, CHANGEDVALUE, ChangeIconButtons);
					DropIconInCorral(corral, icon);
				    }
				}
				runner = runner -> next;
			    }
			}
		    }
		}
	    }
	    theIcons = theIcons -> next;
	}
    }
}

static ObjPtr DropInVisCorral(corral, object, x, y)
ObjPtr corral, object;
int x, y;
/*Drops an icon in a vis corral*/
{
    ObjPtr repObj;
    repObj = GetVar(object, REPOBJ);
    if (repObj)
    {
	real loc[2];
	ObjPtr locArray;
	loc[0] = x;
	loc[1] = y;
	locArray = NewRealArray(1, 2L);
	CArray2Array(locArray, loc);
	if (InClass(repObj, controllerClass))
	{
	    AddControllerToSpace(repObj, FindSpace(selWinInfo), FindMainCorral(selWinInfo), locArray);
	    ResolveController(repObj);
	}
	else
	{
	    AddObjToSpace(object, FindSpace(selWinInfo), FindMainCorral(selWinInfo), locArray, NULLOBJ);
	    IdleAllWindows();
	}
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

void ForAllVisWindows(routine)
void (*routine)();
/*Performs (*routine)(window) on all vis windows*/
{
    WinInfoPtr curWindow;
    curWindow = allWindows;
    while (curWindow)
    {
	if (InClass((ObjPtr) curWindow, visWindowClass))
	{
	    (*routine)(curWindow);
	}
	curWindow = curWindow -> next;
    }
}

void PushNonVisWindows()
/*Pushes all the non vis windows*/
{
    int maxDepth;
    WinInfoPtr curWindow;
    maxDepth = -1;

    curWindow = allWindows;
    while (curWindow)
    {
	if (InClass((ObjPtr) curWindow, visWindowClass))
	{
	    if ((curWindow) -> id)
	    {
		maxDepth = MAX(maxDepth, windepth((curWindow) -> id));
	    }
	}
	curWindow = curWindow -> next;
    }

    curWindow = allWindows;
    while (curWindow)
    {
	if (!InClass((ObjPtr) curWindow, visWindowClass))
	{
	    if ((curWindow) -> id && windepth((curWindow) -> id) < maxDepth)
	    {
		SelectWindow(curWindow -> id);
		--maxDepth;
		winpush();
	    }
	}
	curWindow = curWindow -> next;
    }
}

void Tile(width, height)
int width, height;
/*Tiles all the vis windows in the lower left corner as big as width, height*/
{
    int nWide, nHigh;		/*# of windows wide and high*/
    int i, j;
    int nVisWindows;
    WinInfoPtr curWindow;
    long oldWindow;

    if (logging)
    {
	char cmd[256];
	sprintf(cmd, "tile %d %d\n", width, height);
	Log(cmd);
    }

    /*Count the number of vis windows*/
    nVisWindows = 0;
    curWindow = allWindows;
    while (curWindow)
    {
	if (InClass((ObjPtr) curWindow, visWindowClass))
	{
	    ++nVisWindows;
	}
	curWindow = curWindow -> next;
    }
    if (!nVisWindows) return;

    /*Estimate the number of windows high*/
    nHigh = sqrt((double) nVisWindows) + 0.49;
    nWide = nVisWindows / nHigh;
    if (nWide * nHigh < nVisWindows) ++nHigh;

    /*Go through and tile the windows*/
    i = 0;
    j = nHigh - 1;
    curWindow = allWindows;
    while (curWindow)
    {
	if (InClass((ObjPtr) curWindow, visWindowClass))
	{
	    long l, r, b, t;
	    
	    if (i)
	    {
		l = i * width / nWide + 1;
	    }
	    else
	    {
		l = 0;
	    }
	    r = (i + 1) * width / nWide;
	    if (j)
	    {
		b = j * height / nHigh + 1;
	    }
	    else
	    {
		b = 0;
	    }
	    t = (j + 1) * height / nHigh;
	    if (++i >= nWide)
	    {
		i = 0;
		--j;
	    }

	    /*Move the window*/
	    SelectWindow(curWindow -> id);
	    winposition(l, r, b, t);
	}
	curWindow = curWindow -> next;
    }
}

void DoTileFullScreen()
/*Tiles all the visualization windows within the full screen*/
{
    Tile(XMAXSCREEN, YMAXSCREEN);
}

void DoTileVideoScreen()
/*Tiles all the visualization windows within the full screen*/
{
    Tile(recScrWidth, recScrHeight);
}

static void RGBTask()
/*Task to change the current window to RGB*/
{
    if (selWindow)
    {
	selWinInfo -> flags |= WINRGB;

	SetMode(selWinInfo);
	SelectWindow(selWindow);
	ImInvalid((ObjPtr) selWinInfo);
    }
}

static void CMapTask()
/*Task to change the current window to CMap mode*/
{
    if (selWindow)
    {
	selWinInfo -> flags &= ~WINRGB;

	SetMode(selWinInfo);
	SelectWindow(selWindow);
	ImInvalid((ObjPtr) selWinInfo);
    }
}

static ObjPtr ChangeVisColorMode(radio)
ObjPtr radio;
/*Changes the color mode of the current visualization window according to
  the value of radio*/
{
    int rgbp;

    rgbp = GetInt(GetValue(radio));

    if (rgbp)
    {
	DoTask(RGBTask);
    }
    else
    {
	DoTask(CMapTask);
    }
    return ObjTrue;
}

static void DoDisplayClock()
/*Displays the clock in the space in the current window*/
{
    if (selWinInfo)
    {
	ObjPtr space, clock, panel;
	int l, r, b, t;

	space = FindSpace(selWinInfo);
	if (!space)
	{
	    return;
	}
	panel = GetObjectVar("DoDisplayClock", space, FRONTPANEL);
	if (!panel)
	{
	    return;
	}
	clock = GetObjectVar("DoDisplayClock", space, CLOCK);
	if (!clock)
	{
	    return;
	}
	if (Get2DIntBounds(space, &l, &r, &b, &t))
	{
	    AddClockToSpacePanel(clock, panel, space, l + CLLEFTOFFSET, t - CLTOPOFFSET);
	}
    }
}

static ObjPtr ShowSpacePanelControls(spacePanel, ownerWindow, windowName)
ObjPtr spacePanel;
WinInfoPtr ownerWindow;
char *windowName;
/*Makes a new control window to control a space panel*/
{
    WinInfoPtr controlWindow;
    ObjPtr var;
    ObjPtr panel;
    ObjPtr corral;
    ObjPtr contents;
    real rgb[3], hsv[3];
    WinInfoPtr dialogExists;
    Bool hasBackground;

    dialogExists = DialogExists((WinInfoPtr) 0, spacePanel);
    controlWindow = GetDialog((WinInfoPtr) 0, spacePanel, windowName, 
	SPWINWIDTH, SPWINHEIGHT, SPWINWIDTH,
	SPWINHEIGHT, WINDBUF + WINRGB + WINFIXEDSIZE);
    
    if (!dialogExists)
    {
	long info;
	ObjPtr value;
	
	ObjPtr checkBox, icon, name, colorBar, titleBox, textBox, button;
	ObjPtr colorWheel, slider, radioGroup;
	int left, right, bottom, top;

	SetVar((ObjPtr) controlWindow, REPOBJ, spacePanel);

	/*Set help string*/
	SetVar((ObjPtr) controlWindow, HELPSTRING, NewString("This window \
shows controls for a space panel.  For information about any of the controls \
in the window, use Help In Context on the control.\n"));

	/*Add in a panel*/
	panel = NewPanel(greyPanelClass, 0, SPWINWIDTH, 0, SPWINHEIGHT);
	if (!panel)
	{
	    return ObjFalse;
	}
	contents = GetVar((ObjPtr) controlWindow, CONTENTS);
	PrefixList(contents, panel);
	SetVar(panel, PARENT, (ObjPtr) controlWindow);

	contents = GetVar(panel, CONTENTS);

	/*Add in the group of controls for text color*/
	left = MAJORBORDER;
	top = SPWINHEIGHT - MAJORBORDER;
	right = SPWINWIDTH - MAJORBORDER;
	bottom = MAJORBORDER;

	titleBox = NewTitleBox(left, right, bottom, top, "Background Color");
	PrefixList(contents, titleBox);
	SetVar(titleBox, PARENT, panel);

	left += MINORBORDER;
	right -= MINORBORDER;
	top -= MINORBORDER + TITLEBOXTOP;
	
	/*Get the color for priming the controls*/
	var = GetVar(spacePanel, BACKGROUND);
	if (var && IsArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	{
	    Array2CArray(rgb, var);
	    hasBackground = true;
	}
	else if (var && IsInt(var))
	{
	    rgb[0] = uiColors[GetInt(var)][0] / 255.0;
	    rgb[1] = uiColors[GetInt(var)][1] / 255.0;
	    rgb[2] = uiColors[GetInt(var)][2] / 255.0;
	    hasBackground = true;
	}
	else
	{
	    rgb[0] = rgb[1] = rgb[2] = 0.0;
	    hasBackground = false;
	}
	RGB2HSV(&(hsv[0]), &(hsv[1]), &(hsv[2]), (rgb[0]), (rgb[1]), (rgb[2]));

	/*Make the color wheel*/
	colorWheel = NewColorWheel(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH, top, "Background Color");
	var = NewRealArray(1, 2L);
	CArray2Array(var, hsv);
	SetValue(colorWheel, var);
	SetVar(colorWheel, PARENT, panel);
	SetVar(colorWheel, REPOBJ, spacePanel);
	PrefixList(contents, colorWheel);
	SetMethod(colorWheel, CHANGEDVALUE, ChangeBackgroundColorWheel);
	SetVar(colorWheel, HELPSTRING, NewString("This color wheel controls the \
hue and saturation of the color used to draw the background of the space panel.  \
The final color is a combination of this hue and saturation and the value, or brightness, \
given by the Value slider."));
	
	/*Make the text box below*/
	textBox = NewTextBox(left, left + COLORWHEELWIDTH,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Background Color Label", "Color");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Make the brightness slider*/
	slider = NewSlider(right - SLIDERWIDTH, right, 
		       top - COLORWHEELWIDTH, top,
		       PLAIN, "Background Value");
	SetVar(slider, PARENT, panel);
	PrefixList(contents, slider);
	SetSliderRange(slider, 1.0, 0.0, 0.0);
	SetSliderValue(slider, hsv[2]);
	SetVar(slider, REPOBJ, spacePanel);
	SetMethod(slider, CHANGEDVALUE, ChangeBackgroundColorSlider);
	SetVar(slider, HELPSTRING, NewString("This slider controls the \
value, or brightness, of the color used to draw the background of the space panel.  \
The final color is a combination of this value and the hue and saturation \
given by the Color color wheel."));

	/*Make the text box below*/
	textBox = NewTextBox(right - SLIDERWIDTH - MINORBORDER, right + MINORBORDER,
			top - COLORWHEELWIDTH - TEXTBOXSEP - TEXTBOXHEIGHT,
			top - COLORWHEELWIDTH - TEXTBOXSEP,
			PLAIN, "Text Value Label", "Value");
	SetVar(textBox, PARENT, panel);
	PrefixList(contents, textBox);
	SetTextAlign(textBox, CENTERALIGN);

	/*Cross link the slider and color wheel*/
	SetVar(colorWheel, SLIDER, slider);
	SetVar(slider, COLORWHEEL, colorWheel);

	/*Make the check box*/
	top -= COLORWHEELWIDTH + TEXTBOXSEP + TEXTBOXHEIGHT + MINORBORDER;
	checkBox = NewCheckBox(left, right, 
		top - CHECKBOXHEIGHT, top,
		"No background", hasBackground ? false : true);
	SetVar(checkBox, PARENT, panel);
	SetVar(checkBox, REPOBJ, spacePanel);
	SetVar(checkBox, SLIDER, slider);
	SetVar(checkBox, COLORWHEEL, colorWheel);
	PrefixList(contents, checkBox);
	SetMethod(checkBox, CHANGEDVALUE, ChangeNoBackground);
	SetVar(checkBox, HELPSTRING, NewString("This checkbox controls whether \
a background is shown.  If it is selected, no background is shown, and the \
objects behind can be seen."));

	/*Link slider and color wheel to check box*/
	SetVar(slider, CHECKBOX, checkBox);
	SetVar(colorWheel, CHECKBOX, checkBox);
    }
    return (ObjPtr) controlWindow;
}

WinInfoPtr NewVisWindow(title, flags)
char *title;
long flags;
/*Creates a new visualization window with title title and flags flags*/
{
    WinInfoPtr visWindow;
    ObjPtr space, panel, corral, contents;
    Screencoord left, right, bottom, top;
    ObjPtr lights;
    ThingListPtr runner;

    visWindow = NewObjWindow(visWindowClass, title, flags, VWINWIDTH, VWINHEIGHT, SCRWIDTH, SCRHEIGHT);

    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Turn On", DoTurnOn);
    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Turn Off", DoTurnOff);
    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Show Controls", DoShowControls);
#if 0
    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Set Observer", DoSetObserver);
#endif
    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Make Local Copy", DoMakeLocalCopy);
    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Duplicate", DoCloneIcon);
    DefineMenuItem((ObjPtr) visWindow, OBJECTMENU, "Delete", DoDelete);
    SetVar((ObjPtr) visWindow, HIDEPANEL, ObjTrue);
    DefineMenuItem((ObjPtr) visWindow, WINDOWSMENU, "Show Front Panel Controls", DoShowFrontPanelControls);
    DefineMenuItem((ObjPtr) visWindow, WINDOWSMENU, "Show Back Panel Controls", DoShowBackPanelControls);
    DefineMenuItem((ObjPtr) visWindow, ARRANGEMENU, "Tile Full Screen", DoTileFullScreen);
    DefineMenuItem((ObjPtr) visWindow, ARRANGEMENU, "Tile Video Screen", DoTileVideoScreen);
    DefineMenuItem((ObjPtr) visWindow, TEXTMENU, "New Annotation", DoAddAnnotation);
    DefineMenuItem((ObjPtr) visWindow, TEXTMENU, "Display Clock", DoDisplayClock);

    getviewport(&left, &right, &bottom, &top);

    contents = GetVar((ObjPtr) visWindow, CONTENTS);

    /*Add in the panel behind the space*/
    panel = NewPanel(spaceBackPanelClass, left, right - VWINPANELWIDTH - 1, bottom, top);
    if (!panel)
    {
	return;
    }
/*
    SetVar(panel, BACKGROUND, NewInt(UIWHITE));
*/
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("This panel holds 2-dimensional objects \
and is always drawn behind the space."));
    SetVar(panel, OWNERWINDOW, (ObjPtr) visWindow);
    PrefixList(contents, panel);
    
    /*Add in a space*/
    space = NewSpace(left, right - VWINPANELWIDTH - 1, bottom, top);
    if (!space)
    {
	return;
    }
    SetVar(space, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(space, OWNERWINDOW, (ObjPtr) visWindow);
    PrefixList(contents, space);
    SetVar(space, PARENT, (ObjPtr) visWindow);

    /*Notify the space that the panel is behind the space*/
    SetVar(panel, SPACE, space);
    SetVar(space, BACKPANEL, panel);
    SetVar(panel, PARENT, space);
    SetVar(panel, DRAWCOVER, space);
    SetMethod(panel, NEWCTLWINDOW, ShowSpacePanelControls);

    /*Add in the panel to cover the space*/
    panel = NewPanel(spacePanelClass, left, right - VWINPANELWIDTH - 1, bottom, top);
    if (!panel)
    {
	return;
    }
    SetVar(panel, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("This panel holds 2-dimensional objects \
and is always drawn in front of the space."));
    SetVar(panel, OWNERWINDOW, (ObjPtr) visWindow);
    SetVar(panel, SPACE, space);
    PrefixList(contents, panel);
    SetVar(panel, PARENT, space);

    /*Notify the space that the panel covers the space*/
    SetVar(space, DRAWCOVER, panel);
    SetVar(space, FRONTPANEL, panel);
    SetMethod(panel, NEWCTLWINDOW, ShowSpacePanelControls);

    /*Add in a control panel*/
    panel = NewPanel(greyPanelClass, right - VWINPANELWIDTH, right, bottom, top);
    if (!panel)
    {
	return;
    }
    SetVar(panel, STICKINESS, NewInt(STICKYRIGHT + STICKYBOTTOM + STICKYTOP));
    PrefixList(contents, panel);
    SetVar(panel, PARENT, (ObjPtr) visWindow);
    SetVar(space, PANEL, panel);
    SetVar(panel, TYPESTRING, NewString("panel"));
    SetVar(panel, HELPSTRING, NewString("You can show or hide this panel by choosing the Toggle Panel \
item from the Windows menu."));

    contents = GetVar(panel, CONTENTS);
    if (!contents)
    {
	return;
    }

    /*Add in an icon corral*/
    corral = NewIconCorral(visCorralClass,
		MINORBORDER, VWINPANELWIDTH - MINORBORDER, 
		bottom + ((hasRGB && hasCmap) ? MINORBORDER + MAJORBORDER + CHECKBOXHEIGHT : MINORBORDER),
		top - MINORBORDER,
		BARRIGHT + BARBOTTOM);

    if (!corral)
    {
	return;
    }
    SetVar(corral, NAME, NewString("Space Contents"));
    SetVar(corral, HELPSTRING,
	NewString("This corral shows the objects and space controllers present \
in this visualization window.  You can drag new datasets from the datasets window \
into this corral to visualize them along with the other objects.  You can also \
drag in observer, renderer, clock, and lights icons.  There can only be one \
observer, renderer, and clock in a space, so dragging in these will replace the \
existing controller.  This will probably change soon.\n"));
    SetVar(corral, STICKINESS, NewInt(STICKYLEFT + STICKYRIGHT +
				     STICKYBOTTOM + STICKYTOP));
    PrefixList(contents, corral);
    SetVar(corral, PARENT, panel);
    SetVar(corral, SPACE, space);
    SetVar(space, CORRAL, corral);

    if (hasRGB && hasCmap)
    {
	ObjPtr radio, button;
	/*Make some radio buttons for RGB versus color map mode*/
	radio = NewRadioButtonGroup("Color Mode");
	SetVar(radio, HELPSTRING,
		NewString("These radio buttons control the color mode of the \
visualization window.  Full color mode is more useful for most visualization \
techniques.  Color map mode can be used on computers that do not have enough \
bit planes to do full color well.  It also provides a different way of doing \
interpolation on smooth color shaded visualizations, which is sometimes useful."));
	PrefixList(contents, radio);
	SetVar(radio, PARENT, panel);

	button = NewRadioButton(
		MINORBORDER, VWINPANELWIDTH / 2,
		MINORBORDER, MINORBORDER + CHECKBOXHEIGHT,
		"Color Map");
	SetVar(button, HELPSTRING,
	    NewString("This button puts the window into color map color mode.  \
This prevents lighting and image filtering from working, but it can be useful \
on simple color shaded visualizations."));

	AddRadioButton(radio, button);

	button = NewRadioButton(
		VWINPANELWIDTH / 2, VWINPANELWIDTH - MINORBORDER,
		MINORBORDER, MINORBORDER + CHECKBOXHEIGHT,
		"Full Color");
	SetVar(button, HELPSTRING,
	    NewString("This button puts the window into full color mode.  \
In full color mode, all the lighting and filtering is available."));
	AddRadioButton(radio, button);
	SetValue(radio, NewInt(flags & WINRGB ? 1 : 0));
	SetVar((ObjPtr) visWindow, CMODERADIO, radio);
	SetMethod(radio, CHANGEDVALUE, ChangeVisColorMode);
    }
    /*Add an observer*/
    AddControllerToSpace(NewObserver(), space, corral, NULLOBJ);

    /*Add a renderer*/
    AddControllerToSpace(NewRenderer(), space, corral, NULLOBJ);

    /*Add some lights*/
    if (hasRGB)
    {
	lights = NewLights();
	runner = LISTOF(lights);
	while (runner)
	{
	    AddControllerToSpace(runner -> thing, space, corral, NULLOBJ);
	    runner = runner -> next;
	}
    }

    /*Add a clock*/
    AddControllerToSpace(NewClock(), space, corral, NULLOBJ);

    return visWindow;
}

ObjPtr FindMainCorral(win)
WinInfoPtr win;
/*Finds the main icon corral in a panel within win*/
{
    ObjPtr panel = 0;
    ObjPtr corral;
    ThingListPtr contents;

    /*Search for a panel in win*/
    contents = LISTOF(GetListVar("FindMainCorral", (ObjPtr) win, CONTENTS));
    while (contents)
    {
	if (InClass(contents -> thing, greyPanelClass))
	{
	    panel = contents -> thing;
	    break;
	}
	contents = contents -> next;
    }
    
    if (panel)
    {
	contents = LISTOF(GetListVar("FindMainCorral", panel, CONTENTS));
	while (contents)
	{
	    corral = contents -> thing;
	    if (InClass(corral, corralClass))
	    {
		return corral;
	    }
	    contents = contents -> next;
	}
	Error("FindMainCorral", NOCORRAL, (ObjPtr) win);
    }
    else
    {
	Error("FindMainCorral", NOPANEL, (ObjPtr) win);
    }
    return NULLOBJ;
}

ObjPtr FindSpace(win)
WinInfoPtr win;
/*Finds the space in win*/
{
    ThingListPtr contents;

    /*Search for a space in win*/
    contents = LISTOF(GetVar((ObjPtr) win, CONTENTS));
    while (contents)
    {
	if (InClass(contents -> thing, spaceClass))
	{
	    return contents -> thing;
	}
	contents = contents -> next;
    }
    return NULLOBJ;
}

ObjPtr FindPanel(win)
WinInfoPtr win;
/*Finds the panel in win*/
{
    ThingListPtr contents;

    /*Search for a panel in win*/
    contents = LISTOF(GetVar((ObjPtr) win, CONTENTS));
    while (contents)
    {
	if (InClass(contents -> thing, greyPanelClass))
	{
	    return contents -> thing;
	}
	contents = contents -> next;
    }
    return NULLOBJ;
}

#ifdef PROTO
void RotateWindowTo(WinInfoPtr window, Matrix r)
#else
void RotateWindowTo(window, r)
WinInfoPtr window;
Matrix r;
#endif
/*Rotates the space in window to rotation matrix given by r*/
{
    ObjPtr space;
    ObjPtr xformMatrix;

    space = FindSpace(window);
    if (space)
    {
	ObjPtr observer/*, observers*/;
#if 0
	observers = GetListVar("RotateWindowTo", space, OBSERVERS);
	if (!observers || !LISTOF(observers)) return;
	observer = LISTOF(observers) -> thing;
#else
	observer = GetObjectVar("RotateWindowTo", space, OBSERVER);
#endif
	if (!observer) return;

	xformMatrix = NewMatrix();
	MATCOPY(MATRIXOF(xformMatrix), r);
	SetVar(observer, XFORM, xformMatrix);

	ResolveController(observer);
	if (logging)
	{
	    char cmd[256];
	    MatrixToText(tempStr, MATRIXOF(xformMatrix));
	    sprintf(cmd, "rotate to %s\n", tempStr);
	    Log(cmd);
	}
    }
    else
    {
	ReportError("RotateWindowTo", "No space in window");
    }
}

#ifdef PROTO
void RotateOrthoWindow(WinInfoPtr window, char axis, float amount)
#else
void RotateOrthoWindow(window, axis, amount)
WinInfoPtr window;
char axis;
float amount;
#endif
/*Rotates the space in window about orthogonal axis axis by amount, and 
  stops continuous rotation.*/
{
    ObjPtr space;
    ObjPtr xformMatrix;
    Matrix newXform;

    space = FindSpace(window);
    if (space)
    {
	ObjPtr observer/*, observers*/;

#if 0
	observers = GetListVar("RotateOrthoWindow", space, OBSERVERS);
	if (!observers || !LISTOF(observers)) return;
	observer = LISTOF(observers) -> thing;
#else
	observer = GetObjectVar("RotateOrthoWindow", space, OBSERVER);
#endif
	if (!observer) return;

	/*Get the rotation matrix*/
	xformMatrix = (ObjPtr) GetVar(observer, XFORM);
	if (!xformMatrix || !IsMatrix(xformMatrix))
	{
	    xformMatrix = NewMatrix();
	    SetVar(observer, XFORM, xformMatrix);
	}
	MATCOPY(newXform, MATRIXOF(xformMatrix));
	pushmatrix();
	loadmatrix(newXform);
	rot(amount, axis);
	getmatrix(newXform);
	popmatrix();
	MATCOPY(MATRIXOF(xformMatrix), newXform);
	if (logging)
	{
	    sprintf(tempStr, "rotate %c %g\n", axis, amount);
	    Log(tempStr);
	}
	ResolveController(observer);
    }
}

#ifdef PROTO
void ShearOrthoWindow(WinInfoPtr window, float amount)
#else
void ShearOrthoWindow(window, amount)
WinInfoPtr window;
float amount;
#endif
/*Shears the space in window by amount, and 
  stops continuous rotation.*/
{
    ObjPtr space;

    space = FindSpace(window);
    if (space)
    {
	/*Get the rotation matrix*/
	Matrix shearMatrix;
	ObjPtr matObj;

	MATCOPY(shearMatrix, Identity);
	shearMatrix[2][1] = amount;
	matObj = NewMatrix();
	MATCOPY(MATRIXOF(matObj), shearMatrix);
	SetVar(space, SHEAR, matObj);
	ImInvalid(space);
    }
}

static ObjPtr ChangeWindowToRGB(window, button)
WinInfoPtr window;
int button;
/*If button is 1, changes the color mode of window to RGB*/
{
    if (button == 1)
    {
	ObjPtr radio = NULLOBJ;
	SelectWindow(window -> id);
	if (selWinInfo)
	{
	    radio = GetVar((ObjPtr) selWinInfo, CMODERADIO);
	}
	if (radio)
	{
	    SetValue(radio, NewInt(1));
	}
	return ObjTrue;
    }
    else
    {
	return ObjFalse;
    }
}

ObjPtr SetVisWindowRGB (window)
ObjPtr window;
/*Deferred message to choose whether to make window RGB or not*/
{
    char tempBuf[400];
    WinInfoPtr errWindow;
    if (!hasRGB) return;

    sprintf(tempBuf, "You will see no effects from this action in window %s \
until you change it to full color mode.  Would you like to change it now?",
	((WinInfoPtr) window) -> winTitle);

    errWindow = AlertUser(UIYELLOW, (WinInfoPtr) window, tempBuf,
	ChangeWindowToRGB, 2, "Cancel", "Change");
    SetVar((ObjPtr) errWindow, HELPSTRING,
	NewString("Visualization windows have two color modes: full color and color map.  \
Full color specifies full 24-bit RGB color.  Color map specifies mapped colors using a color \
table.  Some features, such as lighting, only work on windows set to full color mode.  \
If you click on the Change button, the affected window will automatically be set \
to this mode."));
}

void InitVisWindows()
/*Initializes the vis windows*/
{
    visWindowClass = NewObject(objWindowClass, 0);
    AddToReferenceList(visWindowClass);
    SetVar(visWindowClass, HELPSTRING,
	NewString("This is a visualizaton window.  On the right is a panel which \
shows the visualization objects shown in the window, the controllers which \
affect the appearance of the object, and some buttons which affect the window \
as a whole or the selected icons.  If this panel is not shown, it can be brought \
back with the Toggle Panel item in the Window menu.  On the left is a \
3-dimensional space in which \
the visualization object are shown.  There is one clear panel in front of the and \
one clear panel behind the space.  These panels are used to hold 2-dimensional \
objects such as clock readouts and text boxes, which can be created and modified \
with items in the Text menu.")); 
    SetMethod(visWindowClass, SETRGBMESSAGE, SetVisWindowRGB);

    /*Class for the corral in a vis window*/
    visCorralClass = NewObject(corralClass, 0);
    AddToReferenceList(visCorralClass);
    SetMethod(visCorralClass, DROPINCONTENTS, DropInVisCorral);
}

void KillVisWindows()
/*Kills the vis windows*/
{
    DeleteThing(visCorralClass);
    DeleteThing(visWindowClass);
}
