/****************************************************************************\
* 									     *
* File:           FFT.c							     *
* Purpose:        Compute Fourier transform of 1 and 2 D complex arrays	     *
* Last Modified:  9-May-85     by  J. Painter				     *
* 									     *
* The one-d FFT follows algorithm A1.12 on page 490 of Computer Vision by    *
* Ballard and Brown.  The two-D FFT algorithm follows algorithm 2.1 on page  *
* 31 of  Algorithms for Graphics and Image Processing by Pavlidis.	     *
* 									     *
\****************************************************************************/
#include "Complex.h"
#include <stdio.h>
#include <math.h>

char *malloc();

/***************************************************************************\
* 									    *
* Function: ReorderData							    *
* Purpose:  Reorder the Data array by mapping Data[i] to Data[j]	    *
* where j is simply i with the bits in reverse order (treated as a word of  *
* LogN bits.								    *
* 									    *
\***************************************************************************/
static  void ReorderData (N, LogN, Data)
int     N, LogN;
Complex *Data;
{
    int     i,j,index, ibit;
    Complex temp;

    for (i = 0; i < N; i++) {
	j = 0;
	index = i;
	for (ibit = 0; ibit < LogN; ibit++) {
	    j = (j << 1) | (index & 1);
	    index = index >> 1;
	}
	if (j > i) {		/* Swap elements */
	    temp = Data[i];
	    Data[i] = Data[j];
	    Data[j] = temp;
	}
    }
}
/****************************************************\
* 						     *
* Function: ComputeRootsOfUnity			     *
*  Purpose: Compute the complex N'th roots of unity  *
*           and return them in the array W.	     *
* 						     *
\****************************************************/
static  void ComputeRootsOfUnity (N,W)
int     N;
Complex *W;
{
    double  Pi = 3.14159265,
            Theta;
    int     k;

    Theta = 2 * Pi / N;
    W->real = 1;
    W->imag = 0;
    for (k = 1; k < N/2; k++) {
	W[k]  .real = cos (Theta * k);
	W[k]  .imag = sin (Theta * k);
        Conjugate( W[k], W[N-k] );
    }
    W[N/2].real = -1;
    W[N/2].imag = 0;
}

/************************************************************************\
* 									 *
* Function:  FFT_1D   -- Perform a 1-d FFT on the Data array.   	 *
* Arguments: Forward  -- TRUE for forward transform; FALSE for reverse;	 *
*            NSize    -- Size of array to transform.  Should be a power	 *
*                        of 2.						 *
*            Data     -- On input:  Data to be transformed.		 *
*                        On output: Result of transform.		 *
* 									 *
*            Function return:  TRUE if successfull; FALSE if not.	 *
* 									 *
\************************************************************************/
int     FFT_1d (Forward, NSize, Data)
int     Forward, NSize;
Complex Data[];
{
    register int k, IOff, JOff, JWPnt;
    register Complex A, B, Temp;
    int     i, N, LogN,  JPnt, JBk, IStart;

    static int LastN = 0;      /* Size of last FFT done */
    static Complex *W = NULL;  /* Roots of unity from last computation */



                  /* Compute log base 2 of N */
    LogN = 0;
    for (N = 1; N < NSize; N *= 2)
	LogN++;
    if (N != NSize) {
	fprintf (stderr, "%s  %s %d\n",
		"Argument error in FFT.",
		"Array size must be a power of 2.  NSize=",
		NSize);
	return 0;
    }
 
    if ( N != LastN ) {   /* Compute roots of unity */
        if (W != NULL )  free( (char *) W );
        W = (Complex *) malloc ( (unsigned) (N * sizeof (Complex)) );
        ComputeRootsOfUnity (N, W);
        LastN = N;
    }
    ReorderData (N, LogN, Data);  /* Reorder the data */

    JOff = JPnt = N / 2;	/* The heart of the algorithm starts here */
    JBk = 2;
    IOff = 1;
    if (Forward) {
      for (i = 0; i < LogN; i++) {
	for (IStart = 0; IStart < N; IStart += JBk) {
	  JWPnt = 0;
	  for (k = IStart; k < IStart + IOff; k++) {
	    CMultiply ( Data[k+IOff], W[JWPnt], Temp );
	    CAdd ( Temp, Data[k], A );
	    CMultiply ( Data[k+IOff], W[JWPnt+JOff], Temp);
	    CAdd ( Temp, Data[k], B );
	    Data[k] = A;
	    Data[k + IOff] = B;
	    JWPnt += JPnt;
	  }
	}
	JPnt >>= 1;  /* Divide by 2 */
	IOff = JBk;
	JBk <<= 1;   /* Multiply by 2 */
      }
    } else {  /* Inverse transform */
      for (i = 0; i < LogN; i++) {
	for (IStart = 0; IStart < N; IStart += JBk) {
	  JWPnt = 0;
	  for (k = IStart; k < IStart + IOff; k++) {
	    Conjugate ( W[JWPnt], A );
	    CMultiply ( A, Data[k+IOff], Temp );
	    CAdd ( Temp, Data[k], A );
	    Conjugate ( W[JWPnt+JOff], B );
	    CMultiply ( B, Data[k+IOff], Temp );
	    CAdd ( Temp, Data[k], B );
	    Data[k] = A;
	    Data[k + IOff] = B;
	    JWPnt += JPnt;
	  }
	}
	JPnt >>= 1;  /* Divide by 2 */
	IOff = JBk;
	JBk <<= 1;   /* Multiply by 2 */
      }
      for (i = 0; i < N; i++) {
	Data[i].real /= N;
	Data[i].imag /= N;
      }
    }
    return 1;
}

/**************************************************************************\
* 									   *
* Function:  FFT_2d							   *
* Purpose:   Do 2-D FFT on the Data array.				   *
* Arguments: Forward  -- True for forward transform, FALSE for inverse.	   *
*            NColumns -- Size of fasting moving subscript of Data.	   *
*            NRows    -- Size of slowest moving subsrcipt of Data.	   *
*               *Note*   Both NColumns and NRows should be powers of 2.	   *
*            Data     -- On input:  A Complex 2d array to be transformed.  *
*                        Data[NRows][Ncolumns]				   *
*                        On output:  The result of the transform.	   *
* 									   *
\**************************************************************************/
FFT_2d (Forward, NColumns, NRows, Data)
int     Forward, NColumns, NRows;
Complex * Data;
{
    int     i, j, pass;
    Complex *Row, *Column, *Out, *In, *Last;

    /*  The forward tranform is done rows first then columns.  The inverse
     *  transform is done in the reverse order.         
     */
    for (pass = 0; pass < 2; pass++) {
	if ((pass == 0 && Forward) || (pass == 1 && !Forward)) {
	/* 
	 * Do a row-by-row 1-D FFT.
	 */
	    Row = Data;
	    for (i = 0; i < NRows; i++, Row += NColumns ) {
		FFT_1d (Forward, NColumns, Row);
	    }
	}
	else {
	/* 
	 *  Do a column-by-column  1-D FFT.
	 */
	    Column = (Complex *) malloc( (unsigned)(NRows * sizeof (Complex)));
	    Out = Data;
	    for (i = 0; i < NColumns; i++) {

		/*
		 * Extract the i'th column.
		 */
		In = Last = Out;
		Out = Column;
		for (j = 0; j < NRows; j++) {
		    *Out++ = *In;
		    In += NColumns;
		}

		/*
		 *  Transform it.
		 */
		FFT_1d (Forward, NRows, Column);

		/*
		 * Replace it in the Data array
		 */
		Out = Last;
		In = Column;
		for (j = 0; j < NRows; j++) {
		    *Out = *In++;
		    Out += NColumns;
		}
		Out = Last + 1;
	    }
	    free ( (char *) Column);
	}
    }
}
