/*
  File: tiff2wff.c
  Author: K.R. Sloan
  Last Modified: 19 December 1991
  Purpose: convert tiff file to .wff

  Note: maximum bitspersample = 16
 */

#include <assert.h>
#include <stdio.h>
#include <sys/time.h>
#include "wff.h"
#include "tiffio.h"
#include "tiff.h"

#define True (-1)
#define False (0)

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"usage is:\n\t%s [-h][-v][-f tiff] >wff\n", RoutineName);
  exit(-1);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(1);
 }

/* These function deal with WFF files */

static FrameBufferType *SetUpWFFFile(stream, xsize, ysize, zsize, BitsPerBand)
 FILE *stream;
 int xsize, ysize, zsize, BitsPerBand;
 {
  FrameBufferType *FrameBuffer;
  int Bottom, Left, Top, Right;
  char WhatBands[10], Name[NameLength], Value[ValueLength];

  FrameBuffer = (FrameBufferType *)0;

  OpenFB(&FrameBuffer);
  assert(FrameBuffer);

  Bottom = 0; Left = 0; Top = ysize-1; Right = xsize-1;
  SetBounds(FrameBuffer, Bottom, Left, Top, Right);

  switch (zsize)
   {
    case 1: strcpy(WhatBands,"I"  ); break;
    case 2: strcpy(WhatBands,"IA"  ); break;
    case 3: strcpy(WhatBands,"RGB");  break;
    case 4: strcpy(WhatBands,"RGBA");  break;
    default: FatalError("Bad zsize");
   }

  SetColorSystem(FrameBuffer, WhatBands,  BitsPerBand);

  strcpy(Name,"X-CreatedBy");
  strcpy(Value,RoutineName);
  SetDescriptor(FrameBuffer, Name, Value);

  strcpy(Name,"Encoding");
  strcpy(Value,"AIS");
  SetDescriptor(FrameBuffer, Name, Value);

  /* Header operations over, now we can start the output stream */
  PassImageOut(stream, FrameBuffer);
  return (FrameBuffer);
 }  

static void WriteWFFScanline(FB, xsize, Scanline)
 FrameBufferType *FB;
 int xsize;
 unsigned short *Scanline;
 {
  assert (SUCCESS == NextNPixelsOut(FB, xsize, Scanline));
 }

static void Pass(TIFFFileName, fdOut)
 char *TIFFFileName;
 FILE *fdOut;
 {
  FrameBufferType *FB;
  int Left, Bottom, Top, Right, BitsPerBand;
  int BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  double AspectRatio;
  int width, height, depth;
  unsigned short *WFFScanline;
  int WFFScanlineLength;
  unsigned char *TIFFScanline;
  int TIFFScanlineLength;
  int i,w,t,bits,mask,value,x,y,b,row,drow;
  TIFF       *tif;
  TIFFDirectory *td;

  tif = TIFFOpen(TIFFFileName, "r");
  if ((TIFF *)0 == tif) FatalError("opening TIFF file");

  td = &tif->tif_dir;
  if (VERBOSE)
   TIFFPrintDirectory(tif, stderr, True, False, False);

  switch (td->td_photometric)
   {
    case PHOTOMETRIC_MINISBLACK:
    case PHOTOMETRIC_MINISWHITE:
    case PHOTOMETRIC_RGB:
    default:
              BandsPerPixel   = (int)(td->td_samplesperpixel);
              BitsPerBand = (int)(td->td_bitspersample);
              break;
    case PHOTOMETRIC_PALETTE:
              BandsPerPixel   = 3;
              BitsPerBand = 8; break;
  }

  if (BitsPerBand > 16)
   FatalError("too many bistpersample");

  width           = (int)(td->td_imagewidth);
  height          = (int)(td->td_imagelength);
  if (td->td_bitspersample <= 8) depth = 8; else depth = 16;

  if (VERBOSE)
   fprintf(stderr,"%s: %s is %d x %d x %d x %d\n",
      RoutineName, TIFFFileName, width, height, BandsPerPixel, BitsPerBand);

  FB = SetUpWFFFile(stdout, width, height, BandsPerPixel, BitsPerBand);
  assert(FB);

  TIFFScanlineLength = TIFFScanlineSize(tif);
  TIFFScanline = (unsigned char *) malloc(TIFFScanlineLength);
  if ((unsigned char *)0 == TIFFScanline)
   FatalError("no memory for TIFFScanline");

  WFFScanlineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanline = (unsigned short *) malloc(WFFScanlineLength);
  if ((unsigned short *)0 == WFFScanline)
   FatalError("no memory for WFFScanline");

  Bottom = 0; Top = height-1;
  Left = 0; Right = width-1;
  switch(td->td_orientation)
   {
    default:
    case 1: row = Top; drow = -1; break;
    case 2: row = Top; drow = -1; break;
    case 3: row = Bottom; drow = 1; break;
    case 4: row = Bottom; drow = 1; break;
    case 5: 
    case 6:
    case 7:
    case 8: row = Top; drow = -1; break;
   }
  switch(td->td_orientation)
   {
    default:
    case 1:
    case 3: break;

    case 2:
    case 4:
    case 5: 
    case 6:
    case 7:
    case 8: fprintf(stderr,"%s: WARNING: we don't support ORIENTATION = %d\n",
                      RoutineName,td->td_orientation);
            fprintf(stderr,"%s: WARNING: Use Flip to fix your .wff image\n",
                      RoutineName);
   }

  mask = (1 << BitsPerBand) - 1;
  for (y=Bottom; y<=Top; y++,row += drow)
   {
    if (0 > TIFFReadScanline(tif, TIFFScanline, row, 0))
     {
      fprintf(stderr,"HINT: tiffcp -none may solve your problem\n");
      FatalError("TIFFReadScanline failed");
     }
    w = 0; t = 0;
    bits = 0;
    for(x=Left;x<=Right;x++)
     {
      for(b=0;b<(td->td_samplesperpixel);b++)
       {
        int i;

        if (8 == depth)  value = TIFFScanline[t++];
        else if (16 == depth)
         {
          if (FILLORDER_MSB2LSB == tif->tif_fillorder)
           {
            value = TIFFScanline[t++];
            value = (value << 8) + TIFFScanline[t++];
           }
          else 
           {
            value = TIFFScanline[t++];
            value = value + (TIFFScanline[t++] << 8);
           }
         }
        else FatalError("wrong value for depth - call Key Operator");

        bits = depth - BitsPerBand;
        i = mask & (value >> bits);
        if (PHOTOMETRIC_MINISWHITE == td->td_photometric)
         i = (1 << td->td_bitspersample)-1-i;

        if (PHOTOMETRIC_PALETTE == td->td_photometric)
         {
          WFFScanline[w++] = ((td->td_redcolormap[i]  ) >> 8) & 0xff;
          WFFScanline[w++] = ((td->td_greencolormap[i]) >> 8) & 0xff;
          WFFScanline[w++] = ((td->td_bluecolormap[i] ) >> 8) & 0xff;
         }
        else
         WFFScanline[w++] = i;
       }
     }
    if (FAILURE == NextNPixelsOut(FB,width, WFFScanline))
     FatalError("NextNPixelsOut failed");
   }

  TIFFClose(tif);
  CloseFB(&FB);
  free(WFFScanline); free(TIFFScanline);
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *TIFFFileName;

  TIFFFileName = "wff.tiff";
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'f': if (ArgsParsed >= argc)  { usage(); exit(-1);}
                TIFFFileName = argv[ArgsParsed++];
                break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h':  usage(); exit(-1);
     }
   }

  Pass(TIFFFileName, stdout);
  
  exit(0);
 }
