/*
   File: TFFwff.c 
   Authors: Mike Schmidt,
            K.R. Sloan
   Last Modified: 20 September 1990
   Purpose: Take a Complex image file from stdin, perform the reverse Fourier 
            transform on it, output Intensity image file to stdout. 
   Assumptions: Height and width of input image must be powers of 2.
            The original image may have been embedded in a gray
            border to make the dimensions powers of 2.  If so,
            the original dimensions are noted in the header, and we
            restore the original size here.

            As an option (-n) we will normalize the image so that the DC
            value is MaxValue/2, and the range of values compressed (but 
            not expanded) to fit in [0, MaxValue].  This is a useful option
            when images have been filtered in the spectral domain.

            In any event, values are clipped to [0, MaxValue].
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>
#include <Complex.h>

extern char **malloc_2d(); 

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s\n",
           RoutineName);
 }

static void
Pass(fdIn,fdOut,Normalize)
 FILE *fdIn, *fdOut;
 int Normalize;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right;
  int oBottom, oLeft, oTop, oRight, oWidth, oHeight;
  int wWidth, wHeight;
  char Name[NameLength], Value[ValueLength], WhatBands[NameLength];
  char MaxMagString[ValueLength];
  int x,y,n,BitsPerBand,MaxValue;
  double MaxMag,LogMax;
  unsigned short Pixel[2];
  Complex **ComplexArray;
  int passed = 0;
  int NotReal;
  double MinReal, MaxReal, MinImaginary, MaxImaginary, LoRange, HiRange;
  double Scale, DC;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }
  if (FAILURE == OpenFB(&FBout))          { (void)CloseFB(&FBin);  return; }
 
  GetColorSystem(FBin,WhatBands,&BitsPerBand);
  if (0 != strcmp(WhatBands,"mA"))
   {
    fprintf(stderr,"Input file type is not mA!\n");
    (void)CloseFB(&FBin);
    (void)CloseFB(&FBout);
    return;
   }
  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
  wHeight = Top-Bottom+1;
  wWidth  = Right-Left+1;

  oBottom = Bottom, oLeft = Left; oTop = Top; oRight = Right; 
  /*
    Copy over existing NV pairs
    watch for:
      "X-PassedBy"        --  add to it, or add it
      "X-Original-Bottom" --  note these, and delete them
      "X-Original-Top"
      "X-Original-Left"
      "X-Original-Right"
   */

  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      SetDescriptor(FBout, Name, Value);
      passed = 1;
     }
    else if (0 == strcmp(Name,"X-Original-Bottom")) oBottom = atoi(Value);
    else if (0 == strcmp(Name,"X-Original-Top"))    oTop    = atoi(Value);
    else if (0 == strcmp(Name,"X-Original-Left"))   oLeft   = atoi(Value);
    else if (0 == strcmp(Name,"X-Original-Right"))  oRight  = atoi(Value);
    else SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }
  SetBounds(FBout, oBottom, oLeft, oTop, oRight);

  /* Allocate memory for local array of complex numbers. */
  ComplexArray = (Complex **) malloc_2d(wHeight, wWidth,
                                         sizeof (Complex) );
  if (ComplexArray == (Complex **)NULL) 
   {
    fprintf (stderr,"Insufficient memory for transform!/n");
    (void)CloseFB(&FBin);
    return;
   }

  /* Set output file for I image and set up local variables. */ 
  SetColorSystem(FBout,"I",BitsPerBand);
  MaxValue = (1 << BitsPerBand) - 1;
  if (FAILURE == GetDescriptor (FBin,"X-MaxMag",MaxMagString))
   {
    fprintf(stderr,"Input file missing scaling information (X-MaxMag).\n");
    (void)CloseFB(&FBin);
    (void)CloseFB(&FBout);
    return;
   }
  MaxMag = atof (MaxMagString); 
  LogMax = log (MaxMag + 2.0);

  /* Load input input image into complex array. */
  {
   int HalfWidth, HalfHeight, cy, cx;
  
   HalfWidth = wWidth / 2;
   HalfHeight = wHeight / 2;

   for (cy=0, y=HalfHeight;cy<wHeight;cy++,y++)
    {
     if (y == wHeight) y = 0;
     for (cx=0, x=HalfWidth;cx<wWidth; cx++, x++)
      {
       double LogMagnitude, Magnitude, Angle;

       if (x == wWidth) x = 0;
       if (FAILURE == NextPixelIn(FBin,Pixel))
        { (void)CloseFB(&FBin); return; }
       LogMagnitude = ((double)Pixel[0]*LogMax/(double)MaxValue) - 2.0;
       if (1.0 >= 1.0+LogMagnitude)
        Magnitude = 0.0;
       else
        Magnitude = sqrt(exp(LogMagnitude));
       Angle = (double)Pixel[1]*2.0*M_PI/(double)MaxValue;
       ComplexArray[y][x].real=Magnitude*cos(Angle);
       ComplexArray[y][x].imag=Magnitude*sin(Angle);
      } 
    }
  }

  DC = ComplexArray[0][0].real / (wWidth*wHeight);

  /* Perform Inverse Fast Fourier Transform. */
  FFT_2d (0, wWidth, wHeight, ComplexArray[0]);
 
  /*
     calculate statistics
   */

  MaxImaginary = ComplexArray[0][0].imag;  MinImaginary = MaxImaginary;
  MaxReal      = ComplexArray[0][0].real;  MinReal      = MaxReal;
  for(y=0;y<wHeight;y++)
   for(x=0;x<wWidth;x++)
    {
     register double R, I;
     R = ComplexArray[y][x].real;
     I = ComplexArray[y][x].imag;
     if      (R < MinReal) MinReal = R;
     else if (MaxReal < R) MaxReal = R;
     if      (I < MinImaginary) MinImaginary = I;
     else if (MaxImaginary < I) MaxImaginary = I;
    }

  /*
    prepare to translate DC to 0.5 and scale so that both Min and Max fit
   */ 

  Scale = 1.0/MaxValue;
  HiRange = MaxReal-DC; LoRange = DC-MinReal;
  if ( HiRange < LoRange )
   {
    if (MaxValue < (2.0*LoRange)) Scale = 0.5/LoRange;
   }
  else
   {
    if (MaxValue < (2.0*HiRange)) Scale = 0.5/HiRange;
   }
  
  /* Start the output stream. */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  {
   int xBorder, yBorder;
   double Real,Imaginary, Tangent;

   xBorder = (wWidth  - (oRight-oLeft+1)) / 2;
   yBorder = (wHeight - (oTop-oBottom+1)) / 2;

   for (y=yBorder;y<yBorder+(oTop-oBottom+1);y++)
    {
     for (x=xBorder;x<xBorder+(oRight-oLeft+1);x++)
      {
       Real      = ComplexArray[y][x].real;
       Imaginary = ComplexArray[y][x].imag;
       if (Normalize)
        Real = (((Real-DC)*Scale) + 0.5) * MaxValue;
       if (Real<0.0) Real = 0.0; else if (MaxValue<Real) Real = MaxValue;
       Pixel[0] = Real;
       if (FAILURE == NextPixelOut(FBout,Pixel))
        { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
      }
     wffFlush(FBout);
    }
  }
  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);
  fprintf(stderr,"%s: Real Range was      [%f, %f]; DC = %f\n",
          RoutineName, MinReal, MaxReal, DC);
  fprintf(stderr,"%s: Imaginary Range was [%f, %f]\n",
          RoutineName, MinImaginary, MaxImaginary);
  if (Normalize)
   fprintf(stderr,"%s: DC value was moved to %f and values scaled by %f\n",
            RoutineName, 0.5*MaxValue, Scale*MaxValue);
  fprintf(stderr,"%s: values were clipped to [%d, %d]\n",
            RoutineName, 0, MaxValue);
 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int Normalize = 0;
  RoutineName = argv[ArgsParsed++];

  if (ArgsParsed < argc)
   if (  (argv[ArgsParsed][0] == '-')
       &&(argv[ArgsParsed][1] == 'n') )
    {
     Normalize = 1;
     ArgsParsed++;
    }
   else  { usage(); exit (-1); }	
  if (ArgsParsed < argc)  { usage(); exit (-1); }	

  Pass(stdin,stdout);

  exit (0);
}

